#include <string>
#include <sstream>
#include <iostream>

#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbvector.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbdouble.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbchar.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/RefCountedPtr.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbiterator.hpp"


namespace Luban
{
  using std::string;
  using std::istringstream;
  
  LBDEFINE(Luban::LBString, 1, 0 )


  static bool convertIndex(int &index, int sz)
  {
    int i = index<0?(sz+index):index;
    if ( i<0 || i >= sz )
	  return false;
    index = i;
    return true;
  } 


    //static
  LBString* LBString::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() == 0 )
      return new LBString();
    
    switch ( args->numArgs() ) {
    case 1:
      {
	const LBObject* arg0 = args->getArg(0);
	// string can be constructed from everything
	if ( arg0 != 0 )
	  return new LBString(arg0->toString());
	return 0;
	break;
      }
    default:
      ;
    }
    
    return 0;
  }

  LBString::LBString() 
    : _imp( defaultGuts() )
  {
  }

  LBString::LBString(const string& str) 
    : _imp(new LBStringImp(str) )
  {
  }

  LBString::LBString(const char* cstr) 
    : _imp(new LBStringImp(cstr) )
  {
  }

  string LBString::toString() const
  {
    return _imp->_realstring;
  }
  
  ostream& LBString::toStream(ostream& o) const
  {     return o << _imp->_realstring.size()<<' '<<_imp->_realstring ;  }

  istream& LBString::fromStream(istream& i, int major, int minor)
  {
    string s;
    int length=0;
    i>>length;
    if ( i && i.get() ==' ' ) // skip a space
      for( int j = 0; i && j < length; j++ ) s += i.get();
    else
      throw LBException("Corrupted stream, error recovering LBString from stream");
    if (i )
      {
	unicopy(_imp);
	_imp->_realstring =s;
      }
    else
      throw LBException("Corrupted stream, error recovering LBString from stream");
    return i;
  }

  LBDEFAULT_EQUALS_FUNC(Luban::LBString)

  bool LBString::operator==(const LBString& x) const
  {      return _imp == x._imp || _imp->_realstring == x._imp->_realstring ; }

  int LBString::hash() const
  {    return HashFunctions::stringHash(_imp->_realstring); }

  // here to handle the string in human readable format, not streamed out binary
  bool LBString::cast(LBObject *target) const
  {
    if ( target == 0 )
      return false;
    LBString* sp = dynamic_cast<LBString*>(target);
    if ( sp != 0 )
      {
	*sp = *this;
	return true;
      }
    LBDouble* dp = dynamic_cast<LBDouble*>(target);
    istringstream ist(_imp->_realstring);
    if ( dp )
      {
	double x;
	if ( ist >> x )
	  *dp = LBDouble(x);
	else
	  return false;
	return true;
      }
    LBInt* ip = dynamic_cast<LBInt*>(target);
    if ( ip )
      {
	int x;
	if ( ist >> x )
	  *ip = LBInt(x);
	else
	  return false;
	return true;
      }

    return false;
  }

  LBString& LBString::add(const LBObject& lbo)
  {
    const LBString *p = dynamic_cast<const LBString*>(&lbo);
    if ( p )
      {
	unicopy(_imp);
	_imp->_realstring += p->_imp->_realstring;
	return *this;
      }
    const LBChar *p2 = dynamic_cast<const LBChar*>(&lbo);
    if ( p2 )
      {
	unicopy(_imp);
	_imp->_realstring += (char)*p2;
	return *this;
      }

    throw LBException(string("Can not add string with ")+lbo.getType().toString());
  }

  LBString& LBString::mul(const LBObject& lbo)
  {
    const LBInt *p = dynamic_cast<const LBInt*>(&lbo);
    if ( p )
      {
	int x=(int)*p;
	if ( x == 1 )
	  return *this;
	unicopy(_imp);
	if ( x == 0 )
	  _imp->_realstring.clear();
	else
	  if ( x > 0 )
	    {
	      string temp = _imp->_realstring;
	      while(--x)
		_imp->_realstring += temp;
	      return *this;
	    }
	  else
	    throw LBException(string("Can not mulitply string with negative integer ")+p->toString());
      }

    throw LBException(string("Can not multiply LBString with ")+lbo.getType().toString());
  }

  bool LBString::before(const LBObject& lbo) const
  {
    const LBString *p = dynamic_cast<const LBString*>(&lbo);
    if ( p )
      return _imp->_realstring < p->_imp->_realstring;

    throw LBException(string("Can not compare LBString with ")+lbo.getType().toString());
  }

  LBString& LBString::neg()
  {
    int l = _imp->_realstring.size();
    if ( l > 1 )
      {
	unicopy(_imp);
	char temp;
	for( int i = 0 ; i < l/2; i++ )
	  { 
	    temp = _imp->_realstring[i];
	    _imp->_realstring[i]=_imp->_realstring[l-i-1] ;
	    _imp->_realstring[l-i-1]=temp;
	  }
      }
    return *this;
  }


  char& LBString::operator[](int index)
  {
    int idx = index;
    if ( convertIndex(idx, size()) )
      {
	unicopy(_imp);
	return _imp->_realstring[index];
      }
    throw LBException("Index out of range for string subscripting");
  }

  const char& LBString::operator[](int index) const
  {
    int idx = index;
    if ( convertIndex(idx, size()) )
      {
	// to avoid the gcc compiler warning, use a dummy to clarify intention
	const char& dummy = _imp->_realstring[index];
	return dummy;
      }
    throw LBException("Index out of range for string subscripting");
  }

  int LBString::size() const
  { return _imp->_realstring.size(); }
  const string& LBString::str() const
  { return _imp->_realstring; }
  const char* LBString::c_str() const
  {return _imp->_realstring.c_str(); }
  void LBString::clear()
  {  _imp = defaultGuts(); }

  LBString* LBString::subString(int i1, int i2) const
  {
    int idx1=i1, idx2=i2, s=size();
    if ( convertIndex(idx1, s) && convertIndex(idx2, s) && idx2>=idx1 )
      return new LBString(_imp->_realstring.substr(idx1, idx2-idx1+1));

    throw LBException("Invalid substring indexing");
  }

  void LBString::subscriptSet(const LBObject* index, const LBObject* value)
  {
    if ( ! index || ! value )
      throw LBException("Null value or index for string element set function");
    const LBInt *intp = dynamic_cast<const LBInt*>(index);
    if ( intp == 0 )
      throw LBException(index->getType().toString()+" can not be used to index to a string. Use int");
    int i = int(*intp);
    if ( i < 0 || i > size() )
      throw LBException(string("Invalid string index, too big or negative: ")+intp->toString());

    const LBChar *chp = dynamic_cast<const LBChar*>(value);
    if ( chp )
      {
	_imp->_realstring[i]= char(*chp);
	return;
      }

    const LBString *strp = dynamic_cast<const LBString*>(value);
    if ( strp && strp->size() == 1 )
      {
	_imp->_realstring[i] = (*strp)[0];
	return;
      }

    throw LBException(value->getType().toString()+" can not be used to set as element of a string");
  }

  const LBObject* LBString::subscriptConstGet(const LBObject* index) const
  {
    if ( ! index )
      throw LBException("Null index for string element get function");
    const LBInt *intp = dynamic_cast<const LBInt*>(index);
    if ( intp == 0 )
      throw LBException(index->getType().toString()+" can not be used to index to a string. Use int");
    int i = int(*intp);
    if ( ! convertIndex(i, size()) )
      throw LBException("Invalid string index: "+intp->toString());

    // we need a dirty trick to get around the constness of this function
    LBString *thisptr = const_cast<LBString*>(this);

    thisptr->_indexedchar = LBChar(_imp->_realstring[i]);

    return static_cast<const LBObject*>(&_indexedchar);
  }

  class LBStringIterator : public LBConstIterator
  {
  public:
    LBStringIterator(const LBString& s) : _lbstr(s), _index(-1), _end(s.size()), _currentchar('\0'), _mid(0), _output() 
    {
      _mid = &_currentchar;
      _output = LBSimpleVarArgs(&_mid, 1);
    }

    bool next() 
    { 
      if ( ++_index >= _end ) 
	return false; 
      _currentchar = LBChar(_lbstr[_index]);
      return true; 
    }
    const LBVarArgs* getCurrentRow()
    {
      if ( _index >= _end || _index < 0 )
	return 0;
      return &_output;
    }

  private:
    const LBString& _lbstr;
    int _index, _end;
    LBChar _currentchar;
    const LBObject *_mid;
    LBSimpleVarArgs _output;
  };

  LBConstIterator* LBString::getIterator() const
  {
    return new LBStringIterator(*this);
  }
  
  bool LBString::contains(const LBObject* x) const
  {
    if ( ! x )
      return false;
    const LBChar *xp=dynamic_cast<const LBChar*>(x);
    if ( xp )
      return _imp->_realstring.find(char(*xp)) != string::npos;
    const LBString *xsp=dynamic_cast<const LBString*>(x);
    if ( xsp )
      return _imp->_realstring.find(xsp->_imp->_realstring) != string::npos;
    return false;
  }

  LBVector* LBString::split() const
  {
    // use white space as delimeter
    LBVector *result = new LBVector();
    int sz=size();
    if ( sz == 0 )
      {
	result->push(new LBString());
	return result;
      }

    int i=0, j=0;
    for(; i<sz; i=j)
      {
	for( j=i; j<sz && ! isspace(_imp->_realstring[j]); j++);
	if ( i==j )
	  result->push(new LBString());
	else
	  result->push(subString(i,j-1));
	while ( ++j<sz && isspace(_imp->_realstring[j]) );
	if ( j == sz )
	  {
	    result->push(new LBString());
	    return result;
	  }

      }

    return result;

  }


  LBVector* LBString::split(char dlmt) const
  {
    LBVector *result = new LBVector();
    int sz=size();
    if ( sz == 0 )
      {
	result->push(new LBString());
	return result;
      }

    int i=0, j=0;
    for(; i<sz; i=j)
      {
	for( j=i; j<sz && _imp->_realstring[j] != dlmt; j++);
	if ( i==j )
	  result->push(new LBString());
	else
	  result->push(subString(i,j-1));
	j++;
	if ( j == sz )
	  {
	    result->push(new LBString());
	    return result;
	  }
	
      }

    return result;

  }


  LBVector* LBString::split(const string& dlmt) const
  {
    LBVector *result = new LBVector();
    int sz=size();
    int dsz=dlmt.size();
    if ( dsz == 0 )
      throw LBException("Can not use zero size string as delimeter to split string");
    if ( sz == 0 )
      {
	result->push(new LBString());
	return result;
      }

    int i=0, j=0;
    for(; i<sz; i=j)
      {
	j= _imp->_realstring.find( dlmt,i);
	if ( j == string::npos )
	  {
	    result->push(subString(i));
	    return result;
	  }
	else
	  if ( i==j )
	    result->push(new LBString());
	  else
	    result->push(subString(i,j-1));
	j += dsz;
	if ( j == sz )
	  {
	    result->push(new LBString());
	    return result;
	  }
      }

    return result;

  }

  int LBString::find(const string& sub, int startpos) const
  {
    int subsz = sub.size();
    int sz = size();

    if ( subsz == 0 )
      return -1;
    if ( subsz > sz )
      return -1;
    int s = startpos;
    if ( !convertIndex(s,sz) )
      throw LBException("Invalid start index for find function");

    int p = _imp->_realstring.find(sub,s);

    if ( p == string::npos )
      return -1;

    return p;

  }

  int LBString::find(char sub, int startpos) const
  {
    int sz = size();
    if ( sz == 0 )
      return -1;
    int s = startpos;
    if ( !convertIndex(s,sz) )
      throw LBException("Invalid start index for find function");

    int p = _imp->_realstring.find(sub,s);

    if ( p == string::npos )
      return -1;

    return p;

  }

  int LBString::lower()
  {
    int sz = size();
    if ( sz == 0 )
      return 0;
    unicopy(_imp);
    int conv=0;
    for(int i=0; i<size(); i++)
      if ( isupper(_imp->_realstring[i] ) )
      {
	conv++;
	_imp->_realstring[i]= tolower(_imp->_realstring[i]);
      }

    return conv;

  }


  int LBString::upper()
  {
    int sz = size();
    if ( sz == 0 )
      return 0;
    unicopy(_imp);
    int conv=0;
    for(int i=0; i<size(); i++)
      if ( islower(_imp->_realstring[i] ) )
      {
	conv++;
	_imp->_realstring[i]= toupper(_imp->_realstring[i]);
      }

    return conv;

  }

  int LBString::trim(bool back)
  {
    int sz = size();
    if ( sz == 0 )
      return 0;
    if ( back )
      {
	if ( !isspace(_imp->_realstring[sz-1] ) )
	  return 0;

	unicopy(_imp);

	int conv=0;
	sz = _imp->_realstring.size();
	while( sz > 0 && isspace(_imp->_realstring[sz-1]) )
	  {
	    _imp->_realstring.erase(sz-1,1);
	    sz = _imp->_realstring.size();
	    conv++;
	  }
	return conv;
      }

    // if trim front
    if ( !isspace(_imp->_realstring[0] ) )
      return 0;

    unicopy(_imp);

    int conv=0;
    while( _imp->_realstring.size() > 0 && isspace(_imp->_realstring[0]) )
      {
	_imp->_realstring.erase(0,1);
	conv++;
      }

    return conv;

  }


  void LBString::remove(int pos)
  {
    int sz = size();
    int p = pos;
    if ( !convertIndex(p,sz) )
      throw LBException("Invalid string index");

    unicopy(_imp);

    _imp->_realstring.erase(p,1);

  }

  int LBString::remove(int spos, int epos)
  {
    int sz = size();
    int p1 = spos, p2=epos;
    if ( !convertIndex(p1,sz) || !convertIndex(p2,sz) || p1>p2 )
      throw LBException("Invalid string index");

    unicopy(_imp);

    int num = p2-p1+1;
    _imp->_realstring.erase(p1,num);

    return num;

  }

  int LBString::remove(char c, int spos)
  {
    int sz = size();
    if ( sz == 0 )
      return 0;
    int p = spos;
    if ( !convertIndex(p, sz) )
      throw LBException("Invalid string index");

    unicopy(_imp);

    int removed=0;
    string::iterator it = _imp->_realstring.begin()+p; 
    while ( it != _imp->_realstring.end() )
      {
	if ( *it == c )
	  {
	    it = _imp->_realstring.erase(it);
	    removed++;
	    continue;
	  }
	it++;
      }

    return removed;

  }


  int LBString::remove(const string& str, int spos)
  {
    int sz = size();
    int ssz = str.size();
    if ( sz == 0 || ssz == 0 || ssz > sz )
      return 0;
    int p = spos;
    if ( !convertIndex(p, sz) )
      throw LBException("Invalid string index");

    if ( p+ssz > sz )
      return 0;

    int idx = _imp->_realstring.find(str, p);
    if ( idx == string::npos )
      return 0;

    unicopy(_imp);

    int removed=0;
    while ( idx != string::npos )
      {
	_imp->_realstring.erase(idx, ssz);
	removed++;
	idx = _imp->_realstring.find(str, idx);
      }

    return removed;

  }

  int LBString::replace(const string& oldstr, const string& newstr, int spos)
  {
    int sz = size();
    int ssz = oldstr.size();
    if ( ssz == 0 )
      throw LBException("Can not replace zero size string");
    if ( sz == 0 || ssz > sz )
      return 0;
    int p = spos;
    if ( !convertIndex(p, sz) )
      throw LBException("Invalid string index");

    if ( p+ssz > sz )
      return 0;

    int idx = _imp->_realstring.find(oldstr, p);
    if ( idx == string::npos )
      return 0;

    unicopy(_imp);

    int replaced=0;
    int nsz = newstr.size();
    while ( idx != string::npos )
      {
	_imp->_realstring.erase(idx, ssz);
	_imp->_realstring.insert(idx, newstr);
	replaced++;
	idx = _imp->_realstring.find(oldstr, idx+nsz);
      }

    return replaced;

  }

  void LBString::insert(int pos, char c)
  {
    int sz = size();
    if ( pos == -1 || pos == sz )
      {
	unicopy(_imp);
	_imp->_realstring.append(1,c);
	return;
      }
    int p = pos;
    if ( !convertIndex(p,sz) )
      throw LBException("Invalid string index");

    unicopy(_imp);

    if ( pos >= 0 )
      _imp->_realstring.insert(p,1,c);
    else
      _imp->_realstring.insert(p+1,1,c);

  }


  void LBString::insert(int pos, const string&  s)
  {
    int sz = size();
    if ( pos == -1 || pos == sz )
      {
	unicopy(_imp);
	_imp->_realstring.append(s);
	return;
      }
    int p = pos;
    if ( !convertIndex(p,sz) )
      throw LBException("Invalid string index");

    unicopy(_imp);

    if ( pos >= 0 )
      _imp->_realstring.insert(p,s);
    else
      _imp->_realstring.insert(p+1,s);

  }


  // following are the member functions exported to Luban
  // function exported to Luban need to have the same signature
  // and basically act as adapter to pass the call to the "real" member functions
  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_subString, "substr", "string string::substr(int start, int end=endofstring)" ); 
  LBObject* LBString::luban_subString(const LBVarArgs *args)
  {
    int argc=0;
    LBInt i1, i2;
    LBObject *pad[]={ &i1, &i2 };
    if ( args && ( argc = args->numArgs()) )
      {
	if ( argc == 2 ) 
	  {
	    int n = args->assignArgsToLocal(2,pad);
	    if ( n == 2 )
	      return subString((int)i1, (int)i2);
	    throw LBException("Invalid args passed to string::substring(int start, int end<optional>)");
	  }
	if ( argc == 1 )
	  {
	    int n = args->assignArgsToLocal(1, pad);
	    if ( n==1 )
	      return subString((int)i1);
	    throw LBException("Invalid args passed to string::substring(int start, int end<optional>)");
	  }
      }
    throw LBException("Invalid args passed to string::substring(int start, int end<optional>)");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_split, "split", "vector string::split(optional delimeter = whitespace)" ); 
  LBObject* LBString::luban_split(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    int sz = size();
    if ( argc == 0 )
      {
	return split();
      }

    if ( argc == 1 )
      {
	LBChar d;
	if ( args->assignArgToLocal(0,d) )
	  return split(char(d));
	LBString dstr;
	if ( args->assignArgToLocal(0,dstr) )
	  return split(dstr.str());
	throw LBException("Invalid arg type for string::split(), char or string expected");
      }
	    
    throw LBException("Invalid number of args passed to string::split()");

  }


  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_find, "find", "int/null find(string/char substr, optional int startpos = 0)" ); 
  LBObject* LBString::luban_find(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;

    if ( argc == 1 )
      {
	LBString sub;
	if ( args->assignArgToLocal(0,sub) )
	  {
	    int i = find(sub.str());
	    if ( i == -1 )
	      return 0;
	    return new LBInt(i);
	  }
	LBChar subch;
	if ( args->assignArgToLocal(0,subch) )
	  {
	    int i = find(char(subch));
	    if ( i == -1 )
	      return 0;
	    return new LBInt(i);
	  }
	throw LBException("Invalid arg type for string::find(), char or string expected");
      }

    if ( argc == 2 )
      {
	LBInt stpos;
	if ( ! args->assignArgToLocal(1,stpos) )
	  throw LBException("The second argument of string::find function is expected to be int type");
	LBString sub;
	if ( args->assignArgToLocal(0,sub) )
	  {
	    int i = find(sub.str(), int(stpos));
	    if ( i == -1 )
	      return 0;
	    return new LBInt(i);
	  }
	LBChar subch;
	if ( args->assignArgToLocal(0,subch) )
	  {
	    int i = find(char(subch), int(stpos));
	    if ( i == -1 )
	      return 0;
	    return new LBInt(i);
	  }
	throw LBException("Invalid arg type for string::find(), char or string expected");
      }
	    
    throw LBException("Invalid number of args passed to string::find()");

  }


  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_lower, "lower", "int string::lower()" ); 
  LBObject* LBString::luban_lower(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc == 0 )
      return new LBInt(lower());
    throw LBException("Invalid number of args passed to string::lower()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_upper, "upper", "int string::upper()" ); 
  LBObject* LBString::luban_upper(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc == 0 )
      return new LBInt(upper());
    throw LBException("Invalid number of args passed to string::upper()");
  }


  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_trimfront, "trimfront", "int string::trimfront()" ); 
  LBObject* LBString::luban_trimfront(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc == 0 )
      return new LBInt(trim(false));
    throw LBException("Invalid number of args passed to string::trimfront()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_trimback, "trimback", "int string::trimback()" ); 
  LBObject* LBString::luban_trimback(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc == 0 )
      return new LBInt(trim());
    throw LBException("Invalid number of args passed to string::trimback()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_remove, "remove", "null remove(int index), int remove(int start, int end), int remove(string str, optional int startpos=0) or int remove( char c, optional startpos=0)" ); 
  LBObject* LBString::luban_remove(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    switch ( argc )
      {
      case 1:
	{
	  LBInt idx;
	  if ( args->assignArgToLocal(0, idx) )
	    {
	      remove(int(idx));
	      return 0;
	    }
	  LBChar c;
	  if ( args->assignArgToLocal(0, c) )
	    return new LBInt(remove(char(c)));
	  LBString str;
	  if ( args->assignArgToLocal(0, str) )
	    return new LBInt(remove(str.str()));
	  throw LBException("Invalid type of arg passed to string::remove(), expecting int, char or string");
	}
      case 2:
	{
	  LBInt idx1, idx2;
	  if ( args->assignArgToLocal(0, idx1) && args->assignArgToLocal(1, idx2) )
	    return new LBInt(remove(int(idx1), int(idx2)));
	  LBString str;
	  if ( args->assignArgToLocal(0, str) && args->assignArgToLocal(1, idx1) )
	    return new LBInt(remove(str.str(), int(idx1)));
	  LBChar c;
	  if ( args->assignArgToLocal(0, c) && args->assignArgToLocal(1, idx1) )
	    return new LBInt(remove(char(c), int(idx1)));
	  throw LBException("Invalid type of args passed to string::remove(), expecting (int,int), (string,int) or (char,int)");
	}
      }

    throw LBException("Invalid number of args passed to string::remove(), expecting one or two args");

  }

  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_insert, "insert", "null insert(int index, string/char s)" ); 
  LBObject* LBString::luban_insert(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc ==  2 )
      {
	LBInt idx;
	if ( ! args->assignArgToLocal(0, idx) ) 
	  throw LBException("First arg of wrong type for string::insert(), expecting int");
	LBChar c;
	if ( args->assignArgToLocal(1, c) )
	  {
	    insert(int(idx), char(c));
	    return 0;
	  }
	LBString str;
	if ( args->assignArgToLocal(1, str) )
	  {
	    insert(int(idx), str.str());
	    return 0;
	  }
	throw LBException("Second arg of wrong type for string::insert(), expecting string or char");
      }

    throw LBException("Invalid number of args passed to string::insert(), expecting two args");

  }


  LBEXPORT_MEMBER_FUNC(Luban::LBString, luban_replace, "replace", "int replace(char/string old, char/string new, optional int startpos=0)");
  LBObject* LBString::luban_replace(const LBVarArgs *args)
  {
    int argc=args? args->numArgs():0;
    if ( argc == 2 || argc == 3 )
      {
	const LBObject *arg0 = args->getArg(0);
	if ( ! arg0 )
	  throw LBException("First arg is null for string::replace()");
	string os;
	const LBChar *c = dynamic_cast<const LBChar*>(arg0);
	if ( c )
	  os += char(*c);
	else
	  {
	    const LBString *s = dynamic_cast<const LBString*>(arg0);
	    if ( s )
	      os += s->str();
	    else
	      throw LBException("First arg is of wrong type for string::replace(), expecting string/char");
	  }

	const LBObject *arg1 = args->getArg(1);
	if ( ! arg1 )
	  throw LBException("Second arg is null for string::replace()");
	string ns;
	c = dynamic_cast<const LBChar*>(arg1);
	if ( c )
	  ns += char(*c);
	else
	  {
	    const LBString *s = dynamic_cast<const LBString*>(arg1);
	    if ( s )
	      ns += s->str();
	    else
	      throw LBException("Second arg is of wrong type for string::replace(), expecting string/char");
	  }

	if ( argc == 2 )
	  return new LBInt(replace(os,ns));
      
	LBInt pos;
	if ( args->assignArgToLocal(2,pos) )
	  return new LBInt(replace(os,ns,int(pos)));

	throw LBException("Third arg is of wrong type for string::replace(), expecting int");

      }

    throw LBException("Invalid number of args passed to string::replace(), expecting two or three args");

  }


}

