#include <string>
#include <deque>

#include "lbtypes/lbtypespecmultitype.hpp"
#include "lbtypes/lbtypespecrange.hpp"
#include "lbtypes/lbtypespecbasic.hpp"
#include "lbtypes/lbtypespecenum.hpp"
#include "lbtypes/lbtypespectypetype.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbdouble.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbchar.hpp"
#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbbool.hpp"
#include "lbtypes/lbmap.hpp"
#include "lbtypes/lbvector.hpp"
#include "lbtypes/lbset.hpp"
#include "lbtypes/lberror.hpp"


#include "luban/luban_codegen.hpp"
#include "luban/lbvmcode.hpp"
#include "luban/luban_parsetree.hpp"
#include "luban/luban_namespace.hpp"
#include "luban/lbpropertyinfo.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/lbstationarystruct.hpp"
#include "luban/lbabstractstruct.hpp"
#include "luban/lbcompositionstruct.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/lbtypespecstruct.hpp"

namespace Luban
{
  using std::string;

  static LBFullSymbol parseFullName( Tree parsetree )
  {
    LBFullSymbol sym;
    Tree t = parsetree;
    for( ; t && t->nodeType() == TreeNode::FULLNAME; t = t->child(0) )
      {
	const LBSymbol* namesym = t->child(1)->symbol();
	if ( namesym == 0 )
	  return LBFullSymbol(); // error
	sym.insertFront(*namesym);
      }
    const LBSymbol *startname=t->symbol();
    sym.insertFront(*startname);
    return sym;
  }

  static LBFullSymbol parseTypeName(Tree parsetree, const LBFullSymbol& homens)
  {
    if ( parsetree == 0 )
      return LBFullSymbol();
    Tree namenode = parsetree->child(0);
    if ( ! namenode || parsetree->nodeType() != TreeNode::TYPENAME )
      return LBFullSymbol();
    if ( namenode->nodeType() == TreeNode::NAMENODE )
      {
	LBFullSymbol result(homens);
	result.append(*namenode->symbol());
	return result;
      }
    return parseFullName(namenode);
  }


  static bool parseNSSpec(Tree fullnametree, LBFullSymbol& homens, string& errs)
  {
    Tree nametree = fullnametree->child(0);
    homens = parseFullName(nametree);
    // verify the validity of the name space name
    if ( LBNSUtil::findOrCreateNS(homens, errs) == 0 )
	return false;
    return true;
  }

  static LBPropertyInfo::RWPermission rwpermissionFromNodeType( TreeNode::NodeType nt)
  {
    if ( nt == TreeNode::READONLYPMS ) return LBPropertyInfo::READONLY;
    if ( nt == TreeNode::HIDDENPMS ) return LBPropertyInfo::HIDDEN;
    if ( nt == TreeNode::READWRITEPMS ) return LBPropertyInfo::READWRITE;
    return LBPropertyInfo::INVALIDRWP;
  }

  static LBPropertyInfo::ExecAttr execAttrFromNodeType( TreeNode::NodeType nt)
  {
    switch ( nt )
      {
      case TreeNode::INMODE:
	return LBPropertyInfo::IN;
	break;
      case TreeNode::OUTMODE:
        return  LBPropertyInfo::OUT;
	break;
      case TreeNode::STOREMODE:
	return  LBPropertyInfo::STORE;
	break;
      case TreeNode::STATICMODE:
	return  LBPropertyInfo::STATIC;
	break;
      }
    return LBPropertyInfo::INVALIDATTR;
  }


  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::processAUnit( Tree parsetree, LBProcessStruct* &torun, string &errs)
  {
    if ( parsetree->nodeType() != TreeNode::ONESOURCEUNIT )
      {
	errs +="Invalid parse tree, root node is not type ONESOURCEUNIT";
	return FAILURE;
      }

    switch ( parsetree->numChildren() )
      {
      case 1: // process add hoc process statement sequence
	{
	  Tree statements = parsetree->child(0);

	  if ( statements == 0 )
	    {
	      errs +="Empty statment list for adhoc code sequence";
	      return FAILURE;
	    }

	  LBProcessStruct *ps = new LBProcessStruct();
	  LBFullSymbol dummyhomens;
	  LBVMCode::Operand staticlastop;
	  if ( generateStatementsCode(statements, *ps, staticlastop, dummyhomens, errs) == SUCCESS )
	    {
	      torun = ps;
	      if ( torun->metadata()->code().size() == 0 )
		torun->metadata()->setStaticLastOp(staticlastop);
	      return SUCCESS;
	    }
	  delete ps;
	  return FAILURE;
	  break;
	}
      case 2:
	{
	  Tree nsspec = parsetree->child(0);
	  Tree componentdefs = parsetree->child(1);
	  if ( nsspec == 0 )
	    {
	      errs  = "No namespace defined for components";
	      return FAILURE;
	    }
	  LBFullSymbol nsp;
	  if ( ! parseNSSpec(nsspec, nsp, errs) )
	    {
	      errs +="Invalid namespace specification";
	      return FAILURE;
	    }
	  return processComponents(componentdefs, nsp, errs);
	  break;
	}

      default:
	;
      }

    errs +="Invalid top level parse tree, number of children is not 1 or 2"; 
    return FAILURE;

  }
   

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::processComponents( Tree parsetree, const LBFullSymbol& homens, string& errs)
  {
    // Here we only save the parse tree for each symbol
    // further processing only happens when requested by LVM or DSR ( dynamic symbol resolver)
    if ( parsetree->nodeType() != TreeNode::COMPONENTDEFLIST || parsetree->numChildren() != 2 )
      {
	errs +="Invlid parse tree, expecting tree node of type component definition list";
	return FAILURE;
      }
    

    CodeGenStatus allst = SUCCESS;
    bool success = true;
    for (Tree pt = parsetree; pt != 0 && pt->numChildren() == 2; pt = pt->child(0) )
      {
	Tree ptright = pt->pruneChild(1);
	if ( ptright == 0 )
	  {
	    errs +="Invalid parse tree, expecting struct or type definition node";
	    return FAILURE;
	  }

	
	const LBSymbol *name =0;
	switch ( ptright->nodeType() )
	  {
	  case TreeNode::ABSTRACTSTRUCT:
	    name = ptright->child(1)->symbol();
	    break;
	  case TreeNode::STATIONARYSTRUCTDECLARE:
	    name = ptright->child(0)->symbol();
	    break;
	  case TreeNode::PROCESSSTRUCT:
	    name = ptright->child(1)->symbol();
	    break;
	  case TreeNode::COMPOSITIONSTRUCT:
	    name = ptright->child(1)->symbol();
	    break;
	  case TreeNode::TYPENAMEDEF:
	      name = ptright->child(1)->symbol();
	      break;
	    ;
	  default:
	    errs += "Unrecognized node type as struct definition";
	    return FAILURE;
	  }

	LBFullSymbol fullname(homens);
	fullname.append(*name);
	if ( ! LubanSymbolResolver::addSymbol(fullname, ptright, errs) )
	  {
	    errs += "Failed to add symbol "+fullname.toString()+" into name space\n";
	    pt->linkChild(1, ptright); // link the tree back so its memory can be managed by root
	    success =false;
	  }

      }

    // one option is not to do instantiation here, do it at run time
    // Argument is that it is probably good to check the validity earlier
//     for(int i = 0; i < compcache.size(); i++)
//       if ( ! LubanSymbolResolver::instantiateSymbol(compcache[i], errs) )
// 	{
// 	  errs += "Failed to compile symbol "+compcache[i].toString()+"\n";
// 	  success = false;
// 	}

    if ( ! success )
      return FAILURE;

    return SUCCESS;

  }

  //static
  LBStruct* LubanCodeGenerator::generateStructDef( Tree parsetree, LBStructInterface* itfc, const LBFullSymbol& homens, string &errs )
  {
    switch ( parsetree->nodeType() )
      {
      case TreeNode::ABSTRACTSTRUCT:
	return new LBAbstractStruct(itfc);
      case TreeNode::STATIONARYSTRUCTDECLARE:
	return new LBStationaryStruct(itfc);
      case TreeNode::PROCESSSTRUCT:
	return generateProcessStructDef(parsetree, itfc, homens, errs);
      case TreeNode::COMPOSITIONSTRUCT:
	return generateCompositionStructDef(parsetree, itfc, homens, errs);
      default:
	errs += "Unrecognized node type as struct definition";
      }

    return 0;
  }
    
  LBTypeSpec* LubanCodeGenerator::parseTypeDef( Tree ptree, const LBFullSymbol& homens, string& errs )
  {
    if ( ! ptree || ptree->nodeType() != TreeNode::TYPENAMEDEF )
      {
	errs += "Invalid parse tree for typedef";
	return 0;
      }
    return parseTypeExpression( ptree->child(0), homens, errs);
  }

  LBTypeSpec* LubanCodeGenerator::parseTypeExpression( Tree ptree, const LBFullSymbol& homens, string& errs )
  {
    TreeNode::NodeType nt = ptree->nodeType();
    switch ( nt )
      {
      case TreeNode::MULTITYPE:
      {
	LBTypeSpecMultiType *mt = new LBTypeSpecMultiType();
	for( Tree chain = ptree->child(0); chain; chain = chain->child(0) )
	  {
	    Tree onetype = chain->child(1);
	    LBTypeSpec *onet  = parseTypeExpression( onetype, homens, errs);
	    if ( onet == 0 )
	      {
		errs += "Failed to instantiate multi type spec because the sub type failed to instantiate\n";
		delete mt;
		return 0;
	      }
	    mt->addType(onet);
	  }
	return mt;
      }

      case TreeNode::TYPENAME:
      {
	LBFullSymbol tname = parseTypeName( ptree, homens);
	const LBTypeSpec* ts = LubanSymbolResolver::resolveTypeSymbol(tname, errs);
	if ( ts )
	  return static_cast<LBTypeSpec*>(ts->clone());
	return 0;
      }

      case TreeNode::RANGETYPE:
      {
	LBObject *s = ptree->child(0)->literal();
	LBObject *e = ptree->child(1)->literal();
	try {
	  return new LBTypeSpecRange(s,e);
	}
	catch( LBException& exp )
	  {
	    errs += exp.msg()+"\n";
	    delete s, delete e;
	    return 0;
	  }
      }

      case TreeNode::ENUMTYPE:
      {
	LBTypeSpecEnum *etp = new LBTypeSpecEnum();
	for( Tree olist = ptree->child(0); olist; olist = olist->child(0) )
	  {
	    LBObject *eobj = olist->child(1)->literal();
	    etp->addEnum(eobj);
	  }
	return etp;
      }

      case TreeNode::TYPEINFOTYPE:
	return new LBTypeSpecTypeType();
    
      case TreeNode::DOUBLETYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBDouble) );
    
      case TreeNode::INTTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBInt) );

      case TreeNode::STRINGTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBString));

      case TreeNode::BOOLTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBBool) );

      case TreeNode::CHARTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBChar));

      case TreeNode::VECTORTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBVector));

      case TreeNode::MAPTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBMap));

      case TreeNode::SETTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBSet));

      case TreeNode::ERRORTYPE:
	return new LBTypeSpecBasic(LBTYPEID(LBError));
        
      case TreeNode::STRUCT_DECLARATION:
	{
	  LBStructInterface::ExecMode m = (ptree->child(0)->nodeType()== TreeNode::ASYNCHSTRUCT)?LBStructInterface::SYNCH :LBStructInterface::ASYNCH ;
	  return new LBTypeSpecStruct(m);
	}

      case TreeNode::STATIONARY_STRUCT_DEC:
	{
	  return new LBTypeSpecStruct(LBStructInterface::STATIONARY);
	}

      default:
	;
      }

    errs += "Invalid parse tree, unknown node type in type expression\n";
    return 0;

  }

  static LBFullSymbolVec* typeExpExtSymbols(Tree ptree, const LBFullSymbol& homens, LBFullSymbolVec* svec)
  {
    TreeNode::NodeType nt = ptree->nodeType();
    if ( nt == TreeNode::MULTITYPE )
      {
	LBFullSymbolVec* v = svec;
	for( Tree chain = ptree->child(0); chain; chain = chain->child(0) )
	  {
	    Tree onetype = chain->child(1);
	    v  = typeExpExtSymbols( onetype, homens, v); 
	  }
	return v;
      }
    if ( nt == TreeNode::TYPENAME )
      {
	LBFullSymbolVec* v = svec;
	if ( v == 0 )
	   v = new LBFullSymbolVec();
	LBFullSymbol tname = parseTypeName( ptree, homens);
	v->push_back(tname);
	return v;
      }
    
    return svec;
  }

  LBFullSymbolVec* LubanCodeGenerator::getTypeDefExtSymbols( Tree ptree, const LBFullSymbol& homens )
  {
    return getTypeExpressionExtSymbols(ptree->child(0), homens);
  }

  LBFullSymbolVec* LubanCodeGenerator::getTypeExpressionExtSymbols( Tree ptree, const LBFullSymbol& homens )
  {
    return typeExpExtSymbols(ptree, homens, 0);
  }


  LBProcessStruct* LubanCodeGenerator::generateProcessStructDef(Tree ptree, LBStructInterface* itfc,const LBFullSymbol& homens, string& errs)
  {
    LBProcessStruct *prcstr = new LBProcessStruct(itfc);

    Tree statements = ptree->child(4);
    LBVMCode::Operand staticlitop;
    if ( generateStatementsCode(statements, *prcstr, staticlitop, homens, errs) != SUCCESS)
      {
	delete prcstr;
	return 0;
      }

    if ( prcstr->metadata()->code().size() == 0 ) // no code, constatnt expression
      prcstr->metadata()->setStaticLastOp(staticlitop);

    return prcstr;

  }

  LBCompositionStruct* LubanCodeGenerator::generateCompositionStructDef(Tree ptree, LBStructInterface* ifc,const LBFullSymbol& homens, string& errs)
  {
    LBCompositionStruct* compstr = new LBCompositionStruct(ifc);

    Tree compositions = ptree->child(4);
    if ( generateComposition(compositions, *compstr, homens, errs) != SUCCESS)
      {
	delete compstr;
	return 0;
      }
    
    return compstr;

  }


  LBStructInterface* LubanCodeGenerator::generateStructInterface(Tree parsetree, const LBFullSymbol& homens, string &errs)
  {
    if ( parsetree == 0 )
      return 0;
    Tree parents = 0, prplist = 0; 
    const LBSymbol *name=0;
    LBStructInterface::ExecMode md = LBStructInterface::SYNCH;
    switch ( parsetree->nodeType() )
      {
      case TreeNode::ABSTRACTSTRUCT:
      case TreeNode::PROCESSSTRUCT:
      case TreeNode::COMPOSITIONSTRUCT:
	if ( parsetree->child(0)->nodeType() == TreeNode::ASYNCHSTRUCT )
	  md = LBStructInterface::ASYNCH;
	parents = parsetree->child(2);
	prplist = parsetree->child(3);
	name = parsetree->child(1)->symbol();
	break;
      case TreeNode::STATIONARYSTRUCTDECLARE:
	md = LBStructInterface::STATIONARY;
	name = parsetree->child(0)->symbol();
	parents = parsetree->child(1);
	prplist = parsetree->child(2);
	break;
      default:
	errs += "Unrecognized node type as struct definition";
	return 0;
      }

    LBFullSymbol strname(homens);
    strname.append(*name);
    LBStructInterface *itfc = new LBStructInterface(md, strname);
    if ( parents )
      {
	if ( parents->nodeType() != TreeNode::TYPENAMELIST )
	  {
	    errs += "Invalid parse tree, expecting fullname list node type";
	    return 0;
	  }
	for(Tree t = parents; t; t = t->child(0) )
	  {
	    Tree onename = t->child(1);
	    if ( onename == 0 )
	      {
		errs += "Invalide parse tree, corrupted interface inheritance list";
		return 0;
	      }
	    LBFullSymbol symname = parseTypeName(onename, homens);
	    itfc->addParent(symname);
	  }
      }

    // add explicit properteis from the tree
    if ( prplist && addPropertiesFromTree(prplist, *itfc, homens, errs) != SUCCESS )
      {
	delete itfc;
	return 0;
      }
    
    return itfc;

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::addPropertiesFromTree( Tree ptree, LBStructInterface& itfc, const LBFullSymbol& homens, string& errs)
  {
    if ( ! ptree ) return SUCCESS;

    if ( ptree->nodeType() == TreeNode::STATIONARYSTITEMLIST )
      return addStationaryStructPropertiesFromTree(ptree, itfc, homens, errs);

    if ( ptree->nodeType() != TreeNode::PROPERTYLIST )
      return FAILURE;

    LBPropertyInfo::ExecAttr dfattr = LBPropertyInfo::INVALIDATTR;

    // need to reverse the order of the tree
    std::deque<TreeNode*> prps;
    for( Tree prplist = ptree; prplist; prplist = prplist->child(0) )
      prps.push_front(prplist->child(1));

    for(int i=0; i < prps.size(); i++ )
      {
	Tree oneitfitem = prps[i];
	switch ( oneitfitem->nodeType() )
	  {
	  case TreeNode::EXECATTR_DEC:
	    {
	      Tree attrtree =  oneitfitem->child(0);
	      dfattr = execAttrFromNodeType( attrtree->nodeType() );
	      break;
	    }
	  case TreeNode::ONETYPEPROPERTIES:
	    {
	      Tree attrnode = oneitfitem->child(0);
	      Tree rwpmsnode =  oneitfitem->child(1);
	      Tree pnamesnode = oneitfitem->child(2);
	      
	      LBPropertyInfo::ExecAttr attr = dfattr;
	      if ( attrnode )
		attr = execAttrFromNodeType(attrnode->nodeType());

	      LBPropertyInfo::RWPermission rwpms = LBPropertyInfo::defaultRWPermission(attr);
	      if ( rwpmsnode )
		rwpms = rwpermissionFromNodeType(rwpmsnode->nodeType());
	      if ( ! LBPropertyInfo::consistent(attr, rwpms) )
		{
		  errs += "Inconsistent flow type and access permission";
		  return FAILURE;
		}

	      Tree typetree = pnamesnode->pruneChild(0);
	      LBFullSymbolVec *exts=0;
	      if ( typetree )
		exts = LubanCodeGenerator::getTypeExpressionExtSymbols( typetree, homens );

	      for ( Tree namedvtree = pnamesnode->child(1); namedvtree; namedvtree = namedvtree->child(0))
		{
		  Tree onenamedv = namedvtree->child(1);
		  const LBSymbol *pname =onenamedv->child(0)->symbol();
		  Tree initexp = onenamedv->child(1);
		  LBProcessStruct *initprc = 0;
		  int extsym_initprc = 0;
		  if ( initexp )
		    {
		      if ( LBStructInterface::allowInitProcess(itfc.mode(), attr) )
			{
			  initprc = new LBProcessStruct();
			  LBVMCode::Operand staticlitop;
			  if ( generateCondExpCode( initexp, *initprc, staticlitop, homens, errs ) != SUCCESS )
			    {
			      errs += "Failed to generate initialization code for property "+pname->toString()+"\n";
			      delete initprc;
			      return FAILURE;
			    }

			  if ( initprc->metadata()->code().size() == 0 ) // no code, constatnt expression
			    initprc->metadata()->setStaticLastOp(staticlitop);
			  extsym_initprc = initprc->numOfExternalSymbols();
			}
		      else
			{
			  errs += "Initialization expression for property "+pname->toString()+" is not allowed\n";
			  return FAILURE;
			}
		    }

		  Tree typetreecopy = typetree? typetree->clone():0;
		  LBFullSymbolVec *extscopy = exts? new LBFullSymbolVec(*exts):0;
		  if ( extsym_initprc != 0 ) // add all symbol for init process to the ext symbol for the property
		    {
		      if ( ! extscopy )
			extscopy = new LBFullSymbolVec();
		      for( int i=0; i<extsym_initprc; ++i)
			extscopy->push_back(*initprc->externalSymbol(i));
		    }
		  LBPropertyInfo *pinfo = new LBPropertyInfo( *pname, attr, rwpms, typetreecopy, extscopy, homens, initprc);
		  if ( ! itfc.addProperty(pinfo, errs) )
		    {
		      errs += "Failed to add property: "+pname->toString()+"\n";
		      delete pinfo;
		      if ( typetree ) delete typetree;
		      if ( exts ) delete exts;
		      return FAILURE;
		    }
		}

	      if ( typetree ) delete typetree;
	      if ( exts ) delete exts;
	      break;
	    }

	  default:
	    errs += "Invalid parse tree, unrecognized node type in struct property list\n";
	    return FAILURE;

	  }
      }

    return SUCCESS;

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::addStationaryStructPropertiesFromTree( Tree ptree, LBStructInterface& itfc, const LBFullSymbol& homens, string& errs)
  {
    if ( ptree->nodeType() != TreeNode::STATIONARYSTITEMLIST )
      return FAILURE;

    LBPropertyInfo::ExecAttr dfattr = LBPropertyInfo::STORE;

    for( Tree t = ptree; t != 0; t=t->child(0) )
      {
	Tree oneitfitem = t->child(1);
	switch ( oneitfitem->nodeType() )
	  {
	  case TreeNode::STATIONARYSTPROPERTIES:
	  case TreeNode::STATIONARYSTPROPERTIES_ST:
	    {
	      Tree rwpmsnode =  oneitfitem->child(0);
	      Tree pnamesnode = oneitfitem->child(1);

	      LBPropertyInfo::ExecAttr attr = LBPropertyInfo::STATIC;
	      if ( oneitfitem->nodeType() == TreeNode::STATIONARYSTPROPERTIES )
		attr = LBPropertyInfo::STORE;

	      LBPropertyInfo::RWPermission rwpms = LBPropertyInfo::READWRITE;
	      if ( rwpmsnode )
		rwpms = rwpermissionFromNodeType(rwpmsnode->nodeType());
	      if ( ! LBPropertyInfo::consistent(attr, rwpms) )
		{
		  errs += "Inconsistent in/out execution attribute and read/write permission";
		  return FAILURE;
		}

	      Tree typetree = pnamesnode->pruneChild(0);
	      LBFullSymbolVec *exts=0;
	      if ( typetree )
		exts = LubanCodeGenerator::getTypeExpressionExtSymbols( typetree, homens );
	      int extsym_initprc = 0;
	      for ( Tree namedvtree = pnamesnode->child(1); namedvtree; namedvtree = namedvtree->child(0))
		{
		  Tree onenamedv = namedvtree->child(1);
		  try {
		    const LBSymbol *pname =onenamedv->child(0)->symbol();
		    Tree initexp = onenamedv->child(1);
		    LBProcessStruct *initprc = 0;

		    if ( initexp != 0 )
		      {
			LBVMCode::Operand staticop;
			initprc = new LBProcessStruct();
			if ( generateCondExpCode( initexp, *initprc, staticop, homens, errs ) != SUCCESS )
			{
			  errs += "Failed to generate initialization code for property "+pname->toString()+"\n";
			  delete initprc;
			  return FAILURE;
			}

			if ( initprc->metadata()->code().size() == 0 ) // no code, constatnt expression
			  initprc->metadata()->setStaticLastOp(staticop);
			extsym_initprc = initprc->numOfExternalSymbols();

		      }

		    Tree typetreecopy = typetree? typetree->clone():0;
		    LBFullSymbolVec *extscopy = exts? new LBFullSymbolVec(*exts):0;
		    if ( extsym_initprc != 0 ) // add all symbol for init process to the ext symbol for the property
		      {
			if ( ! extscopy )
			  extscopy = new LBFullSymbolVec();
			for( int i=0; i<extsym_initprc; ++i)
			  extscopy->push_back(*initprc->externalSymbol(i));
		      }

		    LBPropertyInfo *pinfo = new LBPropertyInfo( *pname, attr, rwpms, typetreecopy, extscopy, homens, initprc);
		    
		    if ( !itfc.addProperty(pinfo, errs) )
		      {
			errs += "Failed to add property "+pname->toString()+"\n";
			delete initprc;
			if ( typetree ) delete typetree;
			if ( exts ) delete exts;
			return FAILURE;
		      }
		  }
		  catch ( LBException e )
		    {
		      errs += e.msg();
		      if ( typetree ) delete typetree;
		      if ( exts ) delete exts;
		      return FAILURE;
		    }
		}


	      if ( typetree ) delete typetree;
	      if ( exts ) delete exts;


	      break;
	    }

	  default:
	    errs += "Invalid parse tree, unrecognized static struct property node type\n";
	    return FAILURE;
	  }
      }

    return SUCCESS;

  }
  
  // is being done
  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateComposition(Tree comptree, LBCompositionStruct &comps, const LBFullSymbol& homens, string& errs)
  {
    // is being done
    if ( !comptree || comptree->nodeType() != TreeNode::COMPCELLLIST )
      {
	errs += "Invalid parse tree, expecting composition cell list";
	return FAILURE;
      }

    std::deque<TreeNode*> adhocs; // for named ad hoc cells defined

    std::deque<TreeNode*> typed; // this is for statically typed struct cell, to process its named args
    std::deque<int> typedids; // this is for statically typed struct cell, to process its named args

    // first go through all named components and add to the composition struct for future reference
    for( TreeNode *cells = comptree ; cells != 0; cells = cells->child(0) )
      {
	TreeNode *onecell = cells->child(1);
	TreeNode::NodeType celltype = onecell->nodeType();
	if ( celltype != TreeNode::CELL && celltype != TreeNode::ASYNCHCELL )  // ignore the output terminal for now
	  {
	    if ( celltype == TreeNode::COMPCOLLECTOUTPUT )
	      {
		if ( onecell->numChildren() == 3 )
		  {
		    LBFullSymbol structname=parseTypeName(onecell->child(0), homens);
		    bool threading = onecell->child(2)->nodeType() != TreeNode::SEQUENTIAL;
		    int cellid = comps.metadata()->newNonameTypedCell(structname, threading, errs);
		    if ( cellid < 0 )
		      {
			errs += "Can not add cell of type "+structname.toString()+" for output\n";
			return FAILURE;
		      }
		    if ( ! comps.metadata()->linkAllOutputs(cellid, errs) )
		      {
			errs += "Can not link all outputs of cell "+structname.toString()+" to output properties\n";
			return FAILURE;
		      }
		    
		    TreeNode *namedargs = onecell->child(1);
		    typed.push_front(namedargs);
		    typedids.push_front(cellid);
		  }
		else
		  adhocs.push_front(onecell);
	      }
	    else
	      adhocs.push_front(onecell);
	    continue;
	  }
	const LBSymbol *cellname = onecell->child(0)->symbol();
	TreeNode *celldef = onecell->child(1);
	bool threading = celldef->child(1)->nodeType() != TreeNode::SEQUENTIAL;
	bool asynch = celltype == TreeNode::ASYNCHCELL;
	if ( celldef->nodeType() == TreeNode::EXPCELL )
	  {
	    TreeNode *exp = celldef->child(0);
	    if ( exp->nodeType() == TreeNode::STRUCTTYPEEVAL )  // this is a statically typed struct cell
	      {
		LBFullSymbol structtypename = parseTypeName(exp->child(0), homens);
		int cellid = comps.metadata()->newNamedTypedCell(*cellname, structtypename, threading, errs);
		if (  cellid < 0 )
		  {
		    errs += "Can not add cell "+cellname->toString()+"\n";
		    return FAILURE;
		  }
		typed.push_front( exp->child(1) );
		typedids.push_front( cellid );
		continue; // the typed struct cell does not need to be compiled
	      }
	  }

	if ( comps.metadata()->newNamedAdhocCell(*cellname, asynch, threading, errs) == -1  )
	  {
	    errs += "Can not add cell "+cellname->toString()+"\n";
	    return FAILURE;
	  }

	adhocs.push_front(onecell);

      }

    // now compile the named adhoc cells first, and output cells
    for( int i=0; i<adhocs.size(); i++ )
      {
	TreeNode *onecell = adhocs[i];
	TreeNode::NodeType celltype = onecell->nodeType();

	switch ( celltype )
	  {
	  case TreeNode::CELL:
	  case  TreeNode::ASYNCHCELL:
	    {
	      const LBSymbol *cellname = onecell->child(0)->symbol();
	      bool asynch = celltype == TreeNode::ASYNCHCELL;

	      LBProcessStruct *proc = new LBProcessStruct( *cellname, asynch, &comps );

	      TreeNode* celldef = onecell->child(1);
	      if ( celldef->nodeType() == TreeNode::EXPCELL )
		{
		  TreeNode *exp = celldef->child(0);
		  LBVMCode::Operand expresultop;
		  if ( generateCondExpCode(exp, *proc, expresultop, homens, errs) != SUCCESS )
		    {
		      delete proc;
		      errs += "Failed to generate code for internal cell "+cellname->toString()+"\n";
		      return FAILURE;
		    }
		  // add an assign to output here
		  LBVMCode asn(LBVMCode::ASN);
		  asn.op1() = proc->metadata()->getAdhocOutputOperand();
		  asn.op2() = expresultop;
		  proc->metadata()->code().append(asn);
		}
	      else
		{
		  // this is for adhoc compound statement cell
		  TreeNode *compoundstatement = celldef->child(0);
		  LBVMCode::Operand dummy;
		  if ( generateStatementCode(compoundstatement, *proc, dummy, homens, errs) != SUCCESS )
		    {
		      delete proc;
		      errs += "Failed to generate code for internal compound statement cell "+cellname->toString()+"\n";
		      return FAILURE;
		    }
		}

	      comps.metadata()->setNamedAdhocCell(*cellname, proc);
	      continue;
	    }
	  
	    // this is for output property linking
	  case TreeNode::COMPOUTPUT:
	    {
	      const LBSymbol *outpname = onecell->child(0)->symbol();

	      // check the output property against the interface definition
	      int pindex;
	      const LBPropertyInfo *pinfo = comps.interface().propertyInfo(*outpname, pindex);
	      if (  !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
		{
		  errs += "Invalid output property "+outpname->toString()+"\n";
		  return FAILURE;
		}
	      TreeNode *exp = onecell->child(1)->child(0);
	      bool threading = onecell->child(1)->child(1)->nodeType() != TreeNode::SEQUENTIAL;
	      int cellid = comps.metadata()->newNonameAdhocCell(threading);

	      if ( ! comps.metadata()->link(cellid, int(LBCompositionStruct::ComponentList::OUTPUTS), *outpname, pindex, errs) )
		{
		  errs += "Failed to link cell to output property "+outpname->toString()+"\n";
		  return FAILURE;
		}

	      LBProcessStruct *proc = new LBProcessStruct( cellid, &comps );
	      LBVMCode::Operand resultop;
	      if ( generateCondExpCode(exp, *proc, resultop, homens, errs) != SUCCESS )
		{
		  delete proc;
		  errs += "Failed to generate code for output property "+outpname->toString()+"\n";
		  return FAILURE;
		}
	      // add an assign to output here
	      LBVMCode asn(LBVMCode::ASN);
	      asn.op1() = proc->metadata()->getAdhocOutputOperand();
	      asn.op2() = resultop;
	      proc->metadata()->code().append(asn);

	      comps.metadata()->setNonameAdhocCell(cellid, proc);
	      continue;
	    }

	  case TreeNode::COMPCOLLECTOUTPUT:
	    {
	      const LBSymbol *cellname = onecell->child(0)->symbol();
	      int cellid = comps.metadata()->getCellId(*cellname);
	      if ( cellid < 0 )
		{
		  errs += "Component "+cellname->toString()+"does not exist\n";
		  return FAILURE;
		}
	      if ( ! comps.metadata()->linkAllOutputs(cellid, errs) )
		{
		  errs += "Can not link all outputs of cell "+cellname->toString()+" to output properties\n";
		  return FAILURE;
		}	      
	      continue;
	    }
	  default:
	    errs += "Unexpected node type in adhoc struct compile list\n";
	    return FAILURE;
	  }

      }


    // now compile the named and typed struct cells, just process its named arg list
    for(int i=0; i<typed.size(); i++)
      {
	TreeNode *namedargs = typed[i];
	if ( !namedargs )
	  continue; // ignore empty arg list
	int cellid = typedids[i];
	if ( processNamedArgsForTypedCell(cellid, comps, namedargs, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to process named args for cell "+comps.metadata()->getCellName(cellid).toString()+"\n";
	    return FAILURE;
	  }
      }

    return SUCCESS;
  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateStatementsCode(Tree statements, LBProcessStruct &ps, LBVMCode::Operand& litop, const LBFullSymbol& homens, string& errs)
  {
    if ( ! statements  )
      return SUCCESS;
    if ( statements->nodeType() != TreeNode::PRCSTATEMENTLIST )
      {
	errs += "Invalid parse tree, expecting a statement list type tree node";
	return FAILURE;
      }
    Tree substlist = statements->child(0);
    if ( substlist )
      {
	if ( generateStatementsCode(substlist, ps, litop, homens, errs) != SUCCESS )
	  return FAILURE;
      }

    return generateStatementCode(statements->child(1), ps, litop, homens, errs);

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateStatementCode(Tree onest, LBProcessStruct &ps, LBVMCode::Operand& litop, const LBFullSymbol& homens, string& errs)
  {
    switch (onest->nodeType())
      {
      case TreeNode::NOOP:
	return SUCCESS;
      case TreeNode::DISPATCHABLE:
	{
	  bool dispatch = false;
	  if ( onest->child(1)->nodeType() == TreeNode::DISPATCH )
	    dispatch = true;
	  int dspchpoint = -1;
	  if ( dispatch )
	    {
	      dspchpoint = ps.metadata()->code().size();
	      LBVMCode dispcode(LBVMCode::DISPATCH);
	      dispcode.op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, dspchpoint+1);
	      ps.metadata()->code().append(dispcode);

	      ps.metadata()->startDispatch(); 

	    }
	  
	  Tree dcontent = onest->child(0);
	  LBVMCode::Operand resultop;
	  if ( dcontent->nodeType() == TreeNode::PRPBATCHSET )
	    {
	      if ( generatePropertyBatchSetCode(dcontent, ps, resultop, homens, errs ) != SUCCESS )
		return FAILURE;
	    }
	  else
	    if ( generateExpressionCode(dcontent, ps, resultop, homens, errs ) != SUCCESS )
	      return FAILURE;

	  litop = resultop;

	  if ( dispatch )
	    {
	      // backptach the dispatch code
	      (ps.metadata()->code())[dspchpoint].op2() = LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size()-1);
	      LBVMCode::Operand rootop;
	      ps.metadata()->findRootOperand(resultop, rootop);
	      (ps.metadata()->code())[dspchpoint].resultop() = rootop;

	      ps.metadata()->endDispatch(); 

	    }

	  return SUCCESS;
	}

      case TreeNode::TYPED_VAR_INIT:
	{
	  Tree typeexp = onest->child(0);

	  // here we have to reverse the node order first
	  std::deque<TreeNode*> vars;
	  for( Tree varinitlist = onest->child(1); varinitlist; varinitlist = varinitlist->child(0) )
	    vars.push_front(varinitlist->child(1));

	  for( int i=0; i<vars.size(); i++ )
	    {
	      Tree onevarinit = vars[i];
	      const LBSymbol *varname = onevarinit->child(0)->symbol();
	      if ( varname == 0 )
		{
		  errs += "Invalid parse tree, variable name empty";
		  return FAILURE;
		}

	      LBVMCode::Operand varop;
	      if ( !ps.metadata()->addLocalVariable(*varname, typeexp, homens, varop, errs) )
		{
		  errs += "Failed to create local variable "+varname->toString()+"\n";
		  return FAILURE;
		}

	      Tree initexp = onevarinit->child(1);
	      if ( initexp )
		{
		  LBVMCode::Operand resultop;
		  if ( generateExpressionCode(initexp, ps, resultop, homens, errs) != SUCCESS )
		    return FAILURE;
		  LBVMCode initcode(LBVMCode::ASN);
		  initcode.op1()=varop;
		  initcode.op2()=resultop;
		  ps.metadata()->code().append(initcode);
		}

	    }

	  return SUCCESS;

	}

      case TreeNode::IFST:
	{
	  Tree boolexp = onest->child(0);
	  Tree statements = onest->child(1);

	  LBVMCode::Operand expresult;
	  if ( generateCondExpCode(boolexp, ps, expresult, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse logical expression for if statement\n";
	      return FAILURE;
	    }

	  LBVMCode zerojump(LBVMCode::ZEROJUMP);
	  zerojump.op1()=expresult;
	  ps.metadata()->code().append(zerojump);
	  int zindex = ps.metadata()->code().size()-1;

	  LBVMCode::Operand dummy;
	  if ( generateStatementCode(statements, ps, dummy, homens, errs) != SUCCESS )
	    return FAILURE;
	  
	  LBVMCode::Operand jumpto(LBVMCode::Operand::LABEL, ps.metadata()->code().size());
	  (ps.metadata()->code())[zindex].op2()=jumpto;

	  return SUCCESS;
	}

      case TreeNode::IFELSEST:
	{
	  Tree boolexp = onest->child(0);
	  Tree statementsif = onest->child(1);
	  Tree statementselse = onest->child(2);

	  LBVMCode::Operand expresult;
	  if ( generateCondExpCode(boolexp, ps, expresult, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse logical expression for ifelse statement\n";
	      return FAILURE;
	    }

	  LBVMCode zerojump(LBVMCode::ZEROJUMP);
	  zerojump.op1()=expresult;
	  int zindex = ps.metadata()->code().size();
	  ps.metadata()->code().append(zerojump);

	  LBVMCode::Operand dummyop;
	  if ( generateStatementCode(statementsif, ps, dummyop, homens, errs) != SUCCESS )
	    return FAILURE;
	  
	  LBVMCode gotostmt(LBVMCode::GOTO);
	  int gindex = ps.metadata()->code().size();
	  ps.metadata()->code().append(gotostmt);

	  // back patch the conditiona jump
	  LBVMCode::Operand jumpto(LBVMCode::Operand::LABEL, ps.metadata()->code().size());
	  (ps.metadata()->code())[zindex].op2()=jumpto;

	  if ( generateStatementCode(statementselse, ps, dummyop, homens, errs) != SUCCESS )
	    return FAILURE;

	  // back patch the goto st
	  (ps.metadata()->code())[gindex].op1()=LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size());;

	  return SUCCESS;
		  
	}

      case TreeNode::WHILEST:
	{
	  Tree boolexp = onest->child(0);
	  Tree statements = onest->child(1);

	  int startindex = ps.metadata()->code().size();

	  ps.metadata()->openLoop();

	  LBVMCode::Operand expresult;
	  if ( generateCondExpCode(boolexp, ps, expresult, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse logical expression for while statement\n";
	      return FAILURE;
	    }

	  LBVMCode zerojump(LBVMCode::ZEROJUMP);
	  zerojump.op1()=expresult;
	  ps.metadata()->code().append(zerojump);
	  int zindex = ps.metadata()->code().size()-1;

	  LBVMCode::Operand dummyop;
	  if ( generateStatementCode(statements, ps, dummyop, homens, errs) != SUCCESS )
	    return FAILURE;

	  LBVMCode gotostmt(LBVMCode::GOTO);
	  gotostmt.op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, startindex);
	  ps.metadata()->code().append(gotostmt);

	  LBVMCode::Operand endofloop(LBVMCode::Operand::LABEL, ps.metadata()->code().size());
	  (ps.metadata()->code())[zindex].op2()=endofloop;
	  
	  ps.metadata()->closeLoop(startindex, endofloop.opindex());

	  return SUCCESS;

	}
	
      case TreeNode::FORST:
	{
	  Tree initexplist = onest->child(0);
	  Tree stopexp = onest->child(1);
	  Tree endexplist = onest->child(2);
	  Tree statements = onest->child(3);
	  
	  // reverse the node order first
	  std::deque<TreeNode*> exps;
	  for( ; initexplist; initexplist = initexplist->child(0) )
	    exps.push_front(initexplist->child(1));

	  for(int i=0 ; i<exps.size(); i++ )
	    {
	      Tree oneinitexp = exps[i];
	      LBVMCode::Operand dummy;
	      if ( generateExpressionCode(oneinitexp, ps, dummy, homens, errs) != SUCCESS )
		{
		  errs += "Failed to parse initializing expression list for FOR statement\n";
		  return FAILURE;
		}
	    }

	  int loopbackpoint = ps.metadata()->code().size();

	  ps.metadata()->openLoop();

	  int zindex = -1;
	  if ( stopexp )
	    {
	      LBVMCode::Operand stopexpresult;
	      if ( generateCondExpCode(stopexp, ps, stopexpresult, homens, errs) != SUCCESS )
		{
		  errs += "Failed to parse stop condition expression for FOR statement\n";
		  return FAILURE;
		}

	      LBVMCode zerojump(LBVMCode::ZEROJUMP);
	      zerojump.op1()=stopexpresult;
	      ps.metadata()->code().append(zerojump);
	      zindex = ps.metadata()->code().size()-1;
	    }

	  LBVMCode::Operand dummyop;
	  if ( generateStatementCode( statements, ps, dummyop, homens, errs ) != SUCCESS )
	    {
	      errs += "Failed to parse the body of FOR statement\n";
	      return FAILURE;
	    }

	  int contindex = ps.metadata()->code().size();

	  // reverse the node order first
	  exps.clear();
	  for( ; endexplist; endexplist = endexplist->child(0) )
	    exps.push_front(endexplist->child(1));

	  for(int i=0 ; i<exps.size(); i++ )
	    {
	      Tree oneendexp = exps[i];
	      LBVMCode::Operand dummy;
	      if ( generateExpressionCode(oneendexp, ps, dummy, homens, errs) != SUCCESS )
		{
		  errs += "Failed to parse initializing expression list for FOR statement\n";
		  return FAILURE;
		}
	    }

	  LBVMCode gotostmt(LBVMCode::GOTO);
	  gotostmt.op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, loopbackpoint);
	  ps.metadata()->code().append(gotostmt);

	  // populate the jump to lable of conditional jump statment
	  int endofloop = ps.metadata()->code().size();
	  if ( zindex >= 0 )
	    (ps.metadata()->code())[zindex].op2()=LBVMCode::Operand(LBVMCode::Operand::LABEL, endofloop);
	  
	  ps.metadata()->closeLoop(contindex, endofloop);

	  return SUCCESS;

	}

      case TreeNode::FOREACHST:
	{
	  Tree varnamelist = onest->child(0);
	  Tree exp = onest->child(1);
	  Tree statements = onest->child(2);
	  
	  LBVMCode::Operand containerop;
	  if ( generateExpressionCode(exp, ps, containerop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the container expression in FOREACH statement\n";
	      return FAILURE;
	    }


	  LBVMCode getiterator(LBVMCode::GETITER);
	  getiterator.op1()=containerop;
	  LBVMCode::Operand iterop = ps.metadata()->newTempIter();
	  getiterator.resultop() = iterop;
	  ps.metadata()->code().append(getiterator);

	  int loopback = ps.metadata()->code().size(); //the point to loop back

	  ps.metadata()->openLoop();

	  LBVMCode iternext(LBVMCode::ITERNEXT);
	  iternext.op1() = iterop;
	  LBVMCode::Operand iternextresult = ps.metadata()->newTemp();
	  iternext.resultop() = iternextresult;
	  ps.metadata()->code().append(iternext);

	  LBVMCode condjump(LBVMCode::ZEROJUMP);
	  condjump.op1()=iternextresult;
	  int condjumpindex = ps.metadata()->code().size();
	  ps.metadata()->code().append(condjump);
	  
	  // the order of the var names needs to be restored
	  std::deque<TreeNode*> vars;
	  for(;varnamelist; varnamelist = varnamelist->child(0) )
	    vars.push_front(varnamelist->child(1));

	  for(int i=0; i<vars.size(); i++ )
	    {
	      const LBSymbol *varname = vars[i]->symbol();
	      LBVMCode::Operand varop;
	      if ( ! ps.metadata()->getLocalVariable(*varname, varop, errs) )
		{
		  errs += "Failed to get local variable "+varname->toString()+"\n";
		  return FAILURE;
		}
	      LBVMCode iterobj(LBVMCode::ITEROBJ);
	      iterobj.op1()=iterop;
	      iterobj.op2() = LBVMCode::Operand(LBVMCode::Operand::INTVALUE,i);
	      iterobj.resultop()=varop;
	      ps.metadata()->code().append(iterobj);
	    }
	    
	  LBVMCode::Operand dummyop;
	  if ( generateStatementCode(statements, ps, dummyop, homens, errs ) != SUCCESS )
	    {
	      errs += "Failed to generate code for FOREACH statement body\n";
	      return FAILURE;
	    }

	  // add goto loopback
	  LBVMCode gotost(LBVMCode::GOTO);
	  gotost.op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, loopback);
	  ps.metadata()->code().append(gotost);

	  // populate the jump to addres for conditional jump code
	  int endofloop = ps.metadata()->code().size();
	  LBVMCode::Operand endofloopop(LBVMCode::Operand::LABEL,endofloop );
	  (ps.metadata()->code())[condjumpindex].op2()=endofloopop;
	  
	  ps.metadata()->closeLoop(loopback, endofloop);

	  // code to reset the iterator to make it release the temp copy of the container
	  LBVMCode resetiter(LBVMCode::RESETITER);
	  resetiter.op1() = iterop;
	  ps.metadata()->code().append(resetiter);

	  return SUCCESS;

	}

      case TreeNode::BREAKST:
	{
	  LBVMCode gotoendofloop(LBVMCode::GOTO);
	  int gindex = ps.metadata()->code().size();
	  ps.metadata()->code().append(gotoendofloop);
	  if ( ! ps.metadata()->addBreak(gindex) )
	    {
	      errs += "Invalid use of BREAK statement\n";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
	  
      case TreeNode::CONTINUEST:
	{
	  LBVMCode cont(LBVMCode::GOTO);
	  int gindex = ps.metadata()->code().size();
	  ps.metadata()->code().append(cont);
	  if ( ! ps.metadata()->addContinue(gindex) )
	    {
	      errs += "Invalid use of continue statement\n";
	      return FAILURE;
	    }
	  return SUCCESS;
	}

      case TreeNode::FINISHST:
	{
	  LBVMCode stopst(LBVMCode::HALT);
	  ps.metadata()->code().append(stopst);
	  return SUCCESS;
	}

      case TreeNode::WAITST:
	{
	  LBVMCode holdst(LBVMCode::HOLD);
	  ps.metadata()->code().append(holdst);
	  return SUCCESS;
	}

      case TreeNode::CANCELST:
	{
	  LBVMCode cancelst(LBVMCode::CANCEL);
	  ps.metadata()->code().append(cancelst);
	  return SUCCESS;
	}

      case TreeNode::WAITFORST:
	{
	  // the order of the var names needs to be restored
	  std::deque<TreeNode*> vars;
	  for(Tree variablelist = onest->child(0); variablelist; variablelist = variablelist->child(0))
	    vars.push_front(variablelist->child(1));

	  for(int i=0; i<vars.size(); i++ )
	    {
	      TreeNode *vartree = vars[i];
	      LBVMCode::Operand varop;

	      if ( vartree->nodeType() == TreeNode::LOCALVAR )
		{
		  const LBSymbol *simplename = vartree->child(0)->symbol();
		  if ( ! ps.metadata()->getLocalVariable(*simplename, varop, errs) )
		    {
		      errs += "Failed to find local variable "+simplename->toString()+"\n";
		      return FAILURE;
		    }
		}
	      else // this must be a global variable 
		{
		  LBFullSymbol fullname = parseTypeName(vartree->child(0),homens);
		  const LBSymbol *prpname = vartree->child(1)->symbol();
		  if ( ! ps.metadata()->getGlobalVariable(fullname, *prpname, varop) )
		    {
		      errs += "Failed to find global variable "+fullname.toString()+"."+prpname->toString()+"\n";
		      return FAILURE;
		    }
		}

	      LBVMCode holdontost(LBVMCode::HOLDONTO);
	      holdontost.op1() = varop;
	      ps.metadata()->code().append(holdontost);
	    }

	  return SUCCESS;
	    
	}

	
      case TreeNode::PRCCOLLECTOUTPUT:
	{
	  Tree varexp = onest->child(0);
	  LBVMCode::Operand varop;
	  if ( generateExpressionCode(varexp, ps, varop, homens, errs) != SUCCESS )
	    {
	      errs += "Invalid variable expression in output assign statement\n";
	      return FAILURE;
	    }
	  
	  LBVMCode batchout(LBVMCode::BATCHOUT);
	  batchout.op1()=varop;
	  
	  ps.metadata()->code().append(batchout);

	  return SUCCESS;
	}

      case TreeNode::CLOSURE_PRC:
	{
	  // to be done
	  return FAILURE;
	}
	
      case TreeNode::CLOSURE_COMP:
	{
	  // to be done
	  return FAILURE;
	}

      case TreeNode::COMPOUNDST:  // the compound statement
	{
	  return generateStatementsCode(onest->child(0), ps, litop, homens, errs);
	}

      case TreeNode::COMPOUNDSTPAR:  // the compound statement
	{
	  int dspchpoint = ps.metadata()->code().size();
	  LBVMCode dispcode(LBVMCode::DISPATCH);
	  dispcode.op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, dspchpoint+1);
	  ps.metadata()->code().append(dispcode);

	  ps.metadata()->startDispatch(); // set a flag to make all variable operand multi-thread ready

	  if ( generateStatementsCode(onest->child(0), ps, litop, homens, errs) != SUCCESS )
	    return FAILURE;
	  // backptach the dispatch code

	  ps.metadata()->endDispatch(); // set a flag to show all variable operand not-need to be multi-thread  ready

	  (ps.metadata()->code())[dspchpoint].op2() = LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size()-1);
	  return SUCCESS;
	}

      default:
	;

      }

    errs += "Invalid parse tree, unrecognized statement\n";
    return FAILURE;
  
  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generatePropertyBatchSetCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    if ( exptree->nodeType() != TreeNode::PRPBATCHSET )
      {
	errs += "Invalid parse tree, expecting property batch set statement\n";
	return FAILURE;
      }

    Tree thestruct = exptree->child(0);
    Tree namedargs = exptree->child(1);

    LBVMCode::Operand structop;
    if ( generateLValueExpCode(thestruct, ps, structop, homens, errs) != SUCCESS )
      {
	errs += "Failed to parse the atom expression for struct evaluation\n";
	return FAILURE;
      }

    resultop = structop;

    if (namedargs &&  namedargs->numChildren() == 1 )
      {
	LBVMCode vecall(LBVMCode::VECALL);
	vecall.op1() = structop;
	vecall.op2() = LBVMCode::Operand(LBVMCode::Operand::INTVALUE,( namedargs->child(0)->nodeType() == TreeNode::VECOPPAR )?1:0 );
	vecall.resultop() = resultop;
	ps.metadata()->code().append(vecall);
	return SUCCESS;
      }

    LBVMCode::Operand namedargsop;
    if ( namedargs &&  namedargs->numChildren() == 2 && generateNamedArgsCode(namedargs, ps, namedargsop, homens, errs) != SUCCESS )
      {
	errs += "Failed parse named arguments in struct evaluation expression\n";
	return FAILURE;
      }

    LBVMCode batchset(LBVMCode::BATCHSET);
    batchset.op1() = structop;
    batchset.op2() = namedargsop;
    ps.metadata()->code().append(batchset);

    return SUCCESS;
    
  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateExpressionCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    if ( exptree->nodeType() == TreeNode::REGASSIGN )
      {
	Tree varexp = exptree->child(0);
	LBVMCode::Operand varexpop;
	if ( generateLValueExpCode(varexp, ps, varexpop, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse left value expression in ASSIGN statement\n";
	    return FAILURE;
	  }
	Tree assignop = exptree->child(1);
	Tree asnexp = exptree->child(2);
	LBVMCode::Operand expresultop;
	if ( generateExpressionCode( asnexp, ps, expresultop, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse right side expression in ASSIGN statement\n";
	    return FAILURE;
	  }
	    
	LBVMCode asncode;
	switch ( assignop->nodeType() )
	  {
	  case TreeNode::ASSIGNOP:
	    asncode = LBVMCode(LBVMCode::ASN);
	    break;
	  case TreeNode::MULASSIGNOP:
	    asncode = LBVMCode(LBVMCode::MULASN);
	    break;
	  case TreeNode::DIVASSIGNOP:
	    asncode = LBVMCode(LBVMCode::DIVASN);
	    break;
	  case TreeNode::MODASSIGNOP:
	    asncode = LBVMCode(LBVMCode::MODASN);
	    break;
	  case TreeNode::ADDASSIGNOP:
	    asncode = LBVMCode(LBVMCode::ADDASN);
	    break;
	  case TreeNode::SUBASSIGNOP:
	    asncode = LBVMCode(LBVMCode::SUBASN);
	    break;
	  default:
	    errs += "Unrecognized assingment operator\n";
	    return FAILURE;
	  }

	asncode.op1()=varexpop;
	asncode.op2()=expresultop;
	resultop = varexpop;

	ps.metadata()->code().append(asncode);

	return SUCCESS;
      }

    // this is the case for conditional expression as grammar indicates
    return generateCondExpCode(exptree, ps, resultop, homens, errs);

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateCondExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    if ( exptree->nodeType() == TreeNode::CONDEXP )
      {
	Tree boolexp = exptree->child(0);
	Tree exp1 = exptree->child(1);
	Tree exp2 = exptree->child(2);
	
	LBVMCode::Operand boolexpop;
	if ( generateOrExpCode(boolexp, ps, boolexpop, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse the logical expression in conditional expression\n";
	    return FAILURE;
	  }
	
	LBVMCode zerojump(LBVMCode::ZEROJUMP);
	zerojump.op1()=boolexpop;
	int zindex = ps.metadata()->code().size();
	ps.metadata()->code().append(zerojump);

	LBVMCode::Operand exp1op;
	if ( generateExpressionCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse the first expression in conditional expression\n";
	    return FAILURE;
	  }

	// one common temp operand
	LBVMCode::Operand cmnresult = ps.metadata()->newTemp();

	// one code to assing result to the common result
	LBVMCode asncode(LBVMCode::ASN);
	asncode.op1()=cmnresult;
	asncode.op2()=exp1op;
	ps.metadata()->code().append(asncode);

	// one goto instruction
	LBVMCode gotostmt(LBVMCode::GOTO);
	int gindex = ps.metadata()->code().size();
	ps.metadata()->code().append(gotostmt);


	// back patch the zero jump code
	(ps.metadata()->code())[zindex].op2()=LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size());
	
	LBVMCode::Operand exp2op;
	if ( generateExpressionCode(exp2, ps, exp2op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse the first expression in conditional expression\n";
	    return FAILURE;
	  }

	// one code to assing result to the common result
	LBVMCode asncode2(LBVMCode::ASN);
	asncode2.op1()=cmnresult;
	asncode2.op2()=exp2op;
	
	ps.metadata()->code().append(asncode2);

	// back patch the goto instruction
	(ps.metadata()->code())[gindex].op1()=LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size());

	resultop = cmnresult;

	return SUCCESS;
      }

    // this is the case for OrExp in the grammar
    return generateOrExpCode(exptree, ps, resultop,homens, errs);

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateOrExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    if ( exptree->nodeType() == TreeNode::OROP )
      {
	Tree exp1 = exptree->child(0);
	Tree exp2 = exptree->child(1);

	resultop = ps.metadata()->newTemp();

	LBVMCode::Operand exp1op;
	if ( generateOrExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse expression one in OR expression\n";
	    return FAILURE;
	  }

	LBVMCode orjmp(LBVMCode::ORJMP);
	orjmp.op1() = exp1op;
	orjmp.resultop()= resultop;
	int ojpidx = ps.metadata()->code().size();
	ps.metadata()->code().append(orjmp);

	LBVMCode::Operand exp2op;
	if ( generateAndExpCode(exp2, ps, exp2op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse expression two in OR expression\n";
	    return FAILURE;
	  }
	
	LBVMCode orcode(LBVMCode::LGCOR);
	orcode.op1() = exp1op;
	orcode.op2() = exp2op;
	orcode.resultop() = resultop;

	ps.metadata()->code().append(orcode);

	// back patch the ORJMP instruction
	ps.metadata()->code()[ojpidx].op2()=LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size());

	return SUCCESS;

      }

    // this is for and exp
    return generateAndExpCode(exptree, ps, resultop, homens, errs);

  }


  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateAndExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens,  string& errs)
  {
    if ( exptree->nodeType() == TreeNode::ANDOP )
      {
	Tree exp1 = exptree->child(0);
	Tree exp2 = exptree->child(1);

	resultop = ps.metadata()->newTemp();

	LBVMCode::Operand exp1op;
	if ( generateAndExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse expression one in OR expression\n";
	    return FAILURE;
	  }

	LBVMCode andjmp(LBVMCode::ANDJMP);
	andjmp.op1() = exp1op;
	andjmp.resultop()= resultop;
	int ajpidx = 	ps.metadata()->code().size();
	ps.metadata()->code().append(andjmp);

	LBVMCode::Operand exp2op;
	if ( generateNotExpCode(exp2, ps, exp2op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse expression two in AND expression\n";
	    return FAILURE;
	  }
	
	LBVMCode andcode(LBVMCode::LGCAND);
	andcode.op1() = exp1op;
	andcode.op2() = exp2op;
	andcode.resultop() = resultop;

	ps.metadata()->code().append(andcode);

	// back patch the ANDJMP instruction
	ps.metadata()->code()[ajpidx].op2()=LBVMCode::Operand(LBVMCode::Operand::LABEL, ps.metadata()->code().size());

	return SUCCESS;
      }

    return generateNotExpCode(exptree, ps, resultop, homens, errs);

  }

  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateNotExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens,  string& errs)
  {
    if ( exptree->nodeType() == TreeNode::NOTSIGN )
      {
	Tree exp1 = exptree->child(0);

	resultop = ps.metadata()->newTemp();

	LBVMCode::Operand exp1op;
	if ( generateEqualityExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse sub expression one in NOT expression\n";
	    return FAILURE;
	  }

	LBVMCode notcode(LBVMCode::LGCNOT);
	notcode.op1() = exp1op;
	notcode.resultop() = resultop;

	ps.metadata()->code().append(notcode);

	return SUCCESS;
      }

    return generateEqualityExpCode(exptree, ps, resultop, homens, errs);

  }
	
  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateEqualityExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    if ( exptree->nodeType() == TreeNode::EQOP || exptree->nodeType() == TreeNode::NEOP || exptree->nodeType() == TreeNode::ISAOP )
      {
	Tree exp1 = exptree->child(0);
	Tree exp2 = exptree->child(1);
	LBVMCode::Operand exp1op;
	if ( generateEqualityExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse the expression one in equality expression\n";
	    return FAILURE;
	  }
	LBVMCode::Operand exp2op;
	if ( generateRelationalExpCode(exp2, ps, exp2op, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse the expression two in equality expression\n";
	    return FAILURE;
	  }

	resultop=ps.metadata()->newTemp();

	switch ( exptree->nodeType() )
	  {
	  case TreeNode::EQOP:
	    {
	      LBVMCode eqcmp(LBVMCode::EQCMP);
	      eqcmp.op1()=exp1op;
	      eqcmp.op2()=exp2op;
	      eqcmp.resultop()=resultop;

	      ps.metadata()->code().append(eqcmp);

	      break;
	    }
	  case TreeNode::NEOP:
	    {
	      LBVMCode necmp(LBVMCode::NEQCMP);
	      necmp.op1()=exp1op;
	      necmp.op2()=exp2op;
	      necmp.resultop()=resultop;

	      ps.metadata()->code().append(necmp);

	      break;
	    }

	  case TreeNode::ISAOP:
	    {
	      LBVMCode isacheck(LBVMCode::ISA);
	      isacheck.op1()=exp1op;
	      isacheck.op2()=exp2op;
	      isacheck.resultop() = resultop;

	      ps.metadata()->code().append(isacheck);

	      break;

	    }
	    
	  default:
	    errs += "Unknown operator in equality expression\n";
	    return FAILURE;
	    ;
	  }

	return SUCCESS;

      }
    
    return generateRelationalExpCode(exptree, ps, resultop, homens, errs);

  }
  
  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateTypeOfExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    Tree exp = exptree->child(0);
    LBVMCode::Operand expop;
    if ( generateExpressionCode(exp, ps, expop, homens, errs) != SUCCESS )
      {
	errs += "Failed to parse the expression inside TYPEOF operator\n";
	return FAILURE;
      }

    LBVMCode typeofcode(LBVMCode::TYPEOF);
    typeofcode.op1() = expop;
    resultop = ps.metadata()->newTemp();
    typeofcode.resultop()=resultop;

    ps.metadata()->code().append(typeofcode);

    return SUCCESS;
  }

  
  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateRelationalExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    LBVMCode::OPCode opc;
    switch ( exptree->nodeType() )
      {
      case TreeNode::GTOP:
	opc = LBVMCode::GTCMP;
	break;
      case TreeNode::LTOP:
	opc = LBVMCode::LTCMP;
	break;
      case TreeNode::GEOP:
	opc = LBVMCode::GECMP;
	break;
      case TreeNode::LEOP:
	opc = LBVMCode::LECMP;
	break;
      case TreeNode::TYPEOFOP:
	return generateTypeOfExpCode(exptree, ps, resultop, homens, errs);
      case TreeNode::TYPESPEC:
	if ( ps.metadata()->newLiteralTypeSpec( exptree->child(0), homens, resultop, errs ) )
	  return SUCCESS;
	errs += "Failed to parse type expression\n";
	return FAILURE;
      default:
	  return generateArithExpCode(exptree, ps, resultop,homens, errs);
      }


    Tree exp1 = exptree->child(0);
    Tree exp2 = exptree->child(1);

    LBVMCode::Operand exp1op;
    if ( generateRelationalExpCode(exp1, ps, exp1op, homens, errs ) != SUCCESS )
      {
	errs += "Failed to parse the expression one of RELATIONAL expression\n";
	return FAILURE;
      }

    LBVMCode::Operand exp2op;
    if ( generateArithExpCode(exp2, ps, exp2op, homens, errs ) != SUCCESS )
      {
	errs += "Failed to parse the expression two of RELATIONAL expression\n";
	return FAILURE;
      }
	  
    LBVMCode relopcode(opc);
    relopcode.op1()=exp1op;
    relopcode.op2()=exp2op;
    resultop = ps.metadata()->newTemp();
    relopcode.resultop() = resultop;

    ps.metadata()->code().append(relopcode);

    return SUCCESS;

  }

  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateArithExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    TreeNode::NodeType nt = exptree->nodeType();
    LBVMCode::OPCode opc;
    if ( nt == TreeNode::ADD )
      opc = LBVMCode::ADD;
    else
      if ( nt == TreeNode::SUB )
	opc = LBVMCode::MINUS;
      else
	return generateMulDivExpCode(exptree, ps, resultop,homens, errs);

    Tree exp1 = exptree->child(0);
    Tree exp2 = exptree->child(1);    

    LBVMCode::Operand exp1op;
    if ( generateArithExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
      {
	errs += "Failed to parse expression one in arithematic expression\n";
	return FAILURE;
      }
    LBVMCode::Operand exp2op;
    if ( generateMulDivExpCode(exp2, ps, exp2op,homens, errs) != SUCCESS )
      {
	errs += "Failed to parse expression two in arithematic expression\n";
	return FAILURE;
      }

    LBVMCode amcode(opc);
    amcode.op1() = exp1op;
    amcode.op2() = exp2op;
    resultop = ps.metadata()->newTemp();
    amcode.resultop() = resultop;

    ps.metadata()->code().append(amcode);

    return SUCCESS;

  }
  
  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateMulDivExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    TreeNode::NodeType nt = exptree->nodeType();
    LBVMCode::OPCode opc;
    if ( nt == TreeNode::MUL )
      opc = LBVMCode::MUL;
    else
      if ( nt == TreeNode::DIV )
	opc = LBVMCode::DIV;
      else
	if ( nt == TreeNode::MOD )
	  opc = LBVMCode::MOD;
	else
	  return generateUnaryExpCode(exptree, ps, resultop, homens, errs);

    Tree exp1 = exptree->child(0);
    Tree exp2 = exptree->child(1);    

    LBVMCode::Operand exp1op;
    if ( generateMulDivExpCode(exp1, ps, exp1op,homens, errs) != SUCCESS )
      {
	errs += "Failed to parse expression one in arithematic expression\n";
	return FAILURE;
      }
    LBVMCode::Operand exp2op;
    if ( generateUnaryExpCode(exp2, ps, exp2op,homens, errs) != SUCCESS )
      {
	errs += "Failed to parse expression two in arithematic expression\n";
	return FAILURE;
      }

    LBVMCode mdcode(opc);
    mdcode.op1() = exp1op;
    mdcode.op2() = exp2op;
    resultop = ps.metadata()->newTemp();
    mdcode.resultop() = resultop;

    ps.metadata()->code().append(mdcode);

    return SUCCESS;

  }

  LubanCodeGenerator::CodeGenStatus  LubanCodeGenerator::generateUnaryExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    TreeNode::NodeType nt = exptree->nodeType();
    LBVMCode::OPCode opc;
    switch ( nt )
      {
      case TreeNode::PREINC:
	opc = LBVMCode::PP;
	break;
      case TreeNode::PREDEC:
	opc = LBVMCode::MM;
	break;
      case TreeNode::NEGSIGN:
	opc = LBVMCode::NEG;
	break;
      case TreeNode::PLUSSIGN:
	return generateUnaryExpCode(exptree->child(0), ps, resultop,homens, errs);
	break;
      default:
	return generatePostfixExpCode(exptree, ps, resultop,homens, errs);
      }

    Tree exp1 = exptree->child(0);
    LBVMCode::Operand exp1op;
    if ( generateUnaryExpCode(exp1, ps, exp1op,homens, errs) != SUCCESS )
      {
	errs += "Failed parse operand in unary expression\n";
	return FAILURE;
      }

    LBVMCode uncode(opc);
    uncode.op1() = exp1op;
    if ( nt == TreeNode::PREINC || nt == TreeNode::PREDEC )
      resultop = exp1op;
    else
      {
	resultop = ps.metadata()->newTemp();
	uncode.resultop() = resultop;
      }
    ps.metadata()->code().append(uncode);
    
    return SUCCESS;

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generatePostfixExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    TreeNode::NodeType nt = exptree->nodeType();
    LBVMCode::OPCode opc;
    if ( nt == TreeNode::POSTINC )
      opc = LBVMCode::PP;
    else
      if ( nt == TreeNode::POSTDEC )
	opc = LBVMCode::MM;
      else
	return generateRValueExpCode(exptree, ps, resultop, homens, errs);

    Tree exp1 = exptree->child(0);

    LBVMCode::Operand exp1op;
    if ( generatePostfixExpCode(exp1, ps, exp1op, homens, errs) != SUCCESS )
      {
	errs += "Failed to parse operand in postfix expression\n";
	return FAILURE;
      }

    // generate an assign and an single operand operation
    LBVMCode asncode(LBVMCode::ASN);
    asncode.op2() = exp1op;
    resultop = ps.metadata()->newTemp();
    asncode.op1() = resultop;
    ps.metadata()->code().append(asncode);

    LBVMCode pfxcode(opc);
    pfxcode.op1() = exp1op;
    ps.metadata()->code().append(pfxcode);

    return SUCCESS;

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateRValueExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    switch ( exptree->nodeType() )
      {
      case TreeNode::SUBSCRIPT:
	{
	  Tree varexp = exptree->child(0);
	  Tree subexp = exptree->child(1);
	  LBVMCode::Operand varexpop;
	  if ( generateRValueExpCode( varexp, ps, varexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the variable for subscripting expression\n";
	      return FAILURE;
	    }

	  LBVMCode::Operand subexpop;
	  if ( generateExpressionCode(subexp, ps, subexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the subscription value expression\n";
	      return FAILURE;
	    }

	  resultop = ps.metadata()->getSubscriptOperand(varexpop, subexpop); 
	
	  return SUCCESS;

	}

      case TreeNode::MBRFUNC:
	{
	  Tree varexp = exptree->child(0);
	  const LBSymbol *fname = exptree->child(1)->symbol();
	  Tree args = exptree->child(2);

	  LBVMCode::Operand varexpop;
	  if ( generateRValueExpCode(varexp, ps, varexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the left value variable object expression for member function call\n";
	      return FAILURE;
	    }

	  // generate code to init arg op
	  LBVMCode::Operand argop;
	  if ( args && generateAnonymousArgsCode(args, ps, argop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse argument list for object member function call\n";
	      return FAILURE;
	    }

	  LBVMCode::Operand mbrfop = ps.metadata()->getMemberFuncOperand(varexpop, *fname);

	  // reuse the args op to put result obj 
	  LBVMCode mbrfunc(LBVMCode::MBRFUNC);
	  mbrfunc.op1() = mbrfop;
	  mbrfunc.op2() = argop;
	  resultop = ps.metadata()->newTemp();
	  mbrfunc.resultop() = resultop;
	  ps.metadata()->code().append(mbrfunc);
	  

	  return SUCCESS;

	}

      case TreeNode::PROPERTY:
	{
	  Tree varexp = exptree->child(0);
	  const LBSymbol *prptyname = exptree->child(1)->symbol();

	  if ( varexp->nodeType() == TreeNode::LOCALVAR )
	    {
	      const LBSymbol *varname = varexp->child(0)->symbol();
	      if ( ! ps.metadata()->getVarPropertyOperand(*varname, *prptyname, resultop, errs) )
		{
		  errs += "Invalid property expression "+varname->toString()+"."+prptyname->toString()+"\n";
		  return FAILURE;
		}
	      return SUCCESS;
	    }

	  LBVMCode::Operand varexpop;
	  if ( generateRValueExpCode(varexp, ps, varexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the variable for property read expression\n";
	      return FAILURE;
	    }

	  // possible type checking could be done in this call
	  // though may not be necessary, conclusion: not necessary
	  resultop = ps.metadata()->getVarPropertyOperand(varexpop, *prptyname); 
	  
	  return SUCCESS;
	  
	}

      case TreeNode::STRUCTOBJEVAL:
	{
	  Tree thestruct = exptree->child(0);
	  Tree namedargs = exptree->child(1);

	  LBVMCode::Operand structop;
	  if ( generateAtomExpCode(thestruct, ps, structop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the atom expression for struct evaluation\n";
	      return FAILURE;
	    }

	  if (namedargs &&  namedargs->numChildren() == 1 )
	    {
	      LBVMCode vecall(LBVMCode::VECALL);
	      vecall.op1() = structop;
	      vecall.op2() = LBVMCode::Operand(LBVMCode::Operand::INTVALUE,( namedargs->child(0)->nodeType() == TreeNode::VECOPPAR )?1:0 );
	      resultop = ps.metadata()->newTemp();
	      vecall.resultop() = resultop;
	      ps.metadata()->code().append(vecall);
	      return SUCCESS;
	    }

	  LBVMCode::Operand namedargsop;
	  if ( namedargs &&  namedargs->numChildren() == 2 && generateNamedArgsCode(namedargs, ps, namedargsop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed parse named arguments in struct evaluation expression\n";
	      return FAILURE;
	    }

	  LBVMCode namedeval(LBVMCode::STRUCTEVAL);
	  namedeval.op1() = structop;
	  namedeval.op2() = namedargsop;
	  resultop = ps.metadata()->newTemp();
	  namedeval.resultop() = resultop;

	  ps.metadata()->code().append(namedeval);

	  return SUCCESS;


	}

      default:
	;
      }

    return generateAtomExpCode(exptree, ps, resultop, homens, errs);

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateAnonymousArgsCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    LBProcessStruct::LBAnonymousArgs *anonargs = new LBProcessStruct::LBAnonymousArgs();

    //reverse the order first
    std::deque<TreeNode*> exps;
    for(Tree args=exptree; args; args = args->child(0) )
      exps.push_front(args->child(1));

    for(int i=0; i < exps.size(); i++ )
      {
	Tree oneargexp = exps[i];
	LBVMCode::Operand argexpop;
	if ( generateCondExpCode(oneargexp, ps, argexpop, homens, errs) != SUCCESS )
	  {
	    errs += "Failed to parse one argument anonymous argument list\n";
	    return FAILURE;
	  }
	anonargs->pushArg(argexpop);
      }

    resultop = ps.metadata()->newAnonymousArgsOperand(anonargs);

    return SUCCESS;

  }
    

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateAtomExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    switch ( exptree->nodeType() )
      {
      case TreeNode::PARENTHESISEXP:
	{
	  return generateExpressionCode(exptree->child(0), ps, resultop, homens, errs);
	}
      case TreeNode::LITERAL:
	{
	  resultop = ps.metadata()->newLiteral(exptree->literal());
	  return SUCCESS;
	}

	// to do reminder: all the vector, map and set construction could use several new vmcode
	// that only work for vector, map and set. This is more efficient, and it does not assume
	// any exported functions from these types. The C++ header will make sure it work.

      case TreeNode::VECTOR_CONSTRU:
	{
	  LBVMCode::Operand initvec = ps.metadata()->newLiteral( new LBVector() );
	  resultop = ps.metadata()->newTemp();
	  // assign initial empty vector from literal
	  LBVMCode initasn(LBVMCode::ASN);
	  initasn.op1() = resultop;
	  initasn.op2() = initvec;
	  ps.metadata()->code().append(initasn);
	  
	  Tree explist = exptree->child(0);
	  if ( explist )
	    {
	      // the order of the args needs to be restored back from the tree
	      std::deque<TreeNode*> exps;
	      for(; explist; explist = explist->child(0) )
		exps.push_front(explist->child(1));

	      for(int i=0; i<exps.size(); i++ )
		{
		  Tree exp = exps[i];
		  LBVMCode::Operand expop;
		  if ( generateExpressionCode(exp, ps, expop, homens, errs) != SUCCESS )
		    {
		      errs += "Failed to parse element expression in vector construction\n";
		      return FAILURE;
		    }

		  LBVMCode vecins(LBVMCode::VECINS);
		  vecins.resultop() = resultop;
		  vecins.op1() = expop;
		  ps.metadata()->code().append(vecins);
	  
		}
	    }

	  return SUCCESS;

	}

      case TreeNode::MAP_CONSTRU:
	{
	  LBVMCode::Operand initmap = ps.metadata()->newLiteral( new LBMap() );
	  resultop = ps.metadata()->newTemp();
	  // assign initial empty vector from literal
	  LBVMCode initasn(LBVMCode::ASN);
	  initasn.op1() = resultop;
	  initasn.op2() = initmap;
	  ps.metadata()->code().append(initasn);
	  
	  Tree kvlist = exptree->child(0);
	  if ( kvlist )
	    {
	      // the order of the args needs to be restored back from the tree
	      std::deque<TreeNode*> kvs;
	      for(; kvlist; kvlist = kvlist->child(0) )
		kvs.push_front(kvlist->child(1));

	      for(int i=0; i<kvs.size(); i++ )
		{
		  Tree kv = kvs[i];
		  LBVMCode::Operand keyexpop;
		  if ( generateExpressionCode(kv->child(0), ps, keyexpop, homens, errs) != SUCCESS )
		    {
		      errs += "Failed to parse a key expression in map construction\n";
		      return FAILURE;
		    }
		  LBVMCode::Operand valexpop;
		  if ( generateExpressionCode(kv->child(1), ps, valexpop,homens, errs) != SUCCESS )
		    {
		      errs += "Failed to parse a value expression in map construction\n";
		      return FAILURE;
		    }
	      
		  LBVMCode mapins(LBVMCode::MAPINS);
		  mapins.resultop() = resultop;
		  mapins.op1() = keyexpop;
		  mapins.op2() = valexpop;
		  ps.metadata()->code().append(mapins);
	  
		}
	    }

	  return SUCCESS;

	}

      case TreeNode::SET_CONSTRU:
	{
	  LBVMCode::Operand initset = ps.metadata()->newLiteral( new LBSet() );
	  resultop = ps.metadata()->newTemp();
	  // assign initial empty vector from literal
	  LBVMCode initasn(LBVMCode::ASN);
	  initasn.op1() = resultop;
	  initasn.op2() = initset;
	  ps.metadata()->code().append(initasn);

	  Tree explist = exptree->child(0);
	  if ( explist )
	    {
	      // the order of the args needs to be restored back from the tree
	      std::deque<TreeNode*> exps;
	      for(; explist; explist = explist->child(0) )
		exps.push_front(explist->child(1));

	      for(int i=0; i<exps.size(); i++ )
		{
		  Tree exp = exps[i];
		  LBVMCode::Operand expop;
		  if ( generateExpressionCode(exp, ps, expop,homens, errs) != SUCCESS )
		    {
		      errs += "Failed to parse element expression in set construction\n";
		      return FAILURE;
		    }

		  LBVMCode setins(LBVMCode::SETINS);
		  setins.resultop() = resultop;
		  setins.op1() = expop;
		  ps.metadata()->code().append(setins);
		}
	    }

	  return SUCCESS;

	}

      case TreeNode::LOCALVAR:
      case TreeNode::GLOBALVAR:
	// this is a local or global variable
	return resolveVariableOrCell(exptree, ps, resultop, homens, false, errs);
      case TreeNode::INPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::IN, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"\n";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::OUTPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::OUT, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::STOREPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::STORE, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::STATICPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::STATIC, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}

      case TreeNode::STRUCTTYPEEVAL:
	{
	  Tree thestruct = exptree->child(0);
	  Tree namedargs = exptree->child(1);

	  LBVMCode::Operand structop;
	  LBFullSymbol stname = parseTypeName(thestruct,  homens);
	  if ( ! ps.metadata()->getGlobalStructType(stname, structop, errs) )
	    {
	      errs += "Failed to insantiate symbol "+stname.toString()+" as a struct\n";
	      return FAILURE;
	    }

	  if (namedargs &&  namedargs->numChildren() == 1 )
	    {
	      LBVMCode vecall(LBVMCode::VECALL);
	      vecall.op1() = structop;
	      vecall.op2() = LBVMCode::Operand(LBVMCode::Operand::INTVALUE,( namedargs->child(0)->nodeType() == TreeNode::VECOPPAR )?1:0 );
	      resultop = ps.metadata()->newTemp();
	      vecall.resultop() = resultop;
	      ps.metadata()->code().append(vecall);
	      return SUCCESS;
	    }

	  LBVMCode::Operand namedargsop;
	  if ( namedargs &&  namedargs->numChildren() == 2 && generateNamedArgsCode(namedargs, ps, namedargsop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed parse named arguments in struct evaluation expression\n";
	      return FAILURE;
	    }

	  LBVMCode namedeval(LBVMCode::STRUCTEVAL);
	  namedeval.op1() = structop;
	  namedeval.op2() = namedargsop;
	  resultop = ps.metadata()->newTemp();
	  namedeval.resultop() = resultop;
	  ps.metadata()->code().append(namedeval);

	  return SUCCESS;
	}

      case TreeNode::CONSTR:
	{
	  Tree fullname = exptree->child(0);
	  Tree args = exptree->child(1);

	  // handle argument list first
	  // generate code to init arg op
	  LBVMCode::Operand argop;
	  if ( args && generateAnonymousArgsCode(args, ps, argop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse argument list for object member function call\n";
	      return FAILURE;
	    }
	  
	  LBFullSymbol tname = parseTypeName(fullname, homens);
	  LBVMCode::Operand constrop;
	  if ( ! ps.metadata()->getGlobalConstructableType( tname, constrop) )
	    {
	      errs += "Can not resolve symbol "+tname.toString()+" as a constructable type\n";
	      return FAILURE;
	    }

	  LBVMCode objconstru(LBVMCode::OBJCONSTRU);
	  objconstru.op1() = constrop;
	  objconstru.op2() = argop;
	  resultop = ps.metadata()->newTemp();
	  objconstru.resultop() = resultop;
	  ps.metadata()->code().append(objconstru);
	  return SUCCESS;

	}

      case TreeNode::CASTOP:
	{
	  Tree args = exptree->child(1);
	  LBVMCode::Operand argsop;
	  if ( args && generateAnonymousArgsCode(args, ps, argsop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the anonymous argument list for built in types construction/cast\n";
	      return FAILURE;
	    }
	  Tree casttype=exptree->child(0);
	  LBVMCode::Operand typeop;
	  switch ( casttype->nodeType() )
	    {
	    case TreeNode::DOUBLETYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBDouble))));
		break;
	      }
	    case TreeNode::INTTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBInt))));
		break;
	      }
	    case TreeNode::STRINGTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBString))));
		break;
	      }
	    case TreeNode::BOOLTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBBool))));
		break;
	      }
	    case TreeNode::CHARTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBChar))));
		break;
	      }
	    case TreeNode::VECTORTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBVector))));
		break;
	      }
	    case TreeNode::MAPTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBMap))));
		break;
	      }
       	    case TreeNode::SETTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBSet))));
		break;
	      }
	    case TreeNode::ERRORTYPE:
	      {
		typeop = ps.metadata()->newLiteral(static_cast<LBObject*>(new LBTypeSpecBasic(LBTYPEID(LBError))));
		break;
	      }
	    default:
	      {
		errs += "Unrecognized builtin type name, invalid parse tree\n";
		return FAILURE;
	      }
	    }

	  LBVMCode buildbasic(LBVMCode::OBJCONSTRU);
	  buildbasic.op1() = typeop;
	  buildbasic.op2() = argsop;
	  resultop = ps.metadata()->newTemp();
	  buildbasic.resultop()=resultop;
	  ps.metadata()->code().append(buildbasic);

	  return SUCCESS;

	}
      default:
	;
      }

    errs += "Invalid parse tree, uncrecognized Atom Expression node type\n";
    return FAILURE;

  }

  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateLValueExpCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    switch ( exptree->nodeType() )
      {
      case TreeNode::SUBSCRIPT:
	{
	  Tree varexp = exptree->child(0);
	  Tree subexp = exptree->child(1);
	  LBVMCode::Operand varexpop;
	  if ( generateLValueExpCode( varexp, ps, varexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse left value expression for subscripting expression\n";
	      return FAILURE;
	    }
	
	  LBVMCode::Operand subexpop;
	  if ( generateExpressionCode(subexp, ps, subexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the subscription value expression\n";
	      return FAILURE;
	    }
	
	  resultop = ps.metadata()->getSubscriptOperand(varexpop, subexpop); 

	  return SUCCESS;

	}
      case TreeNode::PROPERTY:
	{
	  Tree varexp = exptree->child(0);
	  const LBSymbol *prptyname = exptree->child(1)->symbol();

	  LBVMCode::Operand varexpop;
	  if ( generateLValueExpCode(varexp, ps, varexpop, homens, errs) != SUCCESS )
	    {
	      errs += "Failed to parse the variable for property read expression\n";
	      return FAILURE;
	    }

	  resultop = ps.metadata()->getVarPropertyOperand(varexpop, *prptyname); 	  
	  
	  return SUCCESS;
	  
	}

      case TreeNode::LOCALVAR:
      case TreeNode::GLOBALVAR:
	{
	  // this is a local or global variable
	  return resolveVariableOrCell(exptree, ps, resultop, homens, true, errs);
	}
      case TreeNode::OUTPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::OUT, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::STOREPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::STORE, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::STATICPROPERTYREF:
	{
	  const LBSymbol *pname = exptree->symbol();
	  if ( ! ps.metadata()->getLocalPropertyOperand(*pname, LBPropertyInfo::STATIC, resultop, errs) )
	    {
	      errs += "Can not find property \""+pname->toString()+"\"";
	      return FAILURE;
	    }
	  return SUCCESS;
	}
      case TreeNode::PARENTHESISEXP:
	{
	  return generateLValueExpCode(exptree->child(0), ps, resultop, homens, errs);
	}

      default:
	;
      }

    errs += "Invalid left value expression\n";
    return FAILURE;

  }


  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::generateNamedArgsCode(Tree exptree, LBProcessStruct& ps, LBVMCode::Operand& resultop, const LBFullSymbol& homens, string& errs)
  {
    LBProcessStruct::LBStructEvalArgs *stargs = new LBProcessStruct::LBStructEvalArgs();
    
    // the order of the args needs to be restored back from the tree
    std::deque<TreeNode*> argsq;
    for( Tree args = exptree; args; args = args->child(0) )
      argsq.push_front(args->child(1));
    
    for(int i=0; i<argsq.size(); i++ )
      {
	Tree onenamedarg = argsq[i];
	if ( onenamedarg->nodeType() == TreeNode::NAMEDARG )
	  {
	    const LBSymbol *name = onenamedarg->child(0)->symbol();
	    Tree argexp = onenamedarg->child(2);
	    LBVMCode::Operand argexpop;
	    if ( generateCondExpCode(argexp, ps, argexpop,homens, errs) != SUCCESS )
	      {
		errs += "Failed to parse one expression for named argument list\n";
		return FAILURE;
	      }


	    Tree vecop = onenamedarg->child(1);
	    bool vecit(false), par(false);
	    if ( vecop )
	      {
		vecit = true;
		if ( vecop->nodeType() == TreeNode::VECOPPAR )
		  par = true;
	      }	      
		
	    if ( ! stargs->addNamedArg(*name, argexpop, vecit, par) )
	      {
		errs += "Failed to parse one expression for named argument list\n";
		return FAILURE;
	      }

	    continue;
	  }

	if (  onenamedarg->nodeType() == TreeNode::INARG )
	  {
	    // get all the property names at run time
	    // byte code only remember the type name
	    // so it won't need to be re-generated when the refered type changes
	    Tree batchin = onenamedarg->child(1);
	    Tree vecop =  onenamedarg->child(0);

	    bool threading(false), par(false);
	    if ( vecop )
	      {
		threading = true;
		if ( vecop->nodeType() == TreeNode::VECOPPAR )
		  par = true;
	      }	      

	    switch ( batchin->nodeType() )
	      {
	      case TreeNode::STRUCTCLASSINS:
		{
		  LBVMCode::Operand structop;
		  LBFullSymbol tname = parseTypeName(batchin->child(0), homens);
		  if ( ! ps.metadata()->getGlobalStructType(tname, structop, errs ) )
		    {
		      errs += "Can not resolve symbol "+tname.toString()+" as struct type\n";
		      return FAILURE;
		    }
		  stargs->addStructINs(structop, threading, par);

		  // to be done
		  // decide: what do we do for internal proc of composition?
		  //      a. Adhoc cell doesn't care input/output port id, only care up/down stream 
		  //         so probably we can record the input dependency then directly expose input storage to internal proc
		  //      b. Data joint do care about port id to handle flow, it is now handled as ad hoc input properties
		  //      HOW DO WE RECONCILE a & b?   probably a new operand type dedicated for internal proc for accessing cells?


		  break;
		}
	      case TreeNode::ALLINSOFTHIS:
		{
		  stargs->addThisINs(threading, par);
		  break;
		}
		  
	      default:
		errs += "Invalid parse tree, unknown node type in named arg list\n";
		return FAILURE;
	      }

	  }
  
      }

    resultop = ps.metadata()->newNamedArgsOperand(stargs);

    return SUCCESS;

  }

  // the input tree here represents a full name, this function tries to resolve it to an operand attached to a process struct
  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::resolveVariableOrCell(Tree vartree, LBProcessStruct& ps, LBVMCode::Operand& resultvarop,  const LBFullSymbol& homens, bool forwriting, string& errs)
  {
    // variable name resolve policy is documented in design/symbolresolvingpolicy
    
    // a simple name like "varx" is a local variable
    // a full name like "X::Y::Z.zstaticProperty is a global variable in the form of static property
    // global variable can also be like "::Z.zstaticProperty", ::Z means Z type within home name space

    if ( vartree->nodeType() == TreeNode::LOCALVAR ) // this must be a local 
      {
	const LBSymbol *simplename = vartree->child(0)->symbol();
	if ( simplename == 0 )
	  {
	    errs += "Invalid parse tree empty local variable name\n";
	    return FAILURE;
	  }
	if ( ! ps.metadata()->getLocalVariableOrCell(*simplename, resultvarop, forwriting, errs) )
	  {
	    errs += "Failed to find local variable "+simplename->toString()+"\n";
	    return FAILURE;
	  }

	return SUCCESS;
      }

    if ( vartree->nodeType() == TreeNode::GLOBALVAR ) // this must be a global variable 
      {
	LBFullSymbol fullname = parseTypeName(vartree->child(0),homens);
	const LBSymbol *prpname = vartree->child(1)->symbol();
	if ( prpname == 0 )
	  {
	    errs += "Invalid parse tree empty static property name\n";
	    return FAILURE;
	  }
	if ( ! ps.metadata()->getGlobalVariable(fullname, *prpname, resultvarop) )
	  {
	    errs += "Failed to find global variable "+fullname.toString()+"."+prpname->toString()+"\n";
	    return FAILURE;
	  }
	return SUCCESS;
      }
    
    errs += "Invalid parse tree unknown variable node type\n";
    return FAILURE;

  }


  LubanCodeGenerator::CodeGenStatus LubanCodeGenerator::processNamedArgsForTypedCell(int cellid, LBCompositionStruct& comps, TreeNode *namedargs, const LBFullSymbol& homens, string& errs)
  {
    std::vector<LBSymbol> tovectorize;
    std::vector<LBFullSymbol> batchvec;
    bool vecnpar = false;
    bool vecall = false;

    // process named args
    if ( namedargs->numChildren() == 1 )
      {
	vecnpar = namedargs->child(0)->nodeType() == TreeNode::VECOPPAR;
	vecall = true;
      }
    else
      {
	const LBStructInterface* cellitfc = comps.metadata()->getTypedCellInterface(cellid);
	if ( ! cellitfc )
	  {
	    errs += "Expecting component to be statically typed\n";
	    return FAILURE;
	  }

	LBSymbol cellname = comps.metadata()->getCellName(cellid);

	for( TreeNode *args=namedargs; args; args = args->child(0) )
	  {
	    Tree onenamedarg = args->child(1);
	    if ( onenamedarg->nodeType() == TreeNode::NAMEDARG )
	      {
		const LBSymbol *name = onenamedarg->child(0)->symbol();
		int pindex=-1;
		const LBPropertyInfo *pinfo = cellitfc->propertyInfo(*name, pindex);
		if ( !pinfo || pinfo->attr() != LBPropertyInfo::IN ) // do we want to wire STORE property ? NO
		  {
		    errs += "The property to accept incoming event doesn't exist or is not an input property: "+name->toString()+"\n";
		    return FAILURE;
		  }

		Tree vecop = onenamedarg->child(1);
		if ( vecop )
		  {
		    tovectorize.push_back(*name);
		    if ( vecop->nodeType() == TreeNode::VECOPPAR )
		      vecnpar = true;
		  }	      

		Tree argexp = onenamedarg->child(2);

		// check if it is simple direct link: X.y  X  input.X
		if ( argexp->nodeType() == TreeNode::PROPERTY && argexp->child(0)->nodeType() == TreeNode::LOCALVAR )
		  {
		    const LBSymbol * varname = argexp->child(0)->child(0)->symbol();
		    int varcellid = comps.metadata()->getCellId(*varname);
		    if ( varcellid < 0 ) // invalid cell name
		      {
			errs += "Refered cell "+varname->toString()+" does not exist\n";
			return FAILURE;
		      }
		    const LBStructInterface *varcellitfc = comps.metadata()->getTypedCellInterface(varcellid);
		    if ( varcellitfc )
		      {
			const LBSymbol * prpname = argexp->child(1)->symbol();
			int varpindex=-1;
			const LBPropertyInfo *varpinfo = varcellitfc->propertyInfo(*prpname, varpindex);
			if ( ! varpinfo )
			  {
			    errs += "Refered property "+varname->toString()+"."+prpname->toString()+" does not exist\n";
			    return FAILURE;
			  }
			if ( varpinfo->attr() == LBPropertyInfo::OUT )
			  if ( ! comps.metadata()->link(varcellid, *prpname, cellid, *name, pindex, errs) )
			    {
			      errs += "Failed to create depdency link from "+varname->toString()+"."+prpname->toString()+" to "+cellname.toString()+"."+name->toString()+"\n";
			      return FAILURE;
			    }
			  else
			    continue;
		      }
		  }

		if ( argexp->nodeType() == TreeNode::LOCALVAR )
		  {
		    const LBSymbol * varname = argexp->child(0)->symbol();
		    int varcellid = comps.metadata()->getCellId(*varname);
		    if ( varcellid < 0 ) // invalid cell name
		      {
			errs += "Refered cell "+varname->toString()+" does not exist\n";
			return FAILURE;
		      }
		    if ( ! comps.metadata()->link(varcellid,  cellid, *name, pindex, errs) )
		      {
			errs += "Failed to create depdency link from "+varname->toString()+" to "+cellname.toString()+"."+name->toString()+"\n";
			return FAILURE;
		      }
		    else
		      continue;
		  }

		if ( argexp->nodeType() == TreeNode::INPROPERTYREF )
		  {
		    const LBSymbol * inprpname = argexp->symbol();
		    int dummy=-1;
		    const LBPropertyInfo *prpinfo = comps.interface().propertyInfo(*inprpname, dummy);
		    if ( ! prpinfo )
		      {
			errs += "Refered input property "+inprpname->toString()+" does not exist\n";
			return FAILURE;
		      }
		    if ( ! comps.metadata()->link(int(LBCompositionStruct::ComponentList::INPUTS), *inprpname,  cellid, *name, pindex, errs) )
		      {
			errs += "Failed to create depdency link from input "+inprpname->toString()+" to "+cellname.toString()+"."+name->toString()+"\n";
			return FAILURE;
		      }
		    else
		      continue;
		  }

		int newcellid = comps.metadata()->newNonameAdhocCell(false);
		LBProcessStruct *ps = new LBProcessStruct(newcellid, &comps);
		LBVMCode::Operand resultop;
		if ( generateCondExpCode(argexp, *ps, resultop, homens, errs) != SUCCESS )
		  {
		    errs += "Failed to parse one expression for named argument list\n";
		    delete ps;
		    return FAILURE;
		  }
		// add an assign to output here
		LBVMCode asn(LBVMCode::ASN);
		asn.op1() = ps->metadata()->getAdhocOutputOperand();
		asn.op2() = resultop;
		ps->metadata()->code().append(asn);

		comps.metadata()->setNonameAdhocCell(newcellid, ps);

		if ( ! comps.metadata()->link(newcellid, cellid, *name, pindex, errs) )
		  {
		    errs += "Failed create dependencty link from "+comps.metadata()->getCellName(newcellid).toString()+" to "+cellname.toString()+"."+name->toString()+"\n";
		    return FAILURE;
		  }
		
		continue;
	      }

	    if (  onenamedarg->nodeType() == TreeNode::INARG )
	      {
		Tree batchin = onenamedarg->child(1);
		Tree vecop =  onenamedarg->child(0);

		bool vecit(false);
		if ( vecop )
		  {
		    vecit = true;
		    if ( vecop->nodeType() == TreeNode::VECOPPAR )
		      vecnpar = true;
		  }

		switch ( batchin->nodeType() )
		  {
		  case TreeNode::STRUCTCLASSINS:
		    {
		      LBVMCode::Operand structop;
		      LBFullSymbol tname = parseTypeName(batchin->child(0), homens);
		      if ( ! comps.metadata()->linkBatchInputs(tname, cellid, errs) )
			{
			  errs += "Can not link input properties of "+tname.toString()+" to cell"+comps.metadata()->getCellName(cellid).toString()+"\n";
			  return FAILURE;
			}
		      if ( vecit ) batchvec.push_back(tname);
		      break;
		    }
		  case TreeNode::ALLINSOFTHIS:
		    {
		      if ( ! comps.metadata()->linkAllInputs(cellid, errs) )
			{
			  errs += "Can not link all input properties to cell"+comps.metadata()->getCellName(cellid).toString()+"\n";
			  return FAILURE;
			}
		      if ( vecit ) batchvec.push_back(LBFullSymbol()); // an empty full symbol means ALL INPUTS
		      break;
		    }
		  
		  default:
		    errs += "Invalid parse tree, unknown node type in named arg list\n";
		    return FAILURE;
		  }

	      }
  
	  }

      }

    if ( tovectorize.size() > 0 || batchvec.size() > 0 || vecall )
      comps.metadata()->vectorizeCell(cellid, tovectorize, batchvec, vecnpar);

    return SUCCESS;

  }




}
