#include <iostream>
#include <string>

#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbexception.hpp"

#include "luban/lbabstractstruct.hpp"
#include "luban/lbstructinterface.hpp"
#include "luban/lbrwlockedobjptr.hpp"
#include "luban/lblocalpropertystorage.hpp"
#include "luban/lbcounterwaiter.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/lubanpermissionchecking.hpp"
#include "luban/lubantypechecking.hpp"


namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBAbstractStruct, 1, 0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBAbstractStruct)

  LBAbstractStruct::~LBAbstractStruct()
  { 
    delete _itfc;
    delete _statics;
    delete [] _staticscounters;
  }

  LBAbstractStruct::LBAbstractStruct()
  {
    throw LBException("No default instance for abstract struct");
  }

  LBAbstractStruct::LBAbstractStruct(const LBAbstractStruct& abs)
  { throw LBException("Can not copy abstract struct");}

  LBAbstractStruct& LBAbstractStruct::operator=(const LBAbstractStruct& x)
  {
    throw LBException("No assignment for abstract struct");
  }

  LBAbstractStruct::LBAbstractStruct(LBStructInterface* itfc)
    :_itfc(itfc), _statics(0), _staticscounters(0)
    {

    int numstatics = itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	_statics = new LocalPropertyStorage(numstatics, true);
	_staticscounters = new CounterWaiter[numstatics];
	// init static properties
	for(int i=0; i<numstatics; i++)
	  {
	    const LBProcessStruct *initp = itfc->propertyInfo(i, LBPropertyInfo::STATIC)->initProcess();
	    if ( ! initp )
	      continue;
	    std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	    _statics->writeValue(i, torun->evalForLastObj());
	  }

      }
  }


  string LBAbstractStruct::toString() const
  { return _itfc->name()->toString(); }

  ostream& LBAbstractStruct::toStream( ostream& o ) const
  { 
    throw LBException("Can not stream abstract structure");
    return o; 
  }

  istream& LBAbstractStruct::fromStream( istream& i, int major, int minor )
  {
    throw LBException("Can not stream abstract structure");
    return i; 
  }

  bool LBAbstractStruct::equals(const LBObject& another) const
  {
    const LBAbstractStruct *ap = dynamic_cast<const LBAbstractStruct*>(&another);
    return this == ap;
  }

  const LBStructInterface& LBAbstractStruct::interface() const
  {  return *_itfc;  }

  LBStruct* LBAbstractStruct::structCall(const LBNamedArgs* args) const
  { throw LBException("Can not call abstract struct"); }

  void LBAbstractStruct::setBatchProperties(const LBNamedArgs* args)
  { throw LBException("Can not set properties of abstract struct"); }

  LBObjPtr LBAbstractStruct::getPropertyValue(const LBSymbol& s)
  { throw LBException("Can not get instance properties of abstract struct"); }

  LBObject* LBAbstractStruct::getPropertyPtr(const LBSymbol& s)
  { throw LBException("Can not get instance properties of abstract struct"); }

  void LBAbstractStruct::touchProperty(const LBSymbol& s)
  { throw LBException("Can not touch property of abstract struct"); }

  void LBAbstractStruct::setProperty(const LBSymbol& s, const LBObject& val)
  { throw LBException("Can not set properties of abstract struct"); }

  void LBAbstractStruct::setProperty(const LBSymbol& s, const LBObjPtr& valptr)
  { throw LBException("Can not set properties of abstract struct"); }

  LBObjPtr LBAbstractStruct::getPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  { throw LBException("Can not get properties of abstract struct"); }

  LBObject* LBAbstractStruct::getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr)
  { throw LBException("Can not get properties of abstract struct"); }

  void LBAbstractStruct::touchProperty(int index, LBPropertyInfo::ExecAttr attr)
  { throw LBException("Can not touch property of abstract struct"); }

  void LBAbstractStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val)
  { throw LBException("Can not set properties of abstract struct"); }

  void LBAbstractStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr)
  { throw LBException("Can not set properties of abstract struct"); }

  bool LBAbstractStruct::addPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _staticscounters[staticprpindex].inc();
  }

  bool LBAbstractStruct::removePending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _staticscounters[staticprpindex].dec();
  }

  bool LBAbstractStruct::waitUntilNoPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _staticscounters[staticprpindex].waitDownToZero();
  }


  LBObjPtr LBAbstractStruct::getStaticPropertyValue(int index)
  {
    const LBPropertyInfo *pinfo = _itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Read permission violation: "+errs);
    return _statics->readValue(index);
  }

  void LBAbstractStruct::setStaticPropertyValue(int index, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _statics->writeValue(index, val);
  }

  void LBAbstractStruct::setStaticPropertyValue(int index, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr() ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _statics->writeValue(index, valptr);
  }

  LBWriteLockedObjPtr LBAbstractStruct::getWriteLockedStaticProperty(int staticprpindex)
  {
    const LBPropertyInfo *pinfo = _itfc->propertyInfo(staticprpindex, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _statics->getWriteLockedObj(staticprpindex);
  }

  LBWriteLockedObjPtr LBAbstractStruct::getWriteLockedStoreProperty(int staticprpindex)
  { throw LBException("Can not get store properties of abstract struct"); }

}
