#include "lbtypes/lbtypefactory.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/DLLUtil.hpp"
#include "lbtypes/lbfullsymbol.hpp"

#include "lbthread/lbthread.hpp"

#include <cstring>
#include <cstdlib>

namespace Luban
{
  struct RegistryEntry
  {
    RegistryEntry(char *dll=0) : dllname(dll), lbtype(0),lubantypename(0), cppclassname(0),errmsg(0) {}
    RegistryEntry(LBTypeInfo* lbt, char *dll) : dllname(dll), lbtype(lbt),lubantypename(0), cppclassname(0),errmsg(0) {}
    char* dllname;
    LBTypeInfo* lbtype;
    const char* lubantypename;
    const char* cppclassname;
    const char* errmsg;
  };

  typedef HashFunctions::CString CharStar;
  typedef LBHashMap<CharStar, RegistryEntry*> TypeRegistry;

  static bool charstarEqual(const CharStar &c1, const CharStar &c2)
  {
    return ( c1 == c2 || strcmp(c1,c2) == 0 );
  }

  static TypeRegistry& typeRegistry()
  {
    static TypeRegistry tg(&HashFunctions::charstarHash, &charstarEqual, 7);
    return tg;
  }

  typedef LBHashMap<LBFullSymbol, RegistryEntry*> LubanTypeRegistry;

  static int symbolHash(const LBFullSymbol& s)
  {  return s.hash(); }
  static bool symbolEqual(const LBFullSymbol &s1, const LBFullSymbol &s2)
  {
    return s1 == s2;
  }

  static LubanTypeRegistry& lubanTypeRegistry()
  {
    static LubanTypeRegistry tg(&symbolHash, &symbolEqual, 7);
    return tg;
  }

  static int registerLubanBuiltInTypes()
  {
    static bool done = false;
    static const char *intcpp="Luban::LBInt";
    static const char *intlbn="int";
    static LBFullSymbol intlbnfsym; 
    intlbnfsym.append(LBSymbol(intlbn));
    static const char *doublecpp="Luban::LBDouble";
    static const char *doublelbn="double";
    static LBFullSymbol doublelbnfsym; 
    doublelbnfsym.append(LBSymbol(doublelbn));
    static const char *stringcpp="Luban::LBString";
    static const char *stringlbn="string";
    static LBFullSymbol stringlbnfsym; 
    stringlbnfsym.append(LBSymbol(stringlbn));
    static const char *charcpp="Luban::LBChar";
    static const char *charlbn="char";
    static LBFullSymbol charlbnfsym; 
    charlbnfsym.append(LBSymbol(charlbn));
    static const char *boolcpp="Luban::LBBool";
    static const char *boollbn="bool";
    static LBFullSymbol boollbnfsym; 
    boollbnfsym.append(LBSymbol(boollbn));
    static const char *mapcpp="Luban::LBMap";
    static const char *maplbn="map";
    static LBFullSymbol maplbnfsym; 
    maplbnfsym.append(LBSymbol(maplbn));
    static const char *vectorcpp="Luban::LBVector";
    static const char *vectorlbn="vector";
    static LBFullSymbol vectorlbnfsym; 
    vectorlbnfsym.append(LBSymbol(vectorlbn));
    static const char *setcpp="Luban::LBSet";
    static const char *setlbn="set";
    static LBFullSymbol setlbnfsym; 
    setlbnfsym.append(LBSymbol(setlbn));
    static const char *errcpp="Luban::LBError";
    static const char *errlbn="error";
    static LBFullSymbol errorlbnfsym; 
    errorlbnfsym.append(LBSymbol(errlbn));

    if ( done )
      return 1;
    
    // register for int, double, string, char, bool, map, vector and set 
    
    RegistryEntry *e = new RegistryEntry();
    e->cppclassname = intcpp;
    e->lubantypename = intlbn;
    typeRegistry()[const_cast<char*>(intcpp)] = e;
    lubanTypeRegistry()[intlbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = doublecpp;
    e->lubantypename = doublelbn;
    typeRegistry()[const_cast<char*>(doublecpp)] = e;
    lubanTypeRegistry()[doublelbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = stringcpp;
    e->lubantypename = stringlbn;
    typeRegistry()[const_cast<char*>(stringcpp)] = e;
    lubanTypeRegistry()[stringlbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = charcpp;
    e->lubantypename = charlbn;
    typeRegistry()[const_cast<char*>(charcpp)] = e;
    lubanTypeRegistry()[charlbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = boolcpp;
    e->lubantypename = boollbn;
    typeRegistry()[const_cast<char*>(boolcpp)] = e;
    lubanTypeRegistry()[boollbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = mapcpp;
    e->lubantypename = maplbn;
    typeRegistry()[const_cast<char*>(mapcpp)] = e;
    lubanTypeRegistry()[maplbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = vectorcpp;
    e->lubantypename = vectorlbn;
    typeRegistry()[const_cast<char*>(vectorcpp)] = e;
    lubanTypeRegistry()[vectorlbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = setcpp;
    e->lubantypename = setlbn;
    typeRegistry()[const_cast<char*>(setcpp)] = e;
    lubanTypeRegistry()[setlbnfsym] = e;

    e = new RegistryEntry();
    e->cppclassname = errcpp;
    e->lubantypename = errlbn;
    typeRegistry()[const_cast<char*>(errcpp)] = e;
    lubanTypeRegistry()[errorlbnfsym] = e;

    done = true;

    return 1;
  }


  int LBTypeFactory::registerType( LBTypeInfo* t)
  {

    registerLubanBuiltInTypes();

    bool ok = true;
    // below assume the default value for pointer type is zero
    // which is true according to ISO C++ standard ISO 14882
    char* temp = const_cast<char*>(t->className());
    RegistryEntry* &e = typeRegistry()[temp];
    if ( e == 0 )
      {
	e = new RegistryEntry(t,0);
	e->cppclassname = t->className();
	if ( e->lubantypename )
	  t->setLubanName(e->lubantypename);
      }
    else
      {

	if ( e->lbtype != 0 && e->lbtype != t )  // conflicting entry
	  ok = false;
	else
	  {
	    e->lbtype = t;
	    if ( e->lubantypename )
	      t->setLubanName(e->lubantypename);
	  }
      }

    return ok?1:0;

  }

  int LBTypeFactory::registerDLL(char* cppclassname, char *dllname, char* lubantypename)
  {
    int retval = 1;
    // below assume the default value for pointer type is zero
    // which is true according to ISO C++ standard ISO 14882
    RegistryEntry* &e = typeRegistry()[cppclassname];
    if ( e == 0 )
      {
	e = new RegistryEntry(dllname);
	e->cppclassname = cppclassname;
      }
    else
      {
	if ( e->dllname != 0  )  // conflicting or duplicated entry
	  retval = 0;
	else
	  e->dllname = dllname;
      }

    if ( lubantypename != 0 )
      {
	LBFullSymbol lbs;
	string lbtstr(lubantypename);
	if ( ! LBFullSymbol::stringToFullSymbol(lbtstr, lbs) )
	  {
	    e->errmsg = "Invalid Luban full type symbol";
	    return 0;
	  }
	RegistryEntry *&lbe = lubanTypeRegistry()[lbs];
	if ( lbe == 0 )
	  {
	    lbe = e;
	    e->lubantypename = lubantypename;
	  }
	else
	  if ( lbe == e && e->lubantypename == 0 )
	    e->lubantypename = lubantypename;
	  else
	    retval = 0; // conflicting or duplicated entry
      }

    return retval;
  }

  static LBMutex& LoadDllLock()
  {
    static LBMutex m;
    return m;
  }

  const LBTypeInfo* LBTypeFactory::findType(const char* cppclassname)
  {
    RegistryEntry* e=0;
    if ( !typeRegistry().find(const_cast<char*>(cppclassname), e) )
      return 0;
    if ( e->errmsg ) return 0;
    if ( e->lbtype ) return e->lbtype;
    if ( e->dllname )
      {
	const char* dllerr = 0;

	LBMutexLocker lock(LoadDllLock()); // only one thread can load dll at a time
	
	// check if the dll loaded again after obtain the lock
	if ( e->lbtype ) return e->lbtype;
	if ( e->errmsg ) return 0;

	// assuming the loading of the dll will populate the typeinfo data
	bool loaded = DLLUtil::loadDLL(e->dllname, dllerr);
	if ( !loaded ) 
	  {
	    // this looks like memory leak, though it doesn't matter because
	    // it confined in a single global static hash map which has limited
	    // number of entries
	    e->errmsg = (dllerr==0)?"Failed to load dll":strdup(dllerr);
	    return 0;
	  }
	if ( e->lbtype ) return e->lbtype;
	e->errmsg = "Specified dll does not contain the class";
	return 0;
      }

    e->errmsg = "Empty type registry entry";
    return 0;
  }

  const char* LBTypeFactory::errorMessage(const char* cppclassname)
  {
    RegistryEntry *e=0;
    bool hasit = typeRegistry().find(const_cast<char*>(cppclassname), e);
    if ( !hasit )
      return "Unregistered data type";
    
    return e->errmsg;
  }

  const char* LBTypeFactory::findDLL(const char* cppclassname)
  {
    RegistryEntry *e=0;
    bool hasit = typeRegistry().find(const_cast<char*>(cppclassname), e);
    if ( !hasit )
      return 0;
    
    return e->dllname;
  }


  const LBTypeInfo* LBTypeFactory::findLubanType(const LBFullSymbol& lubanfulltypename)
  {
    RegistryEntry* e=0;
    if ( !lubanTypeRegistry().find(lubanfulltypename, e) )
      return 0;
    if ( e->lbtype ) return e->lbtype;
    if ( e->errmsg ) return 0;

    return findType(e->cppclassname); // avoid duplicating dll loading code
  }

  const char* LBTypeFactory::checkLubanTypeError(const LBFullSymbol& lbfulltypename)
  {
    RegistryEntry* e=0;
    if ( !lubanTypeRegistry().find(lbfulltypename, e) )
      return "Unregistered Luban data type";
    return e->errmsg;
  }

  const char* LBTypeFactory::findCPPClassName(const LBFullSymbol& lbfulltypename)
  {
    RegistryEntry* e=0;
    if ( !lubanTypeRegistry().find(lbfulltypename, e) )
      return 0;
    return e->cppclassname;
  }

    


}
