#include <iostream>
#include <sstream>
#include <string>
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbbool.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbmap.hpp"


namespace Luban
{
  class LBConstIterator;

  const static char LBDELIMETER = '\n';

  using std::string;
  using std::ostringstream;

  // this function should be used to de-serialize all Luban objects
  // static
  LBObject* LBObject::instanceFromStream(istream& i, string &errmsg)
  {
    const LBTypeInfo* lbtype = LBTypeInfo::restoreFromStream(i,errmsg );
    if ( lbtype == 0 )
      return 0; // type not found

    // get object version in the stream
    int mj=-1,mn=-1;
    char dot='\0', sp=dot;
    i >> mj >> dot >> mn;
    i.get(sp);
    if ( ! i || dot != '.' || sp != ' ')
      {
	errmsg += "Corrupted stream, fail to get object version";
	return 0;
      }

    LBObject *lbo=lbtype->newInstance();
    if ( lbo == 0 )
      {
	errmsg += "Fail to create default instance for data type";
	return 0;
      }
    lbo->fromStream(i, mj, mn);
    
    // eat the delimeter and sanity check
    char slashn;
    i.get(slashn);

    if (slashn != LBDELIMETER )
      {
	errmsg += "Corrupted stream, fail to get object delimeter";
	delete lbo;
	return 0; // error, corrupted stream
      }

    return lbo;

  }

  // this is the official interface to serialize Luban objects
  //static
  ostream& LBObject::instanceToStream(ostream& o, const LBObject& lbo)
  {
    lbo.getType().toStream(o); // meta data first
    o<< ' ' << lbo.getType().majorVersion() << '.'<<lbo.getType().minorVersion() << ' ';
    lbo.toStream(o);
    o.put(LBDELIMETER); // add a delimeter
    return o;
  }

  // This function shall not be used, it is only here to demo that
  // all the solid derived types implement the same function
  //static
//   const LBTypeInfo& LBObject::staticTypeInfo()
//   {
//     static LBTypeInfo lbt;
//     return lbt;
//   }

  //static
  int LBObject::objcmp(const LBObject* obj1, const LBObject* obj2)
  {
    // policy about null value
    if ( obj1 == 0 && obj2 == 0 )
      return 0;
    if ( obj1 == 0 && obj2 != 0 )
      return -1;
    if ( obj1 != 0 && obj2 == 0 )
      return 1;

    try {
      if ( obj1->before(*obj2 ) )
	return -1;
    }
    catch(...) {}
    try {
      if ( obj2->before(*obj1) )
	return 1;
    }
    catch (...) {}
    try {
      if ( obj2->equals(*obj1) )
	return 0;
    }
    catch (...) {}

    //Use type comparison, if the instance comparison failed
    const LBTypeInfo& t1=obj1->getType();
    const LBTypeInfo& t2=obj2->getType();
    if ( t1 > t2 ) return 1;
    if ( t1 < t2 ) return -1;
    return 0;
  }

  // static protected
  LBTypeInfo* LBObject::creatLBTypeInfo( const type_info& cppt, const char* cppname,  LBTypeInfo::LBConstructor lbc, int major, int minor)
  {
    return new LBTypeInfo(cppt, cppname, lbc, major,minor);
  }



  // important decision, default hash is zero so all objects regardless if they have good implementation
  // of hash can be put into map, though the perfomance will be better if they do.
  int LBObject::hash() const 
  { return 0; }
    //  { throw LBException(string("hash function does not apply to data type ")+getType().toString());}

  bool LBObject::cast(LBObject* casttotarget ) const
  { throw LBException(string("cast operation does not apply to data type ")+getType().toString());}

  LBObject& LBObject::add(const LBObject& lbo)
  { throw LBException(string("add operation does not apply to data type ")+getType().toString());}

  LBObject& LBObject::sub(const LBObject& lbo)
  { throw LBException(string("subtract operation does not apply to data type ")+getType().toString());}
   
  LBObject& LBObject::mul(const LBObject& lbo)
  { throw LBException(string("multiply operation does not apply to data type ")+getType().toString());}

  LBObject& LBObject::div(const LBObject& lbo)
  { throw LBException(string("division operation does not apply to data type ")+getType().toString());}

  LBObject& LBObject::mod(const LBObject& lbo)
  { throw LBException(string("mod operation does not apply to data type ")+getType().toString());}

  bool LBObject::before(const LBObject& lbo) const
  { throw LBException(string("\"Before\" comparison does not apply to data type ")+getType().toString());}
  
  LBObject& LBObject::neg()
  { throw LBException(string("negation does not apply to data type ")+getType().toString());}

  LBObject& LBObject::plusplus()
  { throw LBException(string("++ does not apply to data type ")+getType().toString());}

  LBObject& LBObject::minusminus()
  { throw LBException(string("-- does not apply to data type ")+getType().toString());}

  void LBObject::subscriptSet(const LBObject* index, const LBObject* value)
  {  throw LBException(string("subscript set does not apply to data type ")+getType().toString());}

  LBObject* LBObject::subscriptGet(const LBObject* index )
  {  throw LBException(string("subscript get does not apply to data type ")+getType().toString()); }

  const LBObject* LBObject::subscriptConstGet(const LBObject* index ) const
  {  throw LBException(string("subscript get does not apply to data type ")+getType().toString()); }

  int LBObject::size() const
  {  throw LBException(string("size query does not apply to data type ")+getType().toString());}

  LBConstIterator* LBObject::getIterator() const
  {  return 0; }			   

  bool LBObject::contains(const LBObject* x) const
  {  throw LBException(string("element query \"contains\" does not apply to data type ")+getType().toString());}

  void LBObject::clear()
  {  throw LBException(string("clear function call does not apply to data type ")+getType().toString());}

  LBObject* LBObject::memberFuncCall(const LBSymbol& funcname, const LBVarArgs* lbargs)
  {
    static const LBSymbol sz("size");
    static const LBSymbol ctns("contains");
    static const LBSymbol hsh("hash");
    static const LBSymbol clr("clear");
    static const LBSymbol srlz("serialize");
    static const LBSymbol invk("invoke"); // used purely in reflection
    static const LBSymbol allfuncs("allfuncs"); // list of funcs for human read

    if ( funcname == sz )
      {
	if ( lbargs && lbargs->numArgs() )
	  throw LBException("Common member function size() expects no arguments");
	int s = size();
	return new LBInt(s);
      }

    if ( funcname == ctns )
      {
	if ( lbargs && lbargs->numArgs() == 1 )
	  {
	    bool c = contains( lbargs->getArg(0) );
	    return new LBBool(c);
	  }
	throw LBException("Common member function contains() expects one argument");
      }

    if ( funcname == hsh )
      {
	if ( lbargs && lbargs->numArgs() )
	  throw LBException("Common member function hash() expects no arguments");
	int h = hash();
	return new LBInt(h);
      }

    if ( funcname == clr )
      {
	if ( lbargs && lbargs->numArgs() )
	  throw LBException("Common member function clear() expects no arguments");
	clear();
	return 0;
      }

    if ( funcname == srlz )
      {
	if ( lbargs && lbargs->numArgs() )
	  throw LBException("Common member function serialize() expects no arguments");
	ostringstream ost;
	instanceToStream(ost, *this);
	return new LBString(ost.str());
      }

    if ( funcname == invk ) // reflection call
      {
	int numargs = lbargs ? lbargs->numArgs():0;
	if ( numargs == 0 )
	  throw LBException("invoke function for reflection expect one or more arguments");
	const LBString *fname = dynamic_cast<const LBString*>(lbargs->getArg(0));
	if ( ! fname )
	  throw LBException("invoke function for reflection expect the first arguement be a function name string");
	LBSymbol fsym(fname->str());
	LBVarArgsShift newargs(1, lbargs);
	return memberFuncCall(fsym, &newargs);
      }

    if ( funcname == allfuncs )
      {
	static const LBString szstr("size"), szdesc("int size()");
	static const LBString ctnsstr("contains"), ctnsdesc("bool contains(object)");
	static const LBString hshstr("hash"), hshdesc("int hash()");
	static const LBString clrstr("clear"), clrdesc("void clear()");
	static const LBString srlzstr("serialize"), srlzdesc("string serialize()");
	static const LBString invkstr("invoke"), invkdesc("object invoke(string funcname, object arg1, ...)");
	static const LBString allfuncsstr("allfuncs"), allfuncsdesc("map allfuncs()");

	LBMap *result = allMemberFuncs();
	result->set( szstr, szdesc);
	result->set( ctnsstr, ctnsdesc);
	result->set( hshstr, hshdesc);
	result->set( clrstr, clrdesc);
	result->set( srlzstr, srlzdesc);
	result->set( invkstr, invkdesc);
	result->set( allfuncsstr, allfuncsdesc);
	
	return result;
      }
    return exportedMemberFunc(funcname, lbargs);

  }

  ostream& operator<< (ostream& o, const LBObject& lbo)
  {
    return o<<lbo.toString();
  }


}
