#include <vector>
#include <iostream>

#include "lbtypes/lbfullsymbol.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/RefCountedPtr.hpp"
#include "lbtypes/LBDefineMacros.hpp"

namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBFullSymbol, 1,0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBFullSymbol)

  LBFullSymbol::LBFullSymbol() : _imp()
  {
  }

  LBFullSymbol::~LBFullSymbol()
  {
  }

  string LBFullSymbol::toString() const
  {
    string s;
    int sz = size();
    for(int i = 0; i < sz; i++ )
      {
	if ( i )
	  s += "::";
	s += _imp->_symbolvec[i].toString();
      }
       
    return s;
  }

  ostream& LBFullSymbol::toStream( ostream& ost ) const
  {
    int sz = size();
    ost << sz;
    ost << ' ';
    for( int i= 0; i<sz; i++)
      {
	_imp->_symbolvec[i].toStream(ost);
	ost.put(' ');
      }
    return ost;
  }

  istream& LBFullSymbol::fromStream( istream& ist, int major, int minor)
  {
    int len = -1;
    ist >> len;
    char sp = ist.get(); // skip space
    if ( ! ist || len < 0 || sp != ' '  ) 
      throw LBException("Corrupted stream, error recovering number of seperate symbols for a full symbol from stream");

    if ( len == 0 )
      {
	_imp = LBFullSymbolImpPtr();
	return ist;
      }

    LBFullSymbolImp *newimp = new LBFullSymbolImp();
    newimp->_symbolvec.reserve(len);
    for(; ist && len >0; len-- )
      {
	LBSymbol sym;
	sym.fromStream(ist);
	sp=ist.get();
	if ( !ist || sp != ' ' )
	  {
	    delete newimp;
	    throw LBException("Corrupted stream, error recovering delimiter of seperate symbols for a full symbol from stream");
	  }
	newimp->_symbolvec.push_back(sym);
	newimp->_hash += sym.hash();
      }

    _imp = LBFullSymbolImpPtr(newimp);

    return ist;
  }

  LBDEFAULT_EQUALS_FUNC(Luban::LBFullSymbol)

  bool LBFullSymbol::operator==(const LBFullSymbol& x) const  
  {
    if ( _imp == x._imp )
      return true;
    if ( ! _imp || ! x._imp )
      return false;
    if ( _imp->_hash != _imp->_hash ) 
      return false;
    if ( size() != x.size() )
      return false;
    for( int i = 0; i < size(); i++ )
      if ( _imp->_symbolvec[i] != x._imp->_symbolvec[i] )
	return false;
    return true;
  }

  int LBFullSymbol::hash() const
  { return _imp->_hash; }

  int LBFullSymbol::size() const
  { return _imp?_imp->_symbolvec.size():0; }

  void LBFullSymbol::append(const LBSymbol& levelname)
  {
    if ( _imp ) 
      unicopy(_imp);
    else
      _imp = LBFullSymbolImpPtr( new LBFullSymbolImp() );
    _imp->_symbolvec.push_back(levelname);
    _imp->_hash += levelname.hash();
  }

  void LBFullSymbol::insertFront(const LBSymbol& levelname)
  {
    if ( _imp )
      unicopy(_imp);
    else
      _imp = LBFullSymbolImpPtr( new LBFullSymbolImp() );

    _imp->_symbolvec.insert(_imp->_symbolvec.begin(),levelname);
    _imp->_hash += levelname.hash();
  }

  const LBSymbol* LBFullSymbol::nameAtLevel(int level) const
  {
    if ( level < 0 || level >=size() )
      return 0;
    return &_imp->_symbolvec[level];
  }

  bool LBFullSymbol::pop(LBSymbol& result)
  {
    if ( size() == 0 )
      return false;
    unicopy(_imp);
    result = _imp->_symbolvec.back();
    _imp->_symbolvec.pop_back();
    _imp->_hash -= result.hash();
    return true;
  }

  // static
  bool LBFullSymbol::stringToFullSymbol(const string& str, LBFullSymbol& result)
  {
    static const char* clncln="::";
    int sz = str.size();
    result = LBFullSymbol();

    if ( sz==0 )
	return true;

    int ps = 0;
    while( ps != string::npos )
      {
	int pe =str.find(clncln,ps);
	if ( pe == ps )
	  return false;
	int len = (pe==string::npos)?(sz-ps):(pe-ps);
	LBSymbol subs(str.substr(ps,len));
	result.append(subs);
	ps = (pe==string::npos)?pe:(pe+2);
	if ( ps >= sz )
	  return false;
      }

    return true;

  }

}
