#ifndef __HASHUTIL_HPP__
#define __HASHUTIL_HPP__

/**********************************************************
 1. No intention to replicate STL style, which may cause brain damage, 
    I mean by merely trying to use STL classes
 2. Intuitive and prgamatic templated map and set class
 3. As fast and efficient as humanly possible.
***********************************************************/

#include <string>

namespace Luban
{
  template< class Key, class Value > class LBHashMap
  {

  public:
    
    typedef int (*HashFunc)(const Key &key);
    typedef bool (*EqualFunc)(const Key &key1, const Key &key2);

  private:
    struct Unit
    {
      Unit(const Key& k, int hs) : key(k), value(), keyhash(hs), next(0) {}
      Key key;
      Value value;
      int keyhash;
      Unit *next;
    };
    typedef Unit *Head;

    Head *_buckets;
    int _numBuckets;
    HashFunc _hashf;
    EqualFunc _equalf;
    int _numKeys;

  public:
    static int defaultHashFunc(const Key &key)
      { return 0; }
    static bool defaultEqualFunc(const Key &key1, const Key &key2)
      {  return &key1 == &key2 ; }
    
    enum { MAXBUCKETSIZE = 30 }; // this means 2^30 size of one giga entries
    LBHashMap(HashFunc hf=&LBHashMap::defaultHashFunc, EqualFunc ef=&LBHashMap::defaultEqualFunc, int bucket = 0 )
      : _buckets(0), _numBuckets(0), _hashf(hf), _equalf(ef), _numKeys(0)
      {
	int realbucket = (bucket<0)?0:((bucket>MAXBUCKETSIZE)?MAXBUCKETSIZE:bucket);
	int realsize = 1<<realbucket;
	_buckets = new Head[realsize];
	for( int i = 0; i < realsize; i++ )  _buckets[i] = 0; /* it should be zero initialized by default, but not */
	_numBuckets = realsize;
      }

    LBHashMap(const LBHashMap& m) 
      : _buckets(0), _numBuckets(0), _hashf(m._hashf), _equalf(m._equalf), _numKeys(0)
      {
	_buckets = new Head[m._numBuckets];
	_numBuckets = m._numBuckets;
	for( int i = 0; i < m._numBuckets; i++ )
	  if ( m._buckets[i] )
	    {
	      Unit *copy = new Unit( *m._buckets[i] );
	      _buckets[i] = copy;
	      while ( copy->next )
		{
		  Unit *tempcopy = new Unit( *copy->next );
		  copy->next = tempcopy;
		  copy = tempcopy;
		}
	    }
	  else
	    _buckets[i] = 0;
	_numKeys = m._numKeys;
      }

    ~LBHashMap()
      {
	Unit *upt = 0, *uptnext=0;
	for( int i = 0; i < _numBuckets; i++ )
	  if ( upt = _buckets[i] )
	    while( upt )
	      {
		uptnext = upt->next;
		delete upt;
		upt = uptnext;
	      }
	        
	delete [] _buckets;
      }


    const LBHashMap& operator=(const LBHashMap& m) 
    {
      clear();
      _hashf = m._hashf;
      _equalf = m._equalf;

      delete [] _buckets;
      _buckets = new Head[m._numBuckets];
      _numBuckets = m._numBuckets;
      for( int i = 0; i < m._numBuckets; i++ )
	if ( m._buckets[i] )
	  {
	    Unit *copy = new Unit( *m._buckets[i] );
	    _buckets[i] = copy;
	    while ( copy->next )
	      {
		Unit *tempcopy = new Unit( *copy->next );
		copy->next = tempcopy;
		copy = tempcopy;
	      }
	  }
      _numKeys = m._numKeys;
      return *this;
    }


    void clear()
      {
	Unit *upt = 0, *uptnext=0;
	for( int i = 0; i < _numBuckets; i++ )
	  if ( upt = _buckets[i] )
	    {
	      while( upt )
		{
		  uptnext = upt->next;
		  delete upt;
		  upt = uptnext;
		}
	      _buckets[i] = 0;
	    }
      }

    int size() const
      { return _numKeys; }

    bool find(const Key& k, Value& v) const
      {
	int hashk = (*_hashf)( k );
	int ind = hashk & ( _numBuckets-1 );
	Unit *b = _buckets[ind];
	while ( b ) // search through
	  {
	    if ( hashk == b->keyhash && (*_equalf)(k, b->key) )
	      {
		v = b->value;
		return true;
	      }
	    b = b->next;
	  }
	return false;
      }


    bool find(const Key& k, Key& insidek, Value& v) const
      {
	int hashk = (*_hashf)( k );
	int ind = hashk & ( _numBuckets-1 );
	Unit *b = _buckets[ind];
	while ( b ) // search through
	  {
	    if (  hashk == b->keyhash && (*_equalf)(k, b->key) )
	      {
		v = b->value;
		insidek = b->key;
		return true;
	      }
	    b = b->next;
	  }
	return false;
      }


    bool erase(const Key& k)
      {
	int hashk = (*_hashf)( k );
	int ind = hashk & ( _numBuckets-1 );
	Unit *b = _buckets[ind], *before = 0;
	while ( b ) // search through
	  {
	    if (  hashk == b->keyhash && (*_equalf)(k, b->key) )
	      {
		if ( before ) 
		  before->next = b->next;
		else
		  _buckets[ind] = b->next;
		delete b;
		_numKeys--;
		if ( _numBuckets >=256 &&  _numKeys < _numBuckets>>1 )
		  shrink();
		return true;
	      }
	    before = b;
	    b = b->next;
	  }
	return false;
      }


    Value& operator[](const Key& k)
      {
	int hashk = (*_hashf)( k );
	int ind = hashk & ( _numBuckets-1 );
	Unit *b = _buckets[ind], *before = 0;
	while ( b ) // search through
	  {
	    if (  hashk == b->keyhash && (*_equalf)(k, b->key) )
	      return b->value;
	    before = b;
	    b = b->next;
	  }
	Unit *newunit = new Unit(k, hashk);
	if ( before ) 
	  before->next = newunit;
	else
	  _buckets[ind] = newunit;
	_numKeys++;
	if ( _numKeys >= _numBuckets ) expand();
	return newunit->value;
      }

  private:
    void expand()
      {
	int newsize = _numBuckets << 1; // double size
	Head *newbkt = new Head[newsize];
	for( int i = 0; i < newsize; i++ )  newbkt[i] = 0; /* it should be zero initialized by default, but not */
	moveto( newbkt, newsize);
      }

    void shrink()
      {
	int newsize = _numBuckets >> 1; // half size
	Head *newbkt = new Head[newsize];
	for( int i = 0; i < newsize; i++ )  newbkt[i] = 0; /* it should be zero initialized by default, but not */
	moveto(newbkt, newsize);
      }

    void moveto(Head* newbkt, int newsize)
      {
	Unit *upt = 0;
	if ( newbkt )
	  {
	    // move stuff over
	    for( int i = 0; i < _numBuckets; i++ )
	      if ( upt = _buckets[i] )
		while( upt )
		  {
		    int newind = upt->keyhash & ( newsize -1 );
		    Unit *newupt = newbkt[newind];
		    if ( newupt )
		      {
			while ( newupt->next ) newupt = newupt->next;
			newupt->next = upt;
		      }
		    else
		      newbkt[newind] = upt;

		    Unit *temp = upt->next;
		    upt->next = 0;
		    upt = temp;
		  }
	    delete [] _buckets;
	    _buckets = newbkt;
	    _numBuckets = newsize;
	  }
      }

 
  public:
    friend class Iterator;

    class Pair
    {
    public:
      Pair() : key(0), value(0) {}
      const Key* key; const Value* value;
    };
    class Iterator
    {
    public:
      Iterator(const LBHashMap& m)
	: _themap(m), _index(-1),_end(m._numBuckets), _pos(0), _pair() {}
      bool next()
      {
	if ( _pos &&_pos->next )
	  _pos = _pos->next;
	else
	  { 
	    _index ++;
	    while ( _index < _end && _themap._buckets[_index] == 0 ) _index++;
	    if ( _index >= _end ) return false;
	    _pos = _themap._buckets[_index];
	  }
	_pair.key = &_pos->key;
	_pair.value = &_pos->value;
	return true;
      }    
	
      const Pair* currentPair() const
      { 
	if ( _index < _end && _index != -1 ) return &_pair; 
	return 0; 
      }
      
    private:
      const LBHashMap& _themap;
      int _index, _end;
      const Unit *_pos;
      Pair _pair;
    };


  };

  template <class Key> class LBHashSet
  {

  private:
    typedef LBHashMap<Key, int> MapType;
    MapType _map;

  public:
    typedef typename MapType::HashFunc HashFunc;
    typedef typename MapType::EqualFunc EqualFunc;

    LBHashSet() : _map() {}
    LBHashSet(HashFunc hf, EqualFunc ef, int bucket ) : _map(hf, ef, bucket) {}

    bool insert(const Key& k)
      {
	int& x=_map[k];
	if ( x==0 )
	  {
	    x = 1;
	    return true; // newly inserted
	  }

	return false;  // set already has the key
      }

    bool find(const Key& k) const
      {
	int dummy;
	return _map.find(k, dummy);
      }

    void clear()
      {_map.clear(); }

    bool erase(const Key& k)
      {	return _map.erase(k); }
	
    int size() const
      { return _map.size(); }

    friend class Iterator;
    class Iterator 
    {
    public:
      Iterator(const LBHashSet& s)
	: _it(s._map) {}
      bool next()
      { return _it.next() ; }
      const Key* currentKey() const
      {
	const typename MapType::Pair *p =  _it.currentPair();
	if ( p == 0 ) return 0;
	return p->key;
      }
    private:
      typename MapType::Iterator _it;
    };

  };

  using std::string;

  class HashFunctions
  {
  public:
    typedef char* CString;
    static int stringHash(const string& s);
    static int charstarHash(const CString &cstring);
  };

  class EqualFunctions
  {
  public:
    static bool stringEqual(const string& s1, const string& s2 )
    {  return s1 == s2; }
  };

}


#endif
