
#include "lbjava/jniwrap.hpp"

#include <cstdlib>
#include <cstring>
#include <iostream>
#include <sstream>

#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbdouble.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbbool.hpp"
#include "lbtypes/lbchar.hpp"
#include "lbtypes/lbmap.hpp"
#include "lbtypes/lbset.hpp"
#include "lbtypes/lbvector.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbsymbol.hpp"

#include "lbjava/LBJavaUtil.hpp"
#include "lbjava/LBJavaObjRef.hpp"

/* to dos:
   1. jvm args config interface // simple env variables, done
   2. iterator and subscript access   // next release
   3. static function and static/non-static fields access // done
   4. com example ???
   5. java os exec example and util
   6. regexp example   // done
   7. a simple date example //done
   8. invocation q close/open function  //done
   9. better error message for method invocation //done
   10. jdbc example  //done
*/


namespace Java
{
  using std::string;
  using std::istream;
  using std::ostream;
  using std::istringstream;
  using Luban::LBVarArgs;
  using Luban::LBException;
  using Luban::LBString;
  using Luban::LBDouble;
  using Luban::LBInt;
  using Luban::LBBool;
  using Luban::LBChar;
  using Luban::LBMap;
  using Luban::LBSet;
  using Luban::LBVector;
  using Luban::LBConstIterator;
  using Luban::LBSymbol;

  class JVMAndTools
  {
  public:
    const char *_errmsg;
    JavaVM *_jvm;

    jclass _java_lang_Class;
    jmethodID _java_lang_Class_forName;
    jmethodID _java_lang_Class_forName3Args;
    jmethodID _java_lang_Class_newInstance;
    jmethodID _java_lang_Class_getMethod;
    jmethodID _java_lang_Class_getField;
    jmethodID _java_lang_Class_getConstructor;
    jmethodID _java_lang_Class_getConstructors;
    jmethodID _java_lang_Class_getMethods;
    jmethodID _java_lang_Class_getName;
    jmethodID _java_lang_Class_isArray;

    jclass _java_lang_ClassLoader;
    jmethodID _java_lang_ClassLoader_getSystemClassLoader;
    jobject _classloader;

    jclass _java_lang_Thread;
    jmethodID _java_lang_Thread_currentThread;
    jmethodID _java_lang_Thread_setContextClassLoader;

    jclass _java_lang_String;

    jclass _java_lang_Throwable;
    jmethodID _java_lang_Throwable_getMessage;
    jmethodID _java_lang_Throwable_getCause;

    jclass _java_lang_IllegalArgumentException;

    jclass _java_lang_reflect_InvocationTargetException;

    jclass _java_lang_reflect_Constructor;
    jmethodID _java_lang_reflect_Constructor_newInstance;

    jclass _java_lang_reflect_Method;
    jmethodID _java_lang_reflect_Method_getName;
    jmethodID _java_lang_reflect_Method_invoke;

    jclass _java_lang_reflect_Field;
    jmethodID _java_lang_reflect_Field_get;
    jmethodID _java_lang_reflect_Field_set;

    jclass _java_lang_Object;
    jmethodID _java_lang_Object_getClass;
    jmethodID _java_lang_Object_hashCode;
    jmethodID _java_lang_Object_toString;
    jmethodID _java_lang_Object_equals;

    jclass _java_lang_Double;
    jmethodID _java_lang_Double_initDouble;

    jclass _java_lang_Integer;
    jmethodID _java_lang_Integer_initInteger;

    jclass _java_lang_Boolean;
    jmethodID _java_lang_Boolean_initBoolean;
    jmethodID _java_lang_Boolean_booleanValue;

    jclass _java_lang_Byte;
    jmethodID _java_lang_Byte_initByte;

    jclass _java_util_HashMap;
    jmethodID _java_util_HashMap_initHashMap;
    jmethodID _java_util_HashMap_put;

    jclass _java_util_HashSet;
    jmethodID _java_util_HashSet_initHashSet;
    jmethodID _java_util_HashSet_add;

    jclass _java_util_Map;
    jmethodID _java_util_Map_keySet;
    jmethodID _java_util_Map_entrySet;
    jmethodID _java_util_Map_get;

    jclass _java_util_Map_Entry;
    jmethodID _java_util_Map_Entry_getKey;
    jmethodID _java_util_Map_Entry_getValue;

    jclass _java_util_Set;
    jmethodID _java_util_Set_toArray;

    jclass _java_lang_Number;
    jmethodID _java_lang_Number_doubleValue;
    jmethodID _java_lang_Number_intValue;
    jmethodID _java_lang_Number_byteValue;

    jclass _java_lang_Character;
    jmethodID _java_lang_Character_charValue;

    jclass _java_io_ByteArrayOutputStream;
    jmethodID _java_io_ByteArrayOutputStream_constructor;
    jmethodID _java_io_ByteArrayOutputStream_toByteArray;

    jclass _java_io_ObjectOutputStream;
    jmethodID _java_io_ObjectOutputStream_constructor;
    jmethodID _java_io_ObjectOutputStream_writeObject;

    jclass _java_io_ByteArrayInputStream;
    jmethodID _java_io_ByteArrayInputStream_constructor;

    jclass _java_io_ObjectInputStream;
    jmethodID _java_io_ObjectInputStream_constructor;
    jmethodID _java_io_ObjectInputStream_readObject;

    jclass  _java_lang_reflect_Array;
    jmethodID _java_lang_reflect_Array_getLength;
    jmethodID _java_lang_reflect_Array_get;

    jclass  _java_util_List;
    jmethodID _java_util_List_size;
    jmethodID _java_util_List_get;

  };

  static inline jclass findClass(JNIEnv *env, const char *classname)
  {
    static string errhead = "Can't find class ";
    static string errhead2 = "Can't get global reference, error: ";
    jclass clazz = env->FindClass(classname);
    if (clazz == 0) 
      throw errhead+classname;
    jclass result = static_cast<jclass>(env->NewGlobalRef(clazz));
    jthrowable err = env->ExceptionOccurred();
    if ( err )
      {
	env->ExceptionClear();
	throw  errhead2+classname;
      }
    if ( ! result )
      throw errhead2+classname;
    env->DeleteLocalRef(clazz);
    return result;
  }

  static inline jmethodID findMethodID(JNIEnv* env, jclass clazz, const char* classname, const char* methodname, const char* signature)
  {
    static string errhead = "Can't find method ";
    jmethodID mid = env->GetMethodID(clazz, methodname , signature);
    if ( mid == 0) 
      throw errhead+classname+"."+methodname;
    return mid;
  }

  static inline jmethodID findStaticMethodID(JNIEnv* env, jclass clazz, const char* classname, const char* methodname, const char* signature)
  {
    static string errhead = "Can't find static method ";
    jmethodID mid = env->GetStaticMethodID(clazz, methodname , signature);
    if ( mid == 0) 
      throw errhead+classname+"."+methodname;
    return mid;
  }

  static int getJVMOptions(JavaVMOption* &options)
  {
    std::vector<char*> optionstrs;
    char *envclasspath = getenv("CLASSPATH");
    int sz = envclasspath?strlen(envclasspath):0;
    if ( sz )
      {
	char *wholepath = new char[sz+20];
	wholepath[0]='\0';
	strcat(wholepath,"-Djava.class.path=");
	strcat(wholepath,envclasspath);
	optionstrs.push_back(wholepath);
      }
    
    const char *lbvmargs = const_cast<const char*>(getenv("LUBAN_JVMOPTIONS"));
    if ( lbvmargs )
      {
	string str(lbvmargs);
	istringstream strstream(str);

	string onestr;
	strstream>>onestr;
	while(strstream)
	  {
	    char *onecstr = new char[onestr.size()+1];
	    strcpy(onecstr, onestr.c_str());
	    optionstrs.push_back(onecstr);
	    strstream>>onestr;
	  } 
      }

    int numopt = optionstrs.size();
    if ( numopt > 0 )
      {
	options = new JavaVMOption[numopt];
	for(int i=0; i<numopt; ++i)
	  options[i].optionString = optionstrs[i];
      }
    return numopt;
  }

  static void deleteJVMOptions(JavaVMOption *opts, int sz)
  {
    for( int i=0; i<sz; ++i)
      delete [] opts[i].optionString;
    if (opts ) delete [] opts;
  }

  static JVMAndTools *getJVMAndTools()
  {
    static JVMAndTools *jvmtools = 0;
    if ( jvmtools )
      return jvmtools;

    jvmtools = new JVMAndTools();

    JavaVMInitArgs vm_args;
    JavaVMOption *options=0;
    int numopt = getJVMOptions(options);
    /* disable JIT */
    //options[0].optionString = "-Djava.compiler=NONE";
    //options[0].optionString = "-verbose:jni";
    //options[1].optionString = "-Xmx128m";
    vm_args.version = JNI_VERSION_1_4;
    vm_args.options = options;
    //vm_args.options = 0;
    vm_args.nOptions = numopt;
    vm_args.ignoreUnrecognized = 0;
     
    /* Create the Java VM */
    JNIEnv *env;
    jint res = JNI_CreateJavaVM(&(jvmtools->_jvm),(void**)&env,&vm_args);
    if (res < 0)
      {
	jvmtools->_jvm = 0;
	jvmtools->_errmsg = "Failed to create jvm";
	deleteJVMOptions(options, numopt);
	return jvmtools;
      }

    deleteJVMOptions(options, numopt);

    try {
      jvmtools->_java_lang_Class = findClass(env, "java/lang/Class");
      jvmtools->_java_lang_Class_forName = findStaticMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "forName", "(Ljava/lang/String;)Ljava/lang/Class;");
      jvmtools->_java_lang_Class_forName3Args = findStaticMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "forName", "(Ljava/lang/String;ZLjava/lang/ClassLoader;)Ljava/lang/Class;");
      jvmtools->_java_lang_Class_newInstance = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "newInstance", "()Ljava/lang/Object;");
      jvmtools->_java_lang_Class_getConstructor = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getConstructor", "([Ljava/lang/Class;)Ljava/lang/reflect/Constructor;");
      jvmtools->_java_lang_Class_getConstructors = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getConstructors", "()[Ljava/lang/reflect/Constructor;");
      jvmtools->_java_lang_Class_getMethods = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getMethods", "()[Ljava/lang/reflect/Method;");
      jvmtools->_java_lang_Class_getMethod = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getMethod", "(Ljava/lang/String;[Ljava/lang/Class;)Ljava/lang/reflect/Method;");
      jvmtools->_java_lang_Class_getField = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getField", "(Ljava/lang/String;)Ljava/lang/reflect/Field;");
    jvmtools->_java_lang_Class_getName = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "getName", "()Ljava/lang/String;");
    jvmtools->_java_lang_Class_isArray = findMethodID(env, jvmtools->_java_lang_Class, "java.lang.Class", "isArray", "()Z");

    jvmtools->_java_lang_ClassLoader = findClass(env, "java/lang/ClassLoader");
    jvmtools->_java_lang_ClassLoader_getSystemClassLoader = findStaticMethodID(env, jvmtools->_java_lang_ClassLoader, "java.lang.ClassLoader", "getSystemClassLoader", "()Ljava/lang/ClassLoader;");
    jobject loaderlocalref = env->CallStaticObjectMethod(jvmtools->_java_lang_ClassLoader, jvmtools->_java_lang_ClassLoader_getSystemClassLoader);
    jvmtools->_classloader = env->NewGlobalRef(loaderlocalref);
    jthrowable err = env->ExceptionOccurred();
    if ( err || ! jvmtools->_classloader )
      {
	if ( err ) env->ExceptionClear();
	throw string("Failed to obtain default system class loader");
      }
    env->DeleteLocalRef(loaderlocalref);

    jvmtools->_java_lang_Thread =  findClass(env, "java/lang/Thread"); ; 
    jvmtools->_java_lang_Thread_currentThread = findStaticMethodID(env,jvmtools->_java_lang_Thread, "java.lang.Thread", "currentThread", "()Ljava/lang/Thread;");
    jvmtools->_java_lang_Thread_setContextClassLoader = findMethodID(env,jvmtools->_java_lang_Thread, "java.lang.Thread", "setContextClassLoader", "(Ljava/lang/ClassLoader;)V");

    jvmtools->_java_lang_String =  findClass(env, "java/lang/String"); ; 

    jvmtools->_java_lang_Throwable = findClass(env, "java/lang/Throwable");
    jvmtools->_java_lang_Throwable_getMessage =findMethodID(env, jvmtools->_java_lang_Throwable, "java.lang.Throwable", "getMessage", "()Ljava/lang/String;");
    jvmtools->_java_lang_Throwable_getCause =findMethodID(env, jvmtools->_java_lang_Throwable, "java.lang.Throwable", "getCause", "()Ljava/lang/Throwable;");

    jvmtools->_java_lang_IllegalArgumentException = findClass(env, "java/lang/IllegalArgumentException");
    jvmtools->_java_lang_reflect_InvocationTargetException = findClass(env, "java/lang/reflect/InvocationTargetException");

    jvmtools->_java_lang_reflect_Constructor = findClass(env, "java/lang/reflect/Constructor");
    jvmtools->_java_lang_reflect_Constructor_newInstance = findMethodID(env, jvmtools->_java_lang_reflect_Constructor,"java.lang.reflect.Constructor", "newInstance", "([Ljava/lang/Object;)Ljava/lang/Object;");

    jvmtools->_java_lang_reflect_Method = findClass(env, "java/lang/reflect/Method");
    jvmtools->_java_lang_reflect_Method_getName =  findMethodID(env, jvmtools->_java_lang_reflect_Method, "java.lang.reflect.Method",  "getName", "()Ljava/lang/String;");
     jvmtools->_java_lang_reflect_Method_invoke =  findMethodID(env, jvmtools->_java_lang_reflect_Method, "java.lang.reflect.Method",  "invoke", "(Ljava/lang/Object;[Ljava/lang/Object;)Ljava/lang/Object;");

    jvmtools->_java_lang_reflect_Field = findClass(env, "java/lang/reflect/Field");
    jvmtools->_java_lang_reflect_Field_get =  findMethodID(env, jvmtools->_java_lang_reflect_Field, "java.lang.reflect.Field",  "get", "(Ljava/lang/Object;)Ljava/lang/Object;");
     jvmtools->_java_lang_reflect_Field_set =  findMethodID(env, jvmtools->_java_lang_reflect_Field, "java.lang.reflect.Field",  "set", "(Ljava/lang/Object;Ljava/lang/Object;)V");

    jvmtools->_java_lang_Object = findClass(env, "java/lang/Object");
    jvmtools->_java_lang_Object_getClass =  findMethodID(env, jvmtools->_java_lang_Object, "java.lang.Object", "getClass", "()Ljava/lang/Class;");
    jvmtools->_java_lang_Object_hashCode =  findMethodID(env, jvmtools->_java_lang_Object, "java.lang.Object", "hashCode", "()I");
    jvmtools->_java_lang_Object_toString =  findMethodID(env, jvmtools->_java_lang_Object, "java.lang.Object", "toString", "()Ljava/lang/String;");
    jvmtools->_java_lang_Object_equals =  findMethodID(env, jvmtools->_java_lang_Object, "java.lang.Object", "equals", "(Ljava/lang/Object;)Z");

    jvmtools->_java_lang_Double = findClass(env, "java/lang/Double");
    jvmtools->_java_lang_Double_initDouble =  findMethodID(env, jvmtools->_java_lang_Double, "java.lang.Double", "<init>", "(D)V");

    jvmtools->_java_lang_Integer = findClass(env, "java/lang/Integer");   
    jvmtools->_java_lang_Integer_initInteger = findMethodID(env, jvmtools->_java_lang_Integer, "java.lang.Integer", "<init>", "(I)V");

    jvmtools->_java_lang_Boolean =  findClass(env,"java/lang/Boolean");
    jvmtools->_java_lang_Boolean_initBoolean = findMethodID(env, jvmtools->_java_lang_Boolean,"java.lang.Boolean", "<init>", "(Z)V");
    jvmtools->_java_lang_Boolean_booleanValue = findMethodID(env, jvmtools->_java_lang_Boolean,"java.lang.Boolean", "booleanValue", "()Z");

    jclass byteclass = findClass(env,"java/lang/Byte");
    jvmtools->_java_lang_Byte = byteclass;
    jmethodID initbyte_mid = findMethodID(env,byteclass, "java.lang.Byte", "<init>", "(B)V");
    jvmtools->_java_lang_Byte_initByte = initbyte_mid;

    jclass mapclass = findClass(env,"java/util/HashMap");
    jvmtools->_java_util_HashMap = mapclass;
    jvmtools->_java_util_HashMap_initHashMap = findMethodID(env,mapclass,"java.util.HashMap", "<init>", "(I)V");
    jvmtools->_java_util_HashMap_put = findMethodID(env,mapclass, "java.util.HashMap", "put", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");

    jclass setclass = findClass(env,"java/util/HashSet");
    jvmtools->_java_util_HashSet = setclass;
    jvmtools->_java_util_HashSet_initHashSet = findMethodID(env,setclass,"java.util.HashSet", "<init>", "(I)V");
    jvmtools->_java_util_HashSet_add = findMethodID(env,setclass,"java.util.HashSet","add", "(Ljava/lang/Object;)Z");

    jclass mapitfclass = findClass(env,"java/util/Map");
    jvmtools->_java_util_Map = mapitfclass;
    jvmtools->_java_util_Map_keySet = findMethodID(env,mapitfclass,"java.util.Map","keySet", "()Ljava/util/Set;");
    jvmtools->_java_util_Map_entrySet = findMethodID(env,mapitfclass,"java.util.Map","entrySet", "()Ljava/util/Set;");
    jvmtools->_java_util_Map_get =findMethodID(env,mapitfclass,"java.util.Map","get", "(Ljava/lang/Object;)Ljava/lang/Object;");

    jclass mapentryitfclass = findClass(env,"java/util/Map$Entry");
    jvmtools->_java_util_Map_Entry = mapentryitfclass;
    jvmtools->_java_util_Map_Entry_getKey = findMethodID(env,mapentryitfclass,"java.util.Map.Entry","getKey", "()Ljava/lang/Object;");
    jvmtools->_java_util_Map_Entry_getValue = findMethodID(env,mapentryitfclass,"java.util.Map.Entry","getValue", "()Ljava/lang/Object;");

    jclass setitfclass = findClass(env,"java/util/Set");
    jvmtools->_java_util_Set=setitfclass;
    jvmtools->_java_util_Set_toArray = findMethodID(env,setitfclass,"java.util.Set","toArray", "()[Ljava/lang/Object;");

    jclass numberclass = findClass(env,"java/lang/Number");
    jvmtools->_java_lang_Number = numberclass;
    jvmtools->_java_lang_Number_doubleValue = findMethodID(env,numberclass,"java.lang.Number", "doubleValue", "()D");
    jvmtools->_java_lang_Number_intValue = findMethodID(env,numberclass,"java.lang.Number", "intValue", "()I");
    jvmtools->_java_lang_Number_byteValue = findMethodID(env,numberclass,"java.lang.Number", "byteValue", "()B");

    jclass charclass = findClass(env,"java/lang/Character");
    jvmtools->_java_lang_Character = charclass;
    jvmtools->_java_lang_Character_charValue = findMethodID(env,charclass,"java.lang.Character", "charValue", "()C");

    jclass byteoutputclass = findClass(env,"java/io/ByteArrayOutputStream");
    jvmtools->_java_io_ByteArrayOutputStream = byteoutputclass;
    jvmtools->_java_io_ByteArrayOutputStream_constructor = findMethodID(env,byteoutputclass,"java.io.ByteArrayOutputStream", "<init>", "()V");
    jvmtools->_java_io_ByteArrayOutputStream_toByteArray = findMethodID(env,byteoutputclass,"java.io.ByteArrayOutputStream", "toByteArray", "()[B");

    jclass objoutputclass = findClass(env,"java/io/ObjectOutputStream");
    jvmtools->_java_io_ObjectOutputStream = objoutputclass;
    jvmtools->_java_io_ObjectOutputStream_constructor = findMethodID(env,objoutputclass,"java.io.ObjectOutputStream", "<init>", "(Ljava/io/OutputStream;)V");
    jvmtools->_java_io_ObjectOutputStream_writeObject = findMethodID(env,objoutputclass,"java.io.ObjectOutputStream", "writeObject", "(Ljava/lang/Object;)V");

    jclass byteinputclass = findClass(env,"java/io/ByteArrayInputStream");
    jvmtools->_java_io_ByteArrayInputStream = byteinputclass;
    jvmtools->_java_io_ByteArrayInputStream_constructor = findMethodID(env,byteinputclass,"java.io.ByteArrayInputStream", "<init>", "([B)V");

    jclass objinputclass = findClass(env,"java/io/ObjectInputStream");
    jvmtools->_java_io_ObjectInputStream = objinputclass;
    jvmtools->_java_io_ObjectInputStream_constructor = findMethodID(env,objinputclass,"java.io.ObjectInputStream", "<init>", "(Ljava/io/InputStream;)V");
    jvmtools->_java_io_ObjectInputStream_readObject = findMethodID(env,objinputclass,"java.io.ObjectInputStream", "readObject", "()Ljava/lang/Object;");

    jclass arrayclass = findClass(env,"java/lang/reflect/Array");
    jvmtools->_java_lang_reflect_Array = arrayclass;
    jvmtools->_java_lang_reflect_Array_getLength = findStaticMethodID(env, arrayclass,"java.lang.reflect.Array", "getLength", "(Ljava/lang/Object;)I");
    jvmtools->_java_lang_reflect_Array_get = findStaticMethodID(env, arrayclass,"java.lang.reflect.Array", "get", "(Ljava/lang/Object;I)Ljava/lang/Object;");

    jclass listclass = findClass(env,"java/util/List");
    jvmtools->_java_util_List = listclass;
    jvmtools->_java_util_List_size = findMethodID(env, listclass,"java.util.List", "size", "()I");
    jvmtools->_java_util_List_get = findMethodID(env, listclass,"java.util.List", "get", "(I)Ljava/lang/Object;");

    }
    catch ( string& errmsg )
      {
	jvmtools->_jvm = 0;
	string *newstr = new string(errmsg);
	jvmtools->_errmsg = newstr->c_str();
      }

    return jvmtools;
    
  }

  static string jstringToString(JNIEnv *env, jstring jstr)
  {    
    if ( ! jstr )
      return string();
    const char *cstr = static_cast<const char*>(env->GetStringUTFChars(jstr,0));
    string result(cstr);
    env->ReleaseStringUTFChars(jstr, cstr);
    return result;
  }


  static string getJavaErrorMessage(JNIEnv *env, jthrowable err)
  {
    if ( err )
      {
	env->ExceptionClear();
	//	jmethodID getmsg_mid = getJVMAndTools()->_java_lang_Throwable_getMessage;
	jmethodID getmsg_mid = getJVMAndTools()->_java_lang_Object_toString;
	jstring errjstr = static_cast<jstring>(env->CallObjectMethod(err, getmsg_mid));
	string result = jstringToString(env, errjstr);
	env->DeleteLocalRef(errjstr);
	return result;
      }
    return string();
  }


  static bool checkAndClearException(JNIEnv *env, string *errmsg)
  {
    jthrowable err = env->ExceptionOccurred();
    if ( ! err ) return false;
    if ( errmsg ) *errmsg = getJavaErrorMessage(env, err);
    env->ExceptionClear();
    env->DeleteLocalRef(err);
    return true;
  }

  jstring LBJavaUtil::object2jstring(JNIEnv *env, const LBObject *obj)
  {
    JVMAndTools *jvmt = getJVMAndTools();
    const LBString* argonestr = dynamic_cast<const LBString*>(obj);
    if ( argonestr )
      {
	jstring jstr = env->NewStringUTF(argonestr->c_str());
	string errmsg;
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to create new string: "+errmsg);
	return jstr;
      }

    const LBJavaObjRef* jobj = dynamic_cast<const LBJavaObjRef*>(obj);
    if ( ! jobj )
      return 0;
    jobject jobjin = jobj->getJavaObj();
    if ( env->IsInstanceOf(jobjin, jvmt->_java_lang_String) )
      return (jstring)jobjin;
    return 0;

  }

  jclass LBJavaUtil::object2jclass(JNIEnv *env, const LBObject *argone, string& errmsg)
  {
    const LBString* argonestr = dynamic_cast<const LBString*>(argone);

    if ( argonestr )
      {
	string clsname = argonestr->str();
	const JavaClassInfo *jclsinfo = JavaClassInfo::getClassInfo(env, clsname);
	if ( ! jclsinfo )
	  {
	    errmsg = "Failed to find class "+clsname;
	    return 0;
	  }
	return jclsinfo->getJavaClassObj();
      }

    const LBJavaObjRef* jobj = dynamic_cast<const LBJavaObjRef*>(argone);
    if ( ! jobj )
      {
	errmsg = "Object is not a classname string or class object";
	return 0;
      }

    jobject argoneobj = jobj->getJavaObj();
    if ( ! argoneobj )
      {
	errmsg = "Object is Java null, not a classname string or class object";
	return 0;
      }

    JVMAndTools *jvmt = getJVMAndTools();
    if ( env->IsInstanceOf(argoneobj, jvmt->_java_lang_Class) )
      return (jclass) argoneobj;

    if ( env->IsInstanceOf(argoneobj, jvmt->_java_lang_String) )
      {
	string clsname = jstringToString(env, (jstring)argoneobj);
	const JavaClassInfo *jclsinfo = JavaClassInfo::getClassInfo(env, clsname);
	if ( ! jclsinfo )
	  {
	    errmsg = "Failed to find class "+clsname;
	    return 0;
	  }
	return jclsinfo->getJavaClassObj();
      }

    errmsg = "Object is not a classname string or class object";
    return 0;

  }

  jobject LBJavaUtil::lubanToJava(JNIEnv *env, const LBObject* lbobj)
  {
    if ( ! lbobj ) return 0;

    JVMAndTools *jvmt = getJVMAndTools();

    // see it is already a java type
    const LBJavaObjRef *javaobj = dynamic_cast<const LBJavaObjRef*>(lbobj);
    if ( javaobj )
      return javaobj->getJavaObj();

    // iterate through all basic types
    const LBString *luban_string = dynamic_cast<const LBString*>(lbobj);
    if ( luban_string )
      return env->NewStringUTF(luban_string->c_str());

    const LBDouble *luban_double = dynamic_cast<const LBDouble*>(lbobj);
    if ( luban_double )
      return env->NewObject(jvmt->_java_lang_Double, jvmt->_java_lang_Double_initDouble, double(*luban_double));

    const LBInt *luban_int = dynamic_cast<const LBInt*>(lbobj);
    if ( luban_int )
      return env->NewObject(jvmt->_java_lang_Integer, jvmt->_java_lang_Integer_initInteger, int(*luban_int));

    const LBBool *luban_bool = dynamic_cast<const LBBool*>(lbobj);
    if ( luban_bool )
      return env->NewObject(jvmt->_java_lang_Boolean, jvmt->_java_lang_Boolean_initBoolean, bool(*luban_bool));

    const LBChar *luban_char = dynamic_cast<const LBChar*>(lbobj);
    if ( luban_char )
      return env->NewObject(jvmt->_java_lang_Byte, jvmt->_java_lang_Byte_initByte, char(*luban_char));

    const LBVector *luban_vector = dynamic_cast<const LBVector*>(lbobj);
    if ( luban_vector )
      {
	int sz = luban_vector->size();
	jobjectArray result = static_cast<jobjectArray>(env->NewObjectArray(sz, jvmt->_java_lang_Object, 0));
	for( int i=0; i<sz; ++i)
	  {
	    LBInt index(i);
	    const LBObject *oneobj = luban_vector->subscriptConstGet(&index);
	    try {
	      jobject onejobj = lubanToJava(env, oneobj);
	      env->SetObjectArrayElement(result, i, onejobj);
	    }
	    catch( LBException& excp)
	      {
		env->DeleteLocalRef(result);
		throw LBException("Luban vector contains element that can not be converted to java object: "+excp.msg());
	      }
	  }
	return result;
      }

    const LBMap *luban_map = dynamic_cast<const LBMap*>(lbobj);
    if ( luban_map )
      {
	if ( jvmt->_java_util_HashMap == 0 )
	  throw LBException("Can not load java.util.HashMap to convert luban map object");
	int sz = luban_map->size();
	jobject jmap = env->NewObject(jvmt->_java_util_HashMap, jvmt->_java_util_HashMap_initHashMap, 2*sz );
	LBConstIterator *it = luban_map->getIterator();

	while( it->next() )
	  {
	    const LBVarArgs *onerow = it->getCurrentRow();
	    const LBObject *key = onerow->getArg(0);
	    const LBObject *val = onerow->getArg(1);
	    jobject jkey = 0, jval = 0;
	    try {
	      jkey = lubanToJava(env, key);
	      jval = lubanToJava(env, val);
	    }
	    catch( LBException excp)
	      {
		if ( jkey ) env->DeleteLocalRef(jkey);
		if ( jval ) env->DeleteLocalRef(jval);
		env->DeleteLocalRef(jmap);
		throw LBException("Luban map contains element that can not be converted to java object: "+excp.msg());
	      }

	    env->CallObjectMethod(jmap, jvmt->_java_util_HashMap_put, jkey, jval);

	  }
	return jmap;
      }

    const LBSet *luban_set = dynamic_cast<const LBSet*>(lbobj);
    if ( luban_set )
      {
	if ( jvmt->_java_util_HashSet == 0 )
	  throw LBException("Can not load java.util.HashSet to convert luban set object");
	int sz = luban_set->size();
	jobject jset = env->NewObject(jvmt->_java_util_HashSet, jvmt->_java_util_HashSet_initHashSet, 2*sz );
	LBConstIterator *it = luban_set->getIterator();

	while( it->next() )
	  {
	    const LBVarArgs *onerow = it->getCurrentRow();
	    const LBObject *key = onerow->getArg(0);
	    jobject jkey = 0;
	    try {
	      jkey = lubanToJava(env, key);
	    }
	    catch( LBException excp)
	      {
		env->DeleteLocalRef(jset);
		throw LBException("Luban set contains element that can not be converted to java object: "+excp.msg());
	      }

	    env->CallBooleanMethod(jset, jvmt->_java_util_HashSet_add, jkey);

	  }
	return jset;
      }

    throw LBException(string("Can not convert luban object of type ")+lbobj->getType().lubanName()+" to Java object");

    return 0;
  }

  static jobjectArray lubanArgsToJavaObjectArray(JNIEnv *env, const LBVarArgs *args, jobjectArray *jclasses=0, int startidx=0)
  {
    JVMAndTools *jvt = getJVMAndTools();
    int sz = args?args->numArgs():0;
    if ( sz <= startidx )
      return 0;

    int argsz = sz-startidx;
    jobjectArray argresult = static_cast<jobjectArray>(env->NewObjectArray(argsz, jvt->_java_lang_Object, 0));
    if ( jclasses )
      *jclasses =  static_cast<jobjectArray>(env->NewObjectArray(argsz, jvt->_java_lang_Class, 0));
    
    for(int i=startidx; i<sz; ++i)
      {
	const LBObject* oneobj = args->getArg(i);
	jobject onejobj = LBJavaUtil::lubanToJava(env, oneobj);
	env->SetObjectArrayElement(argresult, i-startidx, onejobj);
	if (jclasses )
	  {
	    jobject onejobjclass = env->CallObjectMethod(onejobj, jvt->_java_lang_Object_getClass);
	    env->SetObjectArrayElement( *jclasses, i-startidx, onejobjclass);
	  }
      }

    return argresult;
  }

  class JVMThreadAttacher
  {
    JNIEnv *_jnienv;
  public:
    JVMThreadAttacher()
      : _jnienv(0)
    {
      JVMAndTools *jvmtools = getJVMAndTools();
      if ( jvmtools->_jvm )
	{
	  JavaVMAttachArgs args;
	  args.version= JNI_VERSION_1_4;;
	  args.name = 0;
	  args.group = 0;
	  int result = jvmtools->_jvm->AttachCurrentThread((void**)&_jnienv, (void*)&args);
	  if ( result < 0 )
	    {
	      _jnienv = 0;
	      throw LBException("Can not attach JVM thread to obtain jvm env");
	    }
	  jobject currentthread = _jnienv->CallStaticObjectMethod(jvmtools->_java_lang_Thread, jvmtools->_java_lang_Thread_currentThread );
	  _jnienv->CallVoidMethod(currentthread, jvmtools->_java_lang_Thread_setContextClassLoader, jvmtools->_classloader );
	  jthrowable err = _jnienv->ExceptionOccurred();
	  if ( err )
	    {
	      _jnienv->ExceptionClear();
	      _jnienv->DeleteLocalRef(currentthread);
	      _jnienv->DeleteLocalRef(err);
	      throw LBException("Can not set class loader for current thread");
	    }
	  _jnienv->DeleteLocalRef(currentthread);
	}
      else
	throw LBException(string("Failed to init jvm: ")+ (jvmtools->_errmsg?jvmtools->_errmsg:""));
    }
    ~JVMThreadAttacher()
    {
      JVMAndTools *jvmtools = getJVMAndTools();
      jvmtools->_jvm->DetachCurrentThread();
    }

    JNIEnv *getJNIEnv()
    {
      return _jnienv;
    }

  };

  typedef Luban::LBHashMap<string, JavaClassInfo*> ClassInfoMap;
  class JavaClassInfoCache : public ClassInfoMap
  {
  public:
    static JavaClassInfoCache* getCache()
    {
      static JavaClassInfoCache jcache;
      return &jcache;
    }
  private:
    JavaClassInfoCache()
      : ClassInfoMap( Luban::HashFunctions::stringHash, Luban::EqualFunctions::stringEqual, 8)
    {}
  };

  static int symbolHash(const LBSymbol& s) { return s.hash(); }
  static bool symbolEqual(const LBSymbol& s1, const LBSymbol& s2 ) { return s1==s2; }
  JavaClassInfo::JavaClassInfo(const string& clsname, jclass classobj)
    : _classname(clsname), _classobj(classobj), _constructors(), _methods(), _methodmap(symbolHash, symbolEqual, 6)
  {}

  JavaClassInfo *JavaClassInfo::newClassInfo(JNIEnv *env, const string& clsname, jobject clsobj)
  {
    JVMAndTools *jvmtools = getJVMAndTools();
    jobject classobj = clsobj;
    if ( !classobj )
      {
	jstring clsname_jstr = env->NewStringUTF(clsname.c_str());
	classobj = env->CallStaticObjectMethod(jvmtools->_java_lang_Class, jvmtools->_java_lang_Class_forName3Args, clsname_jstr, jboolean(true), jvmtools->_classloader );
	env->DeleteLocalRef(clsname_jstr);
      }
    if ( ! classobj )
      return 0;

    JavaClassInfo *result = new JavaClassInfo(clsname, (jclass)env->NewGlobalRef(classobj));

    // populate constructors
    jobjectArray  constructors = static_cast<jobjectArray>(env->CallObjectMethod(classobj, jvmtools->_java_lang_Class_getConstructors));
    if ( ! checkAndClearException(env, 0) )
      {
	int sz = env->GetArrayLength(constructors);
	for( int i=0; i<sz; ++i)
	  {
	    jobject oneobj = env->GetObjectArrayElement(constructors, i);
	    result->_constructors.push_back(env->NewGlobalRef(oneobj));
	    env->DeleteLocalRef(oneobj);
	  }
      }

    // populate methods
    jobjectArray  methods = static_cast<jobjectArray>(env->CallObjectMethod(classobj, jvmtools->_java_lang_Class_getMethods));
    env->DeleteLocalRef(classobj);
    if ( ! checkAndClearException(env, 0) )
      {
	int sz = env->GetArrayLength(methods);
	for( int i=0; i<sz; ++i)
	  {
	    jobject onemethod = env->GetObjectArrayElement(methods, i);
	    result->_methods.push_back(env->NewGlobalRef(onemethod));
	    jstring fdesc = (jstring)env->CallObjectMethod(onemethod, jvmtools->_java_lang_Object_toString);
	    string fdescstr = jstringToString(env, fdesc);
	    result->_method_descriptions.push_back(fdescstr);

	    jstring fname = (jstring)env->CallObjectMethod(onemethod, jvmtools->_java_lang_reflect_Method_getName);
	    const char *cstr = static_cast<const char*>(env->GetStringUTFChars(fname,0));
	    int idx = result->_methods.size()-1;
	    LBSymbol fnamesym(cstr);
	    MethodIDList *mids = 0;
	    if (! result->_methodmap.find(fnamesym, mids) )
	      {
		mids = new MethodIDList();
		result->_methodmap[fnamesym] = mids;
	      }
	    mids->push_back(idx);
	      
	    env->ReleaseStringUTFChars(fname, cstr);
	    env->DeleteLocalRef(fname);
	    env->DeleteLocalRef(fdesc);
	    env->DeleteLocalRef(onemethod);
	  }
      }

    return result;
  }

  const JavaClassInfo *JavaClassInfo::getClassInfo(JNIEnv *env, jobject obj)
  {
    JVMAndTools *jvmt = getJVMAndTools();
    jobject clsobj = env->CallObjectMethod(obj, jvmt->_java_lang_Object_getClass);
    jstring clsnamejstr = static_cast<jstring>(env->CallObjectMethod(clsobj, jvmt->_java_lang_Class_getName));
    string clsname = jstringToString(env, clsnamejstr);
    env->DeleteLocalRef(clsnamejstr);

    static JavaClassInfoCache *cache = JavaClassInfoCache::getCache();
    JavaClassInfo *clsinfo = 0;
    if ( cache->find(clsname, clsinfo) )
      return clsinfo;

    clsinfo = newClassInfo(env, clsname, clsobj);
    if ( clsinfo )
      (*cache)[clsname] = clsinfo;
    return clsinfo;

  }

  const JavaClassInfo *JavaClassInfo::getClassInfo(JNIEnv *env, const string& clsname)
  {
    static JavaClassInfoCache *cache = JavaClassInfoCache::getCache();
    JavaClassInfo *clsinfo = 0;
    if ( cache->find(clsname, clsinfo) )
      return clsinfo;
    
    clsinfo = newClassInfo(env, clsname);
    if ( clsinfo )
      (*cache)[clsname] = clsinfo;
    return clsinfo;
  }

  jobject JavaClassInfo::newInstance(JNIEnv *env, const LBVarArgs *args) const
  {
    int numargs = args?args->numArgs()-1:0;
    JVMAndTools *jvmtools = getJVMAndTools();
    // handle default instance
    if ( numargs <= 0 )
      {
	jobject newobj = env->CallObjectMethod( _classobj, jvmtools->_java_lang_Class_newInstance);
	string errmsg;
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to create Java object of type "+_classname+", error: "+errmsg);
	return newobj;
      }

    jobjectArray jargclss =0;
    jobjectArray jargs = lubanArgsToJavaObjectArray(env, args, &jargclss,1);
    
    // try to find the exact match
    jobject constructor = env->CallObjectMethod(_classobj, jvmtools->_java_lang_Class_getConstructor, jargclss);
    if ( checkAndClearException(env, 0) )
      {
	// then we have to try the constructors one by one
	for( int i=0; i<numConstructors(); ++i)
	  {
	    jobject newobj = callConstructor(env, i, jargs);
	    if ( ! checkAndClearException(env, 0) )
		return newobj;
	  }
	throw LBException("Can not find a matching constructor in Java type "+_classname);
      }

    jobject newobj = callConstructor(env, constructor, jargs);
    string errmsg;
    if ( checkAndClearException(env, &errmsg) )
      throw LBException("Failed to create Java object of type "+_classname+", error: "+errmsg);
    return newobj;
    
  }

  jobject JavaClassInfo::callConstructor(JNIEnv *env, jobject jcon, jobject jargs)
  {
    JVMAndTools *jvt = getJVMAndTools();
    return env->CallObjectMethod(jcon, jvt->_java_lang_reflect_Constructor_newInstance, jargs);
  }

  jobject JavaClassInfo::callMemberFunc(JNIEnv *env, jobject theobj, const LBSymbol& fname, const LBVarArgs* args) const
  {
    //    std::cout<<"obj "<<theobj<< ' ';
    MethodIDList *mds=0;
    if ( ! _methodmap.find(fname, mds) )
      throw LBException("No such member function "+fname.toString()+" for Java type "+_classname);

    JVMAndTools *jvmt = getJVMAndTools();

    jobject jargs = lubanArgsToJavaObjectArray(env, args);

    jthrowable err;
    string errmsg;
    string argerrmsg;
    string targeterrmsg;
    string othererrmsg;
    for(int i=0; i<mds->size(); ++i)
      {
	jobject method = _methods[(*mds)[i]];
	jobject result = env->CallObjectMethod(method, jvmt->_java_lang_reflect_Method_invoke, theobj, jargs);
	string oneerr;
	err = env->ExceptionOccurred();
	if ( ! err )
	  return result;
	env->ExceptionClear();
	if ( env->IsInstanceOf(err, jvmt->_java_lang_IllegalArgumentException) )
	  {
	    argerrmsg += getJavaErrorMessage(env, err);
	    env->DeleteLocalRef(err);
	  }
	else 	if ( env->IsInstanceOf(err, jvmt->_java_lang_reflect_InvocationTargetException) )
	  {
	    jthrowable cause = (jthrowable)env->CallObjectMethod(err, jvmt->_java_lang_Throwable_getCause);
	    if ( cause )
	      {
		targeterrmsg = getJavaErrorMessage(env, cause);
		env->DeleteLocalRef(cause);
	      }
	    else
	      targeterrmsg = getJavaErrorMessage(env, err);
	    env->DeleteLocalRef(err);
	    break;
	  }
	else
	  {
	    othererrmsg = getJavaErrorMessage(env, err);
	    env->DeleteLocalRef(err);
	    break;
	  }
      }

    if ( targeterrmsg.size() > 0 )
      errmsg = targeterrmsg;
    else
      if ( othererrmsg.size() > 0 )
	errmsg = othererrmsg;
      else
	errmsg = argerrmsg;

    throw LBException("Failed to call Java object member function "+fname.toString()+" error:"+errmsg);
    return 0;
  }

  LBMap* JavaClassInfo::allMemberFuncs() const
  {
    LBMap *result = new LBMap();
    MethodMap::Iterator it(_methodmap);
    while(it.next())
      {
	const MethodMap::Pair *pair = it.currentPair();
	const LBSymbol *key = pair->key;
	MethodIDList ids = **pair->value;
	string funcdesc;
	for(int i=0; i<ids.size(); ++i)
	  {
	    if ( i != 0 ) funcdesc += " ; ";
	    funcdesc += _method_descriptions[ids[i]];
	  }
	result->insert(new LBString(key->toString()), new LBString(funcdesc));
      }
    return result;
  }

  LBJavaObjRef *LBJavaUtil::buildJavaObj(const LBVarArgs* args)
  {
    int numargs = args?args->numArgs():0;
    if ( numargs == 0 || numargs == 1 && args->getArg(0) == 0 ) // return java null object
      return new LBJavaObjRef();

    // obtain a valid JNI env for current thread
    JVMThreadAttacher jatt;
    JNIEnv *jvmenv = jatt.getJNIEnv();

    // now it must be the format of javaobj("java.lang.Double", 3.14)
    const LBString *lbstr = dynamic_cast<const LBString*>(args->getArg(0));
    if ( ! lbstr )
      throw LBException("To build Java object reference, the first argument must be a java class name string");
    string clsname = lbstr->str();
    const JavaClassInfo *jclsinfo = JavaClassInfo::getClassInfo(jvmenv, clsname);
    if ( ! jclsinfo )
      throw LBException("Can not load java class:"+lbstr->str());

    jobject job1 = jclsinfo->newInstance(jvmenv, args);
    if ( ! job1 )
      throw LBException("Failed to construct java object from specified arguments");

    jobject job = jvmenv->NewGlobalRef(job1);
    return new LBJavaObjRef(job, jclsinfo);

  }
  
  LBJavaObjRef *LBJavaUtil::buildJavaObject(JNIEnv *env, jobject jobj)
  {
    if ( ! jobj )
      return 0;
    const JavaClassInfo *objclass = JavaClassInfo::getClassInfo(env, jobj);

    jobject jobjglobal = env->NewGlobalRef(jobj);
    return new LBJavaObjRef(jobjglobal, objclass);

  }

  string LBJavaUtil::toString(const LBJavaObjRef* lbj)
  {
    static string nullstr("null");
    static string jnullstr("Java null");
    if ( ! lbj )
      return nullstr;
    if ( lbj->isNull() )
      return jnullstr;

    // obtain a valid JNI env for current thread
    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();
    JVMAndTools *jvmt = getJVMAndTools();

    jobject theobj = lbj->getJavaObj();
    //   std::cout<<"obj "<<theobj<< ' ';

    jstring jstr = static_cast<jstring>(env->CallObjectMethod(theobj, jvmt->_java_lang_Object_toString));
    
    return jstringToString(env, jstr);
  }
  
  void LBJavaUtil::toStream(const LBJavaObjRef* lbj, ostream& ost)
  {
    jobject theobj = lbj?lbj->getJavaObj():0;

    if ( theobj == 0 )
      {
	ost<< "0 "; // indicate null java obj
	return;
      }

    // obtain a valid JNI env for current thread
    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();
    JVMAndTools *jvmt = getJVMAndTools();

    // serialize java obj into a byte array
    jobject bytestream = env->NewObject(jvmt->_java_io_ByteArrayOutputStream, jvmt->_java_io_ByteArrayOutputStream_constructor);
    jobject objstream = env->NewObject(jvmt->_java_io_ObjectOutputStream, jvmt->_java_io_ObjectOutputStream_constructor, bytestream);
    env->CallVoidMethod(objstream, jvmt->_java_io_ObjectOutputStream_writeObject, theobj);
    string errstr;
    if ( checkAndClearException(env, &errstr) )
      throw LBException("Failed to serialize java object, error: "+errstr);

    // then read the byte array back
    jbyteArray bytes = (jbyteArray)env->CallObjectMethod(bytestream, jvmt->_java_io_ByteArrayOutputStream_toByteArray);
    int sz = env->GetArrayLength(bytes);
    char buffer[sz];
    env->GetByteArrayRegion(bytes, 0, sz, (jbyte*)buffer);
    ost<<sz<<' '; // write out size first
    for(int i=0; i<sz; ++i)
      ost<<buffer[i];

 }

  void LBJavaUtil::fromStream(LBJavaObjRef *lbj, istream& ist)
  {
    int sz =0;
    ist >> sz;
    char sp;
    ist.get(sp);
    if ( sp != ' ' )
      throw LBException("Corrupted stream for Java object, can not get object size");

    if ( sz == 0 )
      {
	*lbj = LBJavaObjRef();
	return;
      }

    char buffer[sz];
    for(int i=0; i<sz ; ++i)
      ist.get(buffer[i]);

    // obtain a valid JNI env for current thread
    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();
    JVMAndTools *jvmt = getJVMAndTools();

    // construct a byte array and restore java object from it
    jbyteArray bytes = env->NewByteArray(sz);
    env->SetByteArrayRegion(bytes, 0, sz, (jbyte*)buffer);
    jobject bytestream = env->NewObject(jvmt->_java_io_ByteArrayInputStream, jvmt->_java_io_ByteArrayInputStream_constructor, bytes );
    jobject objstream = env->NewObject(jvmt->_java_io_ObjectInputStream, jvmt->_java_io_ObjectInputStream_constructor, bytestream);
    jobject theobj = env->CallObjectMethod(objstream, jvmt->_java_io_ObjectInputStream_readObject);
    string errstr;
    if ( checkAndClearException(env, &errstr) )
	throw LBException("Failed to restore java object, error: "+errstr);

    if ( ! theobj )
      {
	*lbj = LBJavaObjRef();
	return;
      }

    const JavaClassInfo *resultclass = JavaClassInfo::getClassInfo(env, theobj);

    jobject theobjglobal = env->NewGlobalRef(theobj);
    *lbj = LBJavaObjRef(theobjglobal, resultclass);

    return;

  }

  bool LBJavaUtil::equals(const LBJavaObjRef *x, const LBJavaObjRef *y)
  {
    if ( x == y )
      return true;

    if ( ! ( x && y ) )
      return false;

    jobject objx = x->getJavaObj();
    jobject objy = y->getJavaObj();

    if ( objx == objy )
      return true;

    if ( ! ( objx && objy ) )
      return false;

    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();
    JVMAndTools *jvmt = getJVMAndTools();

    return env->CallBooleanMethod(objx, jvmt->_java_lang_Object_equals, objy);

  }

  bool LBJavaUtil::handleCommonJavaUtilCall(JNIEnv* env, jobject javaobj, const Luban::LBSymbol& funcname, const Luban::LBVarArgs* args, jobject& result )
  {
    static const LBSymbol java_getClassForName("java_getClassForName");
    static const LBSymbol java_callStaticFunction("java_callStaticFunction");
    static const LBSymbol java_getStaticField("java_getStaticField");
    static const LBSymbol java_setStaticField("java_setStaticField");
    static const LBSymbol java_getField("java_getField");
    static const LBSymbol java_setField("java_setField");
    static const LBSymbol java_instanceof("java_instanceof");

    int argsz = args?args->numArgs():0;

    if ( funcname == java_getClassForName )
      {
	if ( argsz != 1 )
	  throw LBException("java::javaobj::java_getClassForName() require one string argument as class name");
	const LBObject* argone = args->getArg(0);
	const LBString* argonestr = dynamic_cast<const LBString*>(argone);
	if ( ! argonestr )
	  throw LBException("java::javaobj::java_getClassForName() require one string argument as class name");
	string clsname = argonestr->toString();
	const JavaClassInfo *jclassinfo = JavaClassInfo::getClassInfo(env, clsname);
	if ( ! jclassinfo )
	  throw LBException("Can not find Java class "+clsname);
	result = jclassinfo->getJavaClassObj();
	return true;
      }
    
    if ( funcname == java_callStaticFunction )
      {
	if ( argsz < 2 )
	  throw LBException("java::javaobj::java_callStaticFunction() require two or more arguments");
	const LBObject* argone = args->getArg(0);
	const LBObject* argtwo = args->getArg(1);
	const LBString* argonestr = dynamic_cast<const LBString*>(argone);
	const LBString* argtwostr = dynamic_cast<const LBString*>(argtwo);
	if ( ! argtwostr || !argonestr )
	  throw LBException("java::javaobj::java_callStaticFunction() require first two arguments be class name and function name strings");
	string clsname = argonestr->toString();
	const JavaClassInfo *jclassinfo = JavaClassInfo::getClassInfo(env, clsname);
	if ( ! jclassinfo )
	  throw LBException("Can not find Java class "+clsname);
	LBSymbol javafuncname(argtwostr->toString());
	
	Luban::LBVarArgsShift shiftedargs(2, args);
	result = jclassinfo->callMemberFunc(env, 0, javafuncname, &shiftedargs);
	return true;
      }

    JVMAndTools *jvmt = getJVMAndTools();
	
    if ( funcname == java_instanceof )
      {
	if ( argsz != 1 )
	  throw LBException("java::instanceof() require one single arguement as class name string or java class object");
	
	const LBObject *argone=args->getArg(0);
	string errmsg;
	jclass jcls=object2jclass(env, argone, errmsg);
	if ( !jcls )
	  throw LBException("java::instanceof() failed to find class: "+errmsg);

	jboolean resbool = false;
	if ( javaobj )
	  resbool = env->IsInstanceOf(javaobj, jcls);

	jobject jobjbool = env->NewObject(jvmt->_java_lang_Boolean, jvmt->_java_lang_Boolean_initBoolean, resbool);
	if ( !checkAndClearException(env, &errmsg) )
	  {
	    result = jobjbool;
	    return true;
	  }
	throw LBException("Exception when creating new Boolean object:"+errmsg);
      }

    if ( funcname == java_getField )
      {
	if ( argsz != 1 )
	  throw LBException("java::javaobj::java_getField() require one single arguement as field name string");
	
	const LBObject *argone=args->getArg(0);
	jstring jstr = object2jstring(env, argone);
	if ( ! jstr )
	  throw LBException("java::javaobj::java_setField() requires first arguement be field name string");

	jclass jclz = (jclass)env->CallObjectMethod(javaobj, jvmt->_java_lang_Object_getClass);
	string errmsg;
	if ( !jclz || checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call getClass method on object: "+errmsg);

	jobject fieldobj = env->CallObjectMethod(jclz, jvmt->_java_lang_Class_getField, jstr);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call getField method on object: "+errmsg);

	jobject fieldval = env->CallObjectMethod(fieldobj, jvmt->_java_lang_reflect_Field_get, javaobj);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call get object from a java Field object: "+errmsg);

	result = fieldval;
	    
	env->DeleteLocalRef(fieldobj);

	return true;
      }

    if ( funcname == java_setField )
      {
	if ( argsz != 2 )
	  throw LBException("java::javaobj::java_setField() require two arguements, field name string and field value");

	const LBObject *argone=args->getArg(0);
	jstring jstr = object2jstring(env, argone);
	if ( ! jstr )
	  throw LBException("java::javaobj::java_setField() requires first arguement be field name string");
	const LBObject *argtwo=args->getArg(1);
	jobject fieldval = lubanToJava(env, argtwo);

	jclass jclz = 0;
	jclz = (jclass)env->CallObjectMethod(javaobj, jvmt->_java_lang_Object_getClass);
	string errmsg;
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call getClass method on object: "+errmsg);

	jobject fieldobj = env->CallObjectMethod(jclz, jvmt->_java_lang_Class_getField, jstr);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call getField method on object: "+errmsg);

	env->CallVoidMethod(fieldobj, jvmt->_java_lang_reflect_Field_set, javaobj, fieldval);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call set() on a java Field object: "+errmsg);

	result = 0;

	env->DeleteLocalRef(fieldobj);
	    
	return true;
      }

    if ( funcname == java_getStaticField )
      {
	if ( argsz != 2 )
	  throw LBException("java::javaobj::java_getStaticField() require two arguements, class name and field name");

	const LBObject *argone=args->getArg(0);
	string errmsg;
	jclass jclz = object2jclass(env, argone, errmsg);
	if ( !jclz )
	  throw LBException("java::javaobj::java_getStaticField() failed to find class: "+errmsg);

	const LBObject *argtwo=args->getArg(1);
	jstring fieldname=object2jstring(env, argtwo);
	if ( ! fieldname )
	  throw LBException("java::javaobj::java_getStaticField() requires second arguement be field name string");

	jobject fieldobj = env->CallObjectMethod(jclz, jvmt->_java_lang_Class_getField, fieldname);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call get static field: "+errmsg);

	jobject fieldval = env->CallObjectMethod(fieldobj, jvmt->_java_lang_reflect_Field_get, 0);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call get object from a java Field object: "+errmsg);

	result = fieldval;
	    
	env->DeleteLocalRef(fieldobj);

	return true;

      }
	  

    if ( funcname == java_setStaticField )
      {
	if ( argsz != 3 )
	  throw LBException("java::javaobj::java_setStaticField() require three arguements, class name, field name and new field value");

	const LBObject *argone=args->getArg(0);
	string errmsg;
	jclass jclz = object2jclass(env, argone, errmsg);
	if ( !jclz )
	  throw LBException("java::javaobj::java_getStaticField() failed to find class: "+errmsg);

	const LBObject *argtwo=args->getArg(1);
	jstring fieldname = object2jstring(env, argtwo);
	if ( ! fieldname )
	  throw LBException("java::javaobj::java_getStaticField() second arguement has to be field name string");

	jobject fieldobj = env->CallObjectMethod(jclz, jvmt->_java_lang_Class_getField, fieldname);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call get static field: "+errmsg);

	const LBObject *argthree=args->getArg(2);
	jobject fieldval = lubanToJava(env, argthree);

	env->CallVoidMethod(fieldobj, jvmt->_java_lang_reflect_Field_set, 0, fieldval);
	if ( checkAndClearException(env, &errmsg) )
	  throw LBException("Failed to call set() on a java Field object: "+errmsg);

	result = 0;
	    
	env->DeleteLocalRef(fieldobj);

	return true;

      }
	  

    return false;

  }

  LBObject* LBJavaUtil::callMemberFunc(LBJavaObjRef *lbj, const Luban::LBSymbol& funcname, const Luban::LBVarArgs* args )
  {
    if ( ! lbj )
      throw LBException("Can not call member function on null object");

    jobject javaobj = lbj->getJavaObj();

    // obtain a valid JNI env for current thread
    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();

    jobject javaresult=0;

    // handle common func for all Java obj here
    if ( handleCommonJavaUtilCall(env, javaobj, funcname, args, javaresult) )
      return buildJavaObject(env, javaresult);

    const JavaClassInfo *classinfo = lbj->_imp->getJavaClass();

    javaresult = classinfo->callMemberFunc(env, javaobj, funcname, args);

    return buildJavaObject(env, javaresult);

  }

  Luban::LBMap *LBJavaUtil::allMemberFuncs(const LBJavaObjRef* lbj)
  {
    static const LBString java_getClassForName("java_getClassForName"),java_getClassForNameDesc("javaobj/Class javaobj::getClassForName(string classname)") ;
    static const LBString java_callStaticFunction("java_callStaticFunction"),java_callStaticFunctionDesc("javaobj javaobj::java_callStaticFunction(string classname, string funcname, object arg1, arg2,...)");
    static const LBString java_getStaticField("java_getStaticField"),java_getStaticFieldDesc("javaobj javaobj::java_getStaticField(string classname, string fieldname)");
    static const LBString java_setStaticField("java_setStaticField"),java_setStaticFieldDesc("void javaobj::java_setStaticField(string classname, string fieldname, object value)");
    static const LBString java_getField("java_getField"),java_getFieldDesc("javaobj javaobj::java_getField(string fieldname)");
    static const LBString java_setField("java_setField"),java_setFieldDesc("void javaobj::java_setField(string fieldname, object value)");
    static const LBString java_instanceof("java_instanceof"),java_instanceofDesc("javaobj/Boolean java_instanceof(<string, Class> class)");

    Luban::LBMap *funcs = 0;
    if ( lbj && ! lbj->isNull())
      {
	const JavaClassInfo *classinfo = lbj->_imp->getJavaClass();
	funcs = classinfo->allMemberFuncs();
      }
    else
      funcs = new Luban::LBMap();

    funcs->insert(java_getClassForName, java_getClassForNameDesc);
    funcs->insert(java_callStaticFunction, java_callStaticFunctionDesc);
    funcs->insert(java_getStaticField,java_getStaticFieldDesc);
    funcs->insert(java_setStaticField,java_setStaticFieldDesc);
    funcs->insert(java_getField, java_getFieldDesc);
    funcs->insert(java_setField, java_setFieldDesc);
    funcs->insert(java_instanceof, java_instanceofDesc);

    return funcs;
  }

  int LBJavaUtil::hashCode(const LBJavaObjRef* lbj)
  {
    if ( ! lbj || lbj->isNull())
      return 0;

    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();

    JVMAndTools *jvmt = getJVMAndTools();
    return env->CallIntMethod(lbj->getJavaObj(), jvmt->_java_lang_Object_hashCode);    
  }

  void LBJavaUtil::releaseJavaObjRef(jobject javaobj)
  {
    if ( !javaobj )
      return;
    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();

    env->DeleteGlobalRef(javaobj);

  }

  bool LBJavaUtil::cast(const LBJavaObjRef* lbj, LBObject *target)
  {
    if ( ! lbj || lbj->isNull() || ! target  )
      return false;

    JVMThreadAttacher jatt;
    JNIEnv *env = jatt.getJNIEnv();
    JVMAndTools *jvmt = getJVMAndTools();

    jobject jobj = lbj->getJavaObj();
    jobject javaclassobj = lbj->_imp->getJavaClass()->_classobj;
    bool isjavanumber = env->IsInstanceOf(jobj, jvmt->_java_lang_Number);

    LBDouble *lbdouble = dynamic_cast<LBDouble*>(target);
    if ( lbdouble )
      {
	if ( ! isjavanumber ) return false;
	jdouble d = env->CallDoubleMethod(jobj, jvmt->_java_lang_Number_doubleValue);
	*lbdouble = LBDouble(d);
	return true;
      }

    LBInt *lbint = dynamic_cast<LBInt*>(target);
    if ( lbint )
      {
	if ( ! isjavanumber ) return false;
	jint d = env->CallIntMethod(jobj, jvmt->_java_lang_Number_intValue);
	*lbint = LBInt(d);
	return true;
      }
    LBChar *lbchar = dynamic_cast<LBChar*>(target);
    if ( lbchar )
      {
	if ( isjavanumber )
	  {
	    jbyte d = env->CallByteMethod(jobj, jvmt->_java_lang_Number_byteValue);
	    *lbchar = LBChar(d);
	    return true;
	  }
	bool isjavachar = env->IsInstanceOf(jobj, jvmt->_java_lang_Character);
	if ( isjavachar )
	  {
	    jchar jc = env->CallCharMethod(jobj, jvmt->_java_lang_Character_charValue);
	    *lbchar = LBChar(char(jc));
	    return true;
	  }
      }

    LBBool *lbbool = dynamic_cast<LBBool*>(target);
    if ( lbbool )
      {
	bool isjavabool = env->IsInstanceOf(jobj, jvmt->_java_lang_Boolean);
	if ( isjavabool )
	  {
	    bool jb = env->CallBooleanMethod(jobj, jvmt->_java_lang_Boolean_booleanValue);
	    *lbbool = LBBool(jb);
	    return true;
	  }
      }

    // do shallow convert for container types
    LBVector *lbvector = dynamic_cast<LBVector*>(target);
    if ( lbvector )
      {
	bool isarray = env->CallBooleanMethod(javaclassobj, jvmt->_java_lang_Class_isArray);
	if ( isarray )
	  {
	    int sz = env->CallStaticIntMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_getLength, jobj);
	    for(int i=0; i<sz; ++i)
	      {
		jobject oneobj = env->CallStaticObjectMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_get, jobj, i);
		LBJavaObjRef *oneref = buildJavaObject(env, oneobj);
		lbvector->insert(oneref);
		env->DeleteLocalRef(oneobj);
	      }
	    return true;
	  }
	
	bool isvector = env->IsInstanceOf(jobj, jvmt->_java_util_List);
	if ( isvector )
	  {
	    int sz = env->CallIntMethod(jobj, jvmt->_java_util_List_size);
	    for(int i=0; i<sz; ++i)
	      {
		jobject oneobj = env->CallObjectMethod(jobj, jvmt->_java_util_List_get, i);
		LBJavaObjRef *oneref = buildJavaObject(env, oneobj);
		lbvector->insert(oneref);
		env->DeleteLocalRef(oneobj);
	      }
	    return true;
	  }
      }

    LBMap *lbmap = dynamic_cast<LBMap*>(target);
    if ( lbmap )
      {
	bool ismap =  env->IsInstanceOf(jobj, jvmt->_java_util_Map);
	if ( ismap )
	  {
	    jobject entryset = env->CallObjectMethod(jobj, jvmt->_java_util_Map_entrySet);
	    jobject entryarray = env->CallObjectMethod(entryset, jvmt->_java_util_Set_toArray);
	    int sz = env->CallStaticIntMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_getLength, entryarray);
	    string errstr;
	    if ( !checkAndClearException(env, &errstr) )
	      {
		std::vector<LBJavaObjRef*> keyvec(sz);
		std::vector<LBJavaObjRef*> valvec(sz);
		for(int i=0; i<sz; ++i)
		  {
		    jobject oneentry = env->CallStaticObjectMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_get, entryarray, i);
		    if ( checkAndClearException(env, &errstr) ) break;
		    jobject onekey = env->CallObjectMethod(oneentry, jvmt->_java_util_Map_Entry_getKey);
		    if ( checkAndClearException(env, &errstr) ) break;
		    jobject oneval = env->CallObjectMethod(oneentry, jvmt->_java_util_Map_Entry_getValue);
		    //		    jobject oneval = env->CallObjectMethod(jobj, jvmt->_java_util_Map_get, onekey);
		    if ( checkAndClearException(env, &errstr) ) break;

		    LBJavaObjRef *lbkey = buildJavaObject(env, onekey);
		    LBJavaObjRef *lbval = buildJavaObject(env, oneval);
		    env->DeleteLocalRef(onekey);
		    env->DeleteLocalRef(oneval);
		    
		    keyvec[i]=lbkey;
		    valvec[i]=lbval;

		  }
		// keys are vals have to be saved in vec and insert into map once
		// because inserting will call hascode() function and the hashcode
		// function will detach JVM thread upon finishing
		// so no use of "env" after this loop, return
		for(int j=0; j<keyvec.size(); ++j)
		  lbmap->insert((LBJavaObjRef*)keyvec[j], (LBJavaObjRef*)valvec[j]);

		return true;
	      }

	    lbmap->insert(new LBString(errstr), new LBString(errstr));

	    return true;
	  }
      }

    LBSet *lbset = dynamic_cast<LBSet*>(target);
    if ( lbset )
      {
	bool isset =  env->IsInstanceOf(jobj, jvmt->_java_util_Set);
	if ( isset )
	  {
	    jobject entryarray = env->CallObjectMethod(jobj, jvmt->_java_util_Set_toArray);
	    int sz = env->CallStaticIntMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_getLength, entryarray);
	    string errstr;
	    if ( !checkAndClearException(env, &errstr) )
	      {
		std::vector<LBJavaObjRef*> keyvec(sz);
		for(int i=0; i<sz; ++i)
		  {
		    jobject onekey = env->CallStaticObjectMethod(jvmt->_java_lang_reflect_Array, jvmt->_java_lang_reflect_Array_get, entryarray, i);
		    if ( checkAndClearException(env, &errstr) ) break;

		    LBJavaObjRef *lbkey = buildJavaObject(env, onekey);
		    env->DeleteLocalRef(onekey);
		    
		    keyvec[i]=lbkey;

		  }
		// keys are vals have to be saved in vec and insert into set once
		// because inserting will call hascode() function and the hashcode
		// function will detach JVM thread upon finishing
		// so no use of "env" after this loop, return immediately
		for(int j=0; j<keyvec.size(); ++j)
		  lbset->insert((LBJavaObjRef*)keyvec[j]);

		return true;
	      }

	    lbset->insert(new LBString(errstr));

	    return true;
	  }
      }

    return false;

  }



}
