#include <iostream>
#include <string>

#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/LBObjPtr.hpp"

#include "luban/lbstationarystruct.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/lbstructinterface.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/lblocalpropertystorage.hpp"
#include "luban/lbrwlockedobjptr.hpp"
#include "luban/lbnamedargs.hpp"
#include "luban/lubantypechecking.hpp"
#include "luban/lubanpermissionchecking.hpp"
#include "luban/lbcounterwaiter.hpp"



namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBStationaryStruct, 1, 0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBStationaryStruct)

  LBStationaryStruct::~LBStationaryStruct()
  { delete _stores; }

  LBStationaryStruct::LBStationaryStruct()
    :_metadata(), _stores(0)
  {}

  LBStationaryStruct::LBStationaryStruct(const LBStationaryStruct& s)
    : _metadata(s._metadata), _stores(0)
  { 
    if ( s._stores )
      _stores = new LocalPropertyStorage(*s._stores);
    else
      initLocalProperties();
  }

  
  LBStationaryStruct::LBStationaryStruct(LBStructInterface* itfc)
    : _metadata( new STMetaData(itfc) ), _stores(0)
  {}

  void LBStationaryStruct::initLocalProperties()
  {
    if ( _stores )
      return;
    int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
    if ( numstores )
      {
	_stores = new LocalPropertyStorage(numstores, true);
	for( int i = 0; i< numstores; i++)
	  {
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STORE);
	    const LBProcessStruct *initp = pinfo->initProcess();
	    if ( ! initp )
	      continue;
	    std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	    _stores->writeValue(i, torun->evalForLastObj());
	  }
      }
  }

  string LBStationaryStruct::toString() const
  {
    string result("stationary struct ");
    result += _metadata->_itfc->name()->toString()+' ';

    int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
    if ( numstores > 0 )
      {
	result += "store( ";
	for( int i = 0; i < numstores ; i++ )
	  {
	    result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STORE)->name().toString()+'=';
	    LBObjPtr val =_stores->readValue(i); 
	    result += val.toString()+' ';
	  }
	  result += ") ";
      }

    int numstatics = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	result += "static( ";
	for( int i = 0; i < numstatics ; i++ )
	  {
	    result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STATIC)->name().toString()+'=';
	    LBObjPtr val =_metadata->_statics->readValue(i); 
	    result += val.toString()+' ';
	  }
	result += ") ";
      }

    return result;

  }

  ostream& LBStationaryStruct::toStream( ostream& ost ) const
  { 
    _metadata->_itfc->name()->toStream(ost);

    // then stream the state of struct only
    int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
    ost<<numstores<< ' ';
    for( int i = 0; i < numstores; i++)
      {
	_metadata->_itfc->propertyInfo(i,LBPropertyInfo::STORE)->name().toStream(ost);
	ost.put(' ');
	LBObjPtr valp = _stores->readValue(i);
	if ( valp )
	  {
	    ost.put('1'); ost.put(' ');
	    LBObject::instanceToStream(ost, *valp);
	  }
	else
	  {
	    ost.put('0'); 
	    ost.put(' ');
	  }
      }

    return ost;

  }

  istream& LBStationaryStruct::fromStream( istream& ist, int major, int minor )
  {
    if ( _stores ) 
      {
	delete _stores;
	_stores = 0;
      }

    // recover metadata
    LBFullSymbol stname;
    stname.fromStream(ist);
    string errs;
    LBStruct * st = LubanSymbolResolver::resolveStructSymbol(stname, errs);
    if ( ! st )
      throw(LBException("Can not find struct type "+stname.toString()+" when recovering struct from stream: "+errs));
    LBStationaryStruct *pst = dynamic_cast<LBStationaryStruct*>(st);
    if ( ! pst )
      throw(LBException("Struct type "+stname.toString()+" is not a stationary as expected when recovering struct from stream: "));
    _metadata = pst->_metadata;
    
    // recover state
    int numstores=0;
    ist>>numstores;
    char sp, oneorzero;
    ist.get(sp);
    if ( numstores )
      {
	initLocalProperties();
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover stationary struct from stream") );
	for( int i = 0; i < numstores; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( !pinfo || pinfo->attr() != LBPropertyInfo::STORE )
	      throw( LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as store as expected") );

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover stationary struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover stationary struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover property "+pname.toString()+" from stream") );
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());
		
		_stores->writeValue(index, LBObjPtr(obj));
	      }
	    else
	      _stores->writeValue(index, LBObjPtr());		
	  }

      }

    return ist;

  }

  bool LBStationaryStruct::equals(const LBObject& another) const
  { return this == &another; }

  const LBStructInterface& LBStationaryStruct::interface() const
  { return *_metadata->_itfc;  }

  LBStationaryStruct* LBStationaryStruct::structCall(const LBNamedArgs* args) const
  {
    LBStationaryStruct *evalcopy = new LBStationaryStruct(*this);
    try {
      evalcopy->initLocalProperties();
      evalcopy->setBatchProperties(args);
    }
    catch( LBException &lbe)
      {
	delete evalcopy;
	throw lbe;
      }
    return evalcopy;
  }

  void LBStationaryStruct::setBatchProperties(const LBNamedArgs* args)
  { 
    int sz = args?args->numArgs() : 0;
    string errs;
    for( int i = 0; i < sz; i++ )
      {
	const LBObject *argobj = args->getArg(i);
	const LBSymbol *argname = args->getName(i);
	int argindex = 0;
	const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(*argname, argindex);
	if ( ! pinfo  )
	  throw( LBException("The struct instance does not have the property: "+argname->toString()) );

	// property setting policy and type checking here
	if (  LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
	  {
	    if ( argobj )
	      {
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), argobj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+argobj->getType().toString());

		_stores->writeValue(argindex, *argobj);
	      }
	    else
	      _stores->writeValue(argindex, LBObjPtr());
	  }
	else
	  throw( LBException( "The property: "+argname->toString()+ " does not allow setting from outside") );
      }
  }

  LBObjPtr LBStationaryStruct::getPropertyValue(const LBSymbol& s)
  { 
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw( LBException("Failed to pass read permission check for property value reading: "+errs) );
    return _stores->readValue( pindex );

  }

  LBObject* LBStationaryStruct::getPropertyPtr(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH,  errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw( LBException("Failed to pass permission check for property mutation: "+errs) );

    return _stores->getUnsharedPtr( pindex );
  }

  void LBStationaryStruct::touchProperty(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    return;
  }

  void LBStationaryStruct::setProperty(const LBSymbol& s, const LBObject& val)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), val );
  }

  void LBStationaryStruct::setProperty(const LBSymbol& s, const LBObjPtr& valptr)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), valptr );
  }

  LBObjPtr LBStationaryStruct::getPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  { 
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw LBException("Invalid property index");
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Failed to pass read permission check for property value reading: "+errs);
    return _stores->readValue( index );
  }

  LBObject* LBStationaryStruct::getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);
    return _stores->getUnsharedPtr( index );
  }

  void LBStationaryStruct::touchProperty(int index, LBPropertyInfo::ExecAttr attr)
  {
    return;
  }

  void LBStationaryStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);
    
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    _stores->writeValue(index,val);
  }

  void LBStationaryStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );

    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw( LBException("No write permission check for property setting: "+errs) );
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr()) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    _stores->writeValue(index,valptr);

  }

  bool LBStationaryStruct::addPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].inc();
  }

  bool LBStationaryStruct::removePending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].dec();
  }

  bool LBStationaryStruct::waitUntilNoPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].waitDownToZero();
  }


  LBObjPtr LBStationaryStruct::getStaticPropertyValue(int index)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Read permission violation: "+errs);
    return _metadata->_statics->readValue(index);
  }

  void LBStationaryStruct::setStaticPropertyValue(int index, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _metadata->_statics->writeValue(index, val);
  }

  void LBStationaryStruct::setStaticPropertyValue(int index, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr() ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _metadata->_statics->writeValue(index, valptr);
  }

  LBWriteLockedObjPtr LBStationaryStruct::getWriteLockedStaticProperty(int staticprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(staticprpindex, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _metadata->_statics->getWriteLockedObj(staticprpindex);
  }

  LBWriteLockedObjPtr LBStationaryStruct::getWriteLockedStoreProperty(int storeprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(storeprpindex, LBPropertyInfo::STORE );
    if ( pinfo == 0 )
      throw( LBException("Invalid store property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _stores->getWriteLockedObj(storeprpindex);
  }

  LBStationaryStruct::STMetaData::STMetaData(LBStructInterface *itfc)
    : _itfc(itfc), _statics(0), _staticscounters(0)
  {
    int numstatics = itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	_statics = new LocalPropertyStorage(numstatics, true);
	_staticscounters = new CounterWaiter[numstatics];
	// init static properties
	for(int i=0; i<numstatics; i++)
	  {
	    const LBProcessStruct *initp = itfc->propertyInfo(i, LBPropertyInfo::STATIC)->initProcess();
	    if ( ! initp )
	      continue;
	    std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	    _statics->writeValue(i, torun->evalForLastObj());
	  }
      }
  }


  LBStationaryStruct::STMetaData::~STMetaData()
  {
    delete _itfc;
    delete _statics;
    delete [] _staticscounters;
  }


  //The below to export member functions for Luban's reflection mechanism
  // actually functions defined at lbstruct level
  LBEXPORT_MEMBER_FUNC(Luban::LBStationaryStruct, luban_set, "pset", "void set(map kvpairss),set(string key, object value),set(vector keys, vector values)" ); 
  LBEXPORT_MEMBER_FUNC(Luban::LBStationaryStruct, luban_get, "pget", "object get(string key),vector get(vector keys)" ); 

}
