#include "lbthread/lbthread.hpp"

////////////////////////////////////////////////////////////////////////////
//
// This module is platform dependent implementation
//
////////////////////////////////////////////////////////////////////////////

#include <pthread.h> // this is POSIX, including Linux
#include <errno.h>

namespace Luban
{

  class LBThread::LBThreadImp
  {
    // Here is POSIX implementation
  public:
    LBThreadImp(LBRunable *r);
    ~LBThreadImp();
  public:
    pthread_t _tid;
    int _status;

    static void* threadEntranceFunc(void* runable)
    {
      LBRunable *torun = static_cast<LBRunable*>(runable);
      torun->run();
      return 0;
    }

  };

  // POSIX implementation
  LBThread::LBThreadImp::LBThreadImp(LBRunable *r)
    :_tid(), _status(-1)
  {
    _status = pthread_create(&_tid, 0, threadEntranceFunc, static_cast<void*>(r));
  }

  LBThread::LBThreadImp::~LBThreadImp()
  {
  }

  LBThread::LBThread(LBRunable *payload)
    : _threadimp(0)
  {
    _threadimp = new LBThreadImp(payload);
  }

  LBThread::~LBThread()
  {
    delete _threadimp;
  }

  bool LBThread::join()
  {
    //POSIX
    return _threadimp->_status == 0 && pthread_join(_threadimp->_tid, 0) == 0;
  }

  bool LBThread::detach()
  {
    //POSIX
    return _threadimp->_status == 0 && pthread_detach(_threadimp->_tid) == 0;
  }

  bool LBThread::cancel()
  {
    //POSIX
    return _threadimp->_status == 0 && pthread_cancel(_threadimp->_tid) == 0;
  }

  bool LBThread::isValid() const
  {
    //POSIX
    return _threadimp->_status == 0;
  }


  // Mutex class
  class LBMutex::LBMutexImp
  {
  public:
    LBMutexImp();
    ~LBMutexImp();

  public:
    // POSIX
    pthread_mutex_t _pthreadmutex;
    int _status;
  };

  // POSIX
  LBMutex::LBMutexImp::LBMutexImp()
    :_pthreadmutex()
  {
    _status = pthread_mutex_init(&_pthreadmutex,0);
  }
  //POSIX
  LBMutex::LBMutexImp::~LBMutexImp()
  {
    if ( _status == 0 ) pthread_mutex_destroy(&_pthreadmutex);
  }

  LBMutex::LBMutex()
    :_imp( new LBMutexImp() )
  {}

  LBMutex::~LBMutex()
  { delete _imp; }

  bool LBMutex::lock()
  {
    // POSIX
    return (_imp->_status == 0 ) && pthread_mutex_lock(&_imp->_pthreadmutex) == 0;
  }

  bool LBMutex::unlock()
  {
    // POSIX
    return (_imp->_status == 0 ) && pthread_mutex_unlock(&_imp->_pthreadmutex) == 0;
  }

  bool LBMutex::try_lock()
  {
    // POSIX
    return  (_imp->_status == 0 ) && pthread_mutex_trylock(&_imp->_pthreadmutex) == 0;
  }


  // Condition Variable
  class LBCondVar::LBCondVarImp
  {
  public:
    LBCondVarImp();
    ~LBCondVarImp();

  public:
    //POSIX
    pthread_cond_t _pthreadcond;
    int _status;
  };

  LBCondVar::LBCondVarImp::LBCondVarImp()
    : _pthreadcond()
  {
    //POSIX
    _status = pthread_cond_init(&_pthreadcond, 0);
  }

  LBCondVar::LBCondVarImp::~LBCondVarImp()
  {
    //POSIX
    if ( _status == 0 ) pthread_cond_destroy(&_pthreadcond);
  }

  LBCondVar::LBCondVar()
    : _imp( new LBCondVarImp() )
  {}

  LBCondVar::~LBCondVar()
  { delete _imp; }

  bool LBCondVar::wait(LBMutex& mut)
  {
    // POSIX
    return (_imp->_status == 0 ) && pthread_cond_wait( &_imp->_pthreadcond, &mut._imp->_pthreadmutex) == 0;
  }

  bool LBCondVar::signal()
  {
    //POSIX
    return (_imp->_status == 0 ) && pthread_cond_signal( &_imp->_pthreadcond ) == 0;
  }

  bool LBCondVar::broadcast()
  {
    //POSIX
    return (_imp->_status == 0 ) && pthread_cond_broadcast( &_imp->_pthreadcond ) == 0;
  }



  // RW lock class, a platform independent implementation
  class LBReadWriteLock::LBReadWriteLockImp
  {
  public:
    LBReadWriteLockImp(Preference p);
    ~LBReadWriteLockImp();

  public:
    LBMutex _rwmut;
    LBCondVar _readergo;
    LBCondVar _writergo;
    int _activereaders;
    bool _activewriter;
    int _waitingreaders;
    int _waitingwriters;
    Preference _pref;
    bool _goread;
  };

  LBReadWriteLock::LBReadWriteLockImp::LBReadWriteLockImp(Preference p)
    : _rwmut(), _readergo(), _writergo(), _activereaders(0), _activewriter(false), _waitingreaders(0), _waitingwriters(0), _pref(p), _goread(false)
  {}

  LBReadWriteLock::LBReadWriteLockImp::~LBReadWriteLockImp()
  {}

  LBReadWriteLock::LBReadWriteLock(Preference p)
    : _imp( new LBReadWriteLockImp(p) )
  {}

  LBReadWriteLock::~LBReadWriteLock()
  { delete _imp; }

  bool LBReadWriteLock::readLock()
  {
    bool ok=true;
    LBMutexLocker mutlocker( _imp->_rwmut );
    while (  _imp->_activewriter )
      {
	_imp->_waitingreaders++;
	ok = _imp->_readergo.wait( _imp->_rwmut );
	_imp->_waitingreaders--;
	if ( ! ok )
	  return false;
      }
    _imp->_activereaders++;
    return true;
  }

  bool LBReadWriteLock::writeLock()
  {
    bool ok=true;
    LBMutexLocker mutlocker( _imp->_rwmut );
    while (  _imp->_activewriter || _imp->_activereaders )
      {
	_imp->_waitingwriters++;
	ok = _imp->_writergo.wait( _imp->_rwmut );
	_imp->_waitingwriters--;
	if ( ! ok )
	  return false;
      }
    _imp->_activewriter = true;
    return true;
  }


  bool LBReadWriteLock::unlock()
  {
    bool ok=true;
    LBMutexLocker mutlocker( _imp->_rwmut );
    
    if (  _imp->_activewriter )
      _imp->_activewriter = false;
    else 
      if ( _imp->_activereaders )
	_imp->_activereaders--;
      else
	return false;

    if ( _imp->_pref == READER )
      {
	if ( _imp->_waitingreaders )
	  ok = _imp->_readergo.broadcast();
	else
	  if ( _imp->_waitingwriters )
	    ok = _imp->_writergo.signal();
      }
    else
      if ( _imp->_pref == WRITER )
	{
	  if ( _imp->_waitingwriters )
	    ok = _imp->_writergo.signal();
	  else
	    if ( _imp->_waitingreaders )
	      ok = _imp->_readergo.broadcast();
	}
      else  // EQUAL for read/write
	{
	  if ( _imp->_goread )
	    {
	      if ( _imp->_waitingreaders )
		ok = _imp->_readergo.broadcast();
	      else
		if ( _imp->_waitingwriters )
		  ok = _imp->_writergo.signal();
	    }
	  else
	    {
	      if ( _imp->_waitingwriters )
		ok = _imp->_writergo.signal();
	      else
		if ( _imp->_waitingreaders )
		  ok = _imp->_readergo.broadcast();
	    }

	  _imp->_goread = ! _imp->_goread; // turn the table

	}

    return ok;
  }
 

  bool LBReadWriteLock::try_readLock()
  {
    LBMutexLocker mutlocker( _imp->_rwmut );
    if (  _imp->_activewriter )
      return false;
    _imp->_activereaders++;
    return true;
  }

  bool LBReadWriteLock::try_writeLock()
  {
    LBMutexLocker mutlocker( _imp->_rwmut );
    if (  _imp->_activewriter || _imp->_activereaders )
      return false;
    _imp->_activewriter = true;
    return true;
  }
 

}
