#include <string>
#include <iostream>

#include "lbthread/lbthread.hpp"

#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbfullsymbol.hpp"
#include "lbtypes/HashUtil.hpp"

#include "luban/luban_namespace.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/lbcomponentinfo.hpp"

namespace Luban
{
  using std::string;

  static int symbolhash(const LBSymbol& sym)
  { return sym.hash(); }
  static bool symbolequal(const LBSymbol& sym1, const LBSymbol& sym2)
  { return sym1==sym2; }

  LBNameSpace::LBNameSpace(const LBSymbol& nm, LBNameSpace* parent)
    : _name(nm), _fullname(), _parent(parent), _content(symbolhash, symbolequal, 7), _rwlock()
  {
    if ( parent )
      _fullname = parent->fullName();
    _fullname.append(nm);
  }

  LBNameSpace::LBNameSpace() // this is for root NS construction
    : _name(), _fullname(), _parent(0), _content(symbolhash, symbolequal, 7), _rwlock()
  {
  }

  LBNameSpace::~LBNameSpace()
  {
    NSMap::Iterator it(_content);
    while ( it.next() )
      {
	NSItem *item = *it.currentPair()->value;
	delete item;
      }
  }

  LBNameSpace::NSItem::~NSItem()
  {
    if ( _component ) delete _component;
    if ( _ns ) delete _ns;
  }

  bool LBNameSpace::addSymbol(const LBSymbol& n, LBNameSpace *ns)
  {
    LBWriteLocker wlocker(_rwlock);
    NSItem* &itm = _content[n];
    if ( itm != 0 )
      return false;
    itm = new NSItem(ns);
    return true;
  }

  bool LBNameSpace::addSymbol(const LBSymbol& n, LBComponentInfo *comp)
  {
    LBWriteLocker wlocker(_rwlock);
    NSItem* &itm = _content[n];
    if ( itm != 0 )
      return false;
    itm = new NSItem(comp);
    return true;
  }

  bool LBNameSpace::addOrReplaceSymbol(const LBSymbol& n, LBComponentInfo *comp)
  {
    LBWriteLocker wlocker(_rwlock);
    NSItem* &itm = _content[n];
    if ( itm != 0 )
      {
	if ( itm->getNS() )  // the symbol is a name space, can not be replaced with a component
	  return false;
	delete itm;
      }
    itm = new NSItem(comp);
    return true;
  }

  bool LBNameSpace::removeSymbol(const LBSymbol& nm)
  {
    NSItem *item = 0;
    {
      LBWriteLocker wlocker(_rwlock);
      if ( _content.find(nm, item) )
	_content.erase(nm);
    }
    if ( item ) delete item;
    return false;
  }

  bool LBNameSpace::findSymbol(const LBSymbol& name, LBNameSpace* &ns,  LBComponentInfo* &comp)
  {
    NSItem *itm = 0;
    LBReadLocker rlocker(_rwlock);
    if ( _content.find(name, itm) )
      {
	ns = itm->getNS();
	comp = itm->getComponent();
	return true;
      }
    return false;
  }


  LBNameSpace* LBNameSpace::findNS(const LBSymbol& name)
  {
    NSItem *itm = 0;
    LBReadLocker rlocker(_rwlock);
    if ( _content.find(name, itm) )
      return itm->getNS();
    return 0;
  }

  LBComponentInfo* LBNameSpace::findComponent(const LBSymbol& name)
  {
    NSItem *itm = 0;
    LBReadLocker rlocker(_rwlock);
    if ( _content.find(name, itm) )
      return itm->getComponent();
    return 0;
  }





  LBNameSpace* LBNSUtil::rootNS()
  {
    static LBNameSpace *motherofallns=0;
    if ( motherofallns == 0 )
      {
	motherofallns = new LBNameSpace();
	// here we also add all built in types to name space so they can be reflected
	LBFullSymbol doublefsym, intfsym, boolfsym, charfsym, stringfsym;
	LBFullSymbol vectorfsym, mapfsym, setfsym, errfsym;
	LBSymbol doublesym("double"), intsym("int"), boolsym("bool"), charsym("char"), errsym("error");
	LBSymbol stringsym("string"), vectorsym("vector"), mapsym("map"), setsym("set");

	doublefsym.append(doublesym);
	intfsym.append(intsym);
	boolfsym.append(boolsym);
	charfsym.append(charsym);
	stringfsym.append(stringsym);
	vectorfsym.append(vectorsym);
	mapfsym.append(mapsym);
	setfsym.append(setsym);
	errfsym.append(errsym);

	// put them into name space
	motherofallns->addSymbol(doublesym, new LBComponentInfo(doublefsym));
	motherofallns->addSymbol(intsym , new LBComponentInfo(intfsym));
	motherofallns->addSymbol(charsym , new LBComponentInfo(charfsym));
	motherofallns->addSymbol(boolsym , new LBComponentInfo(boolfsym));
	motherofallns->addSymbol(stringsym , new LBComponentInfo(stringfsym));
	motherofallns->addSymbol(vectorsym , new LBComponentInfo(vectorfsym));
	motherofallns->addSymbol(mapsym , new LBComponentInfo(mapfsym));
	motherofallns->addSymbol(setsym , new LBComponentInfo(setfsym));
	motherofallns->addSymbol(errsym , new LBComponentInfo(errfsym));
      }


    return motherofallns;
  }

  LBNameSpace* LBNSUtil::findOrCreateNS(const LBSymbol& name, LBNameSpace* parentns, string& errs)
  {
    if ( ! parentns )
      {
	errs += "Null name space for the symbol:"+name.toString();
	return 0;
      }
    LBComponentInfo *comp = 0;
    LBNameSpace *ns = 0;
    if ( parentns->findSymbol(name, ns, comp) )
      {
	if ( ! ns )
	  {
	    errs += "Symbol: "+name.toString()+" is a terminal compoment symbol instead of a name space";
	    return 0;
	  }
	return ns;
      }
	
    ns = new LBNameSpace(name, parentns);
    parentns->addSymbol( name, ns );
    return ns;
  }

  LBNameSpace* LBNSUtil::findOrCreateNS(const LBFullSymbol& name, string& errs)
  {
    LBNameSpace *ns = rootNS();
    int s = name.size();
    for( int i = 0; i < s && ns ; i++ )
      {
	const LBSymbol *n = name.nameAtLevel(i);
	ns = findOrCreateNS(*n, ns, errs);
      }
    return ns;
  }

  LBNameSpace* LBNSUtil::findNS(const LBFullSymbol& name, STATUS &st)
  {
    LBNameSpace *ns = rootNS();
    LBComponentInfo *dummy = 0;
    int s = name.size();
    for( int i = 0; i < s ; i++ )
      {
	LBNameSpace *nsnext = 0;
	if ( ns->findSymbol( *name.nameAtLevel(i), nsnext, dummy ) )
	  {
	    if ( ! nsnext )
	      {
		st = TERMINAL;
		return 0;
	      }
	    ns = nsnext;
	  }
	else
	  {
	    st = NONEXIST;
	    return 0;
	  }
      }
    return ns;
  }

  static inline void printIndent(ostream& ost, int indent)
  {
    for(int i=0; i<indent; i++)
      ost << '\t';
  }

  ostream& LBNSUtil::printNS(LBNameSpace *ns,  ostream& ost, bool recursive, int indent)
  {
    LBReadLocker rlocker(ns->_rwlock); // lock for reading
    LBNameSpace::NSMap::Iterator mapit(ns->_content);
    while ( mapit.next() )
      {
	printIndent(ost,indent);

	const LBSymbol *itemname = mapit.currentPair()->key;
	LBNameSpace *downns = (*mapit.currentPair()->value)->getNS();
	if ( ! downns )
	  {
	    LBFullSymbol compname( ns->fullName() );
	    compname.append(*itemname);
	    LubanSymbolResolver::printSymbol(compname, ost);
	    ost << '\n';
	  }
	else
	  {
	    ost << itemname->toString() << "\tnamespace\n";
	    if ( recursive )
	      printNS(downns, ost, recursive, indent+1);
	  }
      }
    return ost;
  }

}
