#include <string>
#include <iostream>
#include <iterator>
#include <algorithm>

#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbbool.hpp"

#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/lbvector.hpp"
#include "lbtypes/lbset.hpp"

namespace Luban
{
  using std::string;
  using std::deque;

    

  LBDEFINE(Luban::LBVector, 1, 0 )


  static bool convertIndex(int &index, int sz)
  {
    int i = index<0?(sz+index):index;
    if ( i<0 || i >= sz )
	  return false;
    index = i;
    return true;
  } 



    //static, this one must be implemented, and at least cover default constructor
  LBVector* LBVector::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() == 0 )
      return new LBVector();
    
    switch ( args->numArgs() ) {
    case 1:
      {
	const LBObject* arg0 = args->getArg(0);
	// export copy constructor
	if ( arg0 )
	  {
	    const LBVector *vp = dynamic_cast<const LBVector*>(arg0);
	    if ( vp )
	      return new LBVector(*vp);
	    const LBInt *sz = dynamic_cast<const LBInt*>(arg0);
	    if( sz )
	      return new LBVector(int(*sz));
	  }
	return 0;
	break;
      }
    case 2:
      {
	const LBObject* arg0 = args->getArg(0);
	const LBObject* arg1 = args->getArg(1);
	if ( arg0 )
	  {
	    const LBInt *sz = dynamic_cast<const LBInt*>(arg0);
	    if( sz )
	      return new LBVector(int(*sz), arg1);
	  }
	return 0;
	break;
      }
    default:
      ;
    }
    
    return 0;
  }


  LBVector::LBVector() 
    : _imp( )
  {
  }

  LBVector::LBVector(int n) 
    : _imp( new LBVectorImp(n) )
  {
  }

  // private
  LBVector::LBVector( LBVectorImp *imp) : _imp(imp)
  {
  }

  LBVector::LBVector(int num, const LBObject& val) 
    : _imp( new LBVectorImp(num) )
  {
    LBObject *realp = val.clone();
    LBObjPtr p(realp);
    for (; num >= 0; num-- )
      _imp->_realvector[num]=p;
  }

  LBVector::LBVector(int num, const LBObject* val) 
    : _imp( new LBVectorImp(num) )
  {
    if ( num < 0 )
      throw LBException("Attempt to create vector with length less than zero");
    if ( val )
      {
	LBObject *realp = val->clone();
	LBObjPtr p(realp);
	for (int i=0; i<num; i++ )
	  _imp->_realvector[i]=p;
      }
  }

  string LBVector::toString() const
  {
    if ( _imp )
      {
	IVector::const_iterator b = _imp->_realvector.begin();
	IVector::const_iterator e = _imp->_realvector.end();
	string s = "[";
	while( b != e )
	  {
	    s += (*b).toString();
	    if ( ++b !=e ) s += ',';
	  }
	s += ']';
	return s;
      }
    static const string emptyvec("[]");
    return emptyvec;
  }

  ostream& LBVector::toStream(ostream& ost) const
  {
    int s = size();
    ost<<s<<'\n';
    for( int i = 0; ost && i < s; i++)
      {
	if ( _imp->_realvector[i] )
	  {
	    // use the general object streaming to include the meta data
	    // so it can streamed back in a general way
	    ost.put('1');ost.put(' ');
	    LBObject::instanceToStream(ost, *_imp->_realvector[i]);
	  }
	else
	  {
	    ost.put('0');ost.put(' ');
	  }
      }
    if ( !ost )
      throw LBException("Error happen when streaming vector elements");
    return ost;
  }

  istream& LBVector::fromStream(istream& i, int major, int minor)
  {
    int length=-1;
    i>>length;
    char slashn=i.get();
    if ( ! i || length < 0 || slashn != '\n'  ) 
      throw LBException("Corrupted stream, error recovering number of elements for a vector from stream");
    if ( length ==0 )
      {
	clear();
	return i;
      }

    LBVectorImp *newimp = new LBVectorImp(length);
    string err;
    int j=0;
    for(;  j < length && i; j++ )
      {
	char onezero=i.get();
	char spchar=i.get();
	if ( spchar != ' ' || onezero != '1' && onezero != '0' )
	  throw LBException( string("Corrupted stream format, bad element delimiter for vector object")+spchar+onezero );
	LBObject *obj = 0;
	if ( onezero == '1' )
	  obj = LBObject::instanceFromStream(i,err);
	if ( obj )
	  newimp->_realvector[j] = LBObjPtr(obj);
      }
    if ( !i || j < length )  // premature exit of above loop
      {
	delete newimp;
	throw LBException( string("Failed to retrieve all objects from stream for vector, error: ")+string(err));
      }

    _imp = LBVectorImpPtr(newimp);

    return i;

  }

  // the below two defines that two vectors are equal only when they have the same guts
  LBDEFAULT_EQUALS_FUNC(Luban::LBVector)

  bool LBVector::operator==(const LBVector& x) const  
  {
    if ( _imp == x._imp ) return true ; 
    if ( size() == 0 && x.size() == 0 ) return true;
    if ( size() != x.size() ) return false;
    for( int i=0; i < size() ; i++ )
      {
	const LBObjPtr &p1 = _imp->_realvector[i];
	const LBObjPtr &p2 = x._imp->_realvector[i];
	if ( p1 == p2 )
	  continue;
	if ( p1 && !p2 || !p1 && p2 )
	  return false;
	try {
	if ( ! p1->equals(*p2) )
	  return false;
	}
	catch (...)
	  {
	    return false;
	  }
      }
  }

  bool LBVector::cast(LBObject* target) const
  {

    // cast to a set, duplicated elements will be merged into one in the set
    LBSet *sp = dynamic_cast<LBSet*>(target);
    if ( sp )
      {
	sp->clear();
	if ( _imp )
	  {
	    IVector::const_iterator b = _imp->_realvector.begin();
	    IVector::const_iterator e = _imp->_realvector.end();
	    for(; b != e; b++ )
	      sp->insert(*b);
	    return true;
	  }
      }

    return false;
  }

  LBVector& LBVector::add(const LBObject& lbo)
  {
    const LBVector  *lbp = dynamic_cast<const LBVector*>(&lbo);
    if ( lbp && lbp->_imp )
      {
	IVector::const_iterator b = lbp->_imp->_realvector.begin();
	IVector::const_iterator e = lbp->_imp->_realvector.end();
	if ( b == e )
	  return *this;
	if ( size() == 0 ) // if self is empty, just copy the adder over
	  {
	    _imp = lbp->_imp;
	    return *this;
	  }
	unicopy(_imp);
	for (; b != e;b++ )
	  _imp->_realvector.push_back( *b );
	return *this;
      }

    throw LBException(string("Can not add vector with ")+lbo.getType().toString());
  }

  LBVector& LBVector::mul(const LBObject& lbo)
  {
    const LBInt  *lbp = dynamic_cast<const LBInt*>(&lbo);
    if ( lbp )
      {
	int s = size();
	if ( s == 0 )
	  return *this;
	int x = int(*lbp);
	if ( x == 1 )
	  return *this;
	if ( x == 0 )
	  {
	    clear();
	    return *this;
	  }
	if ( x < 0 )
	  {
	    neg();
	    x = -x;
	  }

	for( int i = 1; i < x ; i++ )
	  for( int j = 0; j < s; j++ )
	    _imp->_realvector.push_back(_imp->_realvector[j] );
	return *this;
      }

    throw LBException(string("Can not multiply vector with ")+lbo.getType().toString());

  }


  // simple ordering, shorter vector before longer vector
  // dictionary order otherwise
  bool LBVector::before(const LBObject& lbo) const
  {
    const LBVector *p = dynamic_cast<const LBVector*>(&lbo);
    if ( p )
      return size() < p->size();
	
    throw LBException(string("Can not compare LBVector with ")+lbo.getType().toString());
  }

  
  LBVector& LBVector::neg()
  {
    int sz = size();
    if ( sz > 1 )
      {
	unicopy(_imp);
	LBObjPtr temp;
	for( int i = 0 ; i < sz/2; i++ )
	  { 
	    temp = _imp->_realvector[i];
	    _imp->_realvector[i]=_imp->_realvector[sz-i-1] ;
	    _imp->_realvector[sz-i-1]=temp;
	  }
      }
    return *this;
  }

  void LBVector::subscriptSet(const LBObject* index, const LBObject* value)
  {
    if ( ! index )
      throw LBException(string("Invalid null index to a vector "));
    const LBInt *intp = dynamic_cast<const LBInt*>(index);
    if ( intp == 0 )
      throw LBException(index->getType().toString()+" can not be used to index a vector. Only integer can");
    int i = int(*intp);
    if ( ! convertIndex(i, size()) )
      throw LBException(string("Invalid index to a vector, either negative or beyond the end of the vector: ")+intp->toString());

    unicopy(_imp);

    // hope the incoming is the same type and can be assigned, for efficiency
    if ( ! value )
      _imp->_realvector[i] = LBObjPtr();
    else
      _imp->_realvector[i].assign(*value);
    return;
  }

  // this is an unsafe function, the reference may become shared or invalid
  // as the vector changes
  LBObject* LBVector::subscriptGet(const LBObject* index)
  {
    if ( ! index )
      throw LBException(string("Invalid null index to a vector "));

    const LBInt *intp = dynamic_cast<const LBInt*>(index);
    if ( intp == 0 )
      throw LBException(index->getType().toString()+" can not be used to index a vector. Only integer can");
    int i = int(*intp);
    if ( ! convertIndex(i, size()) )
      throw LBException(string("Invalid index to a vector, either negative or beyond the end of the vector: ")+intp->toString());
    
    unicopy(_imp);
    LBObjPtr &p = _imp->_realvector[i];
    uniclone(p);
    return p.getRealPtr();
  }

  const LBObject* LBVector::subscriptConstGet(const LBObject* index) const
  {
    if ( ! index )
      throw LBException(string("Invalid null index to a vector "));

    const LBInt *intp = dynamic_cast<const LBInt*>(index);
    if ( intp == 0 )
      throw LBException(index->getType().toString()+" can not be used to index a vector. Only integer can");
    int i = int(*intp);
    if ( ! convertIndex(i, size()) )
      throw LBException(string("Invalid index to a vector, either negative or beyond the end of the vector: ")+intp->toString());

    return _imp->_realvector[i].getConstRealPtr();

  }
  
  int LBVector::size() const
  { return _imp?_imp->_realvector.size():0; }

  class LBVectorConstIterator : public LBConstIterator
  {
  public:
    LBVectorConstIterator(const LBVector& v)
      : _vec(v), _current(-1), _end(v.size()), _mid(0), _output(&_mid, 1)
    {
    }

    bool next() 
    { 
      if ( ++_current >= _end ) 
	return false; 
      _mid = _vec._imp->_realvector[_current].getConstRealPtr();
      return true; 
    }
    const LBVarArgs* getCurrentRow()
    {
      if ( _current >= _end || _current < 0 )
	return 0;
      return &_output;
    }
    

  private:
    const LBVector& _vec;
    int _current, _end;
    const LBObject *_mid;
    LBSimpleVarArgs _output;

  };

  LBConstIterator* LBVector::getIterator() const
  {  return  new LBVectorConstIterator(*this); }

  bool LBVector::contains(const LBObject* x) const
  {   return index(x) >= 0 ; }

  int LBVector::index(const LBObject* x, int startpos, int endpos) const
  {
    int i = startpos, j = endpos, sz=size();
    if ( ! convertIndex(i, sz) || ! convertIndex(j, sz) || i>j )
      throw LBException("Invalid start and/or end position for vector::index() function");
    IVector::const_iterator it = _imp->_realvector.begin()+i;
    IVector::const_iterator eov = _imp->_realvector.begin()+j+1;
    for (; it != eov; it++,i++ )
      if ( ! x  )
	{
	  if (  !( *it )  )
	    return i;
	}
      else 
	if ( *it && ( (*it).getConstRealPtr() == x || (*it)->equals(*x) ) )
	  return i;
    throw LBException("Can not find object in vector");
  }

  LBObject& LBVector::operator[](int index)
  {
    int i = index;
    if ( ! convertIndex(i, size()) )
      throw LBException(string("Invalid index for vector [] operator, either negative or beyond the end of the vector"));

    unicopy(_imp);
    LBObjPtr &p = _imp->_realvector[i];
    if ( p )
      {
	uniclone(p);
	return *p;
      }

    throw LBException("Tried to get reference to null element in vector");

  }

  const LBObject& LBVector::operator[](int index) const
  {
    int i = index;
    if ( ! convertIndex(i,size() ) )
      throw LBException(string("Invalid index for vector [] operator, either negative or beyond the end of the vector"));

    const LBObjPtr &p = _imp->_realvector[i];
    if ( p )
      return *p;

    throw LBException("Tried to get reference to null element in vector");

  }

  void LBVector::insert(LBObject *ownme, int index)
  {
    if ( index == -1 || index == size() )
      {
	push(ownme);
	return;
      }
    int i = index;
    if ( ! convertIndex(i,size()) )
      throw LBException(string("Invalid index for vector insert function, either negative or beyond the end of the vector"));
    
    unicopy(_imp);
    if ( index >= 0 )
      _imp->_realvector.insert(_imp->_realvector.begin()+i, LBObjPtr(ownme));
    else
      _imp->_realvector.insert(_imp->_realvector.begin()+i+1, LBObjPtr(ownme));
  }

  void LBVector::insert(const LBObject& value, int index)
  {
    if ( index == -1 ||index == size() )
      {
	push(value.clone());
	return;
      }
    int i = index;
    if ( ! convertIndex(i, size()) )
      throw LBException(string("Invalid index for vector::insert function, either negative or beyond the end of the vector"));
    
    unicopy(_imp);
    if ( index >= 0 )
      _imp->_realvector.insert(_imp->_realvector.begin()+i,LBObjPtr(value.clone()));
    else
      _imp->_realvector.insert(_imp->_realvector.begin()+i+1, LBObjPtr(value.clone()));


  }

  class StupidBeforeWrapper
  {
  public:
    bool operator()(LBObjPtr const &j1, LBObjPtr const &j2) const
    { 
      return LBObject::objcmp(j1.getConstRealPtr(),j2.getConstRealPtr())==-1;
    }
  };

  class StupidBeforeWrapperReversed
  {
  public:
    bool operator()(LBObjPtr const &j1, LBObjPtr const &j2) const
    { 
      return LBObject::objcmp(j1.getConstRealPtr(),j2.getConstRealPtr()) == 1;
    }
  };

  void LBVector::sort(int starti, int endi, bool reversed)
  {
    int sz = size();
    if ( sz == 0 )
      return;
    int s = starti, e = endi;
    if ( ! convertIndex(s, sz) || ! convertIndex(e,sz) || s>e )
      throw LBException(string("Invalid index to a vector, could be negative or beyond the end of the vector, or start index beyond end index"));
    if ( s==e ) //easy case
      return;

    unicopy(_imp);
    IVector::iterator i1=_imp->_realvector.begin()+s;
    IVector::iterator i2=_imp->_realvector.begin()+e+1;
    if ( ! reversed )
      std::sort(i1, i2, StupidBeforeWrapper());
    else
      std::sort(i1, i2, StupidBeforeWrapperReversed());
    return;
  }

  LBVector* LBVector::subVector(int starti, int endi) const
  {
    int sz = size();
    int s = starti, e = endi;
    if ( ! convertIndex(s,sz) || ! convertIndex(e,sz) || s>e )
      throw LBException(string("Invalid index to a vector, could be negative or beyond the end of the vector, or start index beyond end index"));

    LBVectorImp *newcore = new LBVectorImp(e-s+1);
    for(int i = s; i <= e ; i++ )
      newcore->_realvector[i-s]=_imp->_realvector[i];

    return new LBVector(newcore);
  }

  // private, friendly container types use only
  void LBVector::push(const LBObjPtr& e)
  {
    uniGuts(_imp );
    _imp->_realvector.push_back(e);
  }

  // private
  void LBVector::setElement(int index, const LBObjPtr& e)
  {
    unicopy(_imp);
    _imp->_realvector[index] = e;
  }

  // private
  const LBObjPtr& LBVector::getElement(int index) const
  {
    return _imp->_realvector[index];
  }

  void LBVector::push(const LBObject& val)
  {
    uniGuts(_imp );
    _imp->_realvector.push_back(LBObjPtr(val.clone()));
  }

  void LBVector::push(LBObject* ownme)
  {
    uniGuts(_imp );
    _imp->_realvector.push_back(LBObjPtr(ownme));
  }

  LBObject* LBVector::pop()
  {
    if ( size() == 0 )
      return 0;
    
    unicopy(_imp);
    LBObjPtr &last = _imp->_realvector.back();
    if ( last.isShared() ) // if shared
      {
	LBObject *res = last.getConstRealPtr()->clone();
	_imp->_realvector.pop_back();
	return res;
      }
    LBObject *p = last.release();
    _imp->_realvector.pop_back();
    return p;
  }

  void LBVector::push_front(const LBObject& val)
  {
    uniGuts(_imp );
    _imp->_realvector.push_front(LBObjPtr(val.clone()));
  }

  void LBVector::push_front(LBObject* ownme)
  {
    uniGuts(_imp );
    _imp->_realvector.push_front(LBObjPtr(ownme));
  }

  LBObject* LBVector::pop_front()
  {
    if ( size() == 0 )
      return 0;
    
    unicopy(_imp);
    LBObjPtr &first = _imp->_realvector.front();
    if ( first.isShared() ) // if shared
      {
	LBObject* res =  first.getConstRealPtr()->clone();
	_imp->_realvector.pop_front();
	return res;
      }
    LBObject *p = first.release();
    _imp->_realvector.pop_front();
    return p;
  }

  void LBVector::clear()
  {
    if ( size() == 0 )
      return;
    _imp = LBVectorImpPtr();
  }

  void LBVector::erase( int pos )
  {
    int p = pos;
    if ( ! convertIndex(p,size()) )
      throw LBException(string("Invalid index for vector erase function, either negative or beyond the end of the vector"));

    unicopy(_imp);
    IVector::iterator i = _imp->_realvector.begin()+p;
    _imp->_realvector.erase(i );
    return;
  }

  void LBVector::erase( int spos, int epos )
  {
    int sz = size();
    int p1 = spos, p2 = epos;
    if ( ! convertIndex(p1,sz) || ! convertIndex(p2,sz) || p1>p2 )
      throw LBException(string("Invalid index for vector erase function"));
    
    unicopy(_imp);
    IVector::iterator i1 = _imp->_realvector.begin()+p1;
    IVector::iterator i2 = _imp->_realvector.begin()+p2+1;
    _imp->_realvector.erase(i1,i2);
    return;
  }

  int LBVector::erase(const LBObject* x, int startpos, int endpos)
  {
    int sz = size();
    int sid = startpos;
    int eid = endpos;
    if ( ! convertIndex(sid,sz) || ! convertIndex(eid,sz) || sid>eid )
      throw LBException("Invalid index for vector erase function");
    
    unicopy(_imp);

    IVector::iterator i1 = _imp->_realvector.begin()+sid;
    int count=0;

    for (int toscan = eid-sid+1; toscan > 0 ; i1++, toscan-- )
      if ( ! x  )
	{
	  while (  !( *i1 ) ) 
	    {
	      i1 = _imp->_realvector.erase(i1);
	      count++; toscan--;
	      if ( toscan == 0 )
		return count;
	    }
	}
      else 
	while ( *i1 && ( (*i1).getConstRealPtr() == x || (*i1)->equals(*x) ) )
	  {
	    i1 = _imp->_realvector.erase(i1);
	    count++; toscan--;
	    if  ( toscan == 0 )
	      return count;
	  }

    return count;
  }


  int LBVector::replace(const LBObject* oldobj, const LBObject *newobj, int startpos, int endpos)
  {
    if ( oldobj == newobj )
      return 0;

    int sz = size();
    int sid = startpos;
    int eid = endpos;
    if ( ! convertIndex(sid,sz) || ! convertIndex(eid,sz) || sid>eid )
      throw LBException("Invalid index for vector erase function");
    
    unicopy(_imp);

    IVector::iterator i1 = _imp->_realvector.begin()+sid;
    int count=0;
    LBObjPtr newp;

    for (int toscan = eid-sid+1; toscan > 0 ; i1++, toscan-- )
      if ( ! oldobj  )
	{
	  if ( !( *i1 ) )
	    {
	      if ( ! newp )
		newp.setRealPtr(newobj->clone());
	      (*i1) = newp;
	      count++;
	    }
	}
      else
	if ( *i1 && ( (*i1).getConstRealPtr() == oldobj || (*i1)->equals(*oldobj) ) )
	  {
	    if ( ! newobj )
	      (*i1) = LBObjPtr();
	    else
	      {
		uniorzero( *i1 );
		if  ( *i1 && (*i1)->assign(*newobj) )
		  {
		    count++;
		    continue;
		  }
		if ( ! newp )
		  newp.setRealPtr(newobj->clone());
		(*i1) = newp;
		count++;
	      }
	  }

    return count;
  }


  
  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_append, "append", "void vector::append(object element1, element2,...)" ); 
  LBObject* LBVector::luban_append(const LBVarArgs *args)
  {
    if ( args &&  args->numArgs()>=1 )
      {
	for( int i = 0; i < args->numArgs(); i++)
	  {
	    const LBObject *element = args->getArg(i);
	    push(element?element->clone():0);
	  }
	return 0;
      }
    throw LBException("Invalid number of args passed to vector::append()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_insert, "insert", "void vector::insert(int idx, object element)" ); 
  LBObject* LBVector::luban_insert(const LBVarArgs *args)
  {
    if ( args &&  args->numArgs()==2 )
      {
	LBInt idx;
	if ( ! args->assignArgToLocal(0, idx) )
	  throw LBException("Expecting first argument to be of int type");
	const LBObject *element = args->getArg(1);
	if ( element )
	  insert( *element, int(idx) );
	else
	  insert( 0, int(idx) );
	return 0;
      }
    throw LBException(string("Invalid number of args passed to vector::insert()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_remove, "remove", "void vector::remove(int idx, optional idx2)" ); 
  LBObject* LBVector::luban_remove(const LBVarArgs *args)
  {
    if ( args )
      switch ( args->numArgs() )
	{
	case 1:
	  {
	    LBInt idx;
	    if ( ! args->assignArgToLocal(0, idx) )
	      throw LBException("Expecting argument to be of int type");
	    erase( int(idx) );
	    return 0;
	  }
	case 2:
	  {
	    LBInt idx1, idx2;
	    if ( ! args->assignArgToLocal(0, idx1) || ! args->assignArgToLocal(1, idx2) )
	      throw LBException("Expecting argument to be of int type");
	    erase( int(idx1), int(idx2) );
	    return 0;
	  }
	}
    throw LBException(string("Invalid number of args passed to vector::remove()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_pushfirst, "pushfirst", "void vector::pushfirst(object element)" ); 
  LBObject* LBVector::luban_pushfirst(const LBVarArgs *args)
  {
    if ( args &&  args->numArgs()==1 )
      {
	const LBObject *e = args->getArg(0) ;
	if ( e )
	  push_front(*e);
	else
	  push_front(0);
	return 0;
      }
    throw LBException(string("Invalid number of args passed to vector::pushfirst()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_pushlast, "pushlast", "void vector::pushlast(object element)" ); 
  LBObject* LBVector::luban_pushlast(const LBVarArgs *args)
  {
    if ( args &&  args->numArgs()==1 )
      {
	const LBObject *e = args->getArg(0) ;
	if ( e )
	  push(*e);
	else
	  push(0);
	return 0;
      }
    throw LBException(string("Invalid number of args passed to vector::pushlast()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_poplast, "poplast", "object vector::poplast()" ); 
  LBObject* LBVector::luban_poplast(const LBVarArgs *args)
  {
    if ( ! args ||  args->numArgs()== 0 )
      return pop();
    throw LBException(string("Invalid number of args passed to vector::poplast()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_popfirst, "popfirst", "object vector::popfirst()" ); 
  LBObject* LBVector::luban_popfirst(const LBVarArgs *args)
  {
    if ( ! args ||  args->numArgs()== 0 )
      return pop_front();
    throw LBException(string("Invalid number of args passed to vector::popfirst()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_subvec, "subvec", "vector vector::subvec(int start, int end)" ); 
  LBObject* LBVector::luban_subvec(const LBVarArgs *args)
  {
    if ( args && args->numArgs()==2 )
      {
	LBInt idx1, idx2;
	if ( ! args->assignArgToLocal(0, idx1) || ! args->assignArgToLocal(1, idx2) )
	  throw LBException("Expecting argument to be of int type");
	return static_cast<LBObject*>(subVector( int(idx1), int(idx2) ));
      }
    throw LBException(string("Invalid number of args passed to vector::subvec()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_sort, "sort", "void vector::sort(optional int start and int end, optional bool downorder)" ); 
  LBObject* LBVector::luban_sort(const LBVarArgs *args)
  {
    if ( args )
      switch( args->numArgs() )
	{
	case 2:
	  {
	    LBInt idx1, idx2;
	    if ( ! args->assignArgToLocal(0, idx1) || ! args->assignArgToLocal(1, idx2) )
	      throw LBException("Expecting argument to be of int type");
	    sort( int(idx1), int(idx2) );
	    return 0;
	  }
	case 3:
	  {
	    LBInt idx1, idx2;
	    if ( ! args->assignArgToLocal(0, idx1) || ! args->assignArgToLocal(1, idx2) )
	      throw LBException("Expecting argument to be of int type");
	    LBBool down(false);
	    if ( ! args->assignArgToLocal(2, down) )
	      throw LBException("Expecting argument to be of bool type");
	    sort( int(idx1), int(idx2), bool(down) );
	    return 0;
	  }
	case 1:
	  {
	    LBBool down(false);
	    if ( ! args->assignArgToLocal(0, down) )
	      throw LBException("Expecting argument to be of bool type");
	    sort( 0, -1, bool(down) );
	    return 0;
	  }
	case 0:
	  sort();
	  return 0;
	default:
	  throw LBException(string("Invalid number of args passed to vector::sort()"));
	}
  
    sort();

    return 0;

  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_index, "index", "int vector::index(object value,optional int start and int end)" ); 
  LBObject* LBVector::luban_index(const LBVarArgs *args)
  {
    if ( args )
      switch ( args->numArgs() )
	{
	case 1:
	  return new LBInt( index(args->getArg(0) ) );
	case 2:
	  {
	    LBInt startidx(0);
	    if ( ! args->assignArgToLocal(1, startidx) )
	      throw LBException("Expect 2nd argument to be of int type");
	    return new LBInt( index(args->getArg(0), int(startidx) ) );
	  }
	case 3:
	  {
	    LBInt startidx(0), endidx(-1);
	    if ( ! args->assignArgToLocal(1, startidx) || ! args->assignArgToLocal(2, endidx) )
	      throw LBException("Expect 2nd and 3rd argument to be of int type");
	    return new LBInt( index(args->getArg(0), int(startidx), int(endidx ) ));
	  }
	}
    throw LBException("Invalid number of args passed to vector::index()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_removeobj, "removeobj", "int vector::removeobj(object value,optional int start and int end)" ); 
  LBObject* LBVector::luban_removeobj(const LBVarArgs *args)
  {
    if ( args )
      switch ( args->numArgs() )
	{
	case 1:
	  return new LBInt( erase(args->getArg(0) ) );
	case 2:
	  {
	    LBInt startidx(0);
	    if ( ! args->assignArgToLocal(1, startidx) )
	      throw LBException("Expect 2nd argument to be of int type");
	    return new LBInt( erase(args->getArg(0), int(startidx) ) );
	  }
	case 3:
	  {
	    LBInt startidx(0), endidx(-1);
	    if ( ! args->assignArgToLocal(1, startidx) || ! args->assignArgToLocal(2, endidx) )
	      throw LBException("Expect 2nd and 3rd argument to be of int type");
	    return new LBInt( erase(args->getArg(0), int(startidx), int(endidx ) ) );
	  }
	}
    throw LBException("Invalid number of args passed to vector::removeobj()");
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBVector, luban_replace, "replace", "int vector::replace(object old, object new, optional int start and int end)" ); 
  LBObject* LBVector::luban_replace(const LBVarArgs *args)
  {
    if ( args )
      switch ( args->numArgs() )
	{
	case 2:
	  return new LBInt( replace(args->getArg(0),args->getArg(1)) );
	case 3:
	  {
	    LBInt startidx(0);
	    if ( ! args->assignArgToLocal(2, startidx) )
	      throw LBException("Expect 3rd argument to be of int type");
	    return new LBInt( replace(args->getArg(0),args->getArg(1), int(startidx) ) );
	  }
	case 4:
	  {
	    LBInt startidx(0), endidx(-1);
	    if ( ! args->assignArgToLocal(2, startidx) || ! args->assignArgToLocal(3, endidx) )
	      throw LBException("Expect 3rd and 4th arguments to be of int type");
	    return new LBInt( replace(args->getArg(0),args->getArg(1), int(startidx), int(endidx) ) );
	  }
	}
    throw LBException("Invalid number of args passed to vector::replace()");
  }



}
