#ifndef __LBSET_HPP__
#define __LBSET_HPP__

#include <string>
#include <iosfwd>
#include "lbtypes/lbobject.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/HashUtil.hpp"


namespace Luban
{
  class LBSet : public LBObject
  {
  public:
    LBDECLARE( Luban::LBSet )

    LBSet( );

    // Below are required interface for any LBObject
    string toString() const;   // for human readable printing
    ostream& toStream(ostream& o) const;  // for persistence and network transportation
    // major.minor is the version number of the object IN THE STREAM
    // which are passed in in case it is needed for backward compatible streaming
    istream& fromStream(istream& i, int major=-1, int minor=-1);     // for persistence and network transportation
    bool equals(const LBObject& another) const;  

    // below are optional
    // return true if succeed to assign the casted value to the target, else return false and leave target unchanged
    bool cast(LBObject* casttotarget ) const; // interesting cast among vector/map/set is implemented here
    LBSet& add(const LBObject& lbo); // union of two sets
    LBSet& sub(const LBObject& lbo); // remove the common part of two sets, to get common part of x and y do x-(x-y)
    bool      before(const LBObject& lbo) const; // simple order of vector, a set is before another if it is smaller
    

    // optional, below are container related functions
    int size() const;
    LBConstIterator* getIterator() const;
    bool contains(const LBObject* k) const;
    void clear();

    // LBSet specific interfaces
    bool operator==(const LBSet& v) const;
    bool insert(const LBObject* key); // only insert when the key does not exist, return true when succeed
    bool insert(const LBObject& key) { return insert(&key); }
    bool insert(LBObject *key); // only insert when the key does not exist, return true when succeed

    // functions used by friend containers
    bool insert(const LBObjPtr& actualelement);

    bool erase( const LBObject* key );
    bool erase( const LBObject& key ) { return erase(&key); }

    friend class LBVector;
    friend class LBMap;


    // member functions exported to Luban
    LBObject* luban_insert(const LBVarArgs *args);
    LBObject* luban_remove(const LBVarArgs *args);

  private:

    class LBSetIterator;
    friend class LBSetIterator;

    typedef LBHashSet< LBObjPtr> ISet;

    class LBSetImp
    {
      static int objHash(const LBObjPtr &obj ) { return obj?obj->hash():0; }
      static bool objEqual(const LBObjPtr &j1, const LBObjPtr &j2)
      { return j1==j2 || j1 && j2 && j1->equals(*j2); }
    public:
      ISet _realset;
      LBSetImp() 
	:_realset(&objHash, &objEqual, 4) 
      {
      }  // default number of buckets in a set is 16 (2**4)
    };

    typedef RefCountedPtr<LBSetImp> LBSetImpPtr;

    static void  uniGuts(LBSetImpPtr& imp)
    {
      if ( imp )
	unicopy(imp);
      else
	imp = LBSetImpPtr( new LBSetImp() );
    }

    LBSetImpPtr _imp;

  };






}


#endif
