#ifndef __LBMAP_HPP__
#define __LBMAP_HPP__

#include <string>
#include <iosfwd>
#include "lbtypes/lbobject.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/HashUtil.hpp"


namespace Luban
{
  class LBVector;

  class LBMap : public LBObject
  {
  public:
    LBDECLARE( Luban::LBMap )

    LBMap( );

    // Below are required interface for any LBObject
    string toString() const;   // for human readable printing
    ostream& toStream(ostream& o) const;  // for persistence and network transportation
    // major.minor is the version number of the object IN THE STREAM
    // which are passed in in case it is needed for backward compatible streaming
    istream& fromStream(istream& i, int major=-1, int minor=-1);     // for persistence and network transportation
    bool equals(const LBObject& another) const;  

    // below are optional
    // return true if succeed to assign the casted value to the target, else return false and leave target unchanged
    bool cast(LBObject* casttotarget ) const; // interesting cast among vector/map/set is implemented here
    LBMap& add(const LBObject& lbo); // union, if the same keys from two map have different value, it will take the value from lbo
    bool      before(const LBObject& lbo) const; // simple order of vector, a vector is before another if it is shorter


    // optional, below are container related functions
    void subscriptSet(const LBObject* index, const LBObject* value);
    // use below two functions with care, the reference becomes shared when the vector get copied/assigned afterwards
    // or invalid when the same element is set to value of different type
    LBObject* subscriptGet(const LBObject* index); // return a reference of the internal element, not a copy
    const LBObject* subscriptConstGet(const LBObject* index) const; // return a reference of the internal element, not a copy
    int size() const;
    LBConstIterator* getIterator() const; // return 0 if not a container type, that is the default implementation
    bool contains(const LBObject* key) const;
    void clear();

    // LBMap specific interfaces
    bool operator==(const LBMap& v) const;
    // use below two operators with care, the reference becomes shared when the vector get copied/assigned
    // or invalid when the same element is set to different value
    LBObject& operator[](const LBObject& i);   // throw exception when the indexed element doesn't exist
    const LBObject& operator[](const LBObject& i) const; // throw exception when the indexed element doesn't exist
    bool insert(const LBObject* key, const LBObject* val); // only insert when the key does not exist, return true when succeed
    bool insert(const LBObject& key, const LBObject& val); // only insert when the key does not exist, return true when succeed
    bool insert(LBObject *key, LBObject* value); // only insert when the key does not exist, return true when succeed
    bool erase( const LBObject* key );
    bool erase( const LBObject& key ) { return erase(&key); }
    void set(const LBObject* key, const LBObject* val); // insert when the key does not exist, replace otherwise
    void set(LBObject *key, LBObject* value); // insert when the key does not exist, replace otherwise
    void set(const LBObjPtr &key, const LBObjPtr & value); // insert when the key does not exist, replace otherwise
    LBVector* keys() const; // return all keys in a vector
    LBVector* values() const; // return all valuess in a vector

    // member functions exported to Luban
    LBObject* luban_insert(const LBVarArgs *args);
    LBObject* luban_remove(const LBVarArgs *args);
    LBObject* luban_keys(const LBVarArgs *args);
    LBObject* luban_values(const LBVarArgs *args);

  private: // nested classes and typedefs

    class LBMapIterator;
    friend class LBMapIterator;

    typedef LBHashMap< LBObjPtr, LBObjPtr > IMap;

    class LBMapImp
    {
      static int objHash(const LBObjPtr &obj ) { return obj ? obj->hash():0; }
      static bool objEqual(const LBObjPtr &j1, const LBObjPtr &j2)
      { return j1==j2 || j1 && j2 && j1->equals(*j2); }
    public:
      IMap _realmap;
      LBMapImp() : _realmap(&objHash, &objEqual, 4) {}  // default number of buckets in a map is 16 (2**4)
    };

    typedef RefCountedPtr<LBMapImp> LBMapImpPtr;

    static void  uniGuts(LBMapImpPtr& imp)
    {
      if ( ! imp )
	imp = LBMapImpPtr( new LBMapImp() ); 
      else
	unicopy(imp);
    }

  private: // constructor used internally
    
    LBMap( LBMapImp *imp );

  private:

    LBMapImpPtr _imp;

  };






}


#endif
