#include <string>
#include <iostream>

#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/lbvector.hpp"

#include "lbtypes/lbmap.hpp"
#include "lbtypes/lbset.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbbool.hpp"

namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBMap, 1, 0 )

    //static, this one must be implemented, and at least cover default constructor
  LBMap* LBMap::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() == 0 )
      return new LBMap();
    
    switch ( args->numArgs() ) {
    case 1:
      {
	const LBObject* arg0 = args->getArg(0);
	// export copy constructor
	if ( arg0 )
	  {
	    const LBMap *vp = dynamic_cast<const LBMap*>(arg0);
	    if ( vp )
	      return new LBMap(*vp);
	  }
	return 0;
	break;
      }
    case 2:
      {
	// you can construct a map from two vectors, one for keys and one for values
	const LBObject *arg0 = args->getArg(0);
	const LBObject *arg1 = args->getArg(1);
	if ( ! ( arg0 && arg1 ) )
	  return 0;
	// for two vector case
	const LBVector *v0 = dynamic_cast<const LBVector*>(arg0);
	const LBVector *v1 = dynamic_cast<const LBVector*>(arg1);
	if ( v0 && v1 )
	  {
	    int sz = v0->size();
	    if ( sz != v1->size() )
	      return 0;
	    LBMapImp *newcore = new LBMapImp();
	    for( int i = 0; i<sz; i++)
	      newcore->_realmap[v0->getElement(i)] = v1->getElement(i);
	    return new LBMap(newcore);
	  }
	return 0;
	break;
      }
    default:
      ;
    }
    
    return 0;
  }


  LBMap::LBMap() 
    : _imp( )
  {
  }

  // private
  LBMap::LBMap(LBMapImp *imp) 
    : _imp(imp )
  {
  }

  string LBMap::toString() const
  {
    if ( size() )
      {
	IMap::Iterator it(_imp->_realmap);
	string s = "{";
	bool gogo = it.next();
	while ( gogo )
	  {
	    const IMap::Pair *p = it.currentPair();
	    s += (*p->key).toString();
	    s += ':';
	    s += (*p->value).toString();
	    if ( gogo = it.next() ) s += ',';
	  }
	s += '}';
	return s;
      }
    static const string emptymap("{:}");
    return emptymap;
  }

  ostream& LBMap::toStream(ostream& ost) const
  {
    int len = size();
    ost << len << '\n';
    if ( len > 0 )
      {
	IMap::Iterator it(_imp->_realmap);
	while( it.next() )
	  {
	    const IMap::Pair *p = it.currentPair();
	    if ( *p->key )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *(*p->key) );
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }

	    if ( *p->value )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *(*p->value) );
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
	return ost;
      }
  }

  istream& LBMap::fromStream(istream& ist,  int major, int minor)
  {
    int length=-1;
    ist>>length;
    char slashn=ist.get();
    if ( ! ist || length < 0 || slashn != '\n'  ) 
      throw LBException("Corrupted stream, error recovering number of elements for a map from stream");
    if ( length ==0 )
      {
	clear();
	return ist;
      }

    LBMapImp *newimp = new LBMapImp();
    string err;
    int j=0;
    for(;  j < length && ist; j++ )
      {
	char oneorzero, spchar;
	ist.get(oneorzero); ist.get(spchar);
	if ( spchar != ' ' || oneorzero != '1' && oneorzero != '0' )
	  throw LBException( string("Invalid delimeter in map stream"));
	LBObject *key=0;
	if ( oneorzero == '1' )
	  key = LBObject::instanceFromStream(ist,err);

	ist.get(oneorzero); ist.get(spchar);
	if ( spchar != ' ' || oneorzero != '1' && oneorzero != '0' )
	  throw LBException( string("Invalid delimeter in map stream"));
	LBObject *value=0;
	if ( oneorzero == '1' )
	  value = LBObject::instanceFromStream(ist,err);

	LBObjPtr pkey(key), pval(value);
	newimp->_realmap[pkey] = pval;
      }
    if ( !ist || j < length )  // premature exit of above loop
      {
	delete newimp;
	throw LBException( string("Failed to retrieve all objects from stream for map, error: ")+string(err));
      }

    _imp = LBMapImpPtr(newimp);

    return ist;
  }

  // the below two defines that two maps are equal by if they have the same elements
  LBDEFAULT_EQUALS_FUNC(Luban::LBMap)

  bool LBMap::operator==(const LBMap& x) const  
  {
    if ( _imp == x._imp ) return true ; 
    if ( size() == 0 && x.size() == 0 ) return true;
    if ( size() != x.size() ) return false;

    IMap::Iterator it(_imp->_realmap);
    while( it.next() )
      {
	const IMap::Pair *p = it.currentPair();
	LBObjPtr xvalue;
	if ( ! x._imp->_realmap.find(*p->key, xvalue) )
	  return false;
	if ( *p->value == xvalue )
	  continue;
	if ( ! *p->value || ! xvalue )
	  return false;
	try {
	  if ( ! (*p->value)->equals(*xvalue) )
	    return false;
	}
	catch (...)
	  {
	    return false;
	  }
      }

    return true;

  }

  bool LBMap::cast(LBObject *casttotarget) const
  {
    // can cast to vector and set
    LBVector *v = dynamic_cast<LBVector*>(casttotarget);
    if ( v )
      {
	v->clear();
	if ( size() )
	  {
	    // for performance purpose, use the friend privilege to pass shared ref counted through pointer
	    IMap::Iterator it(_imp->_realmap);
	    while( it.next() )
	      {
		const IMap::Pair *p = it.currentPair();
		v->push( *p->key );
		v->push( *p->value );
	      }
	    return true;
	  }
      }

    // here is for casting to set, only for keys
    LBSet *s = dynamic_cast<LBSet*>(casttotarget);
    if ( s )
      {
	s->clear();
	if ( size() )
	  {
	    IMap::Iterator it(_imp->_realmap);
	    while( it.next() )
	      {
		const IMap::Pair *p = it.currentPair();
		s->insert( *p->key );
	      }
	    return true;
	  }
      }
	
    return false;
  }

  LBMap& LBMap::add(const LBObject& m)
  {
    const LBMap* mp = dynamic_cast<const LBMap*>(&m);
    if ( mp )
      {
	if ( mp->size() == 0 )
	  return *this;
	if ( size() == 0 )
	  {
	    _imp = mp->_imp;
	    return *this;
	  }
	unicopy(_imp); // copy on write
	IMap::Iterator it(mp->_imp->_realmap);
	while( it.next() )
	  {
	    const IMap::Pair *p = it.currentPair();
	    _imp->_realmap[ *p->key ] = *p->value;
	  }
	return *this;
      }

    throw LBException("Can not add map with type "+m.getType().toString());
    
  }

  // simple comparison of size
  bool LBMap::before(const LBObject& m) const
  {
    const LBMap* mp = dynamic_cast<const LBMap*>(&m);
    if ( mp )
      return size() < mp->size();
    throw LBException("Can not compare map with type "+m.getType().toString());
  }

  void LBMap::subscriptSet(const LBObject* k, const LBObject* v)
  {
    uniGuts(_imp);
	    
    LBObjPtr kp(k?k->clone():0), vp(v?v->clone():0);
    _imp->_realmap[kp] = vp;
  }

  LBObject* LBMap::subscriptGet(const LBObject* key)
  {
    if ( size() == 0 )
      throw LBException( string("Map is empty"));
    LBObjPtr vp, kpinside;
    LBObjPtr kp( const_cast<LBObject*>(key) );
    if ( _imp->_realmap.find(kp, kpinside, vp) )
      {
	unicopy(_imp);
	kp.release(); // don't delete the pointer inside at destruction

	LBObjPtr &internal = _imp->_realmap[kpinside];
	uniclone( internal );
	return internal.getRealPtr();
      }

    string kpstr = kp.toString(); // don't delete the pointer inside at destruction
    kp.release();
    throw LBException( string("Map does not contain key: ")+kpstr);
  }


  const LBObject* LBMap::subscriptConstGet(const LBObject* key) const
  {
    if ( size() == 0 )
      throw LBException("Map is empty");
    LBObjPtr vp, kpinside;
    LBObjPtr kp( const_cast<LBObject*>(key) );
    if ( _imp->_realmap.find(kp, kpinside, vp) )
      {
	kp.release(); // don't delete the pointer inside at destruction
	return vp.getRealPtr();
      }

    string kpstr = kp.toString();
    kp.release(); // don't delete the pointer inside at destruction

    throw LBException( string("Map does not contain key: ")+kpstr );
  }

  int LBMap::size() const
  {  return _imp?_imp->_realmap.size():0; }
  
  class LBMap::LBMapIterator : public LBConstIterator
  {

  public:
    LBMapIterator(const LBMap& m)
      :_end(true), _it(0), _row(_pad, 2)
    {
      if ( m.size() ) _it = new IMap::Iterator(m._imp->_realmap);
    }
    ~LBMapIterator()
    { if ( _it ) delete _it; }

    bool next() 
    { 
      if ( _it && _it->next() )
	{
	  _end = false;
	  _pad[0] = _it->currentPair()->key->operator->();
	  _pad[1] = _it->currentPair()->value->operator->();
	  return true;
	}
      _end = true;
      return false;
    }
    const LBVarArgs* getCurrentRow() 
    { 
      if ( ! _end ) 
	return &_row; 
      return 0; 
    }
  private:
    const LBObject *_pad[2];
    bool _end;
    IMap::Iterator *_it;
    LBSimpleVarArgs _row;
  };

  LBConstIterator* LBMap::getIterator() const
  {   return new LBMapIterator(*this); }

  bool LBMap::contains(const LBObject* key) const
  {
    if ( size() == 0 ) return false;
    LBObjPtr dummyvalue;
    LBObjPtr ktemp( const_cast<LBObject*>(key) );
    bool result = _imp->_realmap.find(ktemp, dummyvalue);
    ktemp.release();
    return result;
  }

  LBObject& LBMap::operator[](const LBObject& i)   // throw exception when the indexed element doesn't exist
  {  
    LBObject *obj = subscriptGet(&i);
    if ( ! obj ) 
      throw(LBException("Can not get reference for null emelement in map") );
    return *obj;
  }

  const LBObject& LBMap::operator[](const LBObject& i) const // throw exception when the indexed element doesn't exist
  {
    const LBObject* obj = subscriptConstGet(&i); 
    if ( ! obj ) 
      throw(LBException("Can not get reference for null emelement in map") );
    return *obj;    
  }

  bool LBMap::insert(const LBObject& k, const LBObject& v)
  {    return insert(&k, &v); }

  bool LBMap::insert(const LBObject* k, const LBObject* v)
  {
    LBObjPtr vp, kpinside;
    LBObjPtr kp( const_cast<LBObject*>(k) );
    if ( _imp && _imp->_realmap.find(kp, kpinside, vp) )
      {
	kp.release();
	return false;
      }
    
    kp.release();
    kp = LBObjPtr(k?k->clone():0);
    vp = LBObjPtr(v?v->clone():0);
    uniGuts(_imp); // copy on write
    _imp->_realmap[kp] = vp;
    return true;
  }

  bool LBMap::insert(LBObject* k, LBObject* v)
  {
    LBObjPtr vp, kpinside;
    LBObjPtr kp( k );
    if ( _imp && _imp->_realmap.find(kp, kpinside, vp) )
      {
	kp.release();
	return false;
      }
    
    vp = LBObjPtr(v);
    uniGuts(_imp); // copy on write
    _imp->_realmap[kp] = vp;
    return true;
  }

  void LBMap::clear()
  {
    if ( size() == 0 )
      return;
    _imp = LBMapImpPtr();
  }

  bool LBMap::erase(const LBObject* key)
  {
    if ( size() == 0 )
      return false;
    unicopy(_imp);
    LBObjPtr kp( const_cast<LBObject*>(key) );
    bool d=_imp->_realmap.erase(kp);
    kp.release();
    return d;
  }
    
  void LBMap::set(const LBObject* key, const LBObject* val) // insert when the key does not exist, replace otherwise
  {
    subscriptSet(key, val);
  }

  void LBMap::set(LBObject *key, LBObject* value) // insert when the key does not exist, replace otherwise
  {
    uniGuts(_imp);
    LBObjPtr kp(key), vp(value);
    _imp->_realmap[kp]=vp;
  }

  void LBMap::set(const LBObjPtr &key, const LBObjPtr& value) // insert when the key does not exist, replace otherwise
  {
    uniGuts(_imp);
    _imp->_realmap[key]=value;
  }

  LBVector* LBMap::keys() const
  {
    if ( size() == 0 )
      return new LBVector();

    LBVector *result = new LBVector(size());
    IMap::Iterator it(_imp->_realmap);
    for(int i = 0; it.next(); i++ )
      {
	const IMap::Pair *p = it.currentPair();
	result->setElement( i, *p->key );
      }
    return result;
  }

  LBVector* LBMap::values() const
  {
    if ( size() == 0 )
      return new LBVector();

    LBVector *result = new LBVector(size());
    IMap::Iterator it(_imp->_realmap);
    for(int i = 0; it.next(); i++ )
      {
	const IMap::Pair *p = it.currentPair();
	result->setElement( i, *p->value );
      }
    return result;
  }


  LBEXPORT_MEMBER_FUNC(Luban::LBMap, luban_insert, "insert", "bool map::insert(object key, object value)" ); 
  LBObject* LBMap::luban_insert(const LBVarArgs *args)
  {
    if ( args && ( args->numArgs()==2) )
      {
	const LBObject *key = args->getArg(0);
	const LBObject *value = args->getArg(1);
	bool res = insert(key, value);
	return new LBBool(res);
      }
    throw LBException(string("Invalid number of args passed to map::insert(object key, object value)"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBMap, luban_remove, "remove", "bool map::remove(object key)" ); 
  LBObject* LBMap::luban_remove(const LBVarArgs *args)
  {
    if ( args && ( args->numArgs()==1) )
      {
	const LBObject *key = args->getArg(0);
	bool res = erase(key);
	return new LBBool(res);
      }
    throw LBException(string("Invalid number of args passed to map::remove()"));
  }


  LBEXPORT_MEMBER_FUNC(Luban::LBMap, luban_keys, "keys", "vector map::keys()" ); 
  LBObject* LBMap::luban_keys(const LBVarArgs *args)
  {
    if ( !args || args->numArgs()==0 )
      return keys();
    throw LBException(string("Invalid number of args passed to map::keys()"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBMap, luban_values, "values", "vector map::values()" ); 
  LBObject* LBMap::luban_values(const LBVarArgs *args)
  {
    if ( !args || args->numArgs()==0 )
      return values();
    throw LBException(string("Invalid number of args passed to map::values()"));
  }


  
}
