#include <iostream>

#include "lbthread/lbthread.hpp"

#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbfullsymbol.hpp"
#include "lbtypes/lbtypefactory.hpp"
#include "lbtypes/lbtypespecbasic.hpp"

#include "luban/lbcomponentinfo.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/luban_coderepository.hpp"
#include "luban/luban_namespace.hpp"
#include "luban/lbtypespecstruct.hpp"
#include "luban/luban_codegen.hpp"
#include "luban/lbstruct.hpp"


namespace Luban
{
  static LubanCodeRepository*  &_lcr()
  {
    static LubanCodeRepository *lcr = 0;
    return lcr;
  }

  // here are coarse level multi-thread control tools
  // Struct interface resolving is simply designed to be a global single-threaded routine
  static LBMutex& resolveMutex()
  {
    static LBMutex m;
    return m;
  }
  
  void LubanSymbolResolver::setRepository(LubanCodeRepository* r)
  {
    if ( _lcr() ) delete _lcr();
    _lcr() = r;
  }

  bool LubanSymbolResolver::addSymbol(const LBFullSymbol& fullname, TreeNode* ptree, string& errs)
  {
    LBComponentInfo *comp = new LBComponentInfo(fullname, ptree );
    return addSymbol(fullname, comp, errs);
  }

  bool LubanSymbolResolver::addImportedType(const LBFullSymbol& fullname, string& errs)
  {
    LBComponentInfo *comp = new LBComponentInfo(fullname );
    return addSymbol(fullname, comp, errs);
  }

  bool LubanSymbolResolver::addOrReplaceSymbol(const LBFullSymbol& fullname, TreeNode* ptree, string& errs)
  {
    LBComponentInfo *comp = new LBComponentInfo(fullname, ptree );
    return addOrReplaceSymbol(fullname, comp, errs);
  }

  bool LubanSymbolResolver::instantiateSymbol(const LBFullSymbol& sym, string& errs)
  {
    LBComponentInfo *comp = findOrLoadSymbol(sym, errs);
    if ( ! comp )
      return false;
    return instantiate(comp, errs);
  }

  bool LubanSymbolResolver::hasSymbol(const LBFullSymbol& sym)
  {
    string dummy;
    return findOrLoadSymbol(sym, dummy) != 0;
  }

  ostream& LubanSymbolResolver::printSymbol(const LBFullSymbol& sym, ostream& ost)
  {
    if ( ! hasSymbol(sym) )
      {
	ost << sym.toString() << " does not exist";
	return ost;
      }
    
    string errs;
    const LBTypeSpec *tp = resolveTypeSymbol(sym, errs);
    if ( ! tp )
      {
	ost << sym.toString() << " has error: "<<errs;
	return ost;
      }

    const LBTypeSpecStruct *tps = dynamic_cast<const LBTypeSpecStruct*>(tp);
    if ( tps )
      {
	ost << sym.toString()<< "\t" << tps->getStructInterface()->toString();
	return ost;
      }

    ost << sym.toString()<< "\t" <<tp->toString();
    return ost;

  }

  bool LubanSymbolResolver::addSymbol(const LBFullSymbol& fullname, LBComponentInfo* s, string& errs)
  {
    LBFullSymbol fname(fullname);
    LBSymbol name;
    if ( ! fname.pop(name) )
      {
	errs += "Invalid symbol name: "+fullname.toString();
	return false;
      }
    LBNameSpace *ns = LBNSUtil::findOrCreateNS(fname, errs);
    if ( ! ns )
      {
	errs += "Can not find or create name space: "+fname.toString();
	return false;
      }

    if ( s->_type == LBComponentInfo::TYPEDEF )
      s->_typedefextsymbols = LubanCodeGenerator::getTypeDefExtSymbols(s->_parsetree, s->_homens);
    
    if ( ns->addSymbol(name, s) ) return true;

    errs += "Symbol name conflict, "+fullname.toString()+" already exists in name space\n";
    return false;

  }

  bool LubanSymbolResolver::addOrReplaceSymbol(const LBFullSymbol& fullname, LBComponentInfo* s, string& errs)
  {
    LBFullSymbol fname(fullname);
    LBSymbol name;
    if ( ! fname.pop(name) )
      {
	errs += "Invalid symbol name: "+fullname.toString();
	return false;
      }
    LBNameSpace *ns = LBNSUtil::findOrCreateNS(fname, errs);
    if ( ! ns )
      {
	errs += "Can not find or create name space: "+fname.toString();
	return false;
      }
    
    return ns->addOrReplaceSymbol(name, s);

  }

  LBStruct* LubanSymbolResolver::resolveStructSymbol(const LBFullSymbol& fullname, string& errs)
  {
    LBComponentInfo *comp = findOrLoadSymbol(fullname, errs);
    if ( ! comp )
      {
	errs += "No such struct type "+fullname.toString()+"\n";
	return 0;
      }
    
    if ( comp->_status == LBComponentInfo::ERROR )
      {
	errs += comp->_fullname.toString()+" has error and can not be used\n";
	return 0;
      }

    if ( comp->_status != LBComponentInfo::ALLREADY )
      if ( ! instantiate(comp, errs) )
	return 0;
    if ( ! comp->_struct )
      errs += "Symbol "+fullname.toString()+ " is not a struct\n";
    return comp->_struct;

  }

  const LBTypeSpec* LubanSymbolResolver::resolveTypeSymbol(const LBFullSymbol& fullname, string& errs)
  {
    LBComponentInfo *comp = findOrLoadSymbol(fullname, errs);
    if ( ! comp )
      {
	errs += "No such type symbol "+fullname.toString()+"\n";
	return 0;
      }

    if ( comp->_type == LBComponentInfo::INVALID || comp->_status == LBComponentInfo::ERROR )
      {
	errs += comp->_fullname.toString()+" has error and can not be used\n";
	return 0;
      }
    
    if ( comp->_status != LBComponentInfo::ALLREADY && comp->_status != LBComponentInfo::INTERFACEREADY  )
      if ( ! resolveTypeOrInterface(comp, errs) )
	return 0;

    return comp->_lubantype;

  }

  bool LubanSymbolResolver::saveSymbolToRepository(const LBFullSymbol& s, string& errs)
  {
    if ( ! _lcr() )
      {
	errs += "There is no code repository set to persist Luban components\n";
	return false;
      }
    LBComponentInfo *comp = findOrLoadSymbol(s,errs);
    if ( ! comp )
      {
	errs += "Symbol: "+s.toString()+" does not exist";
	return false;
      }

    // only save compiled components
    if ( ! instantiate(comp, errs) )
      {
	errs += "Symbol: "+s.toString()+" has error and can not be compiled. Error:"+comp->errormsg();
	return false;
      }

    return _lcr()->persistSymbol(s, *comp, errs);
    
  }

  LBComponentInfo* LubanSymbolResolver::findOrLoadSymbol(const LBFullSymbol& s, string& errs)
  {
    LBFullSymbol nsname(s);
    LBSymbol name;
    if ( !nsname.pop(name) )
      {
	errs += "Invalid symbol name "+s.toString();
	return 0;
      }

    LBNSUtil::STATUS st;
    LBNameSpace *ns = LBNSUtil::findNS(nsname,st);
    if ( ns )
      {
	LBComponentInfo *comp = 0;
	LBNameSpace *dummy = 0;
	if ( ns->findSymbol(name, dummy, comp) )
	  if ( comp )
	    return comp;
	  else
	    {
	      errs += s.toString()+" is the name of a name space, not a component\n";
	      return 0;
	    }

	if ( _lcr() )
	  {
	    comp = _lcr()->retrieveSymbol(s, errs);
	    if ( ! comp )
	      {
		errs += "Symbol "+s.toString()+" can not be found in name space or repository\n";
		return 0;
	      }
	    // follwoing code guards against other threads
	    if ( ns->addSymbol(name, comp) )
	      return comp;
	    delete comp;
	    comp = ns->findComponent(name);
	    if ( comp )
	      return comp;
	    errs += "Symbol definition confliction. Other components may define symbol "+s.toString()+" as name space instead of a component\n";
	    return 0;
	  }

	return 0;

      }

    if ( st != LBNSUtil::NONEXIST )
      {
	errs += "Symbol conflict, "+nsname.toString()+" is not a valid name space symbol because there is terminal symbol in its path";
	return 0;
      }

    if ( _lcr() )
      {
	LBComponentInfo *comp = _lcr()->retrieveSymbol(s, errs);
	if ( ! comp )
	  {
	    errs += "Symbol "+s.toString()+" can not be found in name space or repository\n";
	    return 0;
	  }

	LBNameSpace *newns = LBNSUtil::findOrCreateNS(nsname, errs);
	if ( ! newns )
	  {
	    errs += "Can not create new name space:" +nsname.toString();
	    return 0;
	  }

	// follwoing code guards against other threads
	if ( newns->addSymbol(name, comp) )
	  return comp;
	delete comp;
	comp = newns->findComponent(name);
	if ( comp )
	  return comp;
	errs += "Symbol definition confliction. Other components may define symbol "+s.toString()+" as name space instead of component\n";
	return 0;
      }

    errs += "Can not find the symbol "+ s.toString()+ " in name space or code repository\n";
    return 0;

  }


  bool LubanSymbolResolver::instantiate(LBComponentInfo *comp, string& errs)
  {
    if ( comp->_type == LBComponentInfo::INVALID || comp->_status == LBComponentInfo::ERROR)
      return false;

    if ( comp->_status == LBComponentInfo::ALLREADY )
      return true;

    // resolve interface first
    if ( comp->_status != LBComponentInfo::INTERFACEREADY  )
      if ( ! resolveTypeOrInterface( comp, errs ) )
	return false;

    switch ( comp->_type )
      {
      case LBComponentInfo::STRUCT:
	{
	  LBMutexLocker mlock(comp->_instantiationMut);
	  if ( comp->_struct == 0 )
	    {
	      comp->_struct = LubanCodeGenerator::generateStructDef(comp->_parsetree, comp->_structinterface, comp->_homens, errs);
	    }

	  if ( comp->_struct == 0 )
	    {
	      errs += "Failed to generate the definition of struct "+comp->fullname().toString()+"\n";
	      comp->invalidate(errs);
	      return false;
	    }

	  comp->_status = LBComponentInfo::ALLREADY;
	  return true;
	}

      case LBComponentInfo::PRIMITIVETYPE:
	return true; // primitive is completely inited by resolveTypeOrInterface

      case LBComponentInfo::TYPEDEF:
	{
	  if ( comp->_structinterface && ! comp->_struct )
	    {
	      comp->_struct = resolveStructSymbol( *comp->_structinterface->name(), errs);
	      if ( ! comp->_struct )
		{
		  errs += "Failed to instantiate typedef symbol "+comp->_fullname.toString()+"\n";
		  comp->invalidate(errs);
		  return false;
		}
	    }
	  comp->_status = LBComponentInfo::ALLREADY;
	  return true;
	}

      default:
	;
      }

    errs += "Unknown component type\n";
    comp->invalidate(errs);
    return false;

  }


  const LBStructInterface *LubanSymbolResolver::resolveStructInterface(const LBFullSymbol& sname, string& errs)
  {
    LBComponentInfo *comp = findOrLoadSymbol(sname,errs);
    if ( ! comp )
      {
	errs += "Symbol: "+sname.toString()+" does not exist";
	return 0;
      }

    if ( ! resolveTypeOrInterface(comp, errs) )
      return 0;

    if ( ! comp->_structinterface )
      {
	errs += "Symbol "+sname.toString()+" is not a struct\n";
	return 0;
      }

    return comp->_structinterface;

  }

  class AdHocCycleChecker
  {
  public:
    AdHocCycleChecker(bool *inprocess)
      : _inprocess( inprocess ), _checkOK(false)
    {
      if ( ! *_inprocess )
	{
	  _checkOK = true;
	  *_inprocess = true;
	}
    }
    ~AdHocCycleChecker()
    {
      if ( _checkOK )
	*_inprocess = false;
    }

    bool checkOK() { return _checkOK; }
  private:
    bool *_inprocess;
    bool _checkOK;
  };

  // This is a single threaded function
  bool LubanSymbolResolver::resolveInterfaceSingleThread(LBComponentInfo* comp, string& errs)
  {
    if ( comp->_type == LBComponentInfo::INVALID || comp->_status == LBComponentInfo::ERROR )
      {
	errs += "Symbol "+comp->fullname().toString()+ " has error "+ comp->errormsg();
	return false;
      }

    AdHocCycleChecker circhk(&comp->_interfaceResolvingInProcess);

    if ( ! circhk.checkOK() )
      {
	errs += "Circular struct interface reference involve component:"+comp->fullname().toString()+"\n";
	return false;
      }

    if ( comp->_structinterface )
      return true;

    LBStructInterface *itfc = 0;
    // below is a non recursive call
    itfc = LubanCodeGenerator::generateStructInterface(comp->_parsetree, comp->_homens, errs);
	    
    if ( ! itfc )
      {
	comp->invalidate("Can not resolve interface. "+errs);
	return false;
      }

    if (  itfc->parentsUnresolved() )
      for(int i = 0; i < itfc->numOfParents(); i++ )
	{
	  const LBFullSymbol *pname = itfc->parentName(i);
	  LBComponentInfo *parenti = LubanSymbolResolver::findOrLoadSymbol(*pname, errs);
	  if ( ! parenti )
	    {
	      errs += "Parent interface "+pname->toString()+" does not exist\n";
	      comp->invalidate(errs);
	      delete itfc;
	      return false;
	    }
	  if ( ! parenti || ! resolveTypeOrInterfaceSingleThread(parenti, errs) )
	    {
	      errs += "Failed to resolve parent interface "+pname->toString()+"\n";
	      comp->invalidate(errs);
	      delete itfc;
	      return false;
	    }
	  if ( ! parenti->_structinterface )
	    {
	      errs += "Parent interface "+pname->toString()+" is not a struct as expected\n";
	      comp->invalidate(errs);
	      delete itfc;
	      return false;
	    }
	  if ( !itfc->setParentInterface(i, *parenti->_structinterface, errs) )
	    {
	      errs += "Failed to merge parent interface "+pname->toString();+"\n";
	      comp->invalidate(errs);
	      delete itfc;
	      return false;
	    }
	}

    // resolve all other property type symbols used by the interface
    const LBFullSymbolVec *extsvec = itfc->externalSymbols();
    if (  extsvec )
      for( int i=0; i<extsvec->size(); i++)
	if ( ! resolveTypeOrInterfaceSingleThread( (*extsvec)[i], errs ) )
	  {
	    errs += "Failed to resolve property type symbol: "+(*extsvec)[i].toString();
	    comp->invalidate(errs);
	    delete itfc;
	    return false;
	  }

    if ( ! itfc->resolveAllPropertyTypes(errs) )
      {
	errs += "Failed to resolve property types\n";
	comp->invalidate(errs);
	delete itfc;
	return false;
      }

    comp->_structinterface = itfc;

    if ( comp->_lubantype == 0 )
      comp->_lubantype = new LBTypeSpecStruct(itfc);

    comp->_status = LBComponentInfo::INTERFACEREADY;

    return true;
  }


  bool LubanSymbolResolver::resolveTypeOrInterfaceSingleThread(const LBFullSymbol& symname, string& errs)
  {
    LBComponentInfo *extsym = LubanSymbolResolver::findOrLoadSymbol(symname, errs); // non-recursive
    if ( extsym == 0 )
      {
	errs += "Refered external symbol "+ symname.toString()+ " can not be found\n";
	return false;
      }

    return resolveTypeOrInterfaceSingleThread( extsym, errs );

  }

  bool LubanSymbolResolver::resolveTypeOrInterfaceSingleThread(LBComponentInfo* extsym, string& errs)
  {
    switch ( extsym->_type )
      {
      case LBComponentInfo::TYPEDEF:
	return resolveTypeDefSingleThread(extsym, errs);
      case LBComponentInfo::STRUCT:
	return resolveInterfaceSingleThread(extsym, errs);
      case LBComponentInfo::PRIMITIVETYPE: 
	return  resolveTypeOrInterface(extsym, errs);
      default:
	errs += "Invalid component type\n";
	return false;
      }

    return true;

  }

  // this is tne entrance to single thread routines, a global lock is locked here
  bool LubanSymbolResolver::resolveTypeOrInterface(LBComponentInfo *comp, string& errs)
  {
    if ( comp->_lubantype )
      return true;

    if ( comp->_type == LBComponentInfo::PRIMITIVETYPE )
      {
	LBMutexLocker mlock(comp->_instantiationMut);
	  
	if ( ! comp->_lubantype )
	  {
	    // the following call will cause the loading of the dll if not loaded
	    const LBTypeInfo *primitive = LBTypeFactory::findLubanType(comp->_fullname);
	    if ( primitive == 0 )
	      {
		errs += "Can not import data type "+comp->_fullname.toString()+", error: "+string(LBTypeFactory::checkLubanTypeError(comp->_fullname))+"\n";
		comp->invalidate(errs);
		return false;
	      }
	      
	    comp->_lubantype = new LBTypeSpecBasic(primitive);
	    comp->_status = LBComponentInfo::ALLREADY;
	    return true;
	  }
	return true;
      }

    {
      LBMutexLocker mlock( resolveMutex() );
      return resolveTypeOrInterfaceSingleThread(comp, errs);
    }
  }
    
  bool LubanSymbolResolver::resolveTypeDefSingleThread(LBComponentInfo *comp, string& errs)
  {
    if ( comp->_type == LBComponentInfo::INVALID || comp->_status == LBComponentInfo::ERROR )
      {
	errs += "Symbol "+comp->fullname().toString()+ " has error "+ comp->errormsg();
	return false;
      }

    if ( comp->_lubantype )
      return true;

    AdHocCycleChecker circhk(&comp->_interfaceResolvingInProcess);

    if ( ! circhk.checkOK() )
      {
	errs += "Circular type definition involve component:"+comp->fullname().toString()+"\n";
	return false;
      }

    if ( comp->_typedefextsymbols  )
      {
	for(int i = 0; i < comp->_typedefextsymbols->size(); i++ )
	  {
	    const LBFullSymbol& symname = (*comp->_typedefextsymbols)[i];
	    if ( ! resolveTypeOrInterfaceSingleThread(symname, errs) )
	      {
		errs += "Can not resolve external type symbol: "+ symname.toString()+" for type def expression";
		comp->invalidate(errs);
		return false;
	      }
	  }
      }

    // below is a non recursive call
    comp->_lubantype = LubanCodeGenerator::parseTypeDef(comp->_parsetree, comp->_homens, errs);

    if ( comp->_lubantype == 0 )
      {
	comp->invalidate("Can not resolve typedef. "+errs);
	return false;
      }

    LBTypeSpecStruct *temp = dynamic_cast<LBTypeSpecStruct*>(comp->_lubantype);
    if ( temp )
      {
	comp->_structinterface = const_cast<LBStructInterface*>(temp->getStructInterface());
	comp->_status = LBComponentInfo::INTERFACEREADY;
	return true;
      }

    comp->_status = LBComponentInfo::ALLREADY;

    return true;

  }



}
