#include <iostream>
#include <string>

#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbexception.hpp"

#include "luban/lbtypespecstruct.hpp"
#include "luban/lbstructinterface.hpp"
#include "luban/lbstruct.hpp"
#include "luban/luban_symbolresolver.hpp"


namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBTypeSpecStruct, 1,0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBTypeSpecStruct)

  LBTypeSpecStruct::LBTypeSpecStruct()
    : _structitfc(0),_mode(LBStructInterface::SYNCH) 
  {}

  LBTypeSpecStruct::LBTypeSpecStruct( const LBStructInterface* st )
    : _structitfc(st), _mode(LBStructInterface::SYNCH)
  {}

  LBTypeSpecStruct::LBTypeSpecStruct( LBStructInterface::ExecMode m ) 
    : _structitfc(0), _mode(m)  
  {}

  LBTypeSpecStruct::~LBTypeSpecStruct() 
  {}
    
  string LBTypeSpecStruct::toString() const
  {
    if ( _structitfc )
      return _structitfc->name()->toString();
    if ( _mode == LBStructInterface::SYNCH )
      return string("any synch struct type");
    return string("any asynch struct type");
  }

  ostream& LBTypeSpecStruct::toStream( ostream& ost ) const
  {
    if ( _structitfc )
      {
	ost.put('1'); ost.put('0');
	return _structitfc->name()->toStream(ost);
      }
    ost.put('0'); ost.put(' ');
    ost<<(int)_mode;
    ost.put('\n');
    return ost;
  }

  istream& LBTypeSpecStruct::fromStream( istream& ist, int major, int minor )
  {
    char oneorzero = ist.get();
    char sp = ist.get();
    if ( sp != ' ' || oneorzero != '1' && oneorzero != '0' )
      throw LBException("Corrupted stream for struct type spec, invalid delimeter");
    if ( oneorzero == '1' )
      {
	LBFullSymbol fullname;
	fullname.fromStream(ist);
	string errs;
	_structitfc = LubanSymbolResolver::resolveStructInterface(fullname, errs);
	if ( ! _structitfc )
	  throw LBException("Can not resolve struct type "+fullname.toString()+" inside the struct type spec. Err: "+errs);
	return ist;
      }
    int m;
    ist >> m;
    char slashn = ist.get();
    if ( slashn != '\n' )
      throw LBException("Corrupted stream for struct type spec, invalid delimeter");
    _mode = LBStructInterface::ExecMode(m);
    return ist;
  }

  bool LBTypeSpecStruct::checkObj(const LBObject& obj) const
  {
    const LBStruct *stobj = dynamic_cast<const LBStruct*>(&obj);
    if ( ! stobj )
      return false;
    if ( ! _structitfc )
      return stobj->interface().mode() == _mode;
    if ( _structitfc->name() )
      return stobj->interface().hasInterface(*_structitfc->name());
    return false;
  }

  LBObject* LBTypeSpecStruct::createObj(const LBVarArgs *args) const
  {
    int numargs = args?args->numArgs():0;
    if ( numargs != 0 )
      throw LBException("struct component object construction does not take arguments");
    
    string errs;
    LBStruct *result =  LubanSymbolResolver::resolveStructSymbol(*_structitfc->name(), errs);
    if ( ! result )
      throw LBException("Error constructing struct component "+_structitfc->name()->toString()+ " error message: "+errs);
    LBStruct *copy = static_cast<LBStruct*>(result->clone());
    copy->initializeProperties();
    copy->prepareOutputForProcessUse();
    return copy;
  }

  bool LBTypeSpecStruct::operator==(const LBTypeSpecStruct& x) const
  {
    if ( ! _structitfc &&  ! x._structitfc ) return _mode == x._mode; 
    return _structitfc == x._structitfc;
  }

  LBDEFAULT_EQUALS_FUNC( Luban::LBTypeSpecStruct )

  const LBStructInterface* LBTypeSpecStruct::getStructInterface() const
  {
    return _structitfc;
  }

  //The below to export member functions defined at LBTypeSpec level
  LBEXPORT_MEMBER_FUNC(Luban::LBTypeSpecStruct, luban_create, "instance", "object instance(object arg1, arg2,....)" ); 
  

}
