#ifndef __LBVECTOR_HPP__
#define __LBVECTOR_HPP__

#include <string>
#include <iosfwd>
#include <deque>

#include "lbtypes/LBDeclareMacros.hpp"
#include "lbtypes/LBObjPtr.hpp"

namespace Luban
{
  class LBVector : public LBObject
  {
  public:
    LBDECLARE( Luban::LBVector )

    LBVector( );
    LBVector(int numelements );
    LBVector( int numElements, const LBObject& val ); // put numElements of val into vector
    LBVector( int numElements, const LBObject* val ); // put numElements of val into vector

    // Below are required interface for any LBObject
    string toString() const;   // for human readable printing
    ostream& toStream(ostream& o) const;  // for persistence and network transportation
    // major.minor is the version number of the object IN THE STREAM
    // which are passed in in case it is needed for backward compatible streaming
    istream& fromStream(istream& i, int major=-1, int minor=-1);     // for persistence and network transportation
    bool equals(const LBObject& another) const;  

    // below are optional
    // return true if succeed to assign the casted value to the target, else return false and leave target unchanged
    bool cast(LBObject* casttotarget ) const; // interesting cast among vector/map/set is implemented here
    LBVector& add(const LBObject& lbo); // concatenate
    LBVector& mul(const LBObject& lbo); // duplicate and concatenate
    bool      before(const LBObject& lbo) const; // simple order of vector, a vector is before another if it is shorter
    LBVector& neg();  // this does a reverse of the order of all elements


    // optional, below are container related functions
    void subscriptSet(const LBObject* index, const LBObject* value);
    // use below two functions with care, the reference becomes shared when the vector get copied/assigned afterwards
    // or invalid when the same element is set to value of different type
    LBObject* subscriptGet(const LBObject* index); // return a reference of the internal element, not a copy
    const LBObject* subscriptConstGet(const LBObject* index) const; // return a reference of the internal element, not a copy
    int size() const;
    LBConstIterator* getIterator() const; // return 0 if not a container type, that is the default implementation
    bool contains(const LBObject* v) const; //check if the vector contains a value
    void clear();


    // LBVector specific interfaces
    bool operator==(const LBVector& v) const;
    // use below two operators with care, the reference becomes shared when the vector get copied/assigned
    // or invalid when the same element is set to different value
    LBObject& operator[](int index); 
    const LBObject& operator[](int index) const;
    void insert(const LBObject& val, int index=-1); //by default append to the end
    void insert(LBObject* ownme, int index=-1); //by default append to the end
    void sort(int startindex=0, int endindex=-1, bool reversed = false);
    LBVector* subVector(int startindex, int endindex) const;
    void push(const LBObject& val);
    void push(LBObject* ownme);
    LBObject *pop();
    void push_front(const LBObject& val);
    void push_front(LBObject* ownme);
    LBObject *pop_front();
    void erase( int pos );
    void erase( int startpos, int endpos );
    int erase( const LBObject *obj, int from=0, int to=-1 );
    int index(const LBObject* x, int startpos = 0, int endpos=-1 ) const;  // exception if x is not contained in the vector
    int replace( const LBObject *oldobj, const LBObject *newobj, int from=0, int to=-1 );

    friend class LBMap;
    friend class LBSet;
    friend class LBVectorConstIterator;


    // be used by friend container types, to enable sharing
    void push( const LBObjPtr& element ); 
    void setElement(int index, const LBObjPtr& actualelement);
    const LBObjPtr& getElement(int index) const;


    // member functions exported to Luban
    LBObject* luban_append(const LBVarArgs *args);
    LBObject* luban_insert(const LBVarArgs *args);
    LBObject* luban_remove(const LBVarArgs *args);
    LBObject* luban_pushfirst(const LBVarArgs *args);
    LBObject* luban_pushlast(const LBVarArgs *args);
    LBObject* luban_popfirst(const LBVarArgs *args);
    LBObject* luban_poplast(const LBVarArgs *args);
    LBObject* luban_sort(const LBVarArgs *args);
    LBObject* luban_subvec(const LBVarArgs *args);
    LBObject* luban_index(const LBVarArgs *args);
    LBObject* luban_removeobj(const LBVarArgs *args);
    LBObject* luban_replace(const LBVarArgs *args);



  private: // nested class and typedefs
    
    typedef std::deque< LBObjPtr > IVector;

    class LBVectorImp
    {
      friend class LBVector;
      friend class LBVectorConstIterator;
    private:
      IVector _realvector;
      
      LBVectorImp() : _realvector() {}
      LBVectorImp(int s) : _realvector(s) {}
      
    };

    typedef RefCountedPtr<LBVectorImp> LBVectorImpPtr;
    
    static void uniGuts(LBVectorImpPtr& imp)
    {
      if ( imp )
	unicopy(imp);
      else
	imp = LBVectorImpPtr( new LBVector::LBVectorImp() );
    }

    
  private: // data member
    
    LBVectorImpPtr _imp;

  private:  // functions and constructor
    
    LBVector( LBVectorImp *imp);
  };

}

#endif
