#ifndef __LBSTRING_HPP__
#define __LBSTRING_HPP__

#include <string>
#include <iosfwd>
#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbchar.hpp"
#include "lbtypes/RefCountedPtr.hpp"
#include "lbtypes/LBDeclareMacros.hpp"

namespace Luban
{
  using std::string;

  class LBConstIterator;
  class LBVector;

  class LBString : public LBObject
  {
    LBDECLARE( Luban::LBString )

    // LBString constructors
    LBString();
    LBString(const char* cstr);
    LBString(const string& str);
    
    // LBObject required interfaces
    string toString() const;   // for human readable printing
    ostream& toStream(ostream& o) const;  // for persistence and network transportation
    // major.minor is the version number of the object IN THE STREAM
    // which are passed in in case it is needed for backward compatible streaming
    istream& fromStream(istream& i, int major=-1, int minor=-1); // for persistence and network transportation
    virtual bool equals(const LBObject& another) const;  

    // optional LBObject interface implemented
    int hash() const;
    bool cast(LBObject* target) const; // cast self and assign to the target, return a false if fail
    LBString& add(const LBObject& lbo); //string concatenation
    LBString& mul(const LBObject& lbo); // string duplication many times as specified
    bool      before(const LBObject& lbo) const;
    LBString& neg();  // this does a string reverse

    // LBString has a partial container like interface
    void subscriptSet(const LBObject* index, const LBObject* value);  // for a[b]=c;
    const LBObject* subscriptConstGet(const LBObject* index) const; // for c = a[b]
    int size() const;
    LBConstIterator* getIterator() const;
    bool contains(const LBObject* x) const;
    void clear();

    // LBString specific interfaces
    void subscriptSet(const LBObject& index, const LBObject& value)
    {   subscriptSet(&index, &value); }
    bool operator==(const LBString& x) const;
    char& operator[](int index); // exception if out of range
    const char& operator[](int index) const; // exception if out of range
    const string& str() const;
    const char* c_str() const;
    LBString* subString(int i1, int i2=-1) const;
    LBVector* split() const;
    LBVector* split(char dlmt) const;
    LBVector* split(const string& dlmt) const;
    int find(const string& sub, int startpos = 0 ) const ; // throw exception when no match find
    int find(char sub, int startpos = 0 ) const ; // throw exception when no match find
    int lower();
    int upper();
    int trim(bool back=true);
    void remove(int pos);
    int remove(int spos, int epos);
    int remove(char c, int spos=0);
    int remove(const string& str, int spos=0);
    void insert(int pos, char c);
    void insert(int pos, const string& str);
    int replace(const string& oldstr, const string& newstr, int pos=0);



    // member functions exported to Luban
    LBObject* luban_subString(const LBVarArgs *args);
    LBObject* luban_split(const LBVarArgs *args);
    LBObject* luban_find(const LBVarArgs *args);
    LBObject* luban_lower(const LBVarArgs *args);
    LBObject* luban_upper(const LBVarArgs *args);
    LBObject* luban_trimfront(const LBVarArgs *args);
    LBObject* luban_trimback(const LBVarArgs *args);
    LBObject* luban_remove(const LBVarArgs *args);
    LBObject* luban_insert(const LBVarArgs *args);
    LBObject* luban_replace(const LBVarArgs *args);
    
  private:

  class LBStringImp
  {
  public:
    string _realstring;
    LBStringImp(const string& s): _realstring(s) {}
    LBStringImp(const char* s) : _realstring(s) {}
    LBStringImp() : _realstring() {}
  };

  static const RefCountedPtr<LBString::LBStringImp>& defaultGuts()
  {
    static RefCountedPtr<LBString::LBStringImp> dftimp( new LBString::LBStringImp() );
    return dftimp;
  }

    RefCountedPtr<LBStringImp> _imp;
    LBChar _indexedchar;
    
  };

}


#endif
