#ifndef __LBTHREAD_HPP__
#define __LBTHREAD_HPP__

///////////////////////////////////////////////////
//
// Luban platform independent C++ thread API
//
//////////////////////////////////////////////////
namespace Luban
{
    class  LBRunable
    {
    public:
      virtual ~LBRunable() {}
      virtual void run() = 0;
    };

  class  LBThread  // thread is to start running when LBThread is intantiated
    {
    public:
      LBThread(LBRunable *payload); // LBThread does NOT own payload
      ~LBThread();

      bool join();
      bool detach();
      bool cancel();
      bool isValid() const;

    private:
      LBThread(const LBThread&);
      LBThread();
      LBThread& operator = (const LBThread&);
    private:
      class LBThreadImp; // definition this class is platform dependent
      LBThreadImp *_threadimp;

    };

    class  LBMutex 
    {
    public:
      LBMutex();
      ~LBMutex();

      bool lock(); 
      bool try_lock(); // return 1 if locked, otherwise 0
      bool unlock();

      friend class LBCondVar;
    private:
      LBMutex(const LBMutex&);
      LBMutex& operator= (const LBMutex&);
    private:
      class LBMutexImp; //  definition of this class is platform dependent
      LBMutexImp *_imp;
    };

  class  LBMutexLocker 
  {
  public:
    LBMutexLocker(LBMutex& mut) : _mutex(mut) { _mutex.lock(); }
    ~LBMutexLocker() { _mutex.unlock();}
  private:
    LBMutexLocker();
    LBMutexLocker(const LBMutexLocker&);
    LBMutexLocker& operator = (const LBMutexLocker&);

    LBMutex&  _mutex;

  };

  class  LBMutexPtrLocker 
  {
  public:
    LBMutexPtrLocker(LBMutex* mut) : _mutex(mut) { if ( _mutex) _mutex->lock(); }
    ~LBMutexPtrLocker() { if ( _mutex) _mutex->unlock();}
  private:
    LBMutexPtrLocker();
    LBMutexPtrLocker(const LBMutexPtrLocker&);
    LBMutexPtrLocker& operator = (const LBMutexPtrLocker&);

    LBMutex*  _mutex;

  };

  class  LBReadWriteLock
  {
  public:
    enum Preference { READER, WRITER, EQUAL };
    LBReadWriteLock(Preference p=WRITER);
    ~LBReadWriteLock();

    bool readLock();
    bool try_readLock();  // return true if locked

    bool writeLock();
    bool try_writeLock(); // return true if locked

    bool unlock();

    void setPreference(Preference p);

  private:
    LBReadWriteLock(const LBReadWriteLock&);
    LBReadWriteLock& operator= (const LBReadWriteLock&);
  private:
    class LBReadWriteLockImp;
    LBReadWriteLockImp *_imp;

  };

  class LBWriteLocker
  {
  public:
    LBWriteLocker( LBReadWriteLock& rwlock ) : _rwlock(rwlock) { _rwlock.writeLock(); }
    ~LBWriteLocker() { _rwlock.unlock(); }

  private:
    LBWriteLocker();
    LBWriteLocker(const LBWriteLocker&);
    LBWriteLocker& operator= (const LBWriteLocker&);
  private:
    LBReadWriteLock& _rwlock;


  };

  class LBReadLocker
  {
  public:
    LBReadLocker( LBReadWriteLock& rwlock ) : _rwlock(rwlock) { _rwlock.readLock(); }
    ~LBReadLocker() { _rwlock.unlock(); }
  private:
    LBReadLocker();
    LBReadLocker(const LBReadLocker&);
    LBReadLocker& operator= (const LBReadLocker&);
  private:
    LBReadWriteLock& _rwlock;
  };

  class  LBCondVar 
  {
    // Potential porting issue, WIN32 does not have direct support for cond var
    // Though there is a pthread win32 implementation
    // and an article
    // http://www.cs.wustl.edu/~schmidt/win32-cv-1.html
    // Strategies for Implementing POSIX Condition Variables on Win32
    //
    // Douglas C. Schmidt and Irfan Pyarali
    // Department of Computer Science
    // Washington University, St. Louis, Missouri

  public:
    LBCondVar(); 
    ~LBCondVar();
    
    bool wait(LBMutex& associated_mut);
    bool signal();
    bool broadcast();
  private:
    LBCondVar(const LBCondVar&);
    LBCondVar& operator= (const LBCondVar&);
  private:
    class LBCondVarImp; //  definition this class is platform dependent
    LBCondVarImp *_imp;
  };

    
}

#endif
