#include <iostream>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <cstring>
#include <cstdlib>

#include "lbtypes/lbobject.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/lbexception.hpp"


#include "luban/luban_parser.hpp"
#include "luban/luban_parsetree.hpp"
#include "luban/luban_codegen.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/luban_import.hpp"
#include "luban/luban_namespace.hpp"
#include "luban/luban_symbolresolver.hpp"

using Luban::LBProcessStruct;
using Luban::LubanCodeGenerator;
using Luban::LubanParser;
using Luban::TreeNode;
using Luban::LBObject;
using Luban::LBObjPtr;
using Luban::LBException;
using Luban::LBFullSymbol;
using Luban::LubanImporter;
using Luban::LBNSUtil;
using Luban::LBNameSpace;
using Luban::LubanSymbolResolver;

using std::string;
using std::istringstream;
using std::ostringstream;
using std::vector;
using std::cout;
using std::cerr;

static const char *lbhome_envvar="LUBAN_HOME";
static const char *lbhome_default="/usr/lib/luban";
static const char *import_file="imports";
static const char *std_lbnsrc="std.lbn";

static int parserDebug = 0;
static vector<string> *srcfiles = 0;
static int maxerror = 0;
static char *basictypelist = 0;
static char *scriptarg = 0;
static bool interactive = false;

static int parseCommandLine(int argc, char **argv)
{
  static const char * const dashd = "-d";
  static const char * const dashe = "-e";
  static const char * const dasht = "-t";
  static const char * const dashi = "-i";
  static const char * const dashs = "-s";

  for( int i = 1; i < argc; i++)
    {
      if ( strcmp( argv[i], dashd ) == 0 )
	{
	  i++;
	  if ( i < argc )
	    parserDebug = atoi(argv[i]);
	  else
	    return 0; // error
	}
      else
	if ( strcmp( argv[i], dashe ) == 0 )
	  {
	    i++;
	    if ( i < argc )
		maxerror = atoi(argv[i]);
	    else
	      return 0;
	  }
	else
	  if ( strcmp( argv[i], dasht ) == 0 )
	    {
	      i++;
	      if ( i < argc )
		basictypelist = argv[i];
	      else
		return 0; // error
	    }
	  else
	    if ( strcmp( argv[i], dashs ) == 0 )
	      {
		i++;
		if ( i < argc )
		  scriptarg = argv[i];
		else
		  return 0; // error
	      }
	    else
	      if ( strcmp( argv[i], dashi ) == 0 )
		interactive = true;
	      else  // all other arguments taken as file names
		  srcfiles->push_back(string(argv[i]));
    }

  if ( srcfiles->size() == 0 && scriptarg == 0 )
    interactive = true;
  return 1; //ok

}

static const char *getEditorCommand()
{
  static const char *editorenv = "EDITOR";
  static const char *defaulteditor ="vi";
  static const char *fname = 0;
  if ( fname == 0 )
    {
      fname = getenv(editorenv);
      if ( fname == 0 || strlen(fname) == 0 )
	fname = defaulteditor;
    }
  return fname;
}

static const char* getTempScriptFileName()
{
  static const char *prefix = "luban_adhoc";
  static const char *postfix = ".lbn";
  static string thename;
  static bool inited=false;
  
  if ( ! inited )
    for( int i=0; ; ++i)
      {
	ostringstream strm;
	strm << prefix << i << postfix;
	thename = strm.str();
	if ( ! std::ifstream(thename.c_str()) ) // make sure the file doesn't exist
	  {
	    inited = true;
	    return thename.c_str();
	  }
     }

  return thename.c_str();

}

static int processCommand(const string& cmd) // return 0 means quit
{
  static const string script("script"), ls("list"), s("s"),l("l"), quit("quit"), q("q");
  static const string edt("edit"), e("e");

  vector<string> tokens;
  for( int i = 0; i<cmd.size(); i++)
    {
      if ( isspace(cmd[i]) ) continue;
      string onetoken;
      do  onetoken += cmd[i++];  
      while ( i < cmd.size() &&  ! isspace( cmd[i] ) );
      tokens.push_back(onetoken);
    }
	
  int tksz = tokens.size();
  if ( tksz == 0 )
    return 1;

  if ( tokens[0] == quit || tokens[0] == q )
    return 0;

  if ( tksz == 1 && (tokens[0] == script || tokens[0] == s ||  tokens[0] == edt || tokens[0] == e ) )
    {
      TreeNode *parsetree=0;
      int errs = 0;
      if ( tokens[0] == script || tokens[0] == s )
	{
	  std::cerr << "Type your script, press ESC key then ENTER to run\n";
	  string onesession;
	  std::cin.clear();
	  string oneline;
	  while ( getline(std::cin, oneline) ) // ESC is the stop key
	    {
	      if ( oneline.size() > 0 && oneline[0] == 0x1B )
		break;
	      onesession += oneline+'\n';
	    }
	  std::cerr << '\n';

	  if ( onesession.size() == 0 )
	    return 1;

	  istringstream ist(onesession);
	  errs = LubanParser::parseStream(ist, "stdin", parsetree);
	}
      else
	{
	  int st = system(0);
	  if ( st == 0 )
	    {
	      std::cerr<<"Editor is not available in your environment\n";
	      return 1;
	    }
	  const char *tempfile = getTempScriptFileName();
	  string edcmd(getEditorCommand());
	  edcmd += " ";
	  edcmd += string(const_cast<const char*>(tempfile));
	  st = system(edcmd.c_str());
	  if ( st != 0 )
	    {
	      std::cerr<<"Editor error\n";
	      return 1;
	    }
  
	  std::ifstream fback(tempfile);

	  char onech;
	  string all;
	  while( fback.get(onech) )
	    all += onech;

	  std::cout<< all<<'\n';

	  istringstream ist(all);
	  errs = LubanParser::parseStream(ist, tempfile, parsetree);

	}

      if ( errs == 0 )
	{
	  if ( parserDebug == 1 ) parsetree->printTree(std::cout);

	  // pass the tree to code generator
	  LBProcessStruct *adhoc=0;
	  string errstr;
	  LubanCodeGenerator::CodeGenStatus st = LubanCodeGenerator::processAUnit(parsetree, adhoc, errstr);
	  if ( st == LubanCodeGenerator::SUCCESS )
	    {
	      if ( adhoc )
		{
		  try {
		    LBObjPtr lobj(adhoc->evalForLastObj());
		    std::cout << lobj.toString()<< "\n";
		    std::cout << "Script successfully evaluated\n";
		  }
		  catch( LBException& lbe)
		    {
		      std::cerr<<"Error when executing Luban code from stdin:\n"<<lbe.msg()<<"\n";
		      std::cerr<<adhoc->toString()<<"\n";
		    }
		  delete adhoc;
		}
	      else
		std::cout << "New component successfully registered\n";
	    }
	  else
	    std::cerr << "Failed to register parsed symbol to name space or generate intermediate code. Error message: "<< errstr<< "\n";
	}
      else
	std::cerr << "Total number of parse errors: "<< errs<< "\n";

      if ( parsetree ) delete parsetree;
      
      return 1;
    }

  if ( tksz <= 2 && ( tokens[0] == ls || tokens[0] == l ) )
    {
      if ( tksz == 2 )
	{
	  LBFullSymbol compname;
	  if ( ! LBFullSymbol::stringToFullSymbol(tokens[1], compname) )
	    {
	      cerr << "Invalid component symbol "<<tokens[1]<<"\n";
	      return 1;
	    }
	  
	  LBNSUtil::STATUS st;
	  LBNameSpace *ns = LBNSUtil::findNS(compname, st);
	  if ( ns )
	    LBNSUtil::printNS(ns,cout);
	  else
	    {
	      LubanSymbolResolver::printSymbol(compname,cout);
	      cout << '\n';
	    }
	}
      else
	LBNSUtil::printNS(LBNSUtil::rootNS(), cout);

      return 1;
    }

  cerr << "Invalid command "<<cmd<<"\n";

  return 1;
}

int main(int argc, char **argv)
{
  // add standard luban source module into compile list
  srcfiles = new vector<string>;

  if ( !parseCommandLine(argc, argv) )
    {
      std::cerr << "Usage: "<< argv[0]<< "\nLuban source filenames \n-d debugLevelNumber(1,2)\n-e max number of erros before abort \n-t <Imported Type List File Name> \n-i interactive command line\n";
      return 1;
    }

  const char *lbh = getenv(lbhome_envvar); // find Luban home
  if ( ! lbh ) lbh = lbhome_default;
  string stdsrc = string(lbh)+string("/")+string(std_lbnsrc);

  // take ./std.lbn if LUBAN_HOME/std.lbn does not exist
  std::ifstream stdstrm(stdsrc.c_str());
  if ( stdstrm )
    srcfiles->insert(srcfiles->begin(),stdsrc);
  else
    {
      std::ifstream stdstrm2(std_lbnsrc);
      srcfiles->insert(srcfiles->begin(),string(std_lbnsrc));
    }


  // import types from C/C++ and other languages
  int numimports = 0;
  if ( !basictypelist )
    {
      // take ./imports if LUBAN_HOME/imports not exist
      string importfile = string(lbh)+string("/")+string(import_file);
      std::ifstream impstrm(importfile.c_str());
      if ( impstrm )
	numimports = LubanImporter::importBasicTypes(importfile.c_str());
      else
	{
	  std::ifstream impstrm2(import_file);
	  if ( impstrm2 )
	    numimports = LubanImporter::importBasicTypes(import_file);
	}
    }
  else
    numimports = LubanImporter::importBasicTypes(basictypelist);

  if ( interactive )
    std::cerr << numimports << " imported types\n";

  if ( parserDebug == 2 ) LubanParser::setDebug(parserDebug);
  LubanParser::setErrorLimit(maxerror);

  vector<LBProcessStruct*> adhocs;
  vector<char*> adhocnames;

  for( int i = 0; i < srcfiles->size(); i++ )
    {
      std::ifstream scu((*srcfiles)[i].c_str());
      if ( !scu )
	{
	  std::cerr<< "Can not open file "<< (*srcfiles)[i] <<"\n";
	  continue;
	}

	TreeNode *parsetree = 0;

	int errs = LubanParser::parseStream(scu, argv[i], parsetree);

	if ( errs == 0 )
	  {
	    if ( parserDebug == 1 ) parsetree->printTree(std::cout);

	    // pass the tree to code generator
	    LBProcessStruct *adhoc=0;
	    string errs;
	    LubanCodeGenerator::CodeGenStatus st = LubanCodeGenerator::processAUnit(parsetree, adhoc, errs);
	    if ( st == LubanCodeGenerator::SUCCESS )
	      {
		if ( adhoc )
		  {
		    adhocs.push_back(adhoc);
		    adhocnames.push_back(argv[i]);
		  }
		continue;
	      }

	    std::cerr << "Failed to register parsed symbol to name space or generate intermediate code for module "<<argv[i]<<". Error message: "<< errs<< "\n";
	  }
	else
	  std::cerr << "Erros when parsing "<< argv[i]<<   ". Total number of errors: "<< errs<< "\n";

	if ( parsetree ) delete parsetree;

      }

  // run all the adhocs from the source file list
  for( int i=0; i<adhocs.size(); i++)
    {
      LBProcessStruct *adhoc = adhocs[i];
      try {
	adhoc->evalForLastObj();
      }
      catch( LBException& lbe)
	{
	  std::cout<<"Error when executing "<<adhocnames[i]<< " "<<lbe.msg()<<"\n";
	  std::cout<<adhoc->toString();
	}
      delete adhoc;
    }

  // script code on the command line
  if ( scriptarg )
    {
      string s(scriptarg);
      istringstream sstrm(s);
      
      TreeNode *parsetree = 0;

      int errs = LubanParser::parseStream(sstrm, "commandline", parsetree);

      if ( errs == 0 )
	{
	  if ( parserDebug == 1 ) parsetree->printTree(std::cout);

	  // pass the tree to code generator
	  LBProcessStruct *adhoc=0;
	  string errs;
	  LubanCodeGenerator::CodeGenStatus st = LubanCodeGenerator::processAUnit(parsetree, adhoc, errs);
	  if ( st == LubanCodeGenerator::SUCCESS && adhoc )
	    {
	      try {
		LBObjPtr lobj(adhoc->evalForLastObj());
		std::cout << lobj.toString()<< "\n";
	      }
	      catch( LBException& lbe)
		{
		  std::cout<<"Error when executing command line script: "<<lbe.msg()<<"\n";
		  std::cout<<adhoc->toString();
		}
	      delete adhoc;
	    }
	  else
	    cout << "Error in command line script: "<<errs;
	}

    }

  // this part is for interactive command line evaluation
  if ( interactive )
    {
      static const char *prompt="Luban> ";
      std::cerr << "script/s - to start scripting\n";
      std::cerr << "edit/e  - to edit script using $EDITOR or vi\n";
      std::cerr << "list/l <namespace name> - to browse name space\n";
      std::cerr << "quit/q - to quit\n";
      while ( true )
	{
	  std::cerr << prompt;
	  string onecmd;
	  std::cin.clear();
	  if ( ! getline( std::cin, onecmd ) )
	    break;

	  if ( onecmd.size() == 0 )
	    continue;

	  if ( processCommand(onecmd) == 0 )
	    break;
	}
    }

  delete srcfiles;
  return 0;

  
}
