#ifndef __LBVMCODE_HPP__
#define __LBVMCODE_HPP__

#include <vector>
#include <iosfwd>

#include "lbtypes/lbexception.hpp"

namespace Luban
{
  // ******************************************************************************************
  //
  // BEHAVIOUR SPEC OF LUBAN PROCESS VIRTUAL MACHINE
  // List of all Operation code of Luban virtual machine code ( sequential struct )
  //
  //   ZEROJUMP:  if ( ! op1 ) goto op2
  //       GOTO:  goto op1
  //      ORJMP:  if ( op1 == true )   { resultop = op1; goto op2 }
  //              if ( op1 is not a bool ) { resultop = error; goto op2 }
  //               used in logical expression to short cut unneeded evaluation
  //     ANDJMP:  if ( op1 == false )   { resultop = op1; goto op2 }
  //              if ( op1 is not a bool ) { resultop = error; goto op2 }
  //               used in logical expression to short cut unneeded evaluation
  //   DISPATCH:  take code from op1 to op2 on a different thread/process/machine
  //              also increase the running thread count for the process struct
  //              also increase the pending update count on resultop
  //    GETITER:  resultop = op1.getIterator();
  //   ITERNEXT:  resultop = op1.next()
  //    ITEROBJ:  resultop = op1/* iterator */.getObj(op2)
  //  RESETITER:  reset op1 as iterator, release the iterator and its associated temp copy of container
  //       HALT:  stop the evaluation of the struct
  //       HOLD:  wait for all dispatched threads/process to finish
  //   HOLDONTO:  wait for the thread updating a variable to finish
  //              op1 the variable to wait
  //     CANCEL:  terminate ALL running threads except the main thread for the struct
  //   BATCHOUT: take op1 as a struct, get all its output ports and set
  //             to the same named output ports of a process truct
  //             if not all the output ports of op1 can be copied, throw exception and halt evaluation
  //        ASN: op1 = op2
  //     MULASN: op1 *= op2
  //     DIVASN: op1 /= op2
  //     MODASN: op1 %= op2
  //     ADDASN: op1 += op2
  //     SUBASN: op1 -= op2
  //      LGCOR: resultop = op1 or op2
  //     LGCAND: resultop = op1 and op2
  //      EQCMP: resultop = op1 == op2
  //     NEQCMP: resultop = op1 != op2
  //        ISA: resultop = op1 isa op2
  //     TYPEOF: resultop = typeof (op1)
  //      LTCMP: resultop = op1 < op2
  //      GTCMP: resultop = op1 > op2
  //      LECMP: resultop = op1 <= op2
  //      GECMP: resultop = op1 >= op2
  //        ADD: resultop = op1 + op2
  //      MINUS: resultop = op1 - op2
  //        MUL: resultop = op1 * op2
  //        DIV: resultop = op1 / op2
  //        MOD: resultop = op1 % op2
  //     LGCNOT: resultop = not op1    ( assert op1 must be a boolean, else throw exception )
  //         PP: ++op1
  //         MM: --op1
  //        NEG: resultop = op1.neg()   
  //
  //     SUBSCR: resultop = op1[op2] this instruction has been discarded, replaced by an operand
  //             the resulttop contains not only the object pointer which the subscription pointing to
  //             it also contains the pointer to the original container and the subscripting index value
  //             in case a different type obj need to replace the old obj

  //    MBRFUNC: resultop = op1AsMemberfuncOperand( op2 )  
  //             member func call, reuse resultop to pass args and return result
  //
  //   PROPERTY: resultop = op1.op2   this instruction has been discarded, use an operand to represent it
  //             the resultop, similar with SUBSCR, should also contains the pointer to the original
  //             struct, in case the property value need to be reset to a different type.
  //             plus the property update event should be passed to the orignal property
  //             Same care need to be taken for memeber functiona call
  //             example,  x.y.z = 1; should trigger update in x and x.y
  //                       x.y.z.append(1) should trigger update in x and x.y
  //                       x.y.z[0] = 1; should trigger update in x and x.y
  //                 NOTE: x.y does NOT apply to static properties
  //                       static property can only be refereed as XTypeName.y
  //                       so the use of global variable is visible from syntax
  //
  // STRUCTEVAL: resultop = op1( op2 )   // struct eval with named args
  //   BATCHSET: op1 = op1(op2) almost the same as STRUCTEVAL, only not to make copy of the struct
  // OBJCONSTRU: resultop = op1->newInstance(op2 ) || op2->cast( op1->newInstance())
  //     VECALL: resultop.vecAll()
  //              op2 indicate multithreading, 1 yes, 0 no
  //     VECINS: resultop.insert(op1)    // cast resultop to vector type, used for [1,2,3] kind of construction
  //     MAPINS: resultop.insert(op1,op2) // cast resultop to map type, used for {"a":1,"b":2,"c":3}
  //     SETINS: resultop.insert(op1) // cast resultop to map type, used for {"a","b","c"}
  //
  //   
  //
  //*******************************************************************************************

  using std::istream;
  using std::ostream;

  class LBVMCode
  {
  public:
    friend class LBProcessStruct;
    friend class LubanCodeGenerator;

    class Operand
    {
    public:
      // some of this operand types can be moved down to operand container/element level
      enum OperandType { UNUSED=0, LOCALVAR, GLOBALVAR, LOCALPROPERTY, VARPROPERTY, GLOBALTYPE/*object or struct type*/,
			 TEMPOBJ, TEMPITER, LITERALOBJ, LITERALTYPE, LABEL, INTVALUE,
			 TEMPNAMEDARG, TEMPANONARG, MBRFUNC, SUBSCR, JTWRITE, JTREAD };
			 

      //
      // GLOABLVAR abd PROPERTY operand are alike
      // they are both a place keeper, it saves the symbol name of the property and global
      // variable, the real object and/or index will be resolved at run time
      // and the result will be saved and reused in the life of the Luban exec process
      // The reason for this is
      // property can change if the struct inherit other struct's interface, in this way
      // the byte code doesn't need to be changed even if the property index changes
      // For persistency reason, the global variable name need to be persisted, so it is
      // good to put them into a different storage than the temp objects
      // 

      OperandType optype() const { return _opdtype; }
      int opindex() const { return _opdaddress; }

      Operand(OperandType t=UNUSED, int addr=-1)
	: _opdtype(t), _opdaddress(addr)
      {
      }

      OperandType _opdtype;
      int _opdaddress;
    };

    enum OPCode { NOOP=0, ADD, MINUS, MUL, DIV, MOD, NEG,PP,MM, CALL, ASN, MULASN, DIVASN, MODASN, ADDASN, SUBASN,
		  ZEROJUMP, GOTO, DISPATCH, GETITER, ITERNEXT, ITEROBJ, RESETITER, HALT, HOLD, HOLDONTO, CANCEL, 
		  BATCHOUT, LGCOR, LGCAND, LGCNOT, EQCMP, NEQCMP, ISA, TYPEOF, GTCMP, LTCMP, GECMP, LECMP, MBRFUNC,
		  STRUCTEVAL, OBJCONSTRU, VECALL, VECINS, MAPINS, SETINS, ORJMP, ANDJMP, BATCHSET };

    LBVMCode(OPCode opc=NOOP)
      : _opcode(opc), _op1(), _op2(), _result()
    {}
    
    OPCode& opcode() { return _opcode; } 
    Operand& op1() { return _op1; }
    Operand& op2() { return _op2; }
    Operand& resultop() { return _result; }

  private:

    OPCode _opcode;
    Operand _op1;
    Operand _op2;
    Operand _result;

  };
      

  class LBVMCodeSequence
  {
  public:
    void append(const LBVMCode& code)
    {  _codevec.push_back(code); }
    int size() const
    { return _codevec.size(); }

    LBVMCode& operator[](int index)
    {  return _codevec[index]; }
    const LBVMCode& operator[](int index) const
    {  return _codevec[index]; }

    ostream& toStream(ostream& ost) const
    {
      // to be done
      throw LBException("Not implemented yet");
    }

    istream& fromStream(istream& ist)
    {
      // to be done
      throw LBException("Not implemented yet");
    }


  private:
    std::vector<LBVMCode> _codevec;

  };

}

#endif
