#include <vector>

#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbfullsymbol.hpp"

#include "luban/lbpropertyinfo.hpp"
#include "luban/lbstructinterface.hpp"


namespace Luban
{
  LBStructInterface::LBStructInterface(ExecMode m)
    : _mode(m), _name(), _ins(0), _outs(0), _stores(0), _statics(0), _pmap(0), _parents(0), _ancestors(0), _extsymbols(0), _parentsresolved(0)
  {}

  LBStructInterface::LBStructInterface(ExecMode m, const LBFullSymbol& name)
    : _mode(m), _name(name), _ins(0), _outs(0), _stores(0), _statics(0), _pmap(0), _parents(0), _ancestors(0), _extsymbols(0), _parentsresolved(0)
  {}

  LBStructInterface::LBStructInterface(const LBStructInterface& itf)
    : _mode(itf._mode), _name(itf._name), _ins(0), _outs(0), _stores(0), _statics(0), _pmap(0), _parents(0), _ancestors(0), _extsymbols(0 ), _parentsresolved(itf._parentsresolved)
  {
    if ( itf._ins ) 
      {
	int sz = itf._ins->size();
	_ins = new PropertyInfoVec(sz);
	for( int i=0; i<sz; i++)
	  (*_ins)[i] = new LBPropertyInfo(*(*itf._ins)[i]);
      }
    if ( itf._outs ) 
      {
	int sz = itf._outs->size();
	_outs = new PropertyInfoVec(sz);
	for( int i=0; i<sz; i++)
	  (*_outs)[i] = new LBPropertyInfo(*(*itf._outs)[i]);
      }
    if ( itf._stores ) 
      {
	int sz = itf._stores->size();
	_stores = new PropertyInfoVec(sz);
	for( int i=0; i<sz; i++)
	  (*_stores)[i] = new LBPropertyInfo(*(*itf._stores)[i]);
      }
    if ( itf._statics ) 
      {
	int sz = itf._statics->size();
	_statics = new PropertyInfoVec(sz);
	for( int i=0; i<sz; i++)
	  (*_statics)[i] = new LBPropertyInfo(*(*itf._statics)[i]);
      }

    if ( itf._pmap ) 
      {
	PropertyMap::Iterator it(*itf._pmap);
	while ( it.next() )
	  {
	    const PropertyMap::Pair *p = it.currentPair();
	    PropertyInfoIndex pdx(*p->value);
	    switch ( pdx._pinfo->attr() )
	      {
	      case LBPropertyInfo::IN:
		pdx._pinfo = (*_ins)[pdx._index];
		break;
	      case LBPropertyInfo::OUT:
		pdx._pinfo = (*_outs)[pdx._index];
		break;
	      case LBPropertyInfo::STORE:
		pdx._pinfo = (*_stores)[pdx._index];
		break;
	      case LBPropertyInfo::STATIC:
		pdx._pinfo = (*_statics)[pdx._index];
		break;
	      }
	    (*pmap())[*p->key] = pdx;
	  }
      }

    if ( itf._parents )
      _parents = new ParentInfoVec(*itf._parents);
    if ( itf._ancestors )
      _ancestors = new AncestorSet(*itf._ancestors);
    if ( itf._extsymbols )
      _extsymbols = new LBFullSymbolVec(*itf._extsymbols) ;
  }

  LBStructInterface::~LBStructInterface()
  {
    if ( _ins ) 
      {
	for( int i=0; i<_ins->size(); i++)
	  delete (*_ins)[i];
	delete _ins;
      }
    if ( _outs )
      {
	for( int i=0; i<_outs->size(); i++)
	  delete (*_outs)[i];
	delete _outs;
      }

    if ( _stores )
      {
	for( int i=0; i<_stores->size(); i++)
	  delete (*_stores)[i];
	delete _stores;
      }

    if ( _statics )
      {
	for( int i=0; i<_statics->size(); i++)
	  delete (*_statics)[i];
	delete _statics;
      }

    if ( _pmap ) delete _pmap;

    if ( _parents) delete _parents;
    if ( _ancestors) delete _ancestors;

    if ( _extsymbols ) delete _extsymbols;

  }

  const LBPropertyInfo *LBStructInterface::propertyInfo(int index, LBPropertyInfo::ExecAttr attr) const
  {
    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( index >=0 && _ins && index < _ins->size() )
	  return (*_ins)[index];
	return 0;
      case LBPropertyInfo::OUT:
	if ( index >=0 && _outs && index < _outs->size() )
	  return (*_outs)[index];
	return 0;
      case LBPropertyInfo::STORE:
	if ( index >=0 && _stores && index < _stores->size() )
	  return (*_stores)[index];
	return 0;
      case LBPropertyInfo::STATIC:
	if ( index >=0 && _statics && index < _statics->size() )
	  return (*_statics)[index];
	return 0;
      default:
	;
      }
    
    return 0;

  }

  const LBPropertyInfo* LBStructInterface::propertyInfo(const LBSymbol& name, int &index) const
  {
    if ( ! _pmap )
      return 0;
    PropertyInfoIndex p;
    if ( ! _pmap->find(name, p) )
      return 0;
    index = p._index;
    return p._pinfo;
  }

  string LBStructInterface::toString() const
  {
    string result(_name.toString()+'(');
    if ( _ins ) 
	for( int i=0; i<_ins->size(); i++)
	  result += (*_ins)[i]->toString()+";";
    if ( _outs ) 
	for( int i=0; i<_outs->size(); i++)
	  result += (*_outs)[i]->toString()+";";
    if ( _stores ) 
	for( int i=0; i<_stores->size(); i++)
	  result += (*_stores)[i]->toString()+";";
    if ( _statics ) 
	for( int i=0; i<_statics->size(); i++)
	  result += (*_statics)[i]->toString()+";";
    result += ')';
    return result;
  }

  int LBStructInterface::numberOfPropertiesByAttr(LBPropertyInfo::ExecAttr attr) const
  {
    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( _ins ) return  _ins->size();
	return 0;
      case LBPropertyInfo::OUT:
	if (  _outs ) return _outs->size();
	return 0;
      case LBPropertyInfo::STORE:
	if ( _stores ) return _stores->size();
	return 0;
      case LBPropertyInfo::STATIC:
	if ( _statics ) return  _statics->size();
	return 0;
      default:
	;
      }
    
    return 0;

  }

  LBStructInterface::ExecMode LBStructInterface::mode() const
  { return _mode; }

  const LBFullSymbol* LBStructInterface::name() const
  { return  &_name; }

  bool LBStructInterface::hasInterface(const LBFullSymbol& structname) const
  {
    if ( _name == structname )
      return true;
    if ( _ancestors )
      return _ancestors->find( structname );
    return false;
  }

  int LBStructInterface::numOfParents() const
  {
    if ( _parents ) return _parents->size();
    return 0;
  }

  bool LBStructInterface::parentsUnresolved() const
  {
    return _parentsresolved < numOfParents();

  }

  const LBFullSymbol* LBStructInterface::parentName(int index) const
  {
    if ( index>=0 && _parents && index<_parents->size()) return &(*_parents)[index]._parentname;
    return 0;
  }

  bool LBStructInterface::setParentInterface(int parentindex, const LBStructInterface& pf, string& errs, bool forcereset)
  {
    if ( ! _parents )
      {
	errs += "Invalid call to merge parent struct interface, the struct does NOT have parents\n";
	return false;
      }
    if ( parentindex < 0 || parentindex >=_parents->size() )
      {
	errs += "Invalid parent struct interface index\n";
	return false;
      }

    if ( ! forcereset && (*_parents)[parentindex]._resolved )
      return true;
    
    // HERE is the place to put down the INTERFACE INHERITANCE POLICY
    // 1. Only INs and OUTs are inhereited
    // 2. Sub struct can ONLY change the init expression of a inherited property
    //    name, attr and type can NOT be changed
    //
    int numins = pf.numberOfPropertiesByAttr( LBPropertyInfo::IN );
    for(int i=0; i<numins; i++)
      {
	const LBPropertyInfo *oneinp = pf.propertyInfo(i, LBPropertyInfo::IN);
	PropertyInfoIndex existingp;
	if ( pmap()->find(oneinp->name(), existingp ) )
	  {
	    if ( existingp._pinfo->attr() != LBPropertyInfo::IN )
	      {
		errs += "Conflicting definition of property from inheritance for property: "+oneinp->name().toString()+". There is a same named property that is defined as a non-input port";
		return false;
	      }
	    // check types
	    const LBTypeSpec *existingptype = existingp._pinfo->typeSpec();
	    const LBTypeSpec *oneinptype = oneinp->typeSpec();
	    if ( existingptype != 0 && oneinptype == 0 
		 || existingptype == 0 && oneinptype != 0 
		 || existingptype && oneinptype &&  ! existingptype->equals((const LBObject&)(*oneinptype)) )
	      {
		errs += "Conflicting definition of property type for property: "+oneinp->name().toString()+". There is a same named property that is defined as a different type";
		return false;
	      }
	    // check init process
	    if ( existingp._parentindex != -1 && existingp._pinfo->initProcess() != oneinp->initProcess() )
	      {
		errs += "Conflicting definition of initialization expression for property: "+oneinp->name().toString()+"\n";
		return false;
	      }
	    
	    continue;
	  }

	LBPropertyInfo *pinfocopy = new LBPropertyInfo(*oneinp);
	int idx = ins()->size();
	ins()->push_back(pinfocopy);
	(*pmap())[oneinp->name()]= PropertyInfoIndex(pinfocopy, idx, parentindex);

      }

    int numouts = pf.numberOfPropertiesByAttr( LBPropertyInfo::OUT );
    for(int i=0; i<numouts; i++)
      {
	const LBPropertyInfo *oneoutp = pf.propertyInfo(i, LBPropertyInfo::OUT);
	PropertyInfoIndex existingp;
	if ( pmap()->find(oneoutp->name(), existingp ) )
	  {
	    if ( existingp._pinfo->attr() != LBPropertyInfo::OUT )
	      {
		errs += "Conflicting definition of property from inheritance for property: "+oneoutp->name().toString()+". There is a same named property that is defined as a non-output port";
		return false;
	      }
	    const LBTypeSpec *existingptype = existingp._pinfo->typeSpec();
	    const LBTypeSpec *oneoutptype = oneoutp->typeSpec();
	    if ( existingptype == 0 && oneoutptype == 0 )
	      continue;
	    if ( existingptype && oneoutptype )
	      {
		if ( existingptype->equals((const LBObject&)(*oneoutptype)) )
		  continue;
	      }
	    errs += "Conflicting definition of property type for property: "+oneoutp->name().toString()+". There is a same named property that is defined as a different type";
	    return false;
	  }

	LBPropertyInfo *pinfocopy = new LBPropertyInfo(*oneoutp);
	int idx = outs()->size();
	outs()->push_back(pinfocopy);
	(*pmap())[oneoutp->name()] = PropertyInfoIndex(pinfocopy, idx, parentindex);

      }

    // add all ancestors of the parent
    if ( pf._ancestors )
      {
	AncestorSet::Iterator it(*pf._ancestors);
	while(it.next())
	  ancestors()->insert(*it.currentKey());
      }

    return true;

  }
	    

  bool LBStructInterface::addProperty( LBPropertyInfo *p, string& errs)
  {
    PropertyInfoIndex existingp;
    if ( pmap()->find(p->name(), existingp ) )
      {
	errs += "Repeated definition of the same property: "+p->name().toString();
	return false;
      }

    int idx=-1;
    switch ( p->attr() )
      {
      case LBPropertyInfo::IN:
	idx = ins()->size();
	ins()->push_back(p);
	break;
      case LBPropertyInfo::OUT:
	idx = outs()->size();
	outs()->push_back(p);
	break;
      case LBPropertyInfo::STORE:
	idx = stores()->size();
	stores()->push_back(p);
	break;
      case LBPropertyInfo::STATIC:
	idx = statics()->size();
	statics()->push_back(p);
	break;
      default:
	errs += "Invalid property attribute\n";
	return false;
      }
    
    (*pmap())[p->name()]= PropertyInfoIndex(p, idx) ;
    
    // take care of the ext symbols
    const LBFullSymbolVec *exts = p->externalSymbols();
    if ( exts )
      for( int i = 0; i < exts->size(); i++ )
	extsymbols()->push_back((*exts)[i]);

    return true;

  }

  void LBStructInterface::addParent(const LBFullSymbol& pname)
  {
    parents()->push_back(pname);
    ancestors()->insert(pname);
  }

  bool LBStructInterface::resolveAllPropertyTypes(string& errs)
  {
    bool ok = true;
    if ( _ins )
      for( int i=0; i < _ins->size(); i++ )
	ok = ok && (*_ins)[i]->resolveType(errs);
    if ( _outs )
      for( int i=0; i < _outs->size(); i++ )
	ok = ok && (*_outs)[i]->resolveType(errs);
    if ( _stores )
      for( int i=0; i < _stores->size(); i++ )
	ok = ok && (*_stores)[i]->resolveType(errs);
    if ( _statics )
      for( int i=0; i < _statics->size(); i++ )
	ok = ok && (*_statics)[i]->resolveType(errs);

    return ok;

  }

  // private functions used internally
  LBStructInterface::PropertyInfoVec *LBStructInterface::ins()
  {
    if ( _ins == 0 )
      _ins = new PropertyInfoVec();
    return _ins;
  }
  LBStructInterface::PropertyInfoVec *LBStructInterface::outs()
  {
    if ( _outs == 0 )
      _outs = new PropertyInfoVec();
    return _outs;
  }
  LBStructInterface::PropertyInfoVec *LBStructInterface::stores()
  {
    if ( _stores == 0 )
      _stores = new PropertyInfoVec();
    return _stores;
  }
  LBStructInterface::PropertyInfoVec *LBStructInterface::statics()
  {
    if ( _statics == 0 )
      _statics = new PropertyInfoVec();
    return _statics;
  }
  
  static int symbolhash(const LBSymbol& s)
  { return s.hash(); }
  static bool symbolequal(const LBSymbol& s1,const LBSymbol& s2)
  { return s1==s2; }
  static int fullsymbolhash(const LBFullSymbol& s)
  { return s.hash(); }
  static bool fullsymbolequal(const LBFullSymbol& s1,const LBFullSymbol& s2)
  { return s1==s2; }

  LBStructInterface::PropertyMap *LBStructInterface::pmap()
  {
    if ( _pmap == 0 )
      _pmap = new PropertyMap(symbolhash, symbolequal, 5);
    return _pmap;
  }

  LBStructInterface::ParentInfoVec *LBStructInterface::parents()
  {
    if ( _parents == 0 )
      _parents = new ParentInfoVec();
    return _parents;
  }

  LBStructInterface::AncestorSet* LBStructInterface::ancestors()
  {
    if ( _ancestors == 0 )
      _ancestors = new AncestorSet(fullsymbolhash, fullsymbolequal, 3);
    return _ancestors;
  }

  LBFullSymbolVec *LBStructInterface::extsymbols()
  {
    if ( _extsymbols == 0 )
      _extsymbols = new LBFullSymbolVec();
    return _extsymbols;
  }

  //static 
  bool LBStructInterface::allowInitProcess( ExecMode m, LBPropertyInfo::ExecAttr a )
  {
    switch ( a )
      {
      case LBPropertyInfo::OUT: // no init for output properties
	return false; 
      case LBPropertyInfo::IN: // only synch struct allow init for input properties
	if ( m == SYNCH )
	  return true;
	return false;
      case LBPropertyInfo::STORE: // store property always allow init
      case LBPropertyInfo::STATIC: // static property always allow init
	return true;
      }
    return false;
  }


}

