#include <string>

#include "lbtypes/lbtypespec.hpp"

#include "luban/lbpropertyinfo.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/luban_codegen.hpp"


namespace Luban
{
  using std::string;

  LBPropertyInfo::LBPropertyInfo(const LBSymbol& n,  ExecAttr a, RWPermission r, LBProcessStruct *initprc )
    : _name(n), _attr(a), _permission(r), _type(0), _typetree(0), _extsymbols(0), _homens(), _initprc(initprc)
  {
  }

  LBPropertyInfo::LBPropertyInfo(const LBSymbol& n,  ExecAttr a, RWPermission r, Tree typetree, LBFullSymbolVec *exts, const LBFullSymbol& homens, LBProcessStruct *initprc )
    : _name(n), _attr(a), _permission(r), _type(0), _typetree(typetree), _extsymbols(exts), _homens(homens), _initprc(initprc)
  {
  }

  LBPropertyInfo::LBPropertyInfo(const LBSymbol& n,  ExecAttr a, RWPermission r, LBTypeSpec *tspec, LBProcessStruct *initprc )
    : _name(n), _attr(a), _permission(r), _type(tspec), _typetree(0), _extsymbols(0), _homens(), _initprc(initprc)
  {
  }

  LBPropertyInfo::LBPropertyInfo(const LBPropertyInfo& p)
    : _name(p._name), _attr(p._attr), _permission(p._permission), _type(0), _typetree(0), _extsymbols(0), _homens(p._homens), _initprc(0)
  {
    if ( p._type )
      _type = static_cast<LBTypeSpec*>(p._type->clone());
    if ( p._typetree )
      _typetree = p._typetree->clone();
    if ( p._extsymbols )
      _extsymbols = new LBFullSymbolVec(*p._extsymbols);
    if ( p._initprc )
      _initprc = new LBProcessStruct(*p._initprc);
  }


  LBPropertyInfo::~LBPropertyInfo()
  {
    if ( _type ) delete _type;
    if ( _typetree ) delete _typetree;
    if ( _extsymbols ) delete _extsymbols;
  }

  string LBPropertyInfo::toString() const
  {
    string result(attrToString(_attr));
    result += " "+rwpermissionToString(_permission);
    if ( _type )
      result += " "+_type->toString();
    result += " "+_name.toString();
    return result;
  }

  bool LBPropertyInfo::resolveType(string& errs)
  {
    if ( ! _typetree || _type )
      return true;
    _type = LubanCodeGenerator::parseTypeExpression(_typetree, _homens, errs);
    if ( ! _type )
      errs += "Failed to resolve the type for property "+_name.toString()+"\n";
    return _type;
  }

  // static
  LBPropertyInfo::RWPermission LBPropertyInfo::defaultRWPermission(ExecAttr a)
  {
    switch (a)
      {
      case IN:
	return READWRITE;
      case OUT:
	return READONLY;
      case STORE:
	return HIDDEN;
      case STATIC:
	return READWRITE;
      default:
	;
      }

    return INVALIDRWP;
  }

  // static
  const string&  LBPropertyInfo::attrToString(ExecAttr a)
  {
    static const string instr("input"), outstr("output"), storestr("store"), staticstr("static"), invalidstr("INVALID");
    switch (a)
      {
      case IN:
	return instr;
      case OUT:
	return outstr;
      case STORE:
	return storestr;
      case STATIC:
	return staticstr;
      default:
	;
      }

    return invalidstr;
  }

  // static
  const string&  LBPropertyInfo::rwpermissionToString(RWPermission p)
  {
    static const string r("readonly"), rw("readwrite"), h("hidden"), invalid("INVALID");
    switch (p)
      {
      case READONLY:
	return r;
      case READWRITE:
	return rw;
      case HIDDEN:
	return h;
      default:
	;
      }

    return invalid;
  }

  // static
  bool LBPropertyInfo::consistent(ExecAttr a, RWPermission r)
  {
    switch (a)
      {
      case IN:
	if ( r == READWRITE )
	  return true;
	return false;
      case OUT:
	if ( r == READONLY )
	  return true;
	return false;
      case STORE:
	return true;
      case STATIC:
	return true;
      default:
	;
      }

    return false;
  }


}
