#include <typeinfo>
#include <string>

#include "lbthread/lbthread.hpp"

#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbtypefactory.hpp"


namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBSymbol, 1,0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBSymbol)

  class LBSymbol::LBSymbolImp
  {
  public:
    string *_actualstring;
    int    _hash;

  public:
    LBSymbolImp(string *s, int hkey) : _actualstring(s), _hash(hkey) {}
    bool operator==(const LBSymbolImp& simp)
    {
      if ( _hash != simp._hash ) return false;
      return _actualstring == simp._actualstring || *_actualstring == *simp._actualstring  ;
    }
  };

  typedef LBSymbol::LBSymbolImp* SymImpPtr;
  typedef LBHashMap<SymImpPtr,SymImpPtr> SymbolHash;

  static int symHash( const SymImpPtr& s )
  { return s->_hash; }
  static bool symEq( const SymImpPtr& s1, const SymImpPtr& s2  )
  { return s1==s2 || *s1 == *s2; }


  static  SymbolHash& globalSymbolTable()
  {
    static SymbolHash sh(&symHash, &symEq, 10 ); // initial size 1024
    return sh;
  }

  static LBMutex& globalSymbolMut()
  {
    static LBMutex m;
    return m;
  }

  // this function is thread safe
  static SymImpPtr getSingletonImp(const string& s)
  {
    // The hash is done only once in here
    // memory only allocated when necessary
    // one redundent search operation which is not too bad
    // when considering the hash is not repeated
    int hkey = HashFunctions::stringHash(s);
    LBSymbol::LBSymbolImp simp( const_cast<string*>(&s), hkey );
    SymImpPtr sptr=0;
    LBMutexLocker lock(globalSymbolMut());
    if ( ! globalSymbolTable().find( &simp, sptr) )
      {
	sptr = new LBSymbol::LBSymbolImp(new string(s), hkey);
	globalSymbolTable()[sptr] = sptr;
      }
    return sptr;
  }

  static SymImpPtr defaultImp()
  {
    static SymImpPtr dimp = getSingletonImp("");
    return dimp;
  }

  LBSymbol::LBSymbol()
    : _imp( defaultImp() )
  {}
    

  LBSymbol::LBSymbol(const string& s)
  {
    _imp = getSingletonImp(s);
  }

  LBSymbol::LBSymbol(const char* cstr)
  {
    string temp(cstr);
    _imp = getSingletonImp(temp);
  }

  LBSymbol::~LBSymbol()
  {
  }

  string LBSymbol::toString() const
  { return *_imp->_actualstring; }
  
  ostream& LBSymbol::toStream(ostream& o) const
  {     return o << _imp->_actualstring->size()<<' '<<*_imp->_actualstring ;  }

  istream& LBSymbol::fromStream(istream& i, int major, int minor)
  {
    string s;
    int length=0;
    i>>length;
    i.get(); // skip a space
    for( int j = 0; j < length; j++ ) s += i.get();
    _imp = getSingletonImp(s);
    return i;
  }

  bool LBSymbol::equals(const LBObject& another) const
  {
    const LBSymbol *lbsb = dynamic_cast<LBSymbol*>(const_cast<LBObject*>(&another));
    if ( lbsb )
      return lbsb->_imp == _imp;
    return false;
  }

  bool LBSymbol::operator==(const LBSymbol& x) const
  { return _imp == x._imp; }

  int LBSymbol::hash() const
  { return _imp->_hash; }

}
