#include <iostream>
#include <sstream>
#include <memory>
#include <vector>

#include "lbtypes/lbobject.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbfullsymbol.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbbool.hpp"
#include "lbtypes/lbvector.hpp"
#include "lbtypes/lbset.hpp"
#include "lbtypes/lbmap.hpp"
#include "lbtypes/lberror.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/lbtypespecbasic.hpp"
#include "lbtypes/RefCountedPtr.hpp"


#include "luban/lbprocessstruct.hpp"
#include "luban/lbcompositionstruct.hpp"
#include "luban/lbtypespecstruct.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/luban_implicitcast.hpp"
#include "luban/lbvmcode.hpp"
#include "luban/lbnamedargs.hpp"
#include "luban/lubantypechecking.hpp"
#include "luban/lubanpermissionchecking.hpp"
#include "luban/luban_codegen.hpp"
#include "luban/lbstructiopadlocal.hpp"
#include "luban/lbstructiopadasynch.hpp"
#include "luban/lblocalpropertystorage.hpp"
#include "luban/lbstructoutpadcomp.hpp"
#include "luban/luban_namespace.hpp"
#include "luban/lbcounterwaiter.hpp"

#include "lbthread/lbthread.hpp"


namespace Luban
{

  using std::vector;

  // special treatment for asynch struct
  // because the semantic of asynch struct is different from normal data obj
  // This is a hack to fit a live fish into a dead fish pool
  // strategy is two sided:
  //   a. Always copy instead of ref counting when assign an asynch struct
  //   b. No obtaining of exclusive copy in "getWriteLockedObjPtr" operation
  //      because the sharing of the object means to share the same copy
  static inline bool isAsynchStruct(const LBObject *obj)
  {
    const LBStruct *s = dynamic_cast<const LBStruct*>(obj);
    if ( s && s->interface().mode() == LBStructInterface::ASYNCH )
      return true;
    return false;
  }


  LBDEFINE(Luban::LBProcessStruct, 1, 0 )
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBProcessStruct)

  LBProcessStruct::~LBProcessStruct()
  {
    // for internal multi-thread termination of asynch struct
    _mainstop= true;
    cancelInternalThreads(); // it will set up termination flag and wake up sleeping threads
    if ( _asynchmainthread )
      {
	_asynchmainthread->join();
	delete _asynchmainthread;
      }
    if ( _runtime )
      {
	// wait for all threads to finish
	if ( _runtime->_livethreads )
	  _runtime->_livethreads->joinAll();
	delete _runtime;
      }
    if ( _properties ) delete _properties;
  }

  LBProcessStruct::LBProcessStruct()
    : _metadata(new LBProcessStructMetaData()), _synchmtx(0), _runtime(0), _properties(0), _asynchmainthread(0), _mainstop(false)
  {
  }

  LBProcessStruct::LBProcessStruct(LBStructInterface* itfc)
    : _metadata(new LBProcessStructMetaData( itfc )), _synchmtx(0), _runtime(0), _properties(0), _asynchmainthread(0), _mainstop(false)
  {
  }

  LBProcessStruct::LBProcessStruct(const LBProcessStruct& p)
    : _metadata(p._metadata), _synchmtx(0), _runtime(0), _properties(0), _asynchmainthread(0), _mainstop(false)
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
      _synchmtx = new LBMutex();
    else
      _asynchmainthread = new AsynchMainThread( this );

    // decide if we need to run the property init expressions for all copies ?????
    // decision:no, only for the direct copy from name space

    // the properties of asynch struct do not need to be copied
    if ( p._properties != 0 )
      _properties = new LBProcessStructProperties(*p._properties);
  }


  //constructors used by composition struct to construct internal ad hoc process structs
  LBProcessStruct::LBProcessStruct(const LBSymbol& cellname, bool asynch, LBCompositionStruct *compmom )
    : _metadata(new LBProcessStructMetaData(cellname, asynch, compmom)), _synchmtx(0), _runtime(0), _properties(0), _asynchmainthread(0), _mainstop(false)
  {
    if ( asynch )
      _asynchmainthread = new AsynchMainThread(this);
  }
  LBProcessStruct::LBProcessStruct(int cellid, LBCompositionStruct *compmom )
    : _metadata(new LBProcessStructMetaData(cellid, compmom)), _synchmtx(0), _runtime(0), _properties(0), _asynchmainthread(0), _mainstop(false)
  {
  }



  LBProcessStruct& LBProcessStruct::operator=(const LBProcessStruct& p)
  {
    // prevent any assignment
    throw LBException("Assignment is not supported for process struct");
  }

  string LBProcessStruct::toString() const
  {
    static const char *synch ="Synchronized Struct ", *asynch ="Asynchronized Struct ";

    string result;
    bool issynch = _metadata->_itfc->mode() == LBStructInterface::SYNCH;
    if ( issynch )
      result = synch;
    else
      result = asynch;

    if ( _metadata->_nsRecorded )
      result += _metadata->_itfc->name()->toString()+' ';
    else
      result = "Anonymous "+result+' ';;

    // print status
    result += "Status=";

    if ( ! issynch )
      {
	if ( _asynchmainthread->isdead() )
	  result += "terminated ";
	else
	  result += "active ";
      }
    else
      {
	if ( _runtime )
	  {
	    switch ( _runtime->_status )
	      {
	      case LBProcessStructRunTime::EVALERROR:
		result += "ERROR, "+_runtime->_errmsg;
		break;
	      case LBProcessStructRunTime::EVALRUNNING:
		result += "Evaluating";
		break;
	      case LBProcessStructRunTime::EVALOK:
		result += "Evaluated";
		break;
	      case LBProcessStructRunTime::UNEVALED:
		result += "Unevaluated";
		break;
	      }
	  }
	else
	  result += "Unevaluated";
	result += ' ';

      }

    {

      LBMutexPtrLocker mlock(_synchmtx);

      // print properties
      int numins = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
      if ( numins > 0 )
	{
	  result += "input( ";
	  for( int i = 0; i < numins ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::IN)->name().toString()+"=";
	      if ( issynch && _properties )
		{
		  LBObjPtr val =_properties->_ins->readValueByInternal(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}
      int numouts = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
      if ( numouts > 0 )
	{
	  result += "output( ";
	  for( int i = 0; i < numouts ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::OUT)->name().toString()+'=';
	      if ( issynch && _properties )
		{
		  LBObjPtr val =_properties->_outs->readValueByExternal(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}

      int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
      if ( numstores > 0 )
	{
	  result += "store( ";
	  for( int i = 0; i < numstores ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STORE)->name().toString()+'=';
	      if (  _properties )
		{
		  LBObjPtr val =_properties->_stores->readValue(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}
    }

    int numstatics = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	result += "static( ";
	for( int i = 0; i < numstatics ; i++ )
	  {
	    result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STATIC)->name().toString()+'=';
	    LBObjPtr val =_metadata->_statics->readValue(i); 
	    result += val.toString()+' ';
	  }
	result += ") ";
      }

    return result;
  }

  ostream& LBProcessStruct::toStream( ostream& ost) const
  {
    // first stream meta data
    if ( _metadata->_nsRecorded ) // require both side has the same namespace content, so meta data can be recoved by name
      {
	ost.put('1');ost.put(' ');
	_metadata->_itfc->name()->toStream(ost);
      }
    else
      {
	ost.put('0');ost.put(' ');
	_metadata->toStream(ost);
      }

    // then stream the state of SYNCH struct only
    if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH  && _properties )
      {
	ost.put('1');ost.put(' ');

	int numins = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
	ost<<numins<< ' ';
	for( int i = 0; i < numins; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::IN)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _properties->_ins->readValueByInternal(i);
	    if ( valp )
	      {
		
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
	int numouts = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
	ost<<numouts<< ' ';
	for( int i = 0; i < numouts; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::OUT)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _properties->_outs->readValueByExternal(i);
	    if ( valp )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
	int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
	ost<<numstores<< ' ';
	for( int i = 0; i < numstores; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::STORE)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _properties->_stores->readValue(i);
	    if ( valp )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
      }
    else
      {
	ost.put('0'); ost.put(' ');
      }

    return ost;

  }

  istream& LBProcessStruct::fromStream( istream& ist, int major, int minor)
  {
    // destroy the old guts, if any, should not be necessary
    // because stream should go to an empty default instance
    _mainstop= true;
    cancelInternalThreads(); // it will set up termination flag and wake up sleeping threads
    if ( _asynchmainthread )
      {
	_asynchmainthread->join();
	delete _asynchmainthread;
	_asynchmainthread = 0;
      }
    if ( _runtime )
      {
	// wait for all threads to finish
	if ( _runtime->_livethreads )
	  _runtime->_livethreads->joinAll();
	delete _runtime;
	_runtime = 0;
      }
    if ( _properties ) 
      {
	delete _properties;
	_properties = 0;
      }

    _mainstop= false;

    // recover metadata
    char oneorzero, sp;
    ist.get(oneorzero); ist.get(sp);
    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
      throw( LBException("Corrupted stream, can not recover process struct from stream") );
    if ( oneorzero == '1' ) // recover the struct type name
      {
	LBFullSymbol stname;
	stname.fromStream(ist);
	string errs;
	LBStruct * st = LubanSymbolResolver::resolveStructSymbol(stname, errs);
	if ( ! st )
	  throw(LBException("Can not find struct type "+stname.toString()+" when recovering struct from stream: "+errs));
	LBProcessStruct *pst = dynamic_cast<LBProcessStruct*>(st);
	if ( ! pst )
	  throw(LBException("Struct type "+stname.toString()+" is not a process as expected when recovering struct from stream: "));
	_metadata = pst->_metadata;
      }
    else
      _metadata->fromStream(ist);

    if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
      _synchmtx = new LBMutex();
    else
      _asynchmainthread = new AsynchMainThread( this );

    
    // recover state
    ist.get(oneorzero); ist.get(sp);
    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
      throw( LBException("Corrupted stream, can not recover process struct from stream") );
    if ( oneorzero == '1' )
      {
	initializeProperties();
	prepareOutputForProcessUse();

	int numins = -1;
	ist>>numins; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numins; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( ! pinfo || pinfo->attr() != LBPropertyInfo::IN )
	      throw LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as input as expected");

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover input property "+pname.toString()+" from stream") );
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());

		_properties->_ins->writeValueByExternal(index, LBObjPtr(obj));
	      }
	    else
	      _properties->_ins->writeValueByExternal(index, LBObjPtr());		
	  }

	int numouts = -1;
	ist>>numouts; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numouts; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
	      throw( LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as output as expected") );

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover output property "+pname.toString()+" from stream") );

		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());

		_properties->_outs->writeValueByInternal(index, LBObjPtr(obj));

	      }
	    else
	      _properties->_outs->writeValueByInternal(index, LBObjPtr());		
	  }


	int numstores = -1;
	ist>>numstores; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numstores; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( !pinfo || pinfo->attr() != LBPropertyInfo::STORE )
	      throw( LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as store as expected") );

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover output property "+pname.toString()+" from stream") );
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());
		
		_properties->_stores->writeValue(index, LBObjPtr(obj));
	      }
	    else
	      _properties->_stores->writeValue(index, LBObjPtr());		
	  }

      }

    return ist;
	
  }

  bool LBProcessStruct::equals(const LBObject& another) const
  { return this == &another; }

  const LBStructInterface& LBProcessStruct::interface() const
  {  return *_metadata->_itfc; }

  LBProcessStruct* LBProcessStruct::structCall(const LBNamedArgs* args) const
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      throw LBException("Can not call asynchronous structure");
    LBProcessStruct *evalcopy = new LBProcessStruct(*this);
    try {
      evalcopy->initializeProperties();
      evalcopy->prepareOutputForProcessUse();
      evalcopy->setBatchProperties(args);
    }
    catch( LBException &lbe)
      {
	delete evalcopy;
	throw lbe;
      }
    return evalcopy;
  }


  void LBProcessStruct::setBatchProperties(const LBNamedArgs* args)
  {
    // set the args
    int sz = args?args->numArgs() : 0;
    string errs;

    LBMutexPtrLocker lock(_synchmtx);

    for( int i = 0; i < sz; i++ )
      {
	const LBObject *argobj = args->getArg(i);
	const LBSymbol *argname = args->getName(i);
	int argindex = -1;
	const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(*argname, argindex);
	if ( ! pinfo || ( pinfo->attr() != LBPropertyInfo::IN && pinfo->attr() != LBPropertyInfo::STORE ) )
	  throw( LBException("The struct instance does not have the property: "+argname->toString()) );

	// property setting policy and type checking here
	if (  LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
	  {
	    if ( argobj )
	      {
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), argobj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+argobj->getType().toString());

		if ( pinfo->attr() == LBPropertyInfo::IN )
		  _properties->_ins->writeValueByExternal(argindex, *argobj);
		else
		  _properties->_stores->writeValue(argindex, *argobj);
	      }
	    else
	      {
		if ( pinfo->attr() == LBPropertyInfo::IN )
		  _properties->_ins->writeValueByExternal(argindex, LBObjPtr());
		else
		  _properties->_stores->writeValue(argindex, LBObjPtr());
	      }
	  }
	else
	  throw( LBException( "The property: "+argname->toString()+ " does not allow setting from outside") );
      }

    if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
      runAllCode();    // run the code for the synch
    else
      if ( ! _asynchmainthread->start() )
	throw LBException("Can not start asynch struct main thread: "+_asynchmainthread->errmsg() );
  }


  //private
  LBObjPtr LBProcessStruct::readPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  {
    // locking policy and static property access policy defined here
    switch ( attr )
      {
      case  LBPropertyInfo::IN: // no locking necessary for reading inputs
	return _properties->_ins->readValueByInternal(index);
      case LBPropertyInfo::STATIC: // no local instance locking necessary for global static properties
	throw LBException("Can not read static property through struct instance");
      case LBPropertyInfo::OUT:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexPtrLocker lock(_synchmtx);
	      if ( ! _runtime || _runtime->_status == LBProcessStructRunTime::UNEVALED )
		runAllCode();
	      return _properties->_outs->readValueByExternal(index);
	    }
	  
	  // asynch case
	  _asynchmainthread->start();
	  return _properties->_outs->readValueByExternal(index);
	}
      case LBPropertyInfo::STORE:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexPtrLocker lock(_synchmtx);
	      if ( ! _runtime || _runtime->_status == LBProcessStructRunTime::UNEVALED )
		runAllCode();
	      return _properties->_stores->readValue(index);
	    }
	  // asynch
	  return _properties->_stores->readValue(index);

	}

      }
    
    throw LBException("Unrecognized kind of property");
    return LBObjPtr();
  }

  //private, this is for writing purpose
  LBObject* LBProcessStruct::readPropertyUnsharedGuts(int index, LBPropertyInfo::ExecAttr attr)
  {
    // locking policy and static property access policy defined here
    switch ( attr )
      {
      case  LBPropertyInfo::IN: // need to lock because this is for writing
	{
	  // the policy for store property is here and in the VM code interpreter
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexPtrLocker lock(_synchmtx);
	      return _properties->_ins->readUnsharedGuts(index);
	    }
	  throw( LBException( "Can not obtain the unshared internal object pointer for input property of asynch struct") );
	}
      case LBPropertyInfo::STATIC: // static properties are taken care by different interface
	throw( LBException( "Can not access static property through struct instance") );
      case LBPropertyInfo::OUT:
	throw( LBException( "Can not modify output property value from outside") );
      case LBPropertyInfo::STORE:
	{
	  // the policy for store property is here and in the VM code interpreter
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexPtrLocker lock(_synchmtx);
	      return _properties->_stores->getUnsharedPtr(index);
	    }
	  throw( LBException( "Can not obtain the unshared internal object pointer for store property of asynch struct") );
	}
      }
    
    throw( LBException("Unrecognized kind of property"));
    return 0;
  }


  LBObjPtr LBProcessStruct::getPropertyValue(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw( LBException("Failed to pass read permission check for property value reading: "+errs) );
    return readPropertyValue( pindex, pinfo->attr() );
  }

  LBObject* LBProcessStruct::getPropertyPtr(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH,  errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw( LBException("Failed to pass permission check for property mutation: "+errs) );

    return readPropertyUnsharedGuts( pindex, pinfo->attr() );

  }

  void LBProcessStruct::touchProperty(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    if ( pinfo->attr() != LBPropertyInfo::IN )
      return;
    touchProperty( pindex, pinfo->attr() );
  }

  void LBProcessStruct::setProperty(const LBSymbol& s, const LBObject& val)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), val );
  }


  void LBProcessStruct::setProperty(const LBSymbol& s, const LBObjPtr& valptr)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), valptr );    
  }

  LBObjPtr LBProcessStruct::getPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw LBException("Invalid property index");
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Failed to pass read permission check for property value reading: "+errs);
    return readPropertyValue( index, attr );
  }

  LBObject* LBProcessStruct::getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);

    return readPropertyUnsharedGuts( index, attr );

  }

  void LBProcessStruct::touchProperty(int index, LBPropertyInfo::ExecAttr attr)
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      return; // do nothing for an asynch struct

    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );

    if ( pinfo->attr() != LBPropertyInfo::IN )
      return;

    LBMutexPtrLocker lock(_synchmtx);
    runAllCode();
  }

  void LBProcessStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);
    
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    string sname = _metadata->_itfc->name()?_metadata->_itfc->name()->toString():" ";

    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  {
	    if ( _asynchmainthread->isdead() )
	      throw LBException("Unable to run asynch struct "+sname+" error: "+_asynchmainthread->errmsg() );

	    if ( ! _properties->_ins->writeValueByExternal(index, val) )
	      throw LBException("Unable to set input to asynch struct, probably internal thread being terminated" );

	    _asynchmainthread->start(); // start thread if it has not been started yet

	  }
	else
	  {
	    LBMutexPtrLocker lock(_synchmtx);
	    if ( _properties->_ins->writeValueByExternal(index, val) )
	      runAllCode();
	    else
	      throw LBException("Unable to set input to synch struct");
	  }
	return;
	break;
      case LBPropertyInfo::STORE:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _properties->_stores->writeValue(index, val);
	else
	  {
	    LBMutexPtrLocker lock(_synchmtx);
	   _properties->_stores->writeValue(index, val);
	  }
	return;
	break;
      case LBPropertyInfo::STATIC:
	throw LBException("Can not set value of static property through struct instance");
	return;
	break;
      }

      throw( LBException("Set value to property of invalid attribute") );

  }

  void LBProcessStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );

    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw( LBException("No write permission check for property setting: "+errs) );
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), val.getConstRealPtr()) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    string sname = _metadata->_itfc->name()?_metadata->_itfc->name()->toString():" ";

    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  {
	    if ( _asynchmainthread->isdead() )
	      throw LBException("Unable to run asynch struct "+sname+" error: "+_asynchmainthread->errmsg() );

	    if ( ! _properties->_ins->writeValueByExternal(index, val) )
	      throw LBException("Unable to set input to asynch struct, probably internal thread being terminated" );

	    _asynchmainthread->start(); // start thread if it has not been started yet

	  }
	else
	  {
	    LBMutexPtrLocker lock(_synchmtx);
	    if (  _properties->_ins->writeValueByExternal(index, val) )
	      runAllCode();
	    else
	      throw LBException("Unable to set input to synch struct");
	  }
	return;
	break;
      case LBPropertyInfo::STORE:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _properties->_stores->writeValue(index, val);
	else
	  {
	    LBMutexPtrLocker lock(_synchmtx);
	   _properties->_stores->writeValue(index, val);
	  }
	return;
	break;
      case LBPropertyInfo::STATIC:
	throw LBException("Can not set value of static property through struct instance");
	return;
	break;
      }

      throw( LBException("Set value to property of invalid attribute") );

  }

  bool LBProcessStruct::addPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].inc();
  }

  bool LBProcessStruct::removePending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].dec();
  }
  bool LBProcessStruct::waitUntilNoPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].waitDownToZero();
  }

  LBObjPtr LBProcessStruct::getStaticPropertyValue(int index)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Read permission violation: "+errs);
    return _metadata->_statics->readValue(index);
  }

  void LBProcessStruct::setStaticPropertyValue(int index, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _metadata->_statics->writeValue(index, val);
  }

  void LBProcessStruct::setStaticPropertyValue(int index, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Read permission violation: "+errs);
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr() ) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );
    _metadata->_statics->writeValue(index, valptr);
  }

  void LBProcessStruct::waitUntilInputEmptyOrMainFinish()
  {
    if ( _properties && _properties->_ins )
      {
	_properties->_ins->waitUntilInputEmpty();
	return;
      }
    _asynchmainthread->join();
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBProcessStruct, luban_wait, "waitfinish", "void waitfinish()");
  LBObject *LBProcessStruct::luban_wait(const LBVarArgs *args)
  {
    int numargs = args ? args->numArgs() : 0;
    if ( numargs )
      throw LBException("no arguement for wait() function");
    waitUntilInputEmptyOrMainFinish();
    return 0;
  }

  LBWriteLockedObjPtr LBProcessStruct::getWriteLockedStaticProperty(int staticprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(staticprpindex, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _metadata->_statics->getWriteLockedObj(staticprpindex);
  }

  LBWriteLockedObjPtr LBProcessStruct::getWriteLockedStoreProperty(int storeprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(storeprpindex, LBPropertyInfo::STORE );
    if ( pinfo == 0 )
      throw( LBException("Invalid store property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _properties->_stores->getWriteLockedObj(storeprpindex);
  }


  void LBProcessStruct::setInput(int index, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::IN);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr()) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    string sname = _metadata->_itfc->name()?_metadata->_itfc->name()->toString():"anonymous";

    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      {
	if ( _asynchmainthread->isdead() )
	  throw LBException("Unable to run asynch struct: "+sname+" error: "+_asynchmainthread->errmsg() );
	_properties->_ins->writeValueByExternal(index, valptr);
	_asynchmainthread->start(); // start thread if it has not been started yet
      }
    else
      {
	LBMutexPtrLocker lock(_synchmtx);
	_properties->_ins->writeValueByExternal(index, valptr);
      }
    return;
  }

  void LBProcessStruct::setInput(int index, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::IN);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    string sname = _metadata->_itfc->name()?_metadata->_itfc->name()->toString():"anonymous";

    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      {
	if ( _asynchmainthread->isdead() )
	  throw LBException("Unable to run asynch struct: "+sname+" error: "+_asynchmainthread->errmsg() );
	_properties->_ins->writeValueByExternal(index, val);
	_asynchmainthread->start(); // start thread if it has not been started yet
      }
    else
      {
	LBMutexPtrLocker lock(_synchmtx);
	_properties->_ins->writeValueByExternal(index, val);
      }
    return;
  }

  void LBProcessStruct::sync()
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      throw LBException("Can not run sync operation on asynch struct");
    runAllCode();
    // flush output for adhoc
    if ( _runtime->_parentcomp  ) 
      if ( _metadata->isAdhoc() && _runtime->_adhocoutputUpdated )
	{
          _runtime->_parentcomp->writeDataJoint(_metadata->_jtwriteindex, _runtime->_adhocoutput);
	  _runtime->_adhocoutputUpdated =false;
	}
      else
	if ( _properties && _properties->_outs )
	  _properties->_outs->flush();
  }

  bool LBProcessStruct::linkOutputProperty(const LBSymbol& outprpname, int jointid)
  {
    int prpidx=-1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(outprpname, prpidx);
    if ( !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
      return false;
    _properties->_outs->linkTo(prpidx, jointid);
  }

  bool LBProcessStruct::wakeupAsynch()
  {
    if ( ! _asynchmainthread )
      throw LBException("Attempt to start asynch thread on a synch struct" );
    if ( _asynchmainthread->isdead() )
      throw LBException("asynch struct internal error: "+_asynchmainthread->errmsg() );
    if ( ! _asynchmainthread->start() ) // start thread if it has not been started yet
      throw LBException("Can not start asynch struct main thread: "+_asynchmainthread->errmsg() );
    return true;
  }

  bool LBProcessStruct::prepareOutputForCompositionUse()
  {
    if ( _properties && _properties->_outs ) return true;
    int outnum = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
    if ( outnum == 0 )
      return true;
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      _properties->_outs = new LBStructOutPadAsynchComp(outnum);
    else
      _properties->_outs = new LBStructOutPadSynchComp(outnum);
    return true;
  }

  bool LBProcessStruct::setRunTimeParentComposition(LBCompositionStruct *pcomp)
  {
    if ( _properties && _properties->_outs )
      _properties->_outs->setParentComposition(pcomp);
    if ( ! _runtime ) 
      prepareRunTime();
    _runtime->_parentcomp = pcomp;
    return true;
  }

  bool LBProcessStruct::prepareOutputForProcessUse()
  {
    if ( _properties->_outs ) return true;
    int outnum = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
    if ( outnum == 0 )
      return true;
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      _properties->_outs = new LBStructOutPadAsynch(outnum);
    else
      _properties->_outs = new LBStructOutPadLocal(outnum);
    return true;
  }



  int LBProcessStruct::numOfExternalSymbols() const
  {
    if ( ! _metadata ) return 0;
    return _metadata->_extsymbols.size();
  }

  const LBFullSymbol* LBProcessStruct::externalSymbol(int index ) const
  {
    if ( ! _metadata || index >= _metadata->_extsymbols.size() || index < 0  )
      return 0;
    return _metadata->_extsymbols.symbolName(index);
  }

  const LBObjPtr& LBProcessStruct::evalForLastObj( )
  {
    if ( ! _metadata )
      throw LBException("Struct does not have metadata information");

    if ( _metadata->_code.size() == 0 ) 
      return _metadata->_staticlastobj;

    // Lock the thread lock and run the code
    LBMutexPtrLocker lock(_synchmtx);
    runAllCode();
    return _runtime->_lastobj;
  }

  LBProcessStruct::LBProcessStructMetaData::LBProcessStructMetaData(LBStructInterface* itfc)
    : _itfc( itfc ), _numtemps(0), _numtempiters(0), _multithread(false), _extsresolved(false), _nsRecorded(true), _statics(0), _staticscounters(0), _loopbreaks(0),_loopcontinues(0), _compilingDispatch(false), _cellid(-1), _cellname(), _compmom(0), _jtreadops(),_jtwriteindex(-1),_initmtx()
  {

    int numstatics = itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	_statics = new LocalPropertyStorage(numstatics, true);
	_staticscounters = new CounterWaiter[numstatics];
	// init static properties
	for(int i=0; i<numstatics; i++)
	  {
	    const LBProcessStruct *initp = itfc->propertyInfo(i, LBPropertyInfo::STATIC)->initProcess();
	    if ( ! initp )
	      continue;
	    std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	    _statics->writeValue(i, torun->evalForLastObj());
	  }

      }
  }


  LBProcessStruct::LBProcessStructMetaData::LBProcessStructMetaData()
    : _itfc( new LBStructInterface() ), _numtemps(0), _numtempiters(0), _multithread(false), _extsresolved(false), _nsRecorded(false), _statics(0),_staticscounters(0), _loopbreaks(0), _loopcontinues(0), _compilingDispatch(false), _cellid(-1), _cellname(), _compmom(0), _jtreadops(),_jtwriteindex(-1),_initmtx()
  {
  }

  // constructors used to build internal process struct of composition
  LBProcessStruct::LBProcessStructMetaData::LBProcessStructMetaData(const LBSymbol& cellname, bool asynch, LBCompositionStruct *compmom)
    : _itfc( 0 ), _numtemps(0), _numtempiters(0), _multithread(false), _extsresolved(false), _nsRecorded(false), _statics(0),_staticscounters(0), _loopbreaks(0), _loopcontinues(0), _compilingDispatch(false), _cellid(-1), _cellname(cellname), _compmom(compmom), _jtreadops(),_jtwriteindex(-1),_initmtx()
  {
    if ( asynch )
      _itfc = new LBStructInterface(LBStructInterface::ASYNCH);
    else
      _itfc = new LBStructInterface(LBStructInterface::SYNCH);
    _cellid = compmom->metadata()->getCellId(cellname);
  }

  LBProcessStruct::LBProcessStructMetaData::LBProcessStructMetaData(int cellid, LBCompositionStruct *compmom)
    : _itfc( new LBStructInterface() ), _numtemps(0), _numtempiters(0), _multithread(false), _extsresolved(false), _nsRecorded(false), _statics(0),_staticscounters(0), _loopbreaks(0), _loopcontinues(0), _compilingDispatch(false), _cellid(cellid), _cellname(), _compmom(compmom), _jtreadops(),_jtwriteindex(-1),_initmtx()
  {
  }


  LBProcessStruct::LBProcessStructMetaData::~LBProcessStructMetaData()
  {
    delete _itfc;

    int tsz = _literaltypeexps.size();
    if ( tsz != 0 )
      for( int i = 0; i < tsz; i++)
	delete _literaltypeexps[i];

    int anonsz = _anonargs.size();
    if ( anonsz != 0 )
      for( int i = 0; i < anonsz; i++)
	delete _anonargs[i];

    int namedsz = _namedargs.size();
    if ( namedsz != 0 )
      for( int i = 0; i < namedsz; i++)
	delete _namedargs[i];

    if ( _statics ) delete _statics;
    if ( _staticscounters ) delete [] _staticscounters;

    if ( _loopbreaks ) delete _loopbreaks;
    if ( _loopcontinues ) delete _loopcontinues;
  }

    // private functions
  LBVMCodeSequence&  LBProcessStruct::LBProcessStructMetaData::code()
  {
    return _code;
  }

  bool LBProcessStruct::LBProcessStructMetaData::addLocalVariable(const  LBSymbol& name, Tree typeexp, const LBFullSymbol& homens, LBVMCode::Operand& resultvarop, string& errs )
  {
    LBFullSymbolVec *exts = LubanCodeGenerator::getTypeExpressionExtSymbols( typeexp, homens );
    LBTypeSpec *tspec = LubanCodeGenerator::parseTypeExpression( typeexp, homens,errs );
    if (  tspec == 0 )
      {
	errs += "Invalid type expression for local variable initialization\n";
	delete exts;
	return false;
      }


    TypeExpInfo *tfo = 0;
    if ( exts )
      {
	typeexp->pruneParent();
	tfo = new TypeExpInfo(tspec, typeexp, homens);
	delete exts;
      }
    else
      tfo = new TypeExpInfo(tspec);
    
    if ( !  _localvops.addVar(name, tfo, resultvarop, _compilingDispatch ) )
      {
	delete tfo;
	return false;
      }
    return true;
  }


  bool LBProcessStruct::LBProcessStructMetaData::getLocalVariableOrCell(const  LBSymbol& name, LBVMCode::Operand& resultvarop, bool forwriting, string& errs )
  {
    if ( _compmom )
      {
	if ( name == _cellname )
	  {
	    if ( ! forwriting )  // error, can not read self
	      {
		errs += "Can not read cell content of self: "+name.toString()+"\n";
		return false;
	      }
	    resultvarop = getAdhocOutputOperand();
	    return true;
	  }

	int cellid = _compmom->metadata()->getCellId(name);
	if ( cellid >= 0 )
	  {
	    if ( forwriting )
	      {
		errs += "Can not set cell content other than self: "+name.toString()+"\n";
		return false;
	      }

	    return  getAdhocInputOperand(cellid, resultvarop, errs);;
	  }
      }

    return _localvops.getVar(name, resultvarop, _compilingDispatch );
  }

  bool LBProcessStruct::LBProcessStructMetaData::getLocalVariable(const  LBSymbol& name, LBVMCode::Operand& resultvarop, string& errs )
  {
    if ( _compmom && _compmom->metadata()->getCellId(name) >= 0 )
      {
	errs += name.toString()+" is a component cell name, not a local variable\n";
	return false;
      }

    return _localvops.getVar(name, resultvarop, _compilingDispatch );
  }

  bool LBProcessStruct::LBProcessStructMetaData::getGlobalVariable(const LBFullSymbol fullstructname, const LBSymbol& propertyname, LBVMCode::Operand& resultvarop)
  {
    int idx = _extsymbols.getStructIndex(fullstructname);
    if ( idx < 0 )
      return false;

    return _globalvops.getGlobalVar(idx, propertyname, resultvarop);
  }
    

  bool LBProcessStruct::LBProcessStructMetaData::getGlobalStructType(const LBFullSymbol& fullstructname,  LBVMCode::Operand &resultop, string& errs)
  {
    return  _extsymbols.getStructOp(fullstructname, resultop);
  }

  bool LBProcessStruct::LBProcessStructMetaData::getGlobalConstructableType(const LBFullSymbol& fulltypename,  LBVMCode::Operand &resultop)
  {
    return _extsymbols.getConstructableOp(fulltypename, resultop);
  }
	
  bool LBProcessStruct::LBProcessStructMetaData::getLocalPropertyOperand(const LBSymbol& pname, LBPropertyInfo::ExecAttr exat, LBVMCode::Operand &pop, string& errs)
  {
    const LBStructInterface* itfc = _itfc;
    if ( _compmom )
      {
	if ( exat == LBPropertyInfo::OUT )
	  {
	    errs += "Can not directly access output property from within a cell\n";
	    return false;
	  }

	int comp_prpidx;
	const LBPropertyInfo *comp_pinfo = _compmom->interface().propertyInfo(pname, comp_prpidx);
	if ( ! comp_pinfo ||  comp_pinfo->attr() != exat  )
	  {
	    errs += "Invalid property reference \""+pname.toString()+"\" as "+LBPropertyInfo::attrToString(exat)+"\n";
	    return false;
	  }
	if ( exat == LBPropertyInfo::IN )
	  return getAdhocInputOperand(pname, pop, errs);
	itfc = &_compmom->interface();
      }

    // verify the existence of the property
    int prpindex = -1;
    const LBPropertyInfo *pinfo = itfc->propertyInfo(pname, prpindex);
    if ( pinfo == 0 || pinfo->attr() != exat )
      {
	errs += "Invalid property reference \""+pname.toString()+"\" as "+LBPropertyInfo::attrToString(exat)+"\n";
	return false;
      }
    
    return  _localpropertyops.getPropertyOp(pname, exat, prpindex, pop, _compilingDispatch, errs);

  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::getVarPropertyOperand( const LBVMCode::Operand &varop,const LBSymbol& pname)
  {
    return  _varpropertyops.getPropertyOp(varop, pname);
  }

  bool LBProcessStruct::LBProcessStructMetaData::getVarPropertyOperand( const LBSymbol& varname,const LBSymbol& prpname, LBVMCode::Operand &resultop,string& errs)
  {
    if ( _compmom )
      {
	int cid = _compmom->metadata()->getCellId(varname);
	if ( cid >= 0 )
	  {
	    const LBStructInterface *itfc = _compmom->metadata()->getTypedCellInterface(cid);
	    if ( itfc )
	      {
		int dummy;
		const LBPropertyInfo *pinfo = itfc->propertyInfo(prpname, dummy);
		if ( !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
		  {
		    errs += "Can not link property "+prpname.toString()+", which is either nonexisting or non-output property\n";
		    return false;
		  }

		return getAdhocInputOperand(cid, prpname, resultop, errs);
	      }
	    
	    // upstream is an adhoc cell
	    LBVMCode::Operand midop;
	    if ( ! getAdhocInputOperand(cid,midop, errs) )
	      { 
		errs += "Can not create operand for cell "+varname.toString()+"\n";
		return false;
	      }
	    resultop =  _varpropertyops.getPropertyOp(midop, prpname);
	    return true;
	  }
      }

    // get a local variable operand for the variable name
    LBVMCode::Operand varop;
    _localvops.getVar(varname, varop, _compilingDispatch );
    resultop =  _varpropertyops.getPropertyOp(varop, prpname);
    return true;
  }

  bool LBProcessStruct::LBProcessStructMetaData::getAdhocInputOperand(const LBSymbol& inputprp, LBVMCode::Operand& resultop, string& errs)
  {	    
    int jointid=-1, portid=-1;
    if ( ! _compmom->metadata()->link(int(LBCompositionStruct::ComponentList::INPUTS), inputprp, _cellid, jointid, portid, errs) )
      {
	errs += "Can not create depdency link from input "+inputprp.toString()+" to cell "+_compmom->metadata()->getCellName(_cellid).toString()+"\n";
	return false;
      }
    resultop = LBVMCode::Operand(LBVMCode::Operand::JTREAD, _jtreadops.newJTReadOp(jointid, portid));
    return true;
  }

  bool LBProcessStruct::LBProcessStructMetaData::getAdhocInputOperand(int upcellid, LBVMCode::Operand& resultop, string& errs)
  {
    int jointid=-1, downportid=-1; 
    if ( ! _compmom->metadata()->link(upcellid, _cellid, jointid, downportid, errs) )
      {
	errs += "Failed to create depdency link from cell "+_compmom->metadata()->getCellName(upcellid).toString()+" to "+_compmom->metadata()->getCellName(_cellid).toString()+"\n";
	return false;
      }

    resultop = LBVMCode::Operand(LBVMCode::Operand::JTREAD, _jtreadops.newJTReadOp(jointid, downportid));
    return true;

  }

  bool LBProcessStruct::LBProcessStructMetaData::getAdhocInputOperand(int upcellid, const LBSymbol& upcellprpname, LBVMCode::Operand& resultop, string& errs)
  {
    int jointid=-1, downportid=-1; 
    if ( ! _compmom->metadata()->link(upcellid, upcellprpname, _cellid, jointid, downportid, errs) )
      {
	errs += "Failed to create depdency link from cell "+_compmom->metadata()->getCellName(upcellid).toString()+" to "+_compmom->metadata()->getCellName(_cellid).toString()+"\n";
	return false;
      }

    resultop = LBVMCode::Operand(LBVMCode::Operand::JTREAD, _jtreadops.newJTReadOp(jointid, downportid));
    return true;

  }
    

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::getAdhocOutputOperand()
  {
    if ( ! _compmom ) return LBVMCode::Operand();
    int jid = _compmom->metadata()->getOutputDataJointIdForAdhocCell( _cellid );
    _jtwriteindex = jid;
    return LBVMCode::Operand(LBVMCode::Operand::JTWRITE, jid);
  }

  bool LBProcessStruct::LBProcessStructMetaData::isAdhoc()
  {
    return _compmom;
  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::getSubscriptOperand( const LBVMCode::Operand &varop, const LBVMCode::Operand &subexpop)
  {
    return _subscriptops.getSubscriptOp(varop, subexpop);
  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::getMemberFuncOperand( const LBVMCode::Operand &varop, const LBSymbol& funcname)
  {
    return  _memberfuncops.getMemberFuncOp(varop, funcname);
  }
    
  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::newTemp()
  {
    return LBVMCode::Operand(LBVMCode::Operand::TEMPOBJ, _numtemps++);
  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::newTempIter()
  {
    return LBVMCode::Operand(LBVMCode::Operand::TEMPITER, _numtempiters++);
  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::newNamedArgsOperand(LBProcessStruct::LBStructEvalArgs* args)
  {
    _namedargs.push_back(args);
    return LBVMCode::Operand(LBVMCode::Operand::TEMPNAMEDARG, _namedargs.size()-1);
  }

  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::newAnonymousArgsOperand(LBProcessStruct::LBAnonymousArgs *args  )
  {
    _anonargs.push_back(args);
    return LBVMCode::Operand(LBVMCode::Operand::TEMPANONARG, _anonargs.size()-1);
  }


  LBVMCode::Operand LBProcessStruct::LBProcessStructMetaData::newLiteral(LBObject* obj)
  {
    _literalobjs.push_back(LBObjPtr(obj));
    return LBVMCode::Operand(LBVMCode::Operand::LITERALOBJ, _literalobjs.size()-1);
  }

  bool LBProcessStruct::LBProcessStructMetaData::newLiteralTypeSpec(Tree typetree, const LBFullSymbol& homens,  LBVMCode::Operand &resultop, string& errs)
  {
    LBFullSymbolVec *exts = LubanCodeGenerator::getTypeExpressionExtSymbols( typetree, homens );
    LBTypeSpec *tspec = LubanCodeGenerator::parseTypeExpression( typetree, homens,errs );
    if (  tspec == 0 )
      {
	errs += "Invalid type expression\n";
	delete exts;
	return false;
      }

    TypeExpInfo *tfo = 0;
    if ( exts )  // put all refered external type symbol into the external symbol table
      {
// 	for(int i=0; i< exts->size(); i++)
// 	  _extsymbols.addSymbol((*exts)[i]);
// no  need to put the already-resolved symbol to external symbols

	typetree->pruneParent();
	tfo = new TypeExpInfo(tspec, typetree, homens);
      }
    else
      tfo = new TypeExpInfo(tspec);

    _literaltypeexps.push_back(tfo);
    resultop = LBVMCode::Operand(LBVMCode::Operand::LITERALTYPE, _literaltypeexps.size()-1);

    return true;
  }

  void LBProcessStruct::LBProcessStructMetaData::setStaticLastOp(const LBVMCode::Operand& litop)
  {
    if ( litop.optype() == LBVMCode::Operand::LITERALOBJ && _literalobjs.size() > 0  )
      _staticlastobj = _literalobjs.back();
    if ( litop.optype() == LBVMCode::Operand::LITERALTYPE && _literaltypeexps.size() > 0 )
      _staticlastobj = _literaltypeexps.back()->getTypeSpecAsObj();
  }


  bool LBProcessStruct::LBProcessStructMetaData::findRootOperand(const LBVMCode::Operand& subop, LBVMCode::Operand& rootop)
  {
    switch( subop.optype())
      {
      case LBVMCode::Operand::SUBSCR:
	{
	  const LBVMCode::Operand *objop = _subscriptops.objOp(subop.opindex());
	  if ( objop )
	    return findRootOperand(*objop,rootop);
	  return false;
	}
      case LBVMCode::Operand::VARPROPERTY:
	{
	  const LBVMCode::Operand *sop = _varpropertyops.structOp(subop.opindex());
	  if ( sop )
	    return findRootOperand(*sop,rootop);
	  return false;
	}
      default:
	rootop = subop;
      }

    return true;

  }


  bool LBProcessStruct::LBProcessStructMetaData::addBreak(int brkidx)
  {
    if ( _loopbreaks==0 ||  _loopbreaks->size() == 0 )
      return false;
    
    _loopbreaks->back().push_back(brkidx);

    return true;
  }

  bool LBProcessStruct::LBProcessStructMetaData::addContinue(int contidx)
  {
    if ( _loopcontinues==0 ||  _loopcontinues->size() == 0 )
      return false;
    
    _loopcontinues->back().push_back(contidx);

    return true;
  }

  void LBProcessStruct::LBProcessStructMetaData::openLoop()
  {
    if ( _loopbreaks == 0 )
      {
	_loopcontinues = new VVInt();
	_loopbreaks = new VVInt();
      }

    _loopcontinues->push_back(VInt());
    _loopbreaks->push_back(VInt());

  }

  bool  LBProcessStruct::LBProcessStructMetaData::closeLoop(int cont, int brk)
  {
    if ( _loopbreaks == 0 ||   _loopbreaks->size() == 0 )
      return false;

    for( int i=0; i<_loopbreaks->back().size(); i++)
      {
	int gotost = _loopbreaks->back()[i];
	code()[gotost].op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, brk);
      }

    for( int i=0; i<_loopcontinues->back().size(); i++)
      {
	int gotost = _loopcontinues->back()[i];
	code()[gotost].op1() = LBVMCode::Operand(LBVMCode::Operand::LABEL, cont);
      }

    _loopbreaks->pop_back();
    _loopcontinues->pop_back();

    return true;

  }

  ostream&  LBProcessStruct::LBProcessStructMetaData::toStream(ostream& ost) const
  {
    char one='1', zero='0', space=' ';
    if ( _itfc )
      {
	ost.put(one); ost.put(space);
	_itfc->toStream(ost);
      }
    else
      {
	ost.put(zero); ost.put(space);
      }
    
    _code.toStream(ost);
    ost.put(space);
    _localvops.toStream(ost);
    ost.put(space);
    _globalvops.toStream(ost);
    ost.put(space);
    _localpropertyops.toStream(ost);
    ost.put(space);
    _varpropertyops.toStream(ost);
    ost.put(space);
    _subscriptops.toStream(ost);
    ost.put(space);
    _memberfuncops.toStream(ost);
    ost.put(space);

    int lits = _literalobjs.size();
    ost<<lits<<' ';
    int i = 0;
    for(; i < lits; i++)
      {
	if ( _literalobjs[i] )
	  {
	    ost.put(one); ost.put(space);
	    LBObject::instanceToStream(ost, *_literalobjs[i]);
	  }
	else
	  {
	    ost.put(zero); ost.put(space);
	  }
      }

    int litypes = _literaltypeexps.size();
    ost<<litypes<<' ';
    for( i = 0; i < litypes; i++ )
      {
	_literaltypeexps[i]->toStream(ost);
	ost.put(space);
      }

    ost << _numtemps<< space << _numtempiters<<space;

    int nargs = _namedargs.size();
    ost << nargs << space;
    for( i=0; i<nargs; i++)
      {
	_namedargs[i]->toStream(ost);
	ost.put(space);
      }

    int anonargs = _anonargs.size(); 
    ost << anonargs << space;
    for( i=0; i<anonargs; i++)
      {
	_anonargs[i]->toStream(ost);
	ost.put(space);
      }

    
    ost << _multithread<<space;

    _extsymbols.toStream(ost); 

    ost.put(space);

    return ost;
  }

  istream&  LBProcessStruct::LBProcessStructMetaData::fromStream(istream& ist)
  {
    char flag=ist.get(), space=ist.get();
    if ( space != ' ' || flag != '1' && flag != '0' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
    if ( flag == '1' )
      {
	if ( _itfc )
	  delete _itfc;
	_itfc = new LBStructInterface();
	_itfc->fromStream(ist);
      }
    
    _code.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _localvops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _globalvops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _localpropertyops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _varpropertyops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _subscriptops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _memberfuncops.fromStream(ist);
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");


    int lits=-1;
    ist>>lits;
    ist.get(space);
    if ( space != ' ' || lits < 0 )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    int i = 0;
    _literalobjs.clear();
    for(; i < lits; i++)
      {
	ist.get(flag); ist.get(space);
	if ( space != ' ' || flag != '1' && flag != '0' )
	  throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
	if ( flag == '1' )
	  {
	    string errs;
	    LBObject *obj=LBObject::instanceFromStream(ist, errs);
	    if ( ! obj )
	      throw LBException("Failed to recover literal object for process struct: "+errs);
	    _literalobjs.push_back(LBObjPtr(obj));
	  }
      }

    int litypes = -1;
    ist>>litypes;
    ist.get(space);
    if ( space != ' ' || litypes < 0 )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
    for(i=0; i< _literaltypeexps.size();i++ )
      delete _literaltypeexps[i];
    _literaltypeexps.clear();
	
    for( i = 0; i < litypes; i++ )
      {
	TypeExpInfo *tinfo = new TypeExpInfo();
	tinfo->fromStream(ist);
	_literaltypeexps.push_back(tinfo);
	ist.get(space);
	if ( space != ' ' )
	  throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
      }

    ist >> _numtemps;
    ist.get( space );
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
    ist >> _numtempiters;
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    int nargs = -1;
    ist >> nargs;
    ist.get( space );
    if ( space != ' ' || nargs < 0 )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
    for(i=0; i < _namedargs.size(); i++)
      delete _namedargs[i];
    _namedargs.clear();
    for( i=0; i<nargs; i++)
      {
	LBStructEvalArgs *arg = new LBStructEvalArgs();
	arg->fromStream(ist);
	_namedargs.push_back(arg);
	ist.get(space);
	if ( space != ' ' )
	  throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
      }

    int anonargs = -1;
    ist >> anonargs;
    ist.get( space );
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
    for(i=0; i < _anonargs.size(); i++)
      delete _anonargs[i];
    _anonargs.clear();
    
    for( i=0; i<anonargs; i++)
      {
	LBAnonymousArgs *anonarg = new LBAnonymousArgs();
	anonarg->fromStream(ist);
	_anonargs.push_back(anonarg);
	ist.get(space);
	if ( space != ' ' )
	  throw LBException("Corrupted stream, invalid delimeter for process struct metadata");
      }

    ist >> _multithread;
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    _extsymbols.fromStream(ist); 
    ist.get(space);
    if ( space != ' ' )
      throw LBException("Corrupted stream, invalid delimeter for process struct metadata");

    return ist;
  }



  bool LBProcessStruct::resolveExternalSymbolsIfNeeded(string& errs)
  {
    if ( ! _metadata )
      {
	errs += "No metadata information in the process struct\n";
	return false;
      }

    if ( _metadata->_extsresolved )
      return true;
    
    return resolveExternalSymbols(errs);
  }

    
  bool LBProcessStruct::resolveExternalSymbols(string& errs)
  {
    if ( ! _metadata )
      {
	errs += "No metadata information in the process struct\n";
	return false;
      }
    
    LBMutexLocker lock(_metadata->_initmtx);
    if (    _metadata->_extsresolved ) return true;
    bool ok =  _metadata->_extsymbols.resolveAllSymbols(errs);
    ok = ok && reIndexAllGlobalVars(errs);
    ok = ok && reIndexAllPropertyOps(errs);
    ok = ok && resolveAllNamedArgs(errs);

    _metadata->_extsresolved = ok;

    return ok;

  }


  void LBProcessStruct::runAllCode()
  {
    // This function is single threaded,
    // meaning only one thread can run the code through at a time
    // use a lock to ensure that
    if ( ! _metadata || _metadata->_code.size() == 0 )
      return;


    // here are the temp objects and iterators
    TempObjVec temps( _metadata->_numtemps);
    TempIterObjVec tempiters(_metadata->_numtempiters );

    string errmsg;

    // resolve all external symbol
    if ( ! resolveExternalSymbolsIfNeeded(errmsg) )
      throw LBException("Failed to resolve external symbols for struct evaluation: "+errmsg);
    
    _mainstop = false;
    prepareRunTime();

    _runtime->_status = LBProcessStructRunTime::EVALRUNNING;

    // start running
    int ic = 0;
    int numcode = _metadata->_code.size();
    LBVMCode::Operand lastop;
    while (  ic < numcode && ic >= 0 )
      {
	if ( ! runOneCode(ic, &temps, &tempiters, lastop, errmsg) )
	  {
	    throwError(errmsg);
	    return;
	  }
	if ( _mainstop )
	  {
	    throwError("One of the sub-threads died unexpectedly: "+_runtime->_errmsg+"\n");
	    return;
	  }
      }

    // get last result obj in code running
    readObjFromOperand(lastop, &temps,  _runtime->_lastobj, errmsg);

    if ( _metadata->_multithread && _runtime->_livethreads )
      _runtime->_livethreads->waitAll();

    if ( _mainstop )
      {
	throwError("One of the sub-threads died unexpectedly: "+_runtime->_errmsg+"\n");
	return;
      }
    
    _runtime->_status = LBProcessStructRunTime::EVALOK;
    return;
  }

  bool LBProcessStruct::runAllCodeAsynch(string& errs)
  {

    if ( ! _metadata || _metadata->_code.size() == 0 )
      {
	if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	errs += "Null intermidate code list for asynch struct\n";
	throwError(errs, false);
	if ( _runtime->_parentcomp )
	  _runtime->_parentcomp->informComponentAct(false);
	return false;
      }


    string preerrmsg;
    // resolve all external symbol
    if ( ! resolveExternalSymbolsIfNeeded(preerrmsg) )
      {
	if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	errs += "Failed resolve external symbols for component inside composition:\n"+preerrmsg;
	throwError(errs, false);
	if ( _runtime->_parentcomp )
	  _runtime->_parentcomp->informComponentAct(false);
	return false;
      }

    // prepare the run time environment
    _mainstop = false;
    prepareRunTime();

    _runtime->_status = LBProcessStructRunTime::EVALRUNNING;

    // start running
    while ( ! _mainstop )
      {
	// here are the temp objects and iterators
	TempObjVec temps( _metadata->_numtemps);
	TempIterObjVec tempiters(_metadata->_numtempiters );

	string errmsg;
	int ic = 0;
	int numcode = _metadata->_code.size();
	LBVMCode::Operand lastop;
	while ( ic < numcode && ic >= 0  && ! _mainstop)
	  {
	    try {
	      if ( ! runOneCode(ic, &temps, &tempiters, lastop, errmsg) )
		{
		  if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
		  errs += "Error exexuting intermediate code:\n"+errmsg;
		  throwError(errs, false);
		  if ( _runtime->_parentcomp )
		    _runtime->_parentcomp->informComponentAct(false);
		  return false;
		}
	    }
	    catch( LBException& excp )
	      {
		throwError(excp.msg(), false);
		if ( _runtime->_parentcomp )
		  _runtime->_parentcomp->informComponentAct(false);
		return false;
	      }
	    catch( ... )
	      {
		throwError("Unknown error in asynch struct main thread", false);
		if ( _runtime->_parentcomp )
		  _runtime->_parentcomp->informComponentAct(false);
		return false;
	      }
	  }

	if ( _mainstop )
	  break;

	if ( _metadata->_multithread && _runtime->_livethreads )
	  _runtime->_livethreads->waitAll();

	if ( _mainstop )
	  break;
    
	// wait for new inputs
	if ( ( _properties && _properties->_ins ) || _runtime->_adhocDataCounter )
	  {
	    if ( _runtime->_parentcomp )
	      _runtime->_parentcomp->informComponentAct(false);

	    if ( _runtime->_adhocDataCounter )  // embeded asynch
	      {
		if ( ! _runtime->_adhocDataCounter->waitUpFromZero() )
		  break;
	      }
	    else // typed asynch
	      if ( ! _properties->_ins->waitForNewInput() )
		break;
	    if ( _runtime->_parentcomp )
	      _runtime->_parentcomp->informComponentAct(true);
	    _runtime->reset();
	    continue;
	  }
	break;
      }

    if ( _runtime->_parentcomp )
      _runtime->_parentcomp->informComponentAct(false);

    if ( _mainstop )
      errs += "Asynch struct thread terminated\n";

    throwError(errs, false);  // dismiss all wating thread for this struct's output

    if ( _mainstop )
      return false;
    _runtime->_status = LBProcessStructRunTime::EVALOK;
    return true;
  }

  void LBProcessStruct::signalNewData()
  {
    if ( ! _runtime->_parentcomp )
      throw LBException("Fatal error, signalNewData is only to be called by adhoc process struct inside a composition");

    if ( ! _runtime->_adhocDataCounter )
      throw LBException("Fatal error, call signalNewData on a synch struct");
    
    _runtime->_adhocDataCounter->inc();

  }

//   void LBProcessStruct::dismissNewDataWaiters()
//   {
//     if ( ! _runtime->_parentcomp )
//       throw LBException("Fatal error, signalNewData is only to be called by adhoc process struct inside a composition");

//     if ( ! _runtime->_adhocDataCounter )
//       throw LBException("Fatal error, call signalNewData on a synch struct");
    
//     _runtime->_adhocDataCounter->dismissWaitingThreads();

//   }

//   bool LBProcessStruct::isAdhocAsynch() const
//   {
//     return _runtime && _runtime->_parentcomp && _runtime->_adhocDataCounter;
//   }

  // helper classes used only in the VM code interpreter
  class LBProcessStructVarArgs : public LBVarArgs
  {
  public:
    LBProcessStructVarArgs(const vector<LBObjPtr> &objs)
      : _snapshots(objs)
    {}

    int numArgs() const { return _snapshots.size(); }
    const LBObject* getArg(int index) const 
    { 
      if ( index < 0 || index >= _snapshots.size() ) return 0; 
      return _snapshots[index].getConstRealPtr();
    }

  private:
    LBProcessStructVarArgs(const LBProcessStructVarArgs& v);
    LBProcessStructVarArgs& operator=(const LBProcessStructVarArgs& v);

    const vector<LBObjPtr> &_snapshots;

  };
      
  class LBProcessStructNamedArgs : public LBNamedArgs
  {
  public:
    LBProcessStructNamedArgs()
      : _names(0), _values(0), _sz(0)
    {}

    LBProcessStructNamedArgs(const vector<LBSymbol> &names, const vector<LBObjPtr> &values)
      : _names(&names), _values(&values), _sz(names.size())
    {}

    int numArgs() const { return _sz; }
    const LBObject* getArg(int index) const 
    { 
      if ( ! _values ) return 0;
      if ( index < 0 || index >= _sz ) return 0; 
      return (*_values)[index].getConstRealPtr();
    }

    const LBSymbol* getName(int index) const 
    { 
      if ( index < 0 || index >= _sz ) return 0; 
      return &(*_names)[index];
    }

  private:
    
    const vector<LBSymbol> *_names;
    const vector<LBObjPtr> *_values;
    int _sz;
  };
      


  bool LBProcessStruct::runOneCode(int& ic, TempObjVec *tempobjs, TempIterObjVec *tempiterobjs, LBVMCode::Operand& lastop, string& errs)
  {
    LBVMCode onecode = _metadata->_code[ic];
    LBVMCode::OPCode opcode = onecode.opcode();
    LBVMCode::Operand op1 = onecode.op1();
    int op1idx = op1.opindex();
    LBVMCode::Operand::OperandType op1type = op1.optype(); 
    LBVMCode::Operand op2 = onecode.op2();
    int op2idx = op2.opindex();
    LBVMCode::Operand::OperandType op2type = op2.optype(); 
    LBVMCode::Operand resultop = onecode.resultop();
    int resultopidx = resultop.opindex();
    LBVMCode::Operand::OperandType resultoptype = resultop.optype(); 

    string localerrs;

    ic++;

    switch ( opcode )
      {
      case LBVMCode::NOOP:
	return true;
      case LBVMCode::GOTO:
	ic = op1idx;
	return true;
      case LBVMCode::ZEROJUMP:
	{
	  lastop = op1;
	  LBObjPtr objptr;
	  if ( !readObjFromOperand(op1, tempobjs, objptr, errs) )
	    return false;
	  const LBBool *bval = dynamic_cast<const LBBool*>(objptr.getRealPtr());
	  if ( bval == 0 )
	    {
	      errs += "The condition value for conditional statement is not a boolean value: "+objptr.toString()+"\n";;
	      return false;
	    }
	  if ( ! bool(*bval) )
	    ic = op2idx;

	  return true;
	}
      case LBVMCode::ORJMP:
      case LBVMCode::ANDJMP:
	{
	  LBObjPtr objptr;
	  if ( !readObjFromOperand(op1, tempobjs, objptr, errs) )
	    {
	      errs += "Can not read first operand for logical and/or operation";
	      return false;
	    }

	  const LBBool *bval = dynamic_cast<const LBBool*>(objptr.getConstRealPtr());
	  if ( ! bval )
	    {
	      if ( ! writeObjToOperand(resultop, tempobjs, LBError("The first operand for logical and/or operation is not a bool type: "+objptr.toString()), errs) )
		return false;
	      lastop = resultop;
	      ic = op2idx;
	      return true;
	    }



	  if ( opcode == LBVMCode::ORJMP )
	    {
	      if ( bool(*bval) )
 		{
		  if ( !writeObjToOperand(resultop, tempobjs, LBBool(true),errs) )
		    return false;	      
		  lastop = resultop;
		  ic = op2idx;
		}
	    }
	  else
	    if ( ! bool(*bval) )
	      {
		if ( !writeObjToOperand(resultop, tempobjs, LBBool(false),errs) )
		  return false;	      
		lastop = resultop;
		ic = op2idx;
	      }

	  return true;
	}

      case LBVMCode::DISPATCH:
	{
	  LiveThread *onethread = new LiveThread(*this, op1idx, op2idx, resultop);
	  _runtime->_livethreads->add(onethread);

	  // increase running thread counter
	  if ( ! _runtime->_livethreads->incRunning() )
	    {
	      errs += "Can not increase running thread counter at the start of the thread\n";
	      return false;
	    }
	  if ( ! addPendingUpdate( resultop, errs ) )
	    {
	      _runtime->_livethreads->decRunning();
	      errs += "Can not increase pending update number for target variable before dispatching\n";
	      return false;
	    }
	  if ( !onethread->start() )
	    {
	      _runtime->_livethreads->decRunning();
	      removePendingUpdate(resultop, errs);
	      errs += "Failed to start new thread\n";
	      return false;
	    }
	  ic = op2idx+1;
	  return true;
	}

      case LBVMCode::GETITER:
	{
	  LBObjPtr objptr;
	  if ( !readObjFromOperand(op1, tempobjs, objptr, errs) )
	    {
	      errs += "Can not read container operand to do foreach loop\n";
	      return false;
	    }
	  if ( objptr )
	    {
	      LBConstIterator *it = objptr->getIterator();
	      if ( it )
		{
		  (*tempiterobjs)[resultopidx].reset(it, objptr);
		  return true;
		}
	      errs += "Can not get iterator from non-container type: "+objptr->getType().toString();
	      return false;
	    }
	  errs += "Can not get iterator from null object\n";
	  return false;
	}

      case LBVMCode::ITERNEXT:
	{
	  // a little optimization can be done here,
	  // clean up the iterator operand after it is used
	  // or obtain a read locked object in the iterator instead of a snap shot
	  LBBool nxt((*tempiterobjs)[op1idx].getIterator()->next());
	  return writeObjToOperand(resultop, tempobjs, nxt, errs);
	}

      case LBVMCode::ITEROBJ:
	{
	  LBConstIterator *it =  (*tempiterobjs)[op1idx].getIterator();
	  if ( it )
	    {
	      const LBVarArgs *row =it->getCurrentRow();
	      bool writeok = false;
	      if ( row )
		{
		  const LBObject *obj=row->getArg(op2idx);
		  if ( obj )
		    writeok = writeObjToOperand(resultop, tempobjs, *obj, errs);
		  else
		    writeok = writeObjToOperand(resultop, tempobjs, LBObjPtr(), errs);
		}
	      else
		writeok = writeObjToOperand(resultop, tempobjs, LBError("Empty row in an unfinished iterator"), errs);

	      if ( ! writeok ) 
		return false;
	      lastop = resultop;
	      return true;
	    }
	  errs += "Corrupted struct, failed to get iterator from operand\n";
	  return false;
	}

      case LBVMCode::RESETITER:
	{
	  (*tempiterobjs)[op1idx].reset();
	  return true;
	}
      case LBVMCode::HALT:
	if ( _metadata->_multithread && _runtime->_livethreads )
	  {
	    cancelInternalThreads();
	    _runtime->_livethreads->joinAll();
	  }
	ic = -1;
	return true;
      case LBVMCode::HOLD:
	if ( _runtime->_livethreads )
	  _runtime->_livethreads->waitAll();
	return true;
      case LBVMCode::CANCEL:
	cancelInternalThreads();
	// wait for all threads to finish
	if ( _runtime->_livethreads )
	  _runtime->_livethreads->joinAll();
	return true;
      case LBVMCode::HOLDONTO:
	{
	  switch ( op1type )
	    {
	    case LBVMCode::Operand::LOCALVAR:
	      {
		(*_runtime->_localvars)[op1idx].waitUntilNoPending();
		break;
	      }
	    case LBVMCode::Operand::GLOBALVAR:
	      {
		LBStruct *st = 0;
		int pidx = -1;
		if ( resolveGlobalVar(op1idx, st, pidx, errs) )
		  {
		    st->waitUntilNoPending(pidx);
		    break;
		  }
		return false;
	      }
	    default:
	      errs += "Corrupted struct, operand for WAIT instruction is not a variable\n";
	      return false;
	    }
	  return true;
	}

      case LBVMCode::BATCHOUT:
	{
	  LBObjPtr objptr;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr, errs) )
	    return false;
	  LBStruct *st = dynamic_cast<LBStruct*>(objptr.getRealPtr());
	  if ( ! st )
	    {
	      errs += "Tried to take the output properties from a non-struct object\n";
	      return false;
	    }
	  if ( ! _metadata->_itfc->hasInterface( *st->interface().name() ) )
	    {
	      errs += "Struct does not implement the interface of "+st->interface().name()->toString()+". Batch setting of output property can not be done\n";
	      return false;
	    }	       
	  int numouts = st->interface().numberOfPropertiesByAttr(LBPropertyInfo::OUT);
	  for(int i = 0; i < numouts; i++)
	    {
	      const LBPropertyInfo *pinfo = st->interface().propertyInfo(i, LBPropertyInfo::OUT);
	      LBObjPtr objptr = st->getPropertyValue(i, LBPropertyInfo::OUT);
	      int localindex = -1;
	      const LBPropertyInfo *localpinfo = _metadata->_itfc->propertyInfo(pinfo->name(), localindex);
	      if ( localpinfo && localpinfo->attr() == LBPropertyInfo::OUT )
		_properties->_outs->writeValueByInternal(localindex, objptr );
	      else
		{
		  errs += "Batch output from struct contains invalid output property "+pinfo->name().toString()+"\n";
		  return false;
		}
	    }
	  return true;
	}

      case LBVMCode::ASN:
	{
	  LBObjPtr objptr;
	  if ( !readObjFromOperand(op2, tempobjs, objptr, errs) )
	    {
	      errs += " Can not read right side operand in assignment operation";
	      return false;
	    }
	  else
	    if ( !writeObjToOperand(op1, tempobjs, objptr, errs) )
	      return false;
	  
	  lastop = op1;
	  return true;

	}

      case LBVMCode::MULASN:
	{
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( !readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("multiply assign operation on Null object");
		else
		  {
		    try {
		      obj1->mul(*objptr2);
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing multiply assign operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to read left operand for multiply assign operation:"+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing multiply assign\n";
	  return false;

	}

      case LBVMCode::DIVASN:
	{
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( !readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("divide assign operation on Null object");
		else
		  {
		    try {
		      obj1->div(*objptr2);
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing divide assign operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to obtain object for divide assign operation: "+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing divide assign\n";
	  return false;

	}

      case LBVMCode::MODASN:
	{
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( !readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("mod assign operation on Null object");
		else
		  {
		    try {
		      obj1->mod(*objptr2);
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing mod assign operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to obtain object for mod assign operation"+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing mod assign\n";
	  return false;

	}

      case LBVMCode::ADDASN:
	{
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( !readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("add assign operation on Null object");
		else
		  {
		    try {
		      obj1->add(*objptr2);
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing add assign operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to obtain object for add assign operation"+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing add assign\n";
	  return false;

	}

      case LBVMCode::SUBASN:
	{
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( !readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("minus assign operation on Null object");
		else
		  {
		    try {
		      obj1->sub(*objptr2);
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing minus assign operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to obtain object for minus assign operation"+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing minus assign\n";
	  return false;

	}
      case LBVMCode::LGCOR:
      case LBVMCode::LGCAND:
	{
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  LBError errvalue;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;
	  else
	    {
	      LBBool *b1 = dynamic_cast<LBBool*>(objptr1.getRealPtr());
	      LBBool *b2 = dynamic_cast<LBBool*>(objptr2.getRealPtr());
	      if ( b1==0 || b2==0 )
		{
		  if (!writeObjToOperand(resultop, tempobjs, LBError("One more operands for logic OR operation is not boolean type"),errs))
		    return false;
		  lastop = resultop;
		  return true;
		}

	      bool res;
	      if ( opcode == LBVMCode::LGCOR )
		res = bool(*b1) || bool(*b2);
	      else
		res = bool(*b1) && bool(*b2);

	      if ( !writeObjToOperand(resultop, tempobjs, LBBool(res),errs) )
		return false;
	      
	      lastop = resultop;
	      return true;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, errvalue,errs) )
	    return false;
	  
	  lastop = resultop;
	  return true;


	}

      case LBVMCode::EQCMP:
      case LBVMCode::NEQCMP:
	{
	  bool eqjdg = opcode == LBVMCode::EQCMP;
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    {
	      errs += " Failed to read operands for equality comparison operation";
	      return false;
	    }
	  if (  objptr1 == objptr2  ) // pointing to same obj or both are null
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBBool( eqjdg?true:false ),errs) )
		return false;
	      lastop = resultop;
	      return true; 
	    }

	  if (  ! objptr1 || ! objptr2  ) // one of them is null, so not equal
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBBool(eqjdg?false:true),errs) )
		return false;
	      lastop = resultop;
	      return true; 
	    }		

	  string emsg;
	  try {
	    bool res = objptr1->equals(*objptr2);
	    if ( !writeObjToOperand(resultop, tempobjs, LBBool(eqjdg?res:!res),errs) )
	      return false;
	    lastop = resultop;
	    return true;
	  }
	  catch ( LBException& e)
	    {
	      emsg = e.msg();
	    }
	  try {
	    bool res = objptr2->equals(*objptr1);
	    if ( !writeObjToOperand(resultop, tempobjs, LBBool(eqjdg?res:!res),errs) )
	      return false;
	    lastop = resultop;
	    return true;
	  }
	  catch ( LBException& e)
	    {
	      emsg += e.msg();
	    }
	  emsg = "Can not compare two values: "+emsg;
	  if ( !writeObjToOperand(resultop, tempobjs, LBError(emsg),errs) )
	    return false;
	  lastop = resultop;
	  return true;

	}

      case LBVMCode::ISA:
	{
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    {
	      errs += " Failed to read operands for isa type checking operation";
	      return false;
	    }

	  const LBTypeSpec *tspec = dynamic_cast<const LBTypeSpec*>(objptr2.getRealPtr());
	  if ( ! tspec )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Operand two in ISA operation is null or  non-type spec type\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  if ( ! objptr1 ) // null value is not anything
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBBool(false),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }
	  
	  try {
	    bool res = tspec->checkObj(*objptr1);
	    if ( !writeObjToOperand(resultop, tempobjs, LBBool(res),errs) )
		return false;
	  }
	  catch( LBException& e)
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Exception happened when type checking: "+e.msg()),errs) )
		return false;
	    }
	  
	  lastop = resultop;
	  return true;

	}
      case LBVMCode::TYPEOF:
	{
	  LBObjPtr objptr1;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) )
	    {
	      errs += " Fail to read operand in TYPEOF operation";
	      return false;
	    }
	  if ( ! objptr1 )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Operand is null in TYPEOF operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  // handle struct case first
	  const LBStruct *st = dynamic_cast<const LBStruct*>( objptr1.getConstRealPtr() );
	  if ( st )
	    {
	      if ( st->interface().name() ) // a named struct
		{
		  LBObjPtr sttype(static_cast<LBObject*>(new LBTypeSpecStruct(&st->interface())));
		  if ( !writeObjToOperand(resultop, tempobjs, sttype, errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}

	      LBObjPtr adhocst(static_cast<LBObject*>(new LBTypeSpecStruct(st->interface().mode())));
	      if ( !writeObjToOperand(resultop, tempobjs, adhocst, errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }  

	  // now it must be the basic type
	  LBObjPtr res(static_cast<LBObject*>(new LBTypeSpecBasic(objptr1->getType())));
	  if ( !writeObjToOperand(resultop, tempobjs, res,errs) )
	    return false;
	  lastop = resultop;
	  return true;

	}
	
      case LBVMCode::LTCMP:
      case LBVMCode::GTCMP:
      case LBVMCode::LECMP:
      case LBVMCode::GECMP:
	{
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    {
	      errs += " Failed to read operands for < <= > >= comparison operation";
	      return false;
	    }
	  if ( ! objptr1 || ! objptr2 )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("One of the operand is null for relational comparison operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }
	  
	  try {
	    bool res;
	    switch ( opcode )
	      {
	      case LBVMCode::LTCMP:
		res = objptr1->before(*objptr2);
		break;
	      case LBVMCode::GECMP:
		res = ! objptr1->before(*objptr2);
		break;
	      case LBVMCode::LECMP:
		res = ! objptr2->before(*objptr1);
		break;
	      case LBVMCode::GTCMP:
		res = objptr2->before(*objptr1);
		break;
	      }
	    
	    if ( !writeObjToOperand(resultop, tempobjs, LBBool(res),errs) )
	      return false;
	  }
	  catch( LBException& e)
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Error for relational comparison operation: " + e.msg()),errs) )
		return false;
	    }

	  lastop = resultop;
	  return true;
	}

      case LBVMCode::MOD:
	{
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;

	  if ( ! objptr1 || ! objptr2 )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("One of the operand is null for % operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  LBObject *result = objptr1->clone();
	  try {
	  result->mod(*objptr2);
	  }
	  catch( LBException& e)
	    {
	      delete result;
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Error for % operation: " + e.msg()),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBObjPtr(result), errs) )
	      return false;
	  
	  lastop = resultop;
	  return true;


	}

      case LBVMCode::ADD:
      case LBVMCode::MINUS:
      case LBVMCode::MUL:
      case LBVMCode::DIV:
	{
	  LBObjPtr objptr1;
	  LBObjPtr objptr2;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) || ! readObjFromOperand(op2, tempobjs, objptr2, errs) )
	    return false;

	  if ( ! objptr1 || ! objptr2 )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("One of the operand is null for +-*/ operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  LBObject *result = LubanImplicitCast::handleDoubleAndInt(*objptr1, *objptr2, opcode );
	  if ( ! result  )
	    {
	      result = objptr1->clone();
	      try {
		switch ( opcode )
		  {
		  case LBVMCode::ADD:
		    result->add(*objptr2);
		    break;
		  case LBVMCode::MINUS:
		    result->sub(*objptr2);
		    break;
		  case LBVMCode::MUL:
		    result->mul(*objptr2);
		    break;
		  case LBVMCode::DIV:
		    result->div(*objptr2);
		    break;
		  }
	      }
	      catch( LBException& e)
		{
		  delete result;
		  if ( !writeObjToOperand(resultop, tempobjs, LBError("Error for +-*/% operation: " + e.msg()),errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBObjPtr(result), errs) )
	      return false;
	  
	  lastop = resultop;
	  return true;

	}
	  
      case LBVMCode::LGCNOT:
	{
	  LBObjPtr objptr1;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) )
	    return false;
	  
	  LBBool *b = dynamic_cast<LBBool*>(objptr1.getRealPtr());
	  if ( !b )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("The operand is not boolean type for logic \"not\" operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }
	  LBBool res ((*b)? false:true);
	  if ( !writeObjToOperand(resultop, tempobjs, res, errs) )
	    return false;
	  lastop = resultop;
	  return true;	  
	}

      case LBVMCode::PP:
      case LBVMCode::MM:
	{
	  LBError errvalue;

	  {
	    LBWriteLockedObjPtr obj1;
	    if (  getWriteLockedObjFromOperand( op1, tempobjs, obj1, localerrs) )
	      {
		if ( ! obj1 ) // assume the operand is not LOCKED because the returned obj is null
		  errvalue = LBError("++ or -- operation on Null object");
		else
		  {
		    try {
		      if ( opcode == LBVMCode::MM )
			obj1->minusminus();
		      else
			obj1->plusplus();
		      lastop = op1;
		      return true;
		    }
		    catch ( LBException& e )
		      {
			obj1.invalidate();
			errvalue = LBError("Error happened when doing ++ -- operation: "+e.msg());
		      }
		  }
	      }
	    else
	      {
		errvalue = LBError("Failed to obtain object for ++ or -- operation: "+localerrs);
	      }
	  }

	  if ( writeObjToOperand(op1, tempobjs, errvalue, errs) )
	    {
	      lastop = op1;
	      return true;
	    }

	  errs += "Failed to write error value into object when doing ++ or --\n";
	  return false;

	}

      case LBVMCode::NEG:
	{
	  LBObjPtr objptr1;
	  if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) )
	    return false;

	  if ( ! objptr1 )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("The operand is null for negation(-x) operation\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }
	  
	  uniclone(objptr1);
	  try{
	    objptr1->neg();
	    if ( !writeObjToOperand(resultop, tempobjs, objptr1, errs) )
	      return false;
	  }
	  catch( LBException& e )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Error for negation operation: "+e.msg()),errs) )
		return false;
	    }
	  lastop = resultop;
	  return true;
	}

      case LBVMCode::MBRFUNC:
	{
	  const LBVMCode::Operand *objop = _metadata->_memberfuncops.objOperand(op1idx);
	  const LBSymbol *fname =  _metadata->_memberfuncops.funcName(op1idx);
	  if ( ! objop || ! fname )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Corrupted struct, invalid object operand or null function name for member function call\n"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }
	  
	  LBError errvalue;
	  bool errhappened=false;
	  
	  // get all arguments values
	  int numargs = 0;
	  LBAnonymousArgs *argops = 0;
	  if ( op2type != LBVMCode::Operand::UNUSED )
	    {
	      argops= _metadata->_anonargs[op2idx];
	      numargs = argops->size();
	    }

	  vector<LBObjPtr> argvalues(numargs);
	  LBProcessStructVarArgs vargs(argvalues);
	  if ( numargs )
	    {
	      for(int i=0; i < numargs; i++)
		if ( ! readObjFromOperand( *argops->getArgOp(i), tempobjs, argvalues[i], errs ) )
		  return false;
	    }


	  // this block is to contain the scope of the write lock
	  LBObject *resobj=0;
	  if ( ! errhappened )
	  {
	    LBWriteLockedObjPtr obj;
	    if ( ! getWriteLockedObjFromOperand( *objop, tempobjs, obj, localerrs ) || ! obj )
	      {
		errvalue = LBError("Can not obtain the write-locked exclusive instance of an object for member function call or the object is null: "+localerrs);
		errhappened = true;
	      }
	    else
	      {
		// this place could be optimized to use assignment in possible situation
		try {
		  if ( op2type == LBVMCode::Operand::UNUSED || numargs == 0 )
		    resobj = obj->memberFuncCall(*fname, 0 );
		  else
		    resobj = obj->memberFuncCall(*fname, &vargs );
		}
		catch ( LBException& e )
		  {
		    obj.invalidate();
		    errvalue = LBError("Error in member function call operation: "+e.msg());
		    errhappened = true;
		  }
	      }
	  }

	  if ( ! errhappened )
	    {
	      if ( ! writeObjToOperand(resultop, tempobjs, LBObjPtr(resobj), errs ) )
		{
		  errs += "Failed to write result to operand in member function call operation\n";
		  return false;
		}
	      lastop = resultop;
	      return true;
	    }

	  if ( ! writeObjToOperand(resultop, tempobjs, errvalue, errs ) )
	    {
	      errs += "Failed to write result to operand in member function call operation\n";
	      return false;
	    }
	    
	  lastop = resultop;
	  return true;

	}

      case LBVMCode::STRUCTEVAL:
	{
	  lastop = resultop;
	  const LBStruct *st = 0;
	  LBObjPtr objptr1;
	  if ( op1type == LBVMCode::Operand::GLOBALTYPE )
	    st = _metadata->_extsymbols.getOriginalStructPointer(op1idx);
	  else
	    {
	      if ( ! readObjFromOperand(op1, tempobjs, objptr1, errs) )
		return false;

	      st = dynamic_cast<const LBStruct*>(objptr1.getConstRealPtr());
	    }
	  if ( ! st )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("The object is not a struct for struct evaluation"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  LBStruct *evalcopy = 0;
	  LBVectorStruct *vecedst = 0;
	  if ( op2type == LBVMCode::Operand::UNUSED )
	    {
	      try {
		evalcopy = st->structCall();
	      }
	      catch(LBException& lbe)
		{
		  if ( !writeObjToOperand(resultop, tempobjs, LBError("Struct evaluation exception: "+lbe.msg()),errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}
	    }
	  else
	    {
	      LBStructEvalArgs *argdata = _metadata->_namedargs[op2idx];
	      if ( argdata->numVecNames() > 0 )
		{
		  string emsg;
		  vecedst = LBVectorStruct::newVectorStruct(*st, *argdata, emsg);
		  if ( !vecedst )
		    {
		      if ( !writeObjToOperand(resultop, tempobjs, LBError("Failed to vectorize struct for struct evaluation: "+emsg),errs) )
			return false;
		      lastop = resultop;
		      return true;
		    }
		  st = vecedst;
		}

	      // gather the arg values
	      int numargs = argdata->numArgs();
	      if ( numargs == 0 )
		{
		  try {
		    evalcopy = st->structCall();
		  }
		  catch(LBException& lbe)
		    {
		      if ( !writeObjToOperand(resultop, tempobjs, LBError("Struct evaluation exception: "+lbe.msg()),errs) )
			return false;
		      lastop = resultop;
		      return true;
		    }
		}
	      else
		{
		  // populate the args
		  vector<LBObjPtr> argvalues(numargs);
		  vector<LBSymbol> names(numargs);
		  for( int i = 0; i < numargs; i++)
		    {
		      names[i] = (*argdata->_namedargs)[i]._name;
		      if ( (*argdata->_namedargs)[i]._src == LBStructEvalArgs::OneNamedArg::DIRECT )
			{
			  if ( !readObjFromOperand((*argdata->_namedargs)[i]._directop, tempobjs,  argvalues[i], localerrs) )
			    {
			      errs += "Failed to get argument value for named argument "+(*argdata->_namedargs)[i]._name.toString()+", "+localerrs;
			      return false;
			    }
			}
		      else
			argvalues[i] = _properties->_ins->readValueByInternal((*argdata->_namedargs)[i]._thisindex );
		    }
		  LBProcessStructNamedArgs nmargs(names, argvalues);
		  try {
		    evalcopy = st->structCall(&nmargs);
		  }
		  catch(LBException& lbe)
		    {
		      if ( !writeObjToOperand(resultop, tempobjs, LBError("Struct evaluation exception: "+lbe.msg()),errs) )
			return false;
		      lastop = resultop;
		      return true;
		    }
		}
	      if ( vecedst ) delete vecedst;	      
	    }

	  if ( ! evalcopy )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Struct evaluation failure"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBObjPtr(static_cast<LBObject*>(evalcopy)),errs) )
	    return false;
	  lastop = resultop;
	  return true;
	}

      case LBVMCode::BATCHSET:
	{
	  int numargs = 0;
	  LBStructEvalArgs *argdata=0;
	  if ( op2type != LBVMCode::Operand::UNUSED )
	    {
	      argdata = _metadata->_namedargs[op2idx];
	      if ( argdata->numVecNames() > 0 )
		{
		  errs += "Vectorizing is not allowed in batch property setting";
		  return false;
		}

	      // gather the arg values
	      numargs = argdata->numArgs();
	    }

	  // populate the args
	  vector<LBObjPtr> argvalues(numargs);
	  vector<LBSymbol> names(numargs);
	  for( int i = 0; i < numargs; i++)
	    {
	      names[i] = (*argdata->_namedargs)[i]._name;
	      if ( (*argdata->_namedargs)[i]._src == LBStructEvalArgs::OneNamedArg::DIRECT )
		{
		  if ( !readObjFromOperand((*argdata->_namedargs)[i]._directop, tempobjs,  argvalues[i], localerrs) )
		    {
		      errs += "Failed to get argument value for named argument "+(*argdata->_namedargs)[i]._name.toString()+", "+localerrs;
		      return false;
		    }
		}
	      else
		argvalues[i] = _properties->_ins->readValueByInternal((*argdata->_namedargs)[i]._thisindex );
	    }

	  LBProcessStructNamedArgs namedargs(names, argvalues);


	  LBWriteLockedObjPtr structobj;
	  if ( ! getWriteLockedObjFromOperand(op1, tempobjs, structobj, errs) )
	    {
	      errs += "Failed to get struct object for property batch setting operation\n";
	      return false;
	    }
	  if ( ! structobj )
	    {
	      errs += "Null struct object for property batch setting operation\n";
	      return false;
	    }

	  LBStruct *st = dynamic_cast<LBStruct*>(structobj.getRealPtr());
	  if ( ! st )
	    {
	      errs += "Attempted property batch setting operation on a non-struct object\n";
	      return false;
	    }

	  try {
	    st->setBatchProperties(&namedargs);
	  }
	  catch(LBException& lbe)
	    {
	      errs += "Struct batch setting exception: "+lbe.msg();
	      return false;
	    }
	  lastop = op1;
	  return true;

	}


      case LBVMCode::OBJCONSTRU:
	{
	  const LBTypeSpecBasic *tbasic = 0;
	  if ( op1type == LBVMCode::Operand::GLOBALTYPE )
	    {
	      LBObjPtr st(static_cast<LBObject*>(_metadata->_extsymbols.getInitializedStructCopy(op1idx)));
	      if ( st )
		{
		  if ( op2type == LBVMCode::Operand::UNUSED )
		    {
		      if ( !writeObjToOperand(resultop, tempobjs, st,errs) )
			return false;
		      lastop = resultop;
		      return true;
		    }
		  if ( !writeObjToOperand(resultop, tempobjs, LBError("Struct obj construction takes no arguement"),errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}
	      const LBTypeSpec* tspec = _metadata->_extsymbols.getTypeSpec(op1idx);
	      tbasic = dynamic_cast<const LBTypeSpecBasic*>(tspec);
	    }
	  else
	    if ( op1type == LBVMCode::Operand::LITERALOBJ )
	      tbasic = dynamic_cast<const LBTypeSpecBasic*>( _metadata->_literalobjs[op1idx].getConstRealPtr() );
	    else
	      {
		errs += "Corrupted struct, unknown operand type for object construction\n";
		return false;
	      }

	  if ( ! tbasic || tbasic->typeInfo() == 0  )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Unconstructable global type symbol in object construction operation"),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  LBObject *obj=0;

	  try {
	  if ( op2type == LBVMCode::Operand::UNUSED )
	    obj = tbasic->typeInfo()->newInstance();
	  else
	    {
	      // get all arguments values
	      LBAnonymousArgs *argops = _metadata->_anonargs[op2idx];;
	      int numargs = argops->size();
	      vector<LBObjPtr> argvalues(numargs);
	      for(int i=0; i < numargs; i++)
		if ( ! readObjFromOperand( *argops->getArgOp(i), tempobjs, argvalues[i], errs ) )
		  return false;

	      LBProcessStructVarArgs vargs(argvalues);

	      obj = tbasic->typeInfo()->newInstance(&vargs);

	      if ( ! obj && numargs == 1 ) // then try another way around
		{
		  const LBObject* thearg = argvalues[0].getConstRealPtr();
		  if ( thearg )
		    {
		      obj = tbasic->typeInfo()->newInstance();
		      if ( obj )
			{
			  if ( ! thearg->cast(obj) )
			    {
			      delete obj;
			      obj = 0;
			    }
			}
		    }
		}
	    }

	  if ( obj )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBObjPtr(obj) ,errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBError("Can not construct object of type "+tbasic->toString()+" from specified arguments") ,errs) )
	    return false;
		
	  lastop = resultop;
	  return true;

	  }
	  
	  catch( LBException& excp)
	    {
	      localerrs = excp.msg();
	      if ( obj ) delete obj;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBError("Can not construct object of type "+tbasic->toString()+" from specified arguments: "+localerrs) ,errs) )
	    return false;
		
	  lastop = resultop;
	  return true;
	  
	}
	
      case LBVMCode::VECALL:
	{
	  string emsg;
	  LBVectorStruct *veced = 0;

	  if ( op1type == LBVMCode::Operand::GLOBALTYPE )
	    {
	      LBStruct *st = _metadata->_extsymbols.getInitializedStructCopy(op1idx);
	      if ( st )
		veced = LBVectorStruct::newVectorStruct(st, op2idx == 1, emsg );
	      else
		{
		  if ( !writeObjToOperand(resultop, tempobjs, LBError("The object is not a struct for struct evaluation: "+localerrs),errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}
	    }
	  else
	    {
	      const LBStruct *st=0;
	      LBObjPtr objptr1;
	      if (!readObjFromOperand(op1,tempobjs, objptr1, errs) )
		return false;
	      st = dynamic_cast<const LBStruct*>(objptr1.getConstRealPtr());
	      if ( st )
		veced = LBVectorStruct::newVectorStruct(*st, op2idx == 1, emsg );
	      else
		{
		  if ( !writeObjToOperand(resultop, tempobjs, LBError("The object is not a struct for struct evaluation: "+localerrs),errs) )
		    return false;
		  lastop = resultop;
		  return true;
		}
	      
	    }

	  if ( ! veced )
	    {
	      if ( !writeObjToOperand(resultop, tempobjs, LBError("Failed to vectorize a struct:"+emsg),errs) )
		return false;
	      lastop = resultop;
	      return true;
	    }

	  if ( !writeObjToOperand(resultop, tempobjs, LBObjPtr(static_cast<LBObject*>(veced)),errs) )
	    return false;
	  lastop = resultop;
	  return true;

	}

      case LBVMCode::VECINS:
	{
	  LBWriteLockedObjPtr vecobj;
	  if ( ! getWriteLockedObjFromOperand(resultop, tempobjs, vecobj, errs) || ! vecobj )
	    {
	      errs += "Failed to get the base object for vector construction\n";
	      return false;
	    }
	  LBVector *realvec = dynamic_cast<LBVector*>(vecobj.getRealPtr());
	  if ( ! realvec )
	    {
	      errs += "The base object for vector construction is not a vector\n";
	      return false;
	    }
	  LBObjPtr element;
	  if ( ! readObjFromOperand(op1, tempobjs, element, errs ) )
	    return false;

	  realvec->push(element);
	  lastop = resultop;
	  return true;
	}


      case LBVMCode::MAPINS:
	{
	  LBWriteLockedObjPtr mapobj;
	  if ( ! getWriteLockedObjFromOperand(resultop, tempobjs, mapobj, errs) || ! mapobj )
	    {
	      errs += "Failed to get the base object for map construction\n";
	      return false;
	    }
	  LBMap *realmap = dynamic_cast<LBMap*>(mapobj.getRealPtr());
	  if ( ! realmap )
	    {
	      errs += "The base object for vector construction is not a map\n";
	      return false;
	    }
	  LBObjPtr key;
	  if ( ! readObjFromOperand(op1, tempobjs, key, errs ) )
	    return false;
	  LBObjPtr value;
	  if ( ! readObjFromOperand(op2, tempobjs, value, errs ) )
	    return false;

	  realmap->set(key,value);
	  lastop = resultop;	  
	  return true;
	}

      case LBVMCode::SETINS:
	{
	  LBWriteLockedObjPtr setobj;
	  if ( ! getWriteLockedObjFromOperand(resultop, tempobjs, setobj, errs) || ! setobj )
	    {
	      errs += "Failed to get the base object for set construction\n";
	      return false;
	    }
	  LBSet *realset = dynamic_cast<LBSet*>(setobj.getRealPtr());
	  if ( ! realset )
	    {
	      errs += "The base object for set construction is not a set\n";
	      return false;
	    }
	  LBObjPtr element;
	  if ( ! readObjFromOperand(op1, tempobjs, element, errs ) )
	    return false;

	  realset->insert(element);
	  lastop = resultop;
	  return true;
	}

	
      default:
	;
      }

    errs += "Corrupted struct, unrecognized instruction type in process byte code\n";
    return false;
    
  }

  bool LBProcessStruct::readObjFromOperand(const LBVMCode::Operand& op, TempObjVec *tempobjs, LBObjPtr& objptr, string& errs)
  {
    switch( op.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
	objptr=(*_runtime->_localvars)[op.opindex()].readValue();
	return true;
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st = 0;
	  int pidx = -1;
	  if ( resolveGlobalVar(op.opindex(),st, pidx, errs ) )
	    {
	      objptr = st->getStaticPropertyValue(pidx);
	      return true;
	    }
	  return false;
	}
      case LBVMCode::Operand::LOCALPROPERTY:
	{
	  int opindex = op.opindex();
	  LBPropertyInfo::ExecAttr attr = _metadata->_localpropertyops.getAttr(opindex);
	  int prpindex =  _metadata->_localpropertyops.getIndex(opindex);
	  if ( attr == LBPropertyInfo::INVALIDATTR || prpindex < 0 )
	    {
	      errs += "Corrupted struct, invalid local property operand\n";
	      return false;
	    }
	  switch (attr)
	    {
	    case LBPropertyInfo::IN:
	      try {
		objptr = _properties->_ins->readValueByInternal(prpindex);
		return true;
	      }
	      catch ( LBException & e)
		{
		  errs += "Error reading input property: "+e.msg();
		  return false;
		}
	    case LBPropertyInfo::STORE:
	      if ( _metadata->isAdhoc() )
		objptr = _runtime->_parentcomp->readStoreNStaticPropertyByInternalComp(prpindex, LBPropertyInfo::STORE);
	      else
		objptr = _properties->_stores->readValue(prpindex);
	      return true;
	    case LBPropertyInfo::STATIC:
	      if ( _metadata->isAdhoc() )
		objptr = _runtime->_parentcomp->readStoreNStaticPropertyByInternalComp(prpindex, LBPropertyInfo::STATIC);
	      else
		objptr = _metadata->_statics->readValue(prpindex);
	      return true;
	    case LBPropertyInfo::OUT:
	      errs += "Can not read its own output property from within a struct\n";
	      return false;
	    default:
	      ;
	    }
	  errs += "Corrupted struct, unknown property type\n";
	  return false;

	}
      case LBVMCode::Operand::VARPROPERTY:
	{
	  int index = op.opindex();
	  const LBSymbol *pname =_metadata->_varpropertyops.propertyName(index);
	  const LBVMCode::Operand *sop = _metadata->_varpropertyops.structOp(index);
	  if ( ! pname || ! sop )
	    {
	      errs += "Corrupted struct, invalid operand for struct porperty reading operantion\n";
	      return false;
	    }
	  LBObjPtr structobj;
	  if ( ! readObjFromOperand(*sop, tempobjs, structobj, errs) )
	    {
	      errs += "Failed to get struct object for property reading operation\n";
	      return false;
	    }
	  if ( ! structobj )
	    {
	      errs += "Null struct object for property reading operation\n";
	      return false;
	    }

	  LBStruct *st = dynamic_cast<LBStruct*>(structobj.getRealPtr());
	  if ( ! st )
	    {
	      objptr = LBObjPtr(new LBError( "Attempted property reading operation on a non-struct object"));
	      return true;
	    }
	  
	  try {
	    objptr = st->getPropertyValue(*pname);
	    return true;
	  }
	  catch( LBException& e)
	    {
	      objptr = LBObjPtr(new LBError( "Can not read property from struct: "+e.msg()));
	      return true;
	    }

	  return false;
	}
      case LBVMCode::Operand::TEMPOBJ:
	{
	  objptr = (*tempobjs)[op.opindex()];
	  return true;
	}
      case LBVMCode::Operand::LITERALTYPE:
	{
	  objptr = _metadata->_literaltypeexps[op.opindex()]->getTypeSpecAsObj();
	  return true;
	}
      case LBVMCode::Operand::LITERALOBJ:
	{
	  objptr = _metadata->_literalobjs[op.opindex()];
	  return true;
	}
      case LBVMCode::Operand::SUBSCR:
	{
	  int opindex = op.opindex();
	  const LBVMCode::Operand *objop = _metadata->_subscriptops.objOp(opindex);
	  const LBVMCode::Operand *subop = _metadata->_subscriptops.subOp(opindex);
	  if ( objop && subop )
	    {
	      LBObjPtr  sub;
	      if ( ! readObjFromOperand(*subop, tempobjs, sub, errs ) )
		{
		  errs += "Failed to obtain subscript value for subscription read operation\n";
		  return false;
		}
	      LBObjPtr obj;
	      if ( ! readObjFromOperand(*objop, tempobjs, obj, errs ) )
		{
		  errs += "Failed to obtain object value for subscription read operation\n";
		  return false;
		}
	      if ( obj && sub )
		{
		  try {
		    const LBObject *cptr = obj->subscriptConstGet(sub.getConstRealPtr());
		    objptr = LBObjPtr(cptr?cptr->clone():0);
		    return true;
		  }
		  catch (LBException& e )
		    {
		      objptr = LBObjPtr(new LBError( "Subscription operation failure: "+e.msg()) );
		    }
		  return true;
		}
	      objptr = LBObjPtr(new LBError("null object in subscripting operation"));
	      return true;
	    }
	  
	  errs += "Corrupted struct, invalid operand for subscripting operation\n";
	  return false;
	
	}

      // this case is for the ad hoc process with a containing parent composition
      case LBVMCode::Operand::JTREAD:
	{
	  int jid=-1, pid=-1;
	  if ( ! _metadata->_jtreadops.getJointInfo(op.opindex(), jid, pid) )
	    {
	      errs += "Invalid operand to access parent composition data joint\n";
	      return false;
	    }
	  if ( _runtime->_parentcomp->readDataJoint(jid,pid,objptr) )
	    {
	      if ( _runtime->_adhocDataCounter )
		_runtime->_adhocDataCounter->dec();
	      return true;
	    }
	  errs += "Failed to read from parent composition struct, halt execution\n";
	  return false;
	}

      default:
	;
      }

    errs += "Corrupted struct, unknown operand type for object value retrieving\n";
    return false;

  }

  bool LBProcessStruct::writeObjToOperand(const LBVMCode::Operand& op, TempObjVec *tempobjs, const LBObjPtr& obj, string& errs)
  {
    // below is to make sure every assign of an asynch struct is a copy instead of ref counted ptr
    // because the semantic of asynch struct is different from normal data obj
    // This is a hack to fit a live fish into a dead fish pool
    if ( isAsynchStruct(obj.getConstRealPtr()) )
	 return writeObjToOperand(op, tempobjs, *obj, errs);	 

    switch( op.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
        if ( ! (*_runtime->_localvars)[op.opindex()].writeValue(obj) )
	  {
	    errs += "Failed to write to variable"+_metadata->_localvops.varName(op.opindex())->toString()+ " because of type confliction\n";
	    errs += "Required type: "+_metadata->_localvops.typeSpec(op.opindex())->toString()+ " assign obj is of type:"+obj->getType().lubanName()+"\n";

	    return false;
	  }
	return true;
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st=0;
	  int pidx=-1;
	  if ( !resolveGlobalVar(op.opindex(), st, pidx, errs) )
	    {
	      errs += "Invalid global variable operand, corrupted struct\n";
	      return false;
	    }
	  try {
	    st->setStaticPropertyValue(pidx, obj);
	    return true;
	  }
	  catch(LBException& e)
	    {
	      errs += "Failed to write golbal static property value: "+e.msg();
	    }
	  return false;
	}
      case LBVMCode::Operand::LOCALPROPERTY:
	{
	  int opindex = op.opindex();
	  LBPropertyInfo::ExecAttr attr = _metadata->_localpropertyops.getAttr(opindex);
	  int prpindex =  _metadata->_localpropertyops.getIndex(opindex);
	  return writeValueToProperty(prpindex, attr, obj, errs);
	}
      case LBVMCode::Operand::VARPROPERTY:
	{
	  int index = op.opindex();
	  const LBSymbol *pname =_metadata->_varpropertyops.propertyName(index);
	  const LBVMCode::Operand *sop = _metadata->_varpropertyops.structOp(index);
	  if ( ! pname || ! sop )
	    {
	      errs += "Corrupted struct, invalid operand for struct porperty writing operantion\n";
	      return false;
	    }
	  LBWriteLockedObjPtr structobj;
	  if ( ! getWriteLockedObjFromOperand(*sop, tempobjs, structobj, errs) )
	    {
	      errs += "Failed to get struct object for property writing operation\n";
	      return false;
	    }
	  if ( ! structobj )
	    {
	      errs += "Null struct object for property writing operation\n";
	      return false;
	    }

	  LBStruct *st = dynamic_cast<LBStruct*>(structobj.getRealPtr());
	  if ( ! st )
	    {
	      errs += "Attempted property writing operation on a non-struct object\n";
	      return false;
	    }
	  
	  try {
	    st->setProperty(*pname, obj);
	    return true;
	  }
	  catch( LBException& e)
	    {
	      structobj.invalidate();
	      errs += "Can not set property value for struct: "+e.msg();
	    }

	  return false;
	}
      case LBVMCode::Operand::TEMPOBJ:
	{
	  // The design to have one thread to own its own temp area
	  // so following multi-thread unsafe code are ok
	  (*tempobjs)[op.opindex()] = obj;
	  return true;
	}
      case LBVMCode::Operand::SUBSCR:
	{
	  int opindex = op.opindex();
	  const LBVMCode::Operand *objop = _metadata->_subscriptops.objOp(opindex);
	  const LBVMCode::Operand *subop = _metadata->_subscriptops.subOp(opindex);
	  if ( objop && subop )
	    {
	      LBObjPtr  sub;
	      if ( ! readObjFromOperand(*subop, tempobjs, sub, errs ) )
		{
		  errs += "Failed to obtain subscript value for subscription write operation\n";
		  return false;
		}
	      LBWriteLockedObjPtr container;
	      if ( ! getWriteLockedObjFromOperand(*objop, tempobjs, container, errs ) )
		{
		  errs += "Failed to obtain object value for subscription write operation\n";
		  return false;
		}
	      if ( container )
		{
		  try {
		    container->subscriptSet(sub.getConstRealPtr(), obj.getConstRealPtr());
		    return true;
		  }
		  catch (LBException& e )
		    {
		      container.invalidate();
		      errs += "Subscription write operation failure: "+e.msg();
		    }
		  return false;
		}
	      errs += "null object in subscripting operation\n";
	      return false;
	    }
	  
	  errs += "Corrupted struct, invalid operand for subscripting operation\n";
	  return false;
	
	}

      // for ad hoc struct with parent composition
      case LBVMCode::Operand::JTWRITE:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	    return _runtime->_parentcomp->writeDataJoint(op.opindex(), obj);
	  _runtime->_adhocoutput = obj;
	  _runtime->_adhocoutputUpdated = true;
	  return true;
	}

      default:
	;
      }

    errs += "Corrupted struct, unknown operand type for object value writing\n";
    return false;

  }
    

  bool LBProcessStruct::writeObjToOperand(const LBVMCode::Operand& op, TempObjVec *tempobjs, const LBObject& obj, string& errs)
  {
    switch( op.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
        if ( ! (*_runtime->_localvars)[op.opindex()].writeValue(obj) )
	  {
	    errs += "Failed to write to variable "+_metadata->_localvops.varName(op.opindex())->toString()+ " because of type confliction\n";
	    errs += "Required type: "+_metadata->_localvops.typeSpec(op.opindex())->toString()+ " assign obj is of type: "+obj.getType().lubanName()+"\n";
	    return false;
	  }
	return true;
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st=0;
	  int pidx=-1;
	  if ( !resolveGlobalVar(op.opindex(), st, pidx, errs) )
	    {
	      errs += "Invalid global variable operand, corrupted struct\n";
	      return false;
	    }
	  try {
	    st->setStaticPropertyValue(pidx, obj);
	    return true;
	  }
	  catch(LBException& e)
	    {
	      errs += "Failed to write golbal static property value: "+e.msg();
	    }
	  return false;
	}
      case LBVMCode::Operand::LOCALPROPERTY:
	{
	  int opindex = op.opindex();
	  LBPropertyInfo::ExecAttr attr = _metadata->_localpropertyops.getAttr(opindex);
	  int prpindex =  _metadata->_localpropertyops.getIndex(opindex);
	  return writeValueToProperty(prpindex, attr, obj, errs);
	}
      case LBVMCode::Operand::VARPROPERTY:
	{
	  int index = op.opindex();
	  const LBSymbol *pname =_metadata->_varpropertyops.propertyName(index);
	  const LBVMCode::Operand *sop = _metadata->_varpropertyops.structOp(index);
	  if ( ! pname || ! sop )
	    {
	      errs += "Corrupted struct, invalid operand for struct porperty writing operantion\n";
	      return false;
	    }
	  LBWriteLockedObjPtr structobj;
	  if ( ! getWriteLockedObjFromOperand(*sop, tempobjs, structobj, errs) )
	    {
	      errs += "Failed to get struct object for property writing operation\n";
	      return false;
	    }
	  if ( ! structobj )
	    {
	      errs += "Null struct object for property writing operation\n";
	      return false;
	    }

	  LBStruct *st = dynamic_cast<LBStruct*>(structobj.getRealPtr());
	  if ( ! st )
	    {
	      errs += "Attempted property writing operation on a non-struct object\n";
	      return true;
	    }
	  
	  try {
	    st->setProperty(*pname, obj);
	    return true;
	  }
	  catch( LBException& e)
	    {
	      structobj.invalidate();
	      errs += "Can not read property from struct: "+e.msg();
	    }

	  return false;
	}
      case LBVMCode::Operand::TEMPOBJ:
	{
	  // The design to have one thread to own its own temp area
	  // so following multi-thread unsafe code are ok
	  int opindex = op.opindex();
	  (*tempobjs)[opindex].assign(obj);
	  return true;
	}
      case LBVMCode::Operand::SUBSCR:
	{
	  int opindex = op.opindex();
	  const LBVMCode::Operand *objop = _metadata->_subscriptops.objOp(opindex);
	  const LBVMCode::Operand *subop = _metadata->_subscriptops.subOp(opindex);
	  if ( objop && subop )
	    {
	      LBObjPtr  sub;
	      if ( ! readObjFromOperand(*subop, tempobjs, sub, errs ) )
		{
		  errs += "Failed to obtain subscript value for subscription write operation\n";
		  return false;
		}
	      LBWriteLockedObjPtr container;
	      if ( ! getWriteLockedObjFromOperand(*objop, tempobjs, container, errs ) )
		{
		  errs += "Failed to obtain object value for subscription write operation\n";
		  return false;
		}
	      if ( container )
		{
		  try {
		    container->subscriptSet(sub.getConstRealPtr(), &obj);
		    return true;
		  }
		  catch (LBException& e )
		    {
		      container.invalidate();
		      errs += "Subscription write operation failure: "+e.msg();
		    }
		  return false;
		}
	      errs += "null object in subscripting operation\n";
	      return false;
	    }
	  
	  errs += "Corrupted struct, invalid operand for subscripting operation\n";
	  return false;
	
	}

      // for ad hoc struct with parent composition
      case LBVMCode::Operand::JTWRITE:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	    return _runtime->_parentcomp->writeDataJoint(op.opindex(), obj);
	  _runtime->_adhocoutput.assign(obj);
	  _runtime->_adhocoutputUpdated = true;
	  return true;
	}

      default:
	;
      }

    errs += "Corrupted struct, unknown operand type for object value writing\n";
    return false;

  }

  bool LBProcessStruct::getWriteLockedObjFromOperand(const LBVMCode::Operand& op, TempObjVec *tempobjs, LBWriteLockedObjPtr& resultobj, string& errs)
  {
    switch ( op.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
	resultobj = (*_runtime->_localvars)[op.opindex()].getWriteLockedObj();
	return true;
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st=0;
	  int prpindex=-1;
	  if ( !resolveGlobalVar(op.opindex(), st, prpindex, errs) )
	    {
	      errs += "Invalid global variable operand, corrupted struct\n";
	      return false;
	    }
	  try {
	    resultobj = st->getWriteLockedStaticProperty(prpindex);
	    return true;
	  }
	  catch(LBException& e)
	    {
	      errs += "Failed to obtain write locked golbal static property value: "+e.msg();
	    }
	  return false;
	}
      case LBVMCode::Operand::LOCALPROPERTY:
	{
	  int opindex = op.opindex();
	  LBPropertyInfo::ExecAttr attr = _metadata->_localpropertyops.getAttr(opindex);
	  int prpindex =  _metadata->_localpropertyops.getIndex(opindex);
	  switch (attr)
	    {
	    case LBPropertyInfo::IN:
	      errs += "Getting write locked property value from IN property is not allowed\n";
	      return false;
	    case LBPropertyInfo::STORE:
	      if ( _metadata->isAdhoc() )
		_runtime->_parentcomp->getWriteLockedStoreNStaticPropertyByInternalComp(prpindex, LBPropertyInfo::STORE);
	      else
		resultobj = _properties->_stores->getWriteLockedObj(prpindex);
	      return true;
	    case LBPropertyInfo::STATIC:
	      if ( _metadata->isAdhoc() )
		_runtime->_parentcomp->getWriteLockedStoreNStaticPropertyByInternalComp(prpindex, LBPropertyInfo::STATIC);
	      else
		resultobj = _metadata->_statics->getWriteLockedObj(prpindex);
	      return true;
	    case LBPropertyInfo::OUT:
	      errs += "Getting write locked property value from output property is not allowed\n";
	      return false;
	    default:
	      ;
	    }
	  errs += "Corrupted struct, unknown property type\n";
	  return false;

	}
      case LBVMCode::Operand::VARPROPERTY:
	{
	  int index = op.opindex();
	  const LBSymbol *pname =_metadata->_varpropertyops.propertyName(index);
	  const LBVMCode::Operand *sop = _metadata->_varpropertyops.structOp(index);
	  if ( ! pname || ! sop )
	    {
	      errs += "Corrupted struct, invalid operand for struct property writing operantion\n";
	      return false;
	    }

	  if ( ! getWriteLockedObjFromOperand(*sop, tempobjs, resultobj, errs) )
	    {
	      errs += "Failed to get struct object for property mutating operation\n";
	      return false;
	    }
	  if ( ! resultobj )
	    {
	      errs += "Null struct object for property writing operation\n";
	      return false;
	    }

	  LBStruct *st = dynamic_cast<LBStruct*>(resultobj.getRealPtr());
	  if ( ! st )
	    {
	      resultobj.invalidate();
	      errs += "Attempted property writing operation on a non-struct object\n";
	      return false;
	    }
	  
	  try {
	    int prpindex = -1;
	    const LBPropertyInfo *prpinfo = st->interface().propertyInfo(*pname, prpindex);
	    if ( ! prpinfo )
	      {
		errs += "Property "+pname->toString()+" does not exist\n";
		return false;
	      }
	    LBPropertyInfo::ExecAttr attr = prpinfo->attr();
	    if ( attr == LBPropertyInfo::OUT )
	      {	
		errs += "Getting exclusive property pointer for mutating from output property is not allowed, property: "+pname->toString();
		return false;
	      }
	    if ( st->interface().mode() == LBStructInterface::SYNCH )
	      {
		LBObject *currentobj = st->getPropertyPtr(prpindex, prpinfo->attr());
		resultobj.pushObj(currentobj);
		resultobj.addTouch(st, prpindex, prpinfo->attr());
	      }
	    else
	      {
		if ( attr == LBPropertyInfo::STORE )
		  resultobj.pushLock( st->getWriteLockedStoreProperty(prpindex) );
		else
		  {
		    resultobj.invalidate();
		    errs += "Can not modify input properties for asynch struct";
		    return false;
		  }
	      }
	    return true;
	  }
	  catch( LBException& e)
	    {
	      errs += "Failed to get exclusive property pointer from struct: "+e.msg();
	    }

	  return false;
	}
      case LBVMCode::Operand::TEMPOBJ:
	{
	  // The design to have one thread to own its own temp area
	  // so following code are thread safe
	  int opindex = op.opindex();
	  if ( !(*tempobjs)[opindex] )
	    return LBWriteLockedObjPtr();
	  if ( ! isAsynchStruct((*tempobjs)[opindex].getConstRealPtr() ) )
	    uniclone((*tempobjs)[opindex]);
	  resultobj = LBWriteLockedObjPtr(0/* null write lock */,(*tempobjs)[opindex].getRealPtr()); 
	  return true;
	}
      case LBVMCode::Operand::SUBSCR:
	{
	  int opindex = op.opindex();
	  const LBVMCode::Operand *objop = _metadata->_subscriptops.objOp(opindex);
	  const LBVMCode::Operand *subop = _metadata->_subscriptops.subOp(opindex);
	  if ( objop && subop )
	    {
	      LBObjPtr  sub;
	      if ( ! readObjFromOperand(*subop, tempobjs, sub, errs ) )
		{
		  errs += "Failed to obtain subscript value for subscription write operation\n";
		  return false;
		}
	      if ( ! getWriteLockedObjFromOperand(*objop, tempobjs, resultobj, errs ) )
		{
		  errs += "Failed to obtain exclusive object pointer for subscription mutating operation\n";
		  return false;
		}
	      if ( resultobj )
		{
		  try {
		    LBObject *currentobj = resultobj->subscriptGet(sub.getConstRealPtr());
		    resultobj.pushObj(currentobj);
		    return true;
		  }
		  catch (LBException& e )
		    {
		      resultobj.invalidate();
		      errs += "Subscription mutating operation failure: "+e.msg();
		    }
		  return false;
		}
	      errs += "Can not get subscription from null object\n";
	      return false;
	    }
	  
	  errs += "Corrupted struct, invalid operand for subscripting operation\n";
	  return false;
	
	}
      default:
	;
      }

    errs += "Corrupted struct, unknown operand type for mut-exclusive object pointer retrieving\n";
    return false;
  
  }

  void LBProcessStruct::throwError(const string& emsg, bool throwit)
  {
    _mainstop = true;
    if ( _metadata->_multithread && _runtime->_livethreads )
      cancelInternalThreads();

    bool synch = _metadata->_itfc->mode() == LBStructInterface::SYNCH;
    LBError errvalue(emsg);
    if ( _properties && _properties->_outs )
      {
	if ( synch )
	  {
	    for(int i = 0; i < _properties->_outs->size(); i++)
	      _properties->_outs->writeValueByInternal(i,errvalue);
	    if ( _runtime && _runtime->_parentcomp )
	      _properties->_outs->flush();
	  }
	else
	  _properties->_outs->dismissWaitingThreads();
      }
    if ( _metadata->isAdhoc() ) 
      {
	_runtime->_parentcomp->writeDataJoint(_metadata->_jtwriteindex, errvalue);
	return;
      }
    _runtime->_status = LBProcessStructRunTime::EVALERROR;
    _runtime->_errmsg = emsg;
    _runtime->_lastobj = LBObjPtr((LBObject*)(new LBError(emsg)));
    if ( throwit) throw LBException(emsg);
  }

  bool LBProcessStruct::addPendingUpdate(const LBVMCode::Operand& varop, string& errs)
  {
    switch ( varop.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
	{
	  return (*_runtime->_localvars)[varop.opindex()].addPending();
	}
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st=0;
	  int pidx=-1;
	  if ( resolveGlobalVar(varop.opindex(), st, pidx, errs) )
	    return st->addPending(pidx);

	  errs += "Invalid global variable operand, corrupted struct\n";
	  return false;
	}
      default:
	;
      }
    return true;
  }
    

  bool LBProcessStruct::removePendingUpdate(const LBVMCode::Operand& varop, string& errs)
  {
    switch ( varop.optype() )
      {
      case LBVMCode::Operand::LOCALVAR:
	{
	  return (*_runtime->_localvars)[varop.opindex()].removePending();
	}
      case LBVMCode::Operand::GLOBALVAR:
	{
	  LBStruct *st=0;
	  int pidx=-1;
	  if ( resolveGlobalVar(varop.opindex(), st, pidx, errs) )
	    return st->removePending(pidx);

	  errs += "Invalid global variable operand, corrupted struct\n";
	  return false;
	}
      default:
	;
      }
    return true;
  }
    

  void LBProcessStruct::initLocalVarsTypeAndThreading()
  {
    if ( _runtime->_localvars )
      {
	for( int i=0; i < _runtime->_localvars->size(); i++)
	  {
	    if ( _metadata->_localvops.threading(i) ) 
	      (*_runtime->_localvars)[i].initThreading();
	    const LBTypeSpec *tpsp = _metadata->_localvops.typeSpec(i);
	    if ( tpsp )
	      (*_runtime->_localvars)[i].setType(tpsp);
	  }
      }
  }

  bool LBProcessStruct::resolveGlobalVar(int varindex, LBStruct* &resultstruct, int& prpindex, string &errs)
  {
    int sidx = _metadata->_globalvops.getGlobalStructIndex(varindex);
    if ( sidx < 0 )
      {
	errs += "Invalid global variable operand, corrupted struct\n";
	return false;
      }
    LBStruct *st = _metadata->_extsymbols.getOriginalStructPointer(sidx);
    if ( ! st )
      {
	errs += "Failed to get global struct instance for global variable\n";
	return false;
      }
    int pidx = _metadata->_globalvops.getPropertyIndex(varindex);
    if ( pidx < 0 ) // need to resolve the property index first
      {
	const LBSymbol *pname = _metadata->_globalvops.getPropertyName(varindex);
	if ( ! pname )
	  {
	    errs += "Invalid global variable operand, null property name, corrupted struct\n";
	    return false;
	  }
	const LBPropertyInfo* prpinfo = st->interface().propertyInfo(*pname, pidx);
	if ( ! prpinfo || prpinfo->attr() != LBPropertyInfo::STATIC )
	  {
	    errs += "Invalid global variable operand, no such static property: "+pname->toString()+"for struct "+st->interface().name()->toString();
	    return false;
	  }

	_metadata->_globalvops.setPropertyIndex(varindex, pidx);

      }

    resultstruct = st;
    prpindex = pidx;

    return true;

  }
  
  bool LBProcessStruct::reIndexAllGlobalVars(string &errs)
  {
    LBStruct *dummyst;
    int dummyindex;
    bool result = true;
    for( int i = 0; i < _metadata->_globalvops.size(); i++)
      result = result && resolveGlobalVar(i, dummyst, dummyindex, errs);
    return result;
  }
 
 bool LBProcessStruct::reIndexAllPropertyOps(string &errs)
  {
    bool result = true;
    for( int i = 0; i < _metadata->_localpropertyops.size(); i++)
      {
	const LBSymbol *pnm = _metadata->_localpropertyops.getName(i);
	if ( ! pnm )
	  {
	    errs += "Invalid property operand with null property name\n";
	    result = false;
	    continue;
	  }
	int prpindex=-1;
	const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(*pnm, prpindex);
	if ( ! pinfo )
	  {
	    errs += "Invalid property refered in process truct body, property:"+pnm->toString()+" does not exist in interface\n";
	    result = false;
	    continue;
	  }
	if ( pinfo->attr() != _metadata->_localpropertyops.getAttr(i) )
	  {
	    errs += "Invalid property reference in process truct body, property:"+pnm->toString()+" has different attribute definde in interface\n";
	    result = false;
	    continue;
	  }

	_metadata->_localpropertyops.setIndex(i, prpindex);

      }

    return result;
  }

  bool LBProcessStruct::writeValueToProperty(int pindex, LBPropertyInfo::ExecAttr attr, const LBObjPtr& objptr, string& errs)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pindex, attr);
    if ( ! pinfo )
      {
	errs += "Invalid property index for property setting";
	return false;
      }
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), objptr.getConstRealPtr() ) )
      {
	LBError errval("Can not write object to property because of type violation, object type: "+objptr->getType().toString()+" vs required type: "+pinfo->typeSpec()->toString());
	return writeValueToProperty( pindex, attr, errval, errs);
      }

    switch (attr)
      {
      case LBPropertyInfo::IN:
	errs += "Writing to IN property is not allowed\n";
	return false;
      case LBPropertyInfo::STORE:
	if ( _metadata->isAdhoc() )
	  _runtime->_parentcomp->writeStoreNStaticPropertyByInternalComp(pindex, LBPropertyInfo::STORE, objptr);
	else
	  _properties->_stores->writeValue(pindex,objptr);
	return true;
      case LBPropertyInfo::STATIC:
	if ( _metadata->isAdhoc() )
	  _runtime->_parentcomp->writeStoreNStaticPropertyByInternalComp(pindex, LBPropertyInfo::STATIC, objptr);
	else
	  _metadata->_statics->writeValue(pindex,objptr);
	return true;
      case LBPropertyInfo::OUT:
	if ( _properties->_outlocks && (*_properties->_outlocks)[pindex] )
	  {
	    LBMutexLocker mlocker(*(*_properties->_outlocks)[pindex]);
	    return _properties->_outs->writeValueByInternal(pindex,objptr);
	  }

	return _properties->_outs->writeValueByInternal(pindex,objptr);

      default:
	;
      }
    errs += "Corrupted struct, unknown property type\n";
    return false;
  }

  bool LBProcessStruct::writeValueToProperty(int pindex, LBPropertyInfo::ExecAttr attr, const LBObject& obj, string& errs)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pindex, attr);
    if ( ! pinfo )
      {
	errs += "Invalid property index for property setting";
	return false;
      }
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &obj ) )
      return writeValueToProperty( pindex, attr, LBError("Can not write object to property because of type violation, object type: "+obj.getType().toString()+" vs required type: "+pinfo->typeSpec()->toString()), errs);

    switch (attr)
      {
      case LBPropertyInfo::IN:
	errs += "Writing to IN property is not allowed\n";
	return false;
      case LBPropertyInfo::STORE:
	if ( _metadata->isAdhoc() )
	  _runtime->_parentcomp->writeStoreNStaticPropertyByInternalComp(pindex, LBPropertyInfo::STORE, obj);
	else
	  _properties->_stores->writeValue(pindex,obj);
	return true;
      case LBPropertyInfo::STATIC:
	if ( _metadata->isAdhoc() )
	  _runtime->_parentcomp->writeStoreNStaticPropertyByInternalComp(pindex, LBPropertyInfo::STATIC, obj);
	else
	  _metadata->_statics->writeValue(pindex,obj);
	return true;
      case LBPropertyInfo::OUT:
	if ( _properties->_outlocks && (*_properties->_outlocks)[pindex] )
	  {
	    LBMutexLocker mlocker(*(*_properties->_outlocks)[pindex]);
	    return _properties->_outs->writeValueByInternal(pindex,obj);
	  }
	return _properties->_outs->writeValueByInternal(pindex,obj);

      default:
	;
      }
    errs += "Corrupted struct, unknown property type\n";
    return false;
  }

  void LBProcessStruct::cancelInternalThreads()
  {
    if ( _runtime && _runtime->_livethreads )
      {
	// up the flag the threads watches for each vm instruction 
	_runtime->_livethreads->setTerminationFlagAll(true);
      }

    // shake up the threads waiting the inputs
    if ( _properties &&  _properties->_ins )
      _properties->_ins->dismissWaitingThreads();
    if ( _runtime && _runtime->_adhocDataCounter )
      _runtime->_adhocDataCounter->dismissWaitingThreads();
  }

  bool LBProcessStruct::initializeProperties()
  {
    if ( _properties ) return true;

    _properties = new LBProcessStructProperties();
    int numins = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
    int numouts = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
    int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
    bool asynch = ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH );

    if ( numins )
      if ( asynch )
	_properties->_ins = new LBStructInPadAsynch(numins);
      else
	{
	  _properties->_ins = new LBStructInPadLocal(numins);
	  // init if necessary
	  for( int i = 0; i<numins; i++)
	    {
	      const LBProcessStruct *initp = _metadata->_itfc->propertyInfo(i, LBPropertyInfo::IN)->initProcess();
	      if ( ! initp )
		continue;
	      std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	      _properties->_ins->writeValueByExternal(i,torun->evalForLastObj());
	    }
	}
		
    if ( numouts && !asynch )
      for( int i = 0; i< numouts; i++)
	{
	  if ( _metadata->_localpropertyops.getThreading( _metadata->_itfc->propertyInfo(i, LBPropertyInfo::OUT)->name() ) )
	    {
	      if ( ! _properties->_outlocks )
		{
		  _properties->_outlocks = new std::vector<LBMutex*>(numouts,(LBMutex*)0);
		  for( int j = 0; i < numouts; i++)
		    (*_properties->_outlocks)[j] = 0;
		}
	      (*_properties->_outlocks)[i] = new LBMutex();
	    }
	}
	    
    if ( numstores )
      {
	_properties->_stores = new LocalPropertyStorage(numstores);
	for( int i = 0; i< numstores; i++)
	  {
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STORE);
	    if ( _metadata->_localpropertyops.getThreading( pinfo->name() ) || asynch && pinfo->permission() != LBPropertyInfo::HIDDEN )
	      _properties->_stores->initThreading(i);
	    const LBProcessStruct *initp = pinfo->initProcess();
	    if ( ! initp )
	      continue;
	    std::auto_ptr<LBProcessStruct> torun(new LBProcessStruct(*initp));
	    _properties->_stores->writeValue(i, torun->evalForLastObj());
	  }
      }

    return true;
      
  }

  void LBProcessStruct::prepareRunTime()
  {
    bool asynch = _metadata->_itfc->mode() == LBStructInterface::ASYNCH;
    if ( ! _runtime )
      {
	int numvars =  _metadata->_localvops.size();
	_runtime = new LBProcessStructRunTime(numvars, _metadata->_multithread );
	if ( _metadata->_compmom && _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _runtime->_adhocDataCounter = new CounterWaiter(false);
	if ( numvars != 0 )
	  initLocalVarsTypeAndThreading();
      }
    else
      _runtime->reset();
  }


  // this only works after all external symbols has been resolved
  bool LBProcessStruct::resolveAllNamedArgs(string& errs)
  {
    bool ok = true;
    for( int i=0; i < _metadata->_namedargs.size(); i++)
      {
	LBStructEvalArgs *onesetargs = _metadata->_namedargs[i];

	// for struct type in ports
	if ( onesetargs->_structtypes )
	  for( int j = 0; j < onesetargs->_structtypes->size(); j++)
	    {
	      LBStructEvalArgs::BatchNamedArg& onest = (*onesetargs->_structtypes)[j];
	      if ( onest._structtypeop.optype() != LBVMCode::Operand::GLOBALTYPE )
		{
		  errs += "Corrupted struct, wrong operand type in named argument list\n";
		  ok = false;
		}
	      int sidx = onest._structtypeop.opindex();
	      const LBTypeSpec *tspc = _metadata->_extsymbols.getTypeSpec(sidx);
	      const LBFullSymbol *snm = _metadata->_extsymbols.symbolName(sidx);
	      const LBStructInterface *sitfc = 0;
	      if ( tspc )
		{
		  const LBTypeSpecStruct* tspc_st = dynamic_cast<const LBTypeSpecStruct*>(tspc);
		  if ( tspc_st )
		    sitfc = tspc_st->getStructInterface();
		}
	      if ( ! sitfc )
		{
		  if ( snm )
		    errs += "Expected external struct type is not a struct "+snm->toString();
		  else
		    errs += "Corrupted struct, can not find expected external struct type does not exist ";
		  ok = false;
		}
	      int numins = sitfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
	      for( int k = 0; k < numins; k++)
		{
		  const LBSymbol& nm = sitfc->propertyInfo(k,LBPropertyInfo::IN)->name();
		  int prpidxthis = -1;
		  const LBPropertyInfo *inonthis = interface().propertyInfo(nm, prpidxthis);
		  if ( ! inonthis || inonthis->attr() != LBPropertyInfo::IN )
		    {
		      errs += "Can not find IN property "+nm.toString()+" refered by IN property setting with struct type "+snm->toString();
		      ok = false;
		    }
		  onesetargs->addNamedArgFromStructType(nm, j, prpidxthis, onest._vec);
		}
	    }

	// for allThisINs
	if ( onesetargs->_getthisins )
	  {
	    int numthisins = interface().numberOfPropertiesByAttr(LBPropertyInfo::IN);
	    for( int k = 0; k < numthisins; k++)
	      {
		const LBSymbol& nm = interface().propertyInfo(k,LBPropertyInfo::IN)->name();
		onesetargs->addNamedArgFromThisINs(nm, k, onesetargs->_vecthisins);
	      }
	  }
	    
      }

    return ok;

  }


  LBProcessStruct::LBProcessStructRunTime::LBProcessStructRunTime(int numvars, bool multithread)
    : _status(UNEVALED), _localvars(0), _livethreads(0), _lastobj(), _parentcomp(0), _adhocoutput(),_adhocoutputUpdated(false), _adhocDataCounter(0)
  {
    if ( numvars )
      _localvars = new LocalVarObjVec(numvars);

    if ( multithread )
      _livethreads = new LiveThreadPool();

  }

  LBProcessStruct::LBProcessStructRunTime::~LBProcessStructRunTime()
  {
    if ( _livethreads ) delete _livethreads;
    if ( _localvars ) delete _localvars;
    if ( _adhocDataCounter ) delete _adhocDataCounter;
  }

  void LBProcessStruct::LBProcessStructRunTime::reset()
  {
    if ( _localvars )
      for( int i=0; i < _localvars->size(); i++)
	(*_localvars)[i].reset();
    if ( _livethreads )
      _livethreads->reset();

    return;
  }

  LBProcessStruct::LBProcessStructProperties::~LBProcessStructProperties()
  {
    if ( _ins ) delete _ins;
    if ( _outs ) delete _outs;
    if ( _outlocks )
      for( int i=0; i<_outlocks->size(); i++)
	if ( (*_outlocks)[i] ) delete (*_outlocks)[i];
    if ( _stores ) delete _stores;
  }

  LBProcessStruct::LBProcessStructProperties::LBProcessStructProperties( const LBProcessStructProperties& p )
    : _ins(0), _outs(0), _outlocks(0), _stores(0)
  {
    if ( p._ins ) _ins = p._ins->clone();
    if ( p._outs ) _outs =p._outs->clone();
    if ( p._outlocks )
      {
	_outlocks = new LockVec(p._outlocks->size());
	for( int i=0; i<p._outlocks->size(); i++)
	  if ( (*_outlocks)[i] ) 
	    (*_outlocks)[i] = new LBMutex();
	  else
	    (*_outlocks)[i] = 0;
      }

    if ( p._stores ) _stores = new LocalPropertyStorage(*p._stores);

  }
      
  LBProcessStruct::LiveThread::~LiveThread()
  { 
    if ( _runningthread ) 
      delete _runningthread; 
  }

  bool LBProcessStruct::LiveThread::start()
  {
    if ( ! _runningthread )
      _runningthread = new LBThread(&_dispatch);

    return _runningthread && _runningthread->isValid(); 
  }
  
  bool LBProcessStruct::LiveThread::isValid() const
  {
    return _runningthread && _runningthread->isValid();
  }

  void LBProcessStruct::LiveThread::setPoolIndex(int poolindex)
  {
    _dispatch.setPoolIndex(poolindex);
  }

  void LBProcessStruct::LiveThread::setTerminateFlag(bool f)
  {
    _dispatch.setTerminateFlag(f);
  }
  
  
  bool LBProcessStruct::LiveThread::join()
  {
    if ( _runningthread && _runningthread->isValid() )
      return _runningthread->join();
    return false;
  }
  
  void LBProcessStruct::Dispatch::run()
  {
    // we are here, the thread already started
    _status = RUNNING;

    TempObjVec     localtemps(_pstruct._metadata->_numtemps);
    TempIterObjVec localtempiters(_pstruct._metadata->_numtempiters);
    
    string errmsg;
    LBVMCode::Operand lastop;
    for ( int ic = _startic;  ic <= _endic && ic >= _startic;  )
      {
	try {
	  if ( ! _pstruct.runOneCode(ic, &localtemps, &localtempiters, lastop, errmsg) )
	    {
	      _status = RUNERR;
	      _pstruct.removePendingUpdate(_target, errmsg);
	      _pstruct._runtime->_livethreads->decRunning();
	      _pstruct.throwError(errmsg, false);
	      return;
	    }
	}
	catch( LBException& excp )
	  {
	    _pstruct.throwError(excp.msg(), false);
	    return;
	  }
	catch( ... )
	  {
	    _pstruct.throwError("Unknown error in asynch struct main thread", false);
	    return;
	  }
	  

	if ( _terminate  )
	  {
	    _status = TERMINATED;
	    _pstruct.removePendingUpdate(_target, errmsg);
	    _pstruct._runtime->_livethreads->decRunning();
	    return;
	  }
      }

    _pstruct.removePendingUpdate(_target, errmsg);
    _pstruct._runtime->_livethreads->decRunning();
    _status = FINISHED;

    return;

  }

  int LBProcessStruct::LiveThreadPool::add(LiveThread *livethread)
  {
    LBMutexLocker locker( _addMutex );
    int sz = _allthreads.size();
    livethread->setPoolIndex(sz);
    _allthreads.push_back(livethread);
    return sz;
  }

  void LBProcessStruct::LiveThreadPool::reset()
  {
    // this function assume single thread
    joinAll();
    _numrunning = 0;

  }

  bool LBProcessStruct::LiveThreadPool::incRunning()
  {
    LBMutexLocker locker( _runningMutex );
    _numrunning++;
    return true;
  }

  bool LBProcessStruct::LiveThreadPool::decRunning()
  {
    LBMutexLocker locker( _runningMutex );
    if ( --_numrunning == 0 )
      _zeroRunningCond.broadcast();
    return true;
  }

  bool LBProcessStruct::LiveThreadPool::waitAll()
  {
    _runningMutex.lock();
    while ( _numrunning != 0 )
      _zeroRunningCond.wait( _runningMutex );
    _runningMutex.unlock();
    return true;
  }

  bool LBProcessStruct::LiveThreadPool::joinAll()
  {
    for(int i=0; i < _allthreads.size(); i++)
      {
	if ( _allthreads[i]->isValid() )
	  _allthreads[i]->join();
	delete _allthreads[i];
      }
    _allthreads.clear();
    return true;
  }

  void LBProcessStruct::LiveThreadPool::setTerminationFlagAll(bool f)
  {
    for(int i=0; i < _allthreads.size(); i++)
      _allthreads[i]->setTerminateFlag(f);
    return ;
  }
    
  LBProcessStruct::LiveThreadPool::LiveThreadPool()
    : _allthreads(), _numrunning(0), _addMutex(), _runningMutex(), _zeroRunningCond()
  {
  }

  LBProcessStruct::LiveThreadPool::~LiveThreadPool()
  {
    // int i=0, sz=_allthreads.size();
//     for( ; i < sz; i++)
//       _allthreads[i]->cancel();  // this action cancel the thread if it is still running

    // to debug !!!!!!!!!!!!
    // the first join would work, but the second ( i=1) will hang
    // the case is all the threads are waiting on the same cond var and mutex
    //    for( int i=0; i < _allthreads.size(); i++)
    //  _allthreads[i]->join();  // wait for them to finish
    for( int i=0; i < _allthreads.size(); i++)
      delete _allthreads[i]; 
  }

  static int symbolHash(const LBSymbol& s) 
  { return s.hash(); }

  static bool symbolEqual(const LBSymbol& s1, const LBSymbol& s2) 
  { return s1 == s2; }

  LBProcessStruct::LocalVarOpInfoTable::LocalVarOpInfoTable()
    : _varvec(), _varmap(symbolHash, symbolEqual,5) // expecting the size of 32
  {
  }

  bool LBProcessStruct::LocalVarOpInfoTable::addVar(const LBSymbol& vname, TypeExpInfo* tinfo, LBVMCode::Operand &resultop, bool threading )
  {
    LocalVarInfo *vinfo = 0;
    if ( _varmap.find(vname, vinfo ) )
      return false;

    int index = _varvec.size();
    vinfo = new LocalVarInfo(vname, tinfo, threading, index);
    _varvec.push_back( vinfo );
    _varmap[vname]=vinfo;
    
    resultop = LBVMCode::Operand(LBVMCode::Operand::LOCALVAR, index);

    return true;

  }

  bool LBProcessStruct::LocalVarOpInfoTable::getVar(const LBSymbol& vname, LBVMCode::Operand &resultop, bool threading )
  {
    LocalVarInfo *vinfo = 0;
    if ( _varmap.find(vname, vinfo ) )
      {
	if ( threading ) vinfo->_threading = threading;
	resultop = LBVMCode::Operand(LBVMCode::Operand::LOCALVAR, vinfo->_index);
	return true;
      }

    int index = _varvec.size();
    vinfo = new LocalVarInfo(vname, 0, threading, index);
    _varvec.push_back( vinfo );
    _varmap[vname] = vinfo;

    resultop = LBVMCode::Operand(LBVMCode::Operand::LOCALVAR, index);
    
    return true;

  }

  int LBProcessStruct::LocalVarOpInfoTable::size() const
  {
    return _varvec.size();
  }

  const LBSymbol* LBProcessStruct::LocalVarOpInfoTable::varName(int index) const
  {
    if ( index < 0 || index >= size() )
      return 0;
    return &_varvec[index]->_name;
  }

  const LBTypeSpec *LBProcessStruct::LocalVarOpInfoTable::typeSpec(int index) const
  {
    if ( index < 0 || index >= size() )
      return 0;
    if (_varvec[index]->_typeexpinfo )
      return _varvec[index]->_typeexpinfo->typeSpec();
    return 0;
  }

  bool LBProcessStruct::LocalVarOpInfoTable::threading(int index) const
  {
    if ( index < 0 || index >= size() )
      return false;
    return _varvec[index]->_threading;
  }

  LBProcessStruct::LocalVarOpInfoTable::LocalVarOpInfoTable(const LocalVarOpInfoTable& ximp)
    : _varvec(ximp._varvec), _varmap()
  {
    for( int i = 0; i < _varvec.size(); i++)
      {
	LocalVarInfo *info = new LocalVarInfo( *_varvec[i] );
	_varvec[i] = info;
	_varmap[info->_name] = info;
      }
  }

  LBProcessStruct::LocalVarOpInfoTable::~LocalVarOpInfoTable()
  {
    for( int i = 0; i < _varvec.size(); i++)
      delete _varvec[i];
  }

  class LBProcessStruct::LocalVarObj::LockStruct
  {
  public:
    LockStruct() : _rwlock(), _pendings(0) {}
    ~LockStruct() { if ( _pendings ) delete _pendings;}
    // access control
    LBReadWriteLock _rwlock;
    CounterWaiter *_pendings;

  };

  LBProcessStruct::LocalVarObj::~LocalVarObj() 
  { if ( _locks ) delete _locks; }

  void LBProcessStruct::LocalVarObj::initThreading()
  {
    _locks = new LockStruct();
  }

  void LBProcessStruct::LocalVarObj::setType(const LBTypeSpec *tp)
  {
    _typespec = tp;
  }
  
  bool LBProcessStruct::LocalVarObj::addPending()
  {
    if ( ! _locks )
      return false;
    if (  _locks->_pendings )
      return _locks->_pendings->inc();

    LBWriteLocker wlock( _locks->_rwlock );
    if ( ! _locks->_pendings )
      _locks->_pendings = new CounterWaiter();
      
    return  _locks->_pendings->inc();

  }

  bool LBProcessStruct::LocalVarObj::removePending()
  {
    if ( ! _locks )
      return false;
    if ( ! _locks->_pendings )
      return false;

    return _locks->_pendings->dec();
  }

  bool LBProcessStruct::LocalVarObj::waitUntilNoPending()
  {
    if ( ! _locks )
      return false;
    if ( ! _locks->_pendings )
      return false;

    return _locks->_pendings->waitDownToZero();

  }
    
  LBObjPtr LBProcessStruct::LocalVarObj::readValue() const
  {
    if ( _locks )
      {
	LBReadLocker rlocker( _locks->_rwlock );
	return _obj;
      }
    return _obj;
  }
    
  bool LBProcessStruct::LocalVarObj::writeValue(const LBObjPtr& objptr)
  {
    if ( ! LubanTypeChecking::checkType( _typespec, objptr.getConstRealPtr() ) )
      return false;

    if ( _locks )
      {
	LBWriteLocker wlocker( _locks->_rwlock );
       _obj = objptr;
       return true;
      }
    _obj = objptr;
    return true;
  }

  bool LBProcessStruct::LocalVarObj::writeValue(const LBObject& obj)
  {
    if ( ! LubanTypeChecking::checkType( _typespec, &obj ) )
      return false;


    if ( _locks )
      {
	LBWriteLocker wlocker( _locks->_rwlock );
	_obj.assign(obj);
	return true;
	
      }

    _obj.assign(obj);
    return true;
  }

  LBWriteLockedObjPtr LBProcessStruct::LocalVarObj::getWriteLockedObj()
  {
    if ( ! _obj )
      return LBWriteLockedObjPtr();
    if ( _locks )
      {
	LBWriteLockedObjPtr lockedptr( &_locks->_rwlock, 0);
	// special treatment for asynch struct
	if ( ! isAsynchStruct(_obj.getConstRealPtr()) )
	  uniclone(_obj);
	lockedptr.pushObj(_obj.getRealPtr());
	return lockedptr;
      }
    if ( ! isAsynchStruct(_obj.getConstRealPtr()) )
      uniclone(_obj);
    return LBWriteLockedObjPtr(0, _obj.getRealPtr() );
  }


  void LBProcessStruct::LocalVarObj::reset()
  {
    if ( _locks && _locks->_pendings)
	_locks->_pendings->reset();

    static const LBObjPtr nullobj;
    _obj = nullobj;
  }

  static int fullsymbolHash(const LBFullSymbol& s) { return s.hash(); }
  static bool fullsymbolEqual(const LBFullSymbol& s1, const LBFullSymbol& s2) { return s1 == s2; }

  LBProcessStruct::ExternalTypeSymbolTable::ExternalTypeSymbolTable()
    : _symbolmap(fullsymbolHash, fullsymbolEqual, 5), _symbolvec()
  {
  }

  LBProcessStruct::ExternalTypeSymbolTable::~ExternalTypeSymbolTable()
  {
    for( int i = 0; i < _symbolvec.size(); i++)
      delete _symbolvec[i];
  }


  int LBProcessStruct::ExternalTypeSymbolTable::getStructIndex(const LBFullSymbol& name)
  {
    TypeSymbolInfo *tpinfo=0;
    if ( _symbolmap.find(name, tpinfo) )
      {
	tpinfo->_exp = ASTRUCT;
	return tpinfo->_index;
      }

    int index = _symbolvec.size();
    _symbolvec.push_back(new TypeSymbolInfo(name, ASTRUCT, index));
    _symbolmap[name]=_symbolvec[index];
    return index;
  }
    
  bool LBProcessStruct::ExternalTypeSymbolTable::getStructOp(const LBFullSymbol& structname, LBVMCode::Operand &resultop)
  {
    int idx = getStructIndex(structname);
    resultop = LBVMCode::Operand(LBVMCode::Operand::GLOBALTYPE, idx);
    return true;
  }

  bool LBProcessStruct::ExternalTypeSymbolTable::getConstructableOp(const LBFullSymbol& sname, LBVMCode::Operand &resultop)
  {
    TypeSymbolInfo *tpinfo=0;
    int index = -1;
    if ( _symbolmap.find(sname, tpinfo) )
      index = tpinfo->_index;
    else
      {
	index = _symbolvec.size();
	_symbolvec.push_back(new TypeSymbolInfo(sname, ACONSTRUCTABLE, index));
	_symbolmap[sname]=_symbolvec[index];
      }

    resultop = LBVMCode::Operand(LBVMCode::Operand::GLOBALTYPE, index);

    return true;

  }

  int LBProcessStruct::ExternalTypeSymbolTable::size() const
  { return _symbolvec.size(); }

  const LBFullSymbol* LBProcessStruct::ExternalTypeSymbolTable::symbolName(int idx) const
  {
    if ( idx < 0 || idx >= _symbolvec.size() )
      return 0;
    return &_symbolvec[idx]->_symbolname;
  }

  bool LBProcessStruct::ExternalTypeSymbolTable::resolveAllSymbols(string& errs)
  {
    bool allok = true;
    for( int i = 0; i < _symbolvec.size(); i++)
      {
	TypeSymbolInfo *tpinfo = _symbolvec[i];
	// call symbol resolver
  	tpinfo->_structobj = LubanSymbolResolver::resolveStructSymbol(tpinfo->_symbolname, errs);
  	tpinfo->_typespecobj = LubanSymbolResolver::resolveTypeSymbol(tpinfo->_symbolname, errs);
	
	if ( ! tpinfo->_structobj && ! tpinfo->_typespecobj )
	  {
	    errs += "Can not resolve external symbol "+ tpinfo->_symbolname.toString()+"\n";
	    allok =  false;
	    continue;
	  }

	if (  tpinfo->_structobj )
	  continue;
	if ( tpinfo->_exp == ASTRUCT )
	  {
	    errs += "External symbol "+tpinfo->_symbolname.toString()+" is not a struct as it is expected to be\n";
	    allok= false;
	    continue;
	  }
	const LBTypeSpecBasic *tbsc = dynamic_cast<const LBTypeSpecBasic*>(tpinfo->_typespecobj);
	if ( ! tbsc )
	  {
	    errs += "Global symbol "+tpinfo->_symbolname.toString()+" is not a constructable type as it is expected to be\n";
	    allok = false;
	    continue;
	  }
      }

    return allok;

  }

  LBStruct* LBProcessStruct::ExternalTypeSymbolTable::getInitializedStructCopy(int index) const
  {
    if ( index < 0 || index >= _symbolvec.size() )
      return 0;
    TypeSymbolInfo *tpinfo =  _symbolvec[index];
    if ( tpinfo->_structobj )
      {
	LBStruct *result = static_cast<LBStruct*>(tpinfo->_structobj->clone());
	result->initializeProperties();
	result->prepareOutputForProcessUse();
	return result;
      }
    return 0;
  }

  LBStruct* LBProcessStruct::ExternalTypeSymbolTable::getOriginalStructPointer(int index) const
  {
    if ( index < 0 || index >= _symbolvec.size() )
      return 0;
    return  _symbolvec[index]->_structobj;
  }

  const LBTypeSpec* LBProcessStruct::ExternalTypeSymbolTable::getTypeSpec(int index) const
  {
    if ( index < 0 || index >= _symbolvec.size() )
      return 0;
    return _symbolvec[index]->_typespecobj;
  }

  bool LBProcessStruct::GlobalVarOpInfoTable::getGlobalVar(int sindex, const LBSymbol& prpname, LBVMCode::Operand& resultop)
  {
    _globalvaropvec.push_back(GlobalVarOpInfo(sindex, prpname));
    resultop = LBVMCode::Operand(LBVMCode::Operand::GLOBALVAR, _globalvaropvec.size()-1);
    return true;
  }

  int LBProcessStruct::GlobalVarOpInfoTable::getGlobalStructIndex(int opindex)
  {
    if ( opindex < 0 || opindex >= _globalvaropvec.size() )
      return -1;
    return _globalvaropvec[opindex]._structindex;
  }

  const LBSymbol* LBProcessStruct::GlobalVarOpInfoTable::getPropertyName(int opindex)
  {
    if ( opindex < 0 || opindex >= _globalvaropvec.size() )
      return 0;
    return &_globalvaropvec[opindex]._prpname;
  }

  int LBProcessStruct::GlobalVarOpInfoTable::getPropertyIndex(int opindex)
  {
    if ( opindex < 0 || opindex >= _globalvaropvec.size() )
      return 0;
    return _globalvaropvec[opindex]._prpindex;    
  }
  
  bool LBProcessStruct::GlobalVarOpInfoTable::setPropertyIndex(int opindex, int prpindex)
  {
    if ( opindex < 0 || opindex >= _globalvaropvec.size() )
      return false;
    _globalvaropvec[opindex]._prpindex = prpindex; 
    return true;
  }

  LBProcessStruct::LocalPropertyOpInfoTable::LocalPropertyOpInfoTable()
    : _propertymap(symbolHash, symbolEqual, 5), _propertyvec()
  {
  }

  bool LBProcessStruct::LocalPropertyOpInfoTable::getPropertyOp(const LBSymbol& prpname, LBPropertyInfo::ExecAttr attr, int prpindex, LBVMCode::Operand &resultop, bool threading, string& errs)
  {
    int opidx = -1;
    if ( _propertymap.find(prpname, opidx) )
      {
	if ( _propertyvec[opidx]._propertyattr != attr )
	  {
	    errs += "Conflicting property attribute definition for property: "+prpname.toString();
	    return false;
	  }
	if ( threading ) _propertyvec[opidx]._threading = true;
	resultop = LBVMCode::Operand(LBVMCode::Operand::LOCALPROPERTY, opidx);
	return true;
      }

    int sz = _propertyvec.size();
    _propertyvec.push_back(PropertyOpInfo(prpname, attr, threading));
    _propertymap[prpname] = sz;
    resultop = LBVMCode::Operand(LBVMCode::Operand::LOCALPROPERTY, sz);

    return true;

  }

  LBPropertyInfo::ExecAttr LBProcessStruct::LocalPropertyOpInfoTable::getAttr(int opindex) const
  {
    if ( opindex < 0 || opindex >= _propertyvec.size() )
      return LBPropertyInfo::INVALIDATTR;
    return _propertyvec[opindex]._propertyattr;
  }

  int LBProcessStruct::LocalPropertyOpInfoTable::getIndex(int opindex) const
  {
    if ( opindex < 0 || opindex >= _propertyvec.size() )
      return LBPropertyInfo::INVALIDATTR;
    return _propertyvec[opindex]._propertyindex;
  }

  const LBSymbol* LBProcessStruct::LocalPropertyOpInfoTable::getName(int opindex) const
  {
    if ( opindex < 0 || opindex >= _propertyvec.size() )
      return 0;
    return &_propertyvec[opindex]._propertyname;
  }

  bool LBProcessStruct::LocalPropertyOpInfoTable::getThreading(const LBSymbol& prpname) const
  {
    int opidx = -1;
    if ( _propertymap.find(prpname, opidx) )
      return _propertyvec[opidx]._threading ;
    return false;
  }

  int LBProcessStruct::LocalPropertyOpInfoTable::size() const
  {
    return _propertyvec.size();
  }

  bool LBProcessStruct::LocalPropertyOpInfoTable::setIndex(int opindex, int prpindex)
  {
    if ( opindex < 0 || opindex >= _propertyvec.size() )
      return false;
    _propertyvec[opindex]._propertyindex = prpindex;
    return true;
  }



  LBVMCode::Operand LBProcessStruct::VarPropertyOpInfoTable::getPropertyOp(const LBVMCode::Operand& varop, const LBSymbol& prpname)
  {
    _prpinfovec.push_back(VarPropertyInfo(varop, prpname));
    return LBVMCode::Operand(LBVMCode::Operand::VARPROPERTY, _prpinfovec.size()-1);
  }

  const LBVMCode::Operand* LBProcessStruct::VarPropertyOpInfoTable::structOp(int opindex) const
  {
    if ( opindex < 0 || opindex >= _prpinfovec.size() )
      return 0;
    return &_prpinfovec[opindex]._op;
  }

  const LBSymbol* LBProcessStruct::VarPropertyOpInfoTable::propertyName(int opindex) const
  {
    if ( opindex < 0 || opindex >= _prpinfovec.size() )
      return 0;
    return &_prpinfovec[opindex]._prpname;
  }


  LBVMCode::Operand LBProcessStruct::MemberFuncOpInfoTable::getMemberFuncOp(const LBVMCode::Operand& varop, const LBSymbol& prpname)
  {
    _funcinfovec.push_back(MemberFuncInfo(varop, prpname));
    return LBVMCode::Operand(LBVMCode::Operand::MBRFUNC, _funcinfovec.size()-1);
  }

  const LBVMCode::Operand* LBProcessStruct::MemberFuncOpInfoTable::objOperand(int opindex) const
  {
    if ( opindex < 0 || opindex >= _funcinfovec.size() )
      return 0;
    return &_funcinfovec[opindex]._op;
  }

  const LBSymbol* LBProcessStruct::MemberFuncOpInfoTable::funcName(int opindex) const
  {
    if ( opindex < 0 || opindex >= _funcinfovec.size() )
      return 0;
    return &_funcinfovec[opindex]._funcname;
  }


  LBVMCode::Operand LBProcessStruct::SubscriptOpInfoTable::getSubscriptOp(const LBVMCode::Operand& varop, const LBVMCode::Operand& subexpop)
  {
    _subopinfovec.push_back(SubOpInfo(varop, subexpop));
    return LBVMCode::Operand(LBVMCode::Operand::SUBSCR, _subopinfovec.size()-1);
  }

  const LBVMCode::Operand* LBProcessStruct::SubscriptOpInfoTable::objOp(int opindex)
  {
    if ( opindex < 0 || opindex >= _subopinfovec.size() )
      return 0;
    return &_subopinfovec[opindex]._varop;
  }

  const LBVMCode::Operand* LBProcessStruct::SubscriptOpInfoTable::subOp(int opindex)
  {
    if ( opindex < 0 || opindex >= _subopinfovec.size() )
      return 0;
    return &_subopinfovec[opindex]._subexpop;
  }

  const LBTypeSpec* LBProcessStruct::TypeExpInfo::typeSpec()
  {
    if ( _typespec ) return static_cast<const LBTypeSpec*>(_typespec.getConstRealPtr());
    if ( _typetree )
      {
	string errs;
	_typespec = LBObjPtr(static_cast<LBObject*>(LubanCodeGenerator::parseTypeExpression(_typetree, _homens, errs )));
      }
    return static_cast<const LBTypeSpec*>(_typespec.getConstRealPtr());
  }

  LBProcessStruct::LBStructEvalArgs::LBStructEvalArgs()
    : _namedargs(0), _namedargmap(0), _vecedargs(0), _structtypes(0), _getthisins(false), _vecthisins(false), _par(false)
  {}

  LBProcessStruct::LBStructEvalArgs::~LBStructEvalArgs()
  {
    if ( _namedargs ) delete _namedargs;
    if ( _namedargmap ) delete _namedargmap;
    if ( _vecedargs ) delete _vecedargs;
    if ( _structtypes ) delete _structtypes;
  }

  bool LBProcessStruct::LBStructEvalArgs::addNamedArg(const LBSymbol& name, const LBVMCode::Operand& op, bool vecit, bool parallel)
  {
    if ( ! _namedargs )
      {
	_namedargs = new NamedArgVec();
	_namedargmap = new NamedArgMap(symbolHash, symbolEqual, 5);
      }

    OneNamedArg *arg ;
    if ( _namedargmap->find(name, arg) )
      return false; // same named arg

    int idx = _namedargs->size();
    _namedargs->push_back(OneNamedArg(name, op, vecit));
    (*_namedargmap)[name] = &(*_namedargs)[idx];
    
    if ( vecit )
      getVecedArgs()->push_back(idx);
    if ( parallel )
      _par = true;

    return true;

  }

  void LBProcessStruct::LBStructEvalArgs::addStructINs(const LBVMCode::Operand& structop, bool vecthem, bool parallel)
  {
    if ( ! _structtypes )
      _structtypes = new StructTypeOpVec();
    
    _structtypes->push_back(BatchNamedArg(structop, vecthem));
    if ( parallel )
      _par = true;
  }

  void LBProcessStruct::LBStructEvalArgs::addThisINs(bool vecthem, bool parallel)  
  {
    _getthisins = true;
    _vecthisins = vecthem;
    if ( parallel )
      _par =true;
  }

  int LBProcessStruct::LBStructEvalArgs::numVecNames() const
  {
    if ( ! _vecedargs )
      return 0;
    return _vecedargs->size();
  }

  const LBSymbol* LBProcessStruct::LBStructEvalArgs::vecName(int idx) const
  {
    if ( ! _vecedargs || ! _namedargs )
      return 0;
    return &(*_namedargs)[(*_vecedargs)[idx]]._name;
  }

  bool LBProcessStruct::LBStructEvalArgs::parallel() const
  {
    return _par;
  }

  void LBProcessStruct::LBStructEvalArgs::addNamedArgFromStructType(const LBSymbol& nm, int structidx, int inprpidx, bool vec)
  {
    if ( ! _namedargs )
      {
	_namedargs = new NamedArgVec();
	_namedargmap = new NamedArgMap(symbolHash, symbolEqual, 5);
      }

    OneNamedArg *arg ;
    if ( _namedargmap->find(nm, arg) )
      return; // dont overwrite same named arg

    int idx = _namedargs->size();
    _namedargs->push_back(OneNamedArg(nm, structidx, inprpidx, vec));
    (*_namedargmap)[nm] = &(*_namedargs)[idx];
    
    if ( vec )
      getVecedArgs()->push_back(idx);
    return;
  }


  void LBProcessStruct::LBStructEvalArgs::addNamedArgFromThisINs(const LBSymbol& nm, int inprpidx, bool vec)
  {
    if ( ! _namedargs )
      {
	_namedargs = new NamedArgVec();
	_namedargmap = new NamedArgMap(symbolHash, symbolEqual, 5);
      }

    OneNamedArg *arg ;
    if ( _namedargmap->find(nm, arg) )
      return; // dont overwrite same named arg

    int idx = _namedargs->size();
    _namedargs->push_back(OneNamedArg(nm, inprpidx, vec));
    (*_namedargmap)[nm] = &(*_namedargs)[idx];
    
    if ( vec )
      getVecedArgs()->push_back(idx);
    return;
  }

  bool LBProcessStruct::AsynchMainThread::start()
  {
    LBMutexLocker mlock(_initmtx);

    if ( ! _runningthread )
      {
	_runningthread = new LBThread(&_dsp);
	if ( _runningthread && _runningthread->isValid() )
	  {
	    _dsp._st = Dispatch::RUNNING;
	    if ( _dsp._pst->_runtime && _dsp._pst->_runtime->_parentcomp ) 
	      _dsp._pst->_runtime->_parentcomp->informComponentAct(true);
	    return true;
	  }
	_dsp._st = Dispatch::UNRUNABLE;
	_dsp._errmsg = "Can not start thread";
	return false;
      }
    return _runningthread->isValid();
  }

  bool LBProcessStruct::AsynchMainThread::isdead() const
  {
    return _dsp._st == Dispatch::ENDED || _dsp._st == Dispatch::UNRUNABLE ;
  }

  const string& LBProcessStruct::AsynchMainThread::errmsg() const
  {
    return _dsp._errmsg;
  }
  
  bool LBProcessStruct::AsynchMainThread::join()
  {
    if ( _runningthread )
      {
	if ( _runningthread->isValid() )
	  _runningthread->join();
	delete _runningthread;
	_runningthread = 0;
	return true;
      }
    return false;
  }

  LBProcessStruct::AsynchMainThread::~AsynchMainThread()
  {
    if ( _runningthread )
      delete _runningthread;
  }

  void LBProcessStruct::AsynchMainThread::Dispatch::run()
  { 
    bool ok=false;
    string errs;
    try { 
      ok = _pst->runAllCodeAsynch(errs);
    }
    catch ( LBException& lbe)
      {
	_errmsg = lbe.msg();
	_st = UNRUNABLE;
      }
    catch (...)
      {
	_st = UNRUNABLE;
	_errmsg="Unknown exception in struct evaluation";
      }
    if ( !ok )
      _errmsg=errs;
    _st = ENDED;
  }

  //The below to export member functions for Luban's reflection mechanism
  // actually functions defined at lbstruct level
  LBEXPORT_MEMBER_FUNC(Luban::LBProcessStruct, luban_set, "pset", "void set(map kvpairss),set(string key, object value),set(vector keys, vector values)" ); 
  LBEXPORT_MEMBER_FUNC(Luban::LBProcessStruct, luban_get, "pget", "object get(string key),vector get(vector keys)" ); 
  // to handle asynch struct internal thread
  LBEXPORT_MEMBER_FUNC(Luban::LBProcessStruct, luban_start, "start", "null start() start internal thread of an asynch struct");

}




