#include <string>
#include <sstream>
#include <iostream>
#include <cstdio>
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbdouble.hpp"
#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/StreamUtil.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbvarargs.hpp"


namespace Luban
{
  using std::string;
  using std::ostringstream;
  using std::istringstream;

  LBDEFINE(Luban::LBDouble, 1, 0)

  //static
  LBDouble* LBDouble::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() == 0 )
      return new LBDouble();
    
    switch ( args->numArgs() ) {
    case 1:
      {
	const LBObject* arg0 = args->getArg(0);
	if ( arg0 == 0 )
	  return 0;
	// copy constructor
	const LBDouble *dptr = dynamic_cast<const LBDouble*>(arg0);
	if ( dptr )
	  return new LBDouble(*dptr);

	// double can be constructed from one int or string
	const LBInt *intptr = dynamic_cast<const LBInt*>(arg0);
	if ( intptr != 0 )
	  return new LBDouble(double(*intptr));
	// the follow is the LBString case
	const LBString *strptr = dynamic_cast<const LBString*>(arg0);
	if ( strptr )
	  {
	    istringstream ist(strptr->str());
	    double x;
	    if ( ist>>x )
	      return new LBDouble(x);
	  }
	break;
      }
    default:
      ;
    }
    
    return 0;
  }


  string LBDouble::toString() const
  {   
    ostringstream ost;
    ost << _realdouble;
    return ost.str();
  }

  // the following two functions are machine depdent code
  // need to move them into a seperate module and mark as machine dependent
  ostream& LBDouble::toStream(ostream& ost) const
  {
    //call the function to output machine independent stream
    return StreamUtil::serializeDouble(ost, _realdouble);
  }

  istream& LBDouble::fromStream(istream& ist, int major, int minor)
  {
    // call function to retrive double from stream of machine independent format
    _realdouble = StreamUtil::de_serializeDouble(ist);
    return ist;
  }

  bool LBDouble::equals(const LBObject& another) const
  {
    const LBDouble *dblptr = dynamic_cast<const LBDouble*>(&another);
    if ( dblptr )
      return _realdouble == dblptr->_realdouble;
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&another);
    if ( lboptr2 )
      return _realdouble == (double) lboptr2->_realint;
    return false;
  }

  int LBDouble::hash() const
  {   return (int)_realdouble; }

  bool LBDouble::cast(LBObject* target) const
  {
    LBDouble *dptr = dynamic_cast<LBDouble*>(target);
    if ( dptr != 0 )
      {
	dptr->_realdouble = _realdouble;
	return true;
      }
    LBInt *intptr = dynamic_cast<LBInt*>(target);
    if ( intptr != 0 )
      {
	intptr->_realint = int(_realdouble);
	return true;
      }

    return false;
  }

  LBDouble& LBDouble::add(const LBObject& lbo)
  {
    const LBDouble* lboptr = dynamic_cast<const LBDouble*>(&lbo);
    if ( lboptr )
      {
	_realdouble += lboptr->_realdouble;
	return *this;
      }
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&lbo);
    if ( lboptr2 )
      {
	_realdouble += lboptr2->_realint;
	return *this;
      }
    throw LBException(string("Can not add LBDouble with ")+lbo.getType().toString());
  }

  LBDouble& LBDouble::sub(const LBObject& lbo)
  {
    const LBDouble* lboptr = dynamic_cast<const LBDouble*>(&lbo);
    if ( lboptr )
      {
	_realdouble -= lboptr->_realdouble;
	return *this;
      }
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&lbo);
    if ( lboptr2 )
      {
	_realdouble -= lboptr2->_realint;
	return *this;
      }

    throw LBException(string("Can not subtract LBDouble with ")+lbo.getType().toString());
  }

  LBDouble& LBDouble::mul(const LBObject& lbo)
  {
    const LBDouble* lboptr = dynamic_cast<const LBDouble*>(&lbo);
    if ( lboptr )
      {
	_realdouble *= lboptr->_realdouble;
	return *this;
      }
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&lbo);
    if ( lboptr2 )
      {
	_realdouble *= lboptr2->_realint;
	return *this;
      }

    throw LBException(string("Can not multiply LBDouble with ")+lbo.getType().toString());
  }

  LBDouble& LBDouble::div(const LBObject& lbo)
  {
    const LBDouble* lboptr = dynamic_cast<const LBDouble*>(&lbo);
    if ( lboptr )
      {
	_realdouble /= lboptr->_realdouble;
	return *this;
      }
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&lbo);
    if ( lboptr2 )
      {
	_realdouble /= lboptr2->_realint;
	return *this;
      }


    throw LBException(string("Can not divide LBDouble with ")+lbo.getType().toString());
  }

  bool LBDouble::before(const LBObject& lbo) const
  {
    const LBDouble* lboptr = dynamic_cast<const LBDouble*>(&lbo);
    if ( lboptr )
	return _realdouble < lboptr->_realdouble;
    const LBInt* lboptr2 = dynamic_cast<const LBInt*>(&lbo);
    if ( lboptr2 )
      return _realdouble < lboptr2->_realint;

    throw LBException(string("Can not mod LBDouble with ")+lbo.getType().toString());
  }

  LBDouble& LBDouble::neg()
  {
    _realdouble = -_realdouble;
    return *this;
  }
  
  LBDouble& LBDouble::plusplus()
  {
    _realdouble++;
    return *this;
  }
  LBDouble& LBDouble::minusminus()
  {
    _realdouble--;
    return *this;
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBDouble, luban_format, "format", "string format(string fmtstring) format double into a string according the specified format. format is the same as used in C printf")
  LBObject* LBDouble::luban_format(const LBVarArgs *args)
  {
    LBString fmt;
    if ( !args || ! args->assignArgToLocal(0,fmt))
      throw LBException("Expecting one string type argument in double::format function");
    string fmtstr("%");
    fmtstr += fmt.str()+'f';
    char buf[256];
    snprintf(buf,256,fmtstr.c_str(), _realdouble);
    return new LBString(buf);
  }

}
