#ifndef __LBCOMPOSITIONSTRUCT_HPP__
#define __LBCOMPOSITIONSTRUCT_HPP__

#include <vector>
#include <deque>
#include <string>

#include "lbtypes/RefCountedPtr.hpp"
#include "lbtypes/LBDeclareMacros.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbfullsymbol.hpp"

#include "luban/lbstruct.hpp"

namespace Luban
{
  class LBNamedArgs;
  class LocalPropertyStorage;
  class LBStructInPadLocal;
  class LBStructOutPad;
  class CounterWaiter;

  using std::string;

  class LBCompositionStruct : public LBStruct
  {
    LBDECLARE(Luban::LBCompositionStruct);
  public:
    friend class LBStructOutPadAsynchComp;
    friend class LBStructOutPadSynchComp;

    ~LBCompositionStruct();
    LBCompositionStruct();
    LBCompositionStruct(const LBCompositionStruct& abs);
    LBCompositionStruct(LBStructInterface* itfc);

    LBCompositionStruct& operator=(const LBCompositionStruct& c);

    // LBObject required interfaces
    string toString() const;
    ostream& toStream( ostream& o ) const ;
    istream& fromStream( istream& i, int major=-1, int minor=-1 );
    bool equals(const LBObject& another) const;


    // common operations for all kind of structs
    const LBStructInterface& interface() const ;
    LBCompositionStruct* structCall(const LBNamedArgs* args=0) const ;
    void setBatchProperties(const LBNamedArgs* args=0) ;

    LBObjPtr getPropertyValue(const LBSymbol& s) ;
    LBObject* getPropertyPtr(const LBSymbol& s) ;
    void touchProperty(const LBSymbol& s) ;
    void setProperty(const LBSymbol& s, const LBObject& val) ; //exception if set failed
    void setProperty(const LBSymbol& s, const LBObjPtr& valptr) ; //exception if set failed

    LBObjPtr getPropertyValue(int index, LBPropertyInfo::ExecAttr attr)  ;
    LBObject* getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr) ;
    void touchProperty(int index, LBPropertyInfo::ExecAttr attr) ;
    void setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val) ; //exception if set failed
    void setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr) ; //exception if set failed

    bool addPending(int staticprpindex) ; // this only applies to static property
    bool removePending(int staticprpindex) ; // this only applies to static property
    bool waitUntilNoPending(int staticprpindex) ; // this only applies to static property
    LBObjPtr getStaticPropertyValue(int index) ;
    void setStaticPropertyValue(int index, const LBObject& val) ;
    void setStaticPropertyValue(int index, const LBObjPtr& valptr) ;


    void waitUntilInputEmptyOrMainFinish(); // for asynch struct, caller hangs until the input q get to zero
    LBObject* luban_wait(const LBVarArgs *args); // to export above function


    LBWriteLockedObjPtr getWriteLockedStaticProperty(int staticprpindex) ; // this only applies to static property
    LBWriteLockedObjPtr getWriteLockedStoreProperty(int index) ; // this is for asynch struct

    // helper interface for compositio
    void setInput(int index, const LBObject& val);
    void setInput(int index, const LBObjPtr& valptr);
    void sync();
    bool linkOutputProperty(const LBSymbol& outprpname, int jointid); 
    bool prepareOutputForCompositionUse();
    bool prepareOutputForProcessUse();
    bool setRunTimeParentComposition(LBCompositionStruct *parentcomp);
    bool wakeupAsynch();
    bool initializeProperties();

    class LBCompositionStructMetaData;
    LBCompositionStructMetaData* metadata() { return _metadata.getRealPtr(); }

    // functions to call before the composition can be evaluated
    bool prepareToRun(string& errs); // this is the head function, which calls ALL the prepare functions in metadata and runtime

    void prepareRunTime();

    bool asynchCompMainLoop(string& errs);

    void informComponentAct(bool wakeup); // called asynch cells

    // called by internal adhoc cell
    bool readDataJoint(int jid, int pid, LBObjPtr& objval ); // called by ad hoc process
    bool writeDataJoint(int jid, const LBObjPtr& valptr); // called by ad hoc process
    bool writeDataJoint(int jid, const LBObject& val); // called by ad hoc process
    LBObjPtr readStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr);
    void writeStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr);
    void writeStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr, const LBObject& val);
    LBWriteLockedObjPtr getWriteLockedStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr);


    // property access help functions they have the locking mechanism and policy inside
  private:
    LBObjPtr readPropertyValue(int index, LBPropertyInfo::ExecAttr attr);
    LBObject* readPropertyUnsharedGuts(int index, LBPropertyInfo::ExecAttr attr);

  private:
    void cancelInternalThreads();
    void throwError(const string& errmsg, bool throwup=true); // error handling when can not continue
 
  public:
    class ComponentInfo;

    class ComponentList
    {
    public:
      friend class LBCompositionStruct;

      enum {INPUTS=0, OUTPUTS=1}; // speciall cell id designated to input and outputs
      ComponentList(bool asynch);
      ~ComponentList();

      int index(const LBSymbol& cname) const;
      const LBSymbol& name(int cid) const;
      const LBStructInterface *getTypedInterface(int cid) const;
      bool isTyped(int cid) const;
      bool isAsynch(int cid) const;
      LBStruct *getLiveComp(int cid) const;
      int topOrder(int cid) const;

      int addNamedTyped(const LBSymbol& cname, const LBFullSymbol& structtypename, bool threading, string& errs); // return -1 if faield to add
      int addNamedAdhoc(const LBSymbol& cname, bool asynch, bool threading, string& errs); // return -1 if faield to add
      int addNonameAdhoc(bool threading);
      int addNonameTyped(const LBFullSymbol& structtypename, bool threading);
      bool setAdhocProc(int compid, LBProcessStruct *proc);

      // for individual port connection
      bool recordDownStream(int cellid, int downstreamcellid); // down to adhoc cell
      bool recordDownStream(int cellid, const LBSymbol& outputname, int jointid, int downstreamcellid); // down to typed or adhoc cell or output property
      bool recordUpStream(int cellid, int upstreamcellid); // up to adhoc cel
      bool recordUpStream(int cellid, const LBSymbol& inputprpname, int jointid, int portid, int upstreamcellid); // up to typed cell or input property

      // batch input connection
      bool recordUpStream(int cellid, const LBFullSymbol& structtypename); // link up to a batch input properties indicated by the structtypename parameter 
      bool recordDownStream( const LBFullSymbol& structtypename, int cellid); // link down from a batch input properties indicated by the structtypename parameter 

      // batch output connection
      bool recordUpStream(const LBFullSymbol& structtypename, int cellid ); // link up from a batch output properties indicated by the structtypename parameter 
      bool recordDownStream( int cellid, const LBFullSymbol& structtypename); // link down to a batch of output properties indicated by the structtypename parameter

      // topologically sort all components
      bool topsort(string& errs);
      int toplevel() const  { return _toporder; } // max number of top order
      int size() const;

      ComponentInfo *getComponentInfo(int cid) { return _complist[cid]; }

    private:
      ComponentList(const ComponentList& c);
      ComponentList& operator=(const ComponentList& c);

    private: // data
      LBHashMap<LBSymbol, int> _namedcellmap;
      std::vector<ComponentInfo*> _complist;

      std::vector<int> *_asynchcells; // all asynch type cells
      std::vector<int> *_zeroincells; // all synch cell with zero input link 
      int _toporder;
    };

    class ComponentInfo
    {
    public:
      friend class ComponentList;
      friend class LBCompositionStructMetaData;

      ComponentInfo(const LBSymbol& cname, const LBFullSymbol& structtypename, bool threading);
      ComponentInfo(const LBSymbol& cname, bool asynch, bool threading);
      ComponentInfo(int cellid, bool threading);
      ComponentInfo(const LBFullSymbol& structtypename, bool threading);  // batch output 

      ~ComponentInfo();

      const LBSymbol& name() const;
      const LBStructInterface* typedInterface() const;
      bool setAdhocProc(LBProcessStruct *proc);

      // dependency
      bool downStream(int downcellid);
      bool downStream(const LBSymbol& outprpname,int jointid, int downcellid);
      bool upStream(int upcellid);
      bool upStream(const LBSymbol& inprpname, int jointid, int portid, int upcellid); // upcell must a typed cell

      // batch dependency
      // below either called from an output cell or the downstream is an input cell
      bool upStream(const LBFullSymbol& structtypename, int upcellid); // properties implied by the struct type name
      // below either called from an input cell or the downstream is an output cell
      bool downStream(const LBFullSymbol& structtypename, int downcellid); // properties implied by the struct type name

      int indegree() { return _ups.size(); }
      bool isAsynch() { return _asynch; }
      bool isAnonymous() { return _anonymous; }

    private:
      ComponentInfo(const ComponentInfo& c);
      ComponentInfo& operator=(const ComponentInfo& c);
      struct TypedUpLink
      {
	TypedUpLink()
	  :_jointid(-1), _portid(-1)
	{}
	TypedUpLink(int j, int p)
	  : _jointid(j), _portid(p)
	{}
	int _jointid;
	int _portid;
      };
      struct BatchLink
      {
	BatchLink()
	  : _structtypename(), _cellid(-1)
	{}
	BatchLink(const LBFullSymbol& sname, int cellid)
	  : _structtypename(sname), _cellid(cellid)
	{}
	LBFullSymbol _structtypename;
	int _cellid;
      };
    private:
      LBSymbol _name;
      bool _typed;
      bool _anonymous;
      LBFullSymbol _structtypename;
      bool _asynch;
      bool _threading;
      int _toporder;
	
      std::vector<int> _ups;
      std::vector<int> _downs;

      typedef LBHashMap<LBSymbol, int> DownPortMap; // negative number means added from batch instead of explicit ref
      DownPortMap *_downmap;
      typedef LBHashMap<LBSymbol, TypedUpLink> UpPortMap;
      UpPortMap *_upmap;
      std::vector<BatchLink> *_batchups;
      std::vector<BatchLink> *_batchdowns;

      LBStruct *_str;
      const LBStructInterface *_typeditfc;
      bool _erred;
      string _errmsg;

    };


    class DataJointKey
    {
    public:
      DataJointKey(int cellid)
	: _cellid(cellid), _prpname(), _prpindex(-1)
      {}
      DataJointKey(int cellid, const LBSymbol& outputname, int prpidx=-1)
	: _cellid(cellid), _prpname(outputname), _prpindex(prpidx)
      {}
      int hash() const { _cellid + _prpname.hash(); }
      bool operator==(const DataJointKey& k) const
      {   return _cellid == k._cellid && _prpname == k._prpname; }
      int _cellid;
      LBSymbol _prpname;
      int _prpindex;
    };

    typedef DataJointKey PortKey;
   
    class DataJoint
    {
    public:
      DataJoint(const DataJointKey& djk);
	
      int getPort(int downcellid);
      int getPort(int downcellid, const LBSymbol& inprpname, int prpidx);
    private:
      DataJoint();
      DataJoint(const DataJoint& d);
      DataJoint& operator=(const DataJoint& d);
    public:
      std::vector<PortKey> _portvec;
      LBHashMap<PortKey, int> _portmap;
      DataJointKey _djtkey;
    };

    class DataJointList
    {
    public:
      DataJointList();
      ~DataJointList();

      int getJointIndex(int cellid); // adhoc
      int getJointIndex(int cellid, const LBSymbol& outputname); // typed or input property

      int getPortIndex(int jointindex, int downcellid);
      int getPortIndex(int jointindex, int downcellid, const LBSymbol& inprpname, int inprpidx);

      int size() const;
      const DataJoint& getJoint(int jid) const;

      std::vector<DataJoint*> _jointvec;
      LBHashMap<DataJointKey, int> _jointmap;

    };

    class ExternalStructSymbolTable
    {
    public:
      ExternalStructSymbolTable();
      void addSymbol(const LBFullSymbol& structtypename, int componentid);
      int size();
      const LBFullSymbol& symbolName(int i);
      int cellId(int i);
    private:
      std::vector<LBFullSymbol> _syms;
      std::vector<int> _cids;
    };

    class InputPropertyToDataJointMap
    {
    public:
      InputPropertyToDataJointMap(int numins);

      int getInputJointid(int inprpindex); // negative number for unlinked
      void setMap(int inprpindex, int jointid);
      int numUnlinked();

      int size();
    private:
      std::vector<int> _inprp2jid;
      int _unlinked;
    };

    class LiveJoint
    {
    public:
      LiveJoint(const DataJoint& djt, int jointid, LBCompositionStruct *parentcomp);
      LiveJoint(const LiveJoint& ljt, LBCompositionStruct *parentcomp);
      ~LiveJoint();

      // below two functions also add to be run cell to the queue
      bool setNewValue(const LBObjPtr& valptr);
      bool setNewValue(const LBObject& val);

      // this is called by the main sync thread
      void asynchFlush(); // pop the top of the asynch que to the sync down ports

      bool readValue(int portid, LBObjPtr& val); // for adhoc process, synch or asynch

      void dismissThreads(); // put a stop sign on this joint, and dismiss all waiting threads, called by composition destructor

    private:
      struct LivePort
      {
	LivePort(int downcell, int prpidx, bool asynch);
	LivePort(const LivePort& p);
	~LivePort();
	bool isAdhoc() const { return _downprpid == -1; }

	int _downcell;
	int _downprpid;
	bool _asynch;

	std::deque<LBObjPtr> *_adhocasynchq;
	LBMutex *_adhocasynchqmtx;
	LBCondVar *_adhocnewdata;

	LBObjPtr _adhocsynchobj;
      };

    private:
      int _jointid;
      bool _asynch;
      LBCompositionStruct *_parentcomp;
      
      std::deque<LBObjPtr> *_asynchq;
      LBMutex *_asynchqmtx;
      
      std::vector<LivePort*> _liveports;

      bool _stop;

    };

    class LBCompositionStructMetaData
    {
    public:
      friend class LBCompositionStruct;
      friend class LiveJoint;

      LBCompositionStructMetaData(LBStructInterface *itfc);
      ~LBCompositionStructMetaData();

      // code generator uses these functions
      int newNamedTypedCell(const LBSymbol& cellname, const LBFullSymbol& structtypename, bool threading, string& errs); // return -1 for failure
      int newNamedAdhocCell(const LBSymbol& cellname, bool asynch, bool threading, string& errs);
      bool setNamedAdhocCell(const LBSymbol& cellname, LBProcessStruct *proc);
      int  newNonameAdhocCell(bool threading); // obtain a place keeper id for an anonymous struct
      bool setNonameAdhocCell(int cellid, LBProcessStruct *proc);
      int newNonameTypedCell(const LBFullSymbol& structtypename, bool threading, string& errs);

      const LBSymbol& getCellName(int cellid) const; //convenience function
      int getCellId(const LBSymbol& name) const;  // return -1 if cell does not exist
      const LBStructInterface *getTypedCellInterface(int cellid) const;

      int getOutputDataJointIdForAdhocCell(int cellid); // for adhoc cell that has single output

      bool linkBatchInputs(const LBFullSymbol& structtypename, int cellid, string& errs); // link structtypename implied bunch of inputs from interface to cellid
      bool linkAllInputs(int cellid, string& errs); // link all inputs from interface to cellid
      bool linkAllOutputs(int cellid, string& errs); // link all outputs from cellid to interface

      bool link(int upcellid, int downcellid, int& jointid, int& jointportid, string& errs); // adhoc-adhoc
      bool link(int upcellid, const LBSymbol& upcellprpname, int downcellid, int& jointid, int& jointportid, string& errs); // typed/input-adhoc
      bool link(int upcellid, const LBSymbol& upoutprpname, int downcellid,const LBSymbol& downinprpname, int downprpidx, string& errs); // typed/input-typed
      bool link(int upcellid, int downcellid,const LBSymbol& downinprpname, int downprpidx, string& errs); // adhoc-typed/output



      bool vectorizeCell(int cellid, const std::vector<LBSymbol>& prpnames, const std::vector<LBFullSymbol>& structnames, bool par)
      { /* to be done */  return true; }

      // functions to call before the composition can be evaluated
      bool prepareMetaData(string& errs); // head function for all below
      bool resolveExternalSymbols(string& errs, bool& sortagain); //resolve all refered external struct types symbols
      bool resolveBatchLinks(string& errs); // resolve all batch links to/from output/input
      bool resolvePropertyReferences(string& errs); // match all input/output property references against interface

    private:
      LBStructInterface *_itfc;
      ComponentList    _components;
      DataJointList _joints;
      ExternalStructSymbolTable _extsyms;
      InputPropertyToDataJointMap _inpmap;

      LocalPropertyStorage *_statics;
      CounterWaiter *_staticscounters;

      LBMutex _preparemtx;
      
      enum METASTATUS { UNPREPARED, PREPARED, ERRED };
      METASTATUS _status;
      string _errmsg;
    };

    class CompositionEvalQueue
    {
    public:
      CompositionEvalQueue( int toplevel, int numofcomps );

      void addCell(int cellid, int toporder);
      bool nextToEval(int& cellid);
      void resetCursor();

      int toplevel() const;

    private:
      typedef std::deque<int> CompQ;
      typedef std::vector<CompQ> CompQVec;
      CompQVec _cqvec;
      std::vector<bool> _compsinq;
      int _cursor;
    };

    class AsynchThreadsNDataProxy
    {
    public:
      AsynchThreadsNDataProxy(int alljoints);
      AsynchThreadsNDataProxy(const AsynchThreadsNDataProxy& p);

      bool addAsynchJoint(int jointid);

      bool hasNewData(); // any joint has new data
      bool hasNewData(int idx); // idx joint has new data
      void informDataConsumed(int idx); // decrease data count of joint idx 
      void informNewData(int jointid); // inform new data
      int waitForNewData(); // just wait on data count
      int waitForNewDataIfAnyActive(); // return 0 immediately if no active asynch thread, and no new data

      void waitUntilNoData(); // block, only returns when data count become zero and main thread is waiting

      int liveJointIndex(int idx) const; // map the idx to the actual LiveJoint index
      int asynchJoints() const;

      int numActive(); // number of active asynch components
      int incActive();
      int decActive();

      void dismissAll(); // call by destructor to wake up all threads and tell them to go home

    private:
      std::vector<int> _jointid2idx;
      std::vector<int> _idx2jointid;
      std::vector<int> _datacounts;
      int _totalcount;
      bool _mainwaiting;
      LBMutex _mtx;
      LBCondVar _newdata;
      LBCondVar _nodata;
      int _actives;
      bool _allgohome;
    };

    class AsynchMainThread
    {
    public:
      AsynchMainThread(LBCompositionStruct *s): _dsp(s), _runningthread(0), _initmtx() {}
      ~AsynchMainThread();

      bool start();
      bool join();
      bool isdead() const; // check if the thread is dead
      const string& errmsg() const;

      class Dispatch : public LBRunable
      {
      public:
	friend class AsynchMainThread;
	enum Status { TOBERUN, RUNNING, UNRUNABLE, ENDED};
	Dispatch( LBCompositionStruct *compst ): _compst(compst), _st(TOBERUN), _errmsg() {}

	void run();

      private:
	LBCompositionStruct *_compst;
	Status _st;
	string _errmsg;
      };

    private:
      Dispatch _dsp;
      LBThread *_runningthread;
      LBMutex _initmtx;

    };

    class LBCompositionStructRunTime
    {
    public:
      LBCompositionStructRunTime(int inprps, int storeprps, int joints, int comps, int toplevel, bool asynch);
      LBCompositionStructRunTime(const LBCompositionStructRunTime& rt, LBCompositionStruct *parentcomp);
      ~LBCompositionStructRunTime();
      
      void signalNewData(int jointid); // used by LiveJoint wake up sync thread if it is sleeping

      std::vector<LiveJoint*> _livejoints;
      std::vector<LBStruct*> _livecomps;
      CompositionEvalQueue _evalque;
      bool _zeroinsEvaled; // flag to indicate if the components that are not linked to input have been evaled

      LBStructInPadLocal *_ins; // inputs buffer, only for synch struct
      LocalPropertyStorage *_stores; // only for store properties,
      LBStructOutPad  *_outs; // for output

      // handling asynch components
      AsynchThreadsNDataProxy *_asynchproxy;
      
      LBCompositionStruct *_parentcomp;
    private:
      LBCompositionStructRunTime& operator=(const LBCompositionStructRunTime& rt);
      LBCompositionStructRunTime(const LBCompositionStructRunTime& rt);
    };

  private:

    typedef RefCountedPtr<LBCompositionStructMetaData> MetaDataPtr;
    MetaDataPtr _metadata;
    mutable LBMutex _synchmtx; // struct-wide lock for synch struct running, and for asynch struct runtime data init
    LBCompositionStructRunTime *_runtime;

    AsynchMainThread *_asynchmain;
    bool _asynchmainstop;
  };

}

#endif
