#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <deque>
#include <algorithm>

#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbobject.hpp"

#include "luban/lbcompositionstruct.hpp"
#include "luban/lbprocessstruct.hpp"
#include "luban/lbstruct.hpp"
#include "luban/lbstructinterface.hpp"
#include "luban/lbrwlockedobjptr.hpp"
#include "luban/luban_symbolresolver.hpp"
#include "luban/lubanpermissionchecking.hpp"
#include "luban/lubantypechecking.hpp"
#include "luban/lblocalpropertystorage.hpp"
#include "luban/lbstructiopad.hpp"
#include "luban/lbstructiopadlocal.hpp"
#include "luban/lbstructiopadasynch.hpp"
#include "luban/lbstructoutpadcomp.hpp"
#include "luban/lbnamedargs.hpp"
#include "luban/lbcounterwaiter.hpp"


namespace Luban
{
  LBDEFINE(Luban::LBCompositionStruct, 1, 0)
  LBDEFAULT_STATIC_CONSTRUCTOR(Luban::LBCompositionStruct)

  LBCompositionStruct::~LBCompositionStruct()
  {
    if ( _asynchmain )
      {
	_asynchmainstop = true;
	cancelInternalThreads();
	_asynchmain->join();
	delete _asynchmain;
      }
    if ( _runtime ) delete _runtime;
  }

  LBCompositionStruct::LBCompositionStruct()
    : _metadata(), _synchmtx(), _runtime(0), _asynchmain(0), _asynchmainstop(false)
  {
  }

  LBCompositionStruct::LBCompositionStruct(const LBCompositionStruct& abs)
    : _metadata(abs._metadata), _synchmtx(), _runtime(0), _asynchmain(0), _asynchmainstop(false)
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH)
      _asynchmain = new AsynchMainThread(this);
    
    if ( abs._runtime )
      _runtime = new LBCompositionStructRunTime(*abs._runtime, this);
  }

  LBCompositionStruct::LBCompositionStruct(LBStructInterface* itfc)
    : _metadata( new LBCompositionStructMetaData(itfc) ), _synchmtx(), _runtime(0), _asynchmain(0), _asynchmainstop(false)
  {}

  LBCompositionStruct& LBCompositionStruct::operator=(const LBCompositionStruct& abs)
  {
    throw LBException("No assignment for composition struct");
  }

  string LBCompositionStruct::toString() const
  { 
    static const string synch("Synchronized Struct "), asynch("Asynchronized Struct ");

    string result;
    bool issynch = _metadata->_itfc->mode() == LBStructInterface::SYNCH;
    if ( issynch )
      result = synch;
    else
      result = asynch;

    // print name
    result += _metadata->_itfc->name()->toString();

    if ( ! issynch )
      {
	if ( _asynchmain->isdead() )
	  result += " terminated ";
	else
	  result += " active ";
      }
    else
	result += ' ';

    {

      LBMutexLocker mlock(_synchmtx);

      // print properties
      int numins = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
      if ( numins > 0 )
	{
	  result += "input( ";
	  for( int i = 0; i < numins ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::IN)->name().toString()+"=";
	      if ( issynch && _runtime )
		{
		  LBObjPtr val =_runtime->_ins->readValueByInternal(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}
      int numouts = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
      if ( numouts > 0 )
	{
	  result += "output( ";
	  for( int i = 0; i < numouts ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::OUT)->name().toString()+'=';
	      if ( issynch && _runtime )
		{
		  LBObjPtr val =_runtime->_outs->readValueByExternal(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}

      int numstores = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
      if ( numstores > 0 )
	{
	  result += "store( ";
	  for( int i = 0; i < numstores ; i++ )
	    {
	      result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STORE)->name().toString()+'=';
	      if (  _runtime )
		{
		  LBObjPtr val =_runtime->_stores->readValue(i); 
		  result += val.toString()+' ';
		}
	      else
		result += ' ';
	    }
	  result += ") ";
	}

    }

    int numstatics = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics > 0 )
      {
	result += "static( ";
	for( int i = 0; i < numstatics ; i++ )
	  {
	    result += _metadata->_itfc->propertyInfo(i, LBPropertyInfo::STATIC)->name().toString()+'=';
	    LBObjPtr val =_metadata->_statics->readValue(i); 
	    result += val.toString()+' ';
	  }
	result += ") ";
      }

    return result;

  }

  ostream& LBCompositionStruct::toStream( ostream& ost ) const
  {
    _metadata->_itfc->name()->toStream(ost);

    if (  _metadata->_itfc->mode() == LBStructInterface::SYNCH && _runtime )
      {
	ost.put('1');ost.put(' ');

	int numins = _runtime->_ins ? _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN):0;
	ost<<numins<< ' ';
	for( int i = 0; i < numins; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::IN)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _runtime->_ins->readValueByInternal(i);
	    if ( valp )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
	int numouts = _runtime->_outs?_metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT):0;
	ost<<numouts<< ' ';
	for( int i = 0; i < numouts; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::OUT)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _runtime->_outs->readValueByExternal(i);
	    if ( valp )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
	int numstores = _runtime->_stores?_metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE):0;
	ost<<numstores<< ' ';
	for( int i = 0; i < numstores; i++)
	  {
	    _metadata->_itfc->propertyInfo(i,LBPropertyInfo::STORE)->name().toStream(ost);
	    ost.put(' ');
	    LBObjPtr valp = _runtime->_stores->readValue(i);
	    if ( valp )
	      {
		ost.put('1'); ost.put(' ');
		LBObject::instanceToStream(ost, *valp);
	      }
	    else
	      {
		ost.put('0'); ost.put(' ');
	      }
	  }
      }
    else
      {
	ost.put('0'); ost.put(' ');
      }

    return ost;

  }

  istream& LBCompositionStruct::fromStream( istream& ist, int major, int minor )
  {
    // destroy the old guts, if any, should not be necessary
    // because stream should go to an empty default instance
    if ( _asynchmain )
      {
	_asynchmainstop = true;
	cancelInternalThreads();
	_asynchmain->join();
	delete _asynchmain;
	_asynchmain = 0;
	_asynchmainstop = false;
      }
    if ( _runtime ) 
      {
	delete _runtime;
	_runtime = 0;
      }
   

    // recover metadata
    LBFullSymbol stname;
    stname.fromStream(ist);
    string errs;
    LBStruct * st = LubanSymbolResolver::resolveStructSymbol(stname, errs);
    if ( ! st )
      throw(LBException("Can not find struct type "+stname.toString()+" when recovering struct from stream: "+errs));
    LBCompositionStruct *cmpst = dynamic_cast<LBCompositionStruct*>(st);
    if ( ! cmpst )
      throw(LBException("Struct type "+stname.toString()+" is not a process as expected when recovering struct from stream: "));
    _metadata = cmpst->_metadata;

    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH)
      _asynchmain = new AsynchMainThread(this);
    
    char oneorzero, sp;

    // recover state
    ist.get(oneorzero); ist.get(sp);
    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
      throw( LBException("Corrupted stream, can not recover process struct from stream") );
    if ( oneorzero == '1' )
      {
	initializeProperties();
	prepareOutputForProcessUse();

	int numins = -1;
	ist>>numins; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numins; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( ! pinfo || pinfo->attr() != LBPropertyInfo::IN )
	      throw LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as input as expected");

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover input property "+pname.toString()+" from stream") );
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());

		_runtime->_ins->writeValueByExternal(index, LBObjPtr(obj));
	      }
	    else
	      _runtime->_ins->writeValueByExternal(index, LBObjPtr());		
	  }

	int numouts = -1;
	ist>>numouts; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numouts; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
	      throw( LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as output as expected") );

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover output property "+pname.toString()+" from stream") );

		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());

		_runtime->_outs->writeValueByInternal(index, LBObjPtr(obj));

	      }
	    else
	      _runtime->_outs->writeValueByInternal(index, LBObjPtr());		
	  }


	int numstores = -1;
	ist>>numstores; ist.get(sp);
	if ( sp != ' ' )
	  throw( LBException("Corrupted stream, can not recover process struct from stream") );
	for( int i = 0; i < numstores; i++)
	  {
	    LBSymbol pname;
	    pname.fromStream(ist);
	    // recheck the index here, in case the struct definition changed
	    int index = -1;
	    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(pname, index);
	    if ( !pinfo || pinfo->attr() != LBPropertyInfo::STORE )
	      throw( LBException("Struct type definition does not match with struct object from stream. Property "+pname.toString()+" is not defined as store as expected") );

	    ist.get(sp);
	    if ( sp != ' ' )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    ist.get(oneorzero); ist.get(sp);
	    if ( sp != ' ' || ! ( oneorzero == '1' || oneorzero == '0' ) )
	      throw( LBException("Corrupted stream, can not recover process struct from stream") );
	    if ( oneorzero == '1' )
	      {
		string errs;
		LBObject *obj = LBObject::instanceFromStream(ist, errs);
		if ( ! obj )
		  throw( LBException("Corrupted stream, can not recover output property "+pname.toString()+" from stream") );
		if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), obj) )
		  throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+obj->getType().toString());
		
		_runtime->_stores->writeValue(index, LBObjPtr(obj));
	      }
	    else
	      _runtime->_stores->writeValue(index, LBObjPtr());		
	  }

      }

    return ist;
	
  }

  bool LBCompositionStruct::equals(const LBObject& another) const
  {  return this == &another; }

  const LBStructInterface& LBCompositionStruct::interface() const
  { return *(_metadata->_itfc);  }

  LBCompositionStruct* LBCompositionStruct::structCall(const LBNamedArgs* args) const
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      throw LBException("Can not call asynchronous structure");
    LBCompositionStruct *evalcopy = new LBCompositionStruct(*this);
    try {
      evalcopy->setBatchProperties(args);
    }
    catch( LBException &lbe )
      {
	delete evalcopy;
	throw lbe;
      }
    return evalcopy; 
  }

  void LBCompositionStruct::setBatchProperties(const LBNamedArgs* args)
  {
    string errs;
    if ( ! prepareToRun(errs) )
      throw LBException("Failed to initialize composition struct for evaluation: "+errs);
    prepareOutputForProcessUse();

    bool synch = _metadata->_itfc->mode() == LBStructInterface::SYNCH ;
    LBMutex *tolock = synch?( &_synchmtx) : 0;
    LBMutexPtrLocker lockif(tolock);

    // set the args
    int sz = args?args->numArgs() : 0;
    if ( sz == 0 )
      {
	// to make it consistent with process struct, no arg setting means to set all property with
	// the same value as before and re-sync everything
	if ( synch )
	  {
	    for( int i=0; _runtime->_ins && i<_runtime->_ins->size(); i++)
	      setInput(i, _runtime->_ins->readValueByInternal(i));
	    _runtime->_zeroinsEvaled = false;
	  }
      }
    else
      for( int i = 0; i < sz; i++ )
	{
	  const LBObject *argobj = args->getArg(i);
	  const LBSymbol *argname = args->getName(i);
	  int argindex = -1;
	  const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(*argname, argindex);
	  if ( ! pinfo || ( pinfo->attr() != LBPropertyInfo::IN && pinfo->attr() != LBPropertyInfo::STORE ) )
	    throw( LBException("The struct does not have the property: "+argname->toString()) );

	  // property setting policy and type checking here
	  if (  LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
	    {
	      if ( argobj )
		{
		  if ( ! LubanTypeChecking::checkType(pinfo->typeSpec(), argobj) )
		    throw LBException("Attempt to set object of wrong type into property: "+pinfo->name().toString()+". Required type: "+pinfo->typeSpec()->toString()+". Object type: "+argobj->getType().toString());

		  if ( pinfo->attr() == LBPropertyInfo::IN )
		    setInput(argindex, *argobj);
		  else
		    _runtime->_stores->writeValue(argindex, *argobj);
		}
	      else
		{
		  if ( pinfo->attr() == LBPropertyInfo::IN )
		    setInput(argindex, LBObjPtr());
		  else
		    _runtime->_stores->writeValue(argindex, LBObjPtr());
		}
	    }
	  else
	    throw LBException( "The property: "+argname->toString()+ " does not allow to be set from outside");
	}

    if ( synch ) 
      sync();
    else
      wakeupAsynch();
 
  }

  LBObjPtr LBCompositionStruct::getPropertyValue(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw LBException("No such property: "+s.toString());
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw( LBException("Failed to pass read permission check for property value reading: "+errs) );
    return readPropertyValue( pindex, pinfo->attr() );
  }

  // private
  LBObjPtr LBCompositionStruct::readPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  {
    // locking policy and static property access policy defined here
    switch ( attr )
      {
      case  LBPropertyInfo::IN: // no locking necessary for reading inputs
	return _runtime->_ins->readValueByInternal(index);
      case LBPropertyInfo::STATIC: // no local instance locking necessary for global static properties
	throw LBException("Can not read static property through struct instance");
      case LBPropertyInfo::OUT:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexLocker lock(_synchmtx);
	      return _runtime->_outs->readValueByExternal(index);
	    }
	  
	  // asynch case
	  _asynchmain->start();
	  return _runtime->_outs->readValueByExternal(index);
	}
      case LBPropertyInfo::STORE:
	{
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexLocker lock(_synchmtx);
	      return _runtime->_stores->readValue(index);
	    }
	  // asynch
	  return _runtime->_stores->readValue(index);
	}

      }
    
    throw LBException("Unrecognized kind of property");
    return LBObjPtr();
  }


  LBObject* LBCompositionStruct::getPropertyPtr(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH,  errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw( LBException("Failed to pass permission check for property mutation: "+errs) );

    return readPropertyUnsharedGuts( pindex, pinfo->attr() );

  }


  //private, this is for writing purpose
  LBObject* LBCompositionStruct::readPropertyUnsharedGuts(int index, LBPropertyInfo::ExecAttr attr)
  {
    // locking policy and static property access policy defined here
    switch ( attr )
      {
      case  LBPropertyInfo::IN: // need to lock because this is for writing
	{
	  // the policy for store property is here and in the VM code interpreter
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexLocker lock(_synchmtx);
	      return _runtime->_ins->readUnsharedGuts(index);
	    }
	  throw( LBException( "Can not obtain the unshared internal object pointer for input property of asynch struct") );
	}
      case LBPropertyInfo::STATIC: // static properties are taken care by different interface
	throw LBException( "Can not access static property through struct instance");
      case LBPropertyInfo::OUT:
	throw LBException( "Can not modify output property value from outside");
      case LBPropertyInfo::STORE:
	{
	  // the policy for store property is here and in the VM code interpreter
	  if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
	    {
	      LBMutexLocker lock(_synchmtx);
	      return _runtime->_stores->getUnsharedPtr(index);
	    }
	  throw LBException( "Can not obtain the unshared internal object pointer for store property of asynch struct");
	}
      }
    
    throw( LBException("Unrecognized kind of property"));
    return 0;
  }


  void LBCompositionStruct::cancelInternalThreads()
  {
    _asynchmainstop = true; // tell asynchMainLoop() to stop
    if ( _runtime )
      {
	for( int i=0; i<_runtime->_livejoints.size();i++)
	  _runtime->_livejoints[i]->dismissThreads();
	if ( _runtime->_asynchproxy )
	  _runtime->_asynchproxy->dismissAll();
      }
  }

  void LBCompositionStruct::throwError(const string& emsg, bool throwup)
  {
    cancelInternalThreads();
    bool synch = _metadata->_itfc->mode() == LBStructInterface::SYNCH;
    LBError errvalue(emsg);
    if ( _runtime && _runtime->_outs )
      {
	if ( synch )
	  {
	    for(int i = 0; i < _runtime->_outs->size(); i++)
	      _runtime->_outs->writeValueByInternal(i,errvalue);
	    if ( _runtime->_parentcomp )
	      _runtime->_outs->flush();
	  }
	else
	  _runtime->_outs->dismissWaitingThreads();
      }
    if (  _runtime && _runtime->_parentcomp  ) 
      return;
    if ( throwup )
      throw LBException("Composition struct evaluation failure: "+emsg);
  }


  void LBCompositionStruct::touchProperty(const LBSymbol& s)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( pinfo == 0 )
      throw( LBException("No such property: "+s.toString()) );
    if ( pinfo->attr() != LBPropertyInfo::IN )
      return;
    touchProperty( pindex, pinfo->attr() );
  }

  void LBCompositionStruct::setProperty(const LBSymbol& s, const LBObject& val)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( ! pinfo )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), val );
  }

  void LBCompositionStruct::setProperty(const LBSymbol& s, const LBObjPtr& valptr)
  {
    int pindex = -1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(s, pindex);
    if ( ! pinfo )
      throw( LBException("No such property: "+s.toString()) );
    setProperty( pindex, pinfo->attr(), valptr );
  }

  LBObjPtr LBCompositionStruct::getPropertyValue(int index, LBPropertyInfo::ExecAttr attr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw LBException("Invalid property index");
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) )
      throw LBException("Failed to pass read permission check for property value reading: "+errs);
    return readPropertyValue( index, attr );
  }

  LBObject* LBCompositionStruct::getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkReadPermission(*pinfo, _metadata->_itfc->mode() == LBStructInterface::ASYNCH, errs) || ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);

    return readPropertyUnsharedGuts( index, attr );
  }

  void LBCompositionStruct::touchProperty(int index, LBPropertyInfo::ExecAttr attr)
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      return; // do nothing for an asynch struct

    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index for touchProperty operation") );

    if ( pinfo->attr() != LBPropertyInfo::IN )
      return;

    int jid = _metadata->_inpmap.getInputJointid(index);
    if ( jid >= 0 )
      {
	LBMutexLocker lock(_synchmtx);
	_runtime->_livejoints[jid]->setNewValue(_runtime->_ins->readValueByInternal(index));
	sync();
      }

  }

  void LBCompositionStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val)
  {  
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);
    
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	    setInput(index, val);
	else
	  {
	    LBMutexLocker lock(_synchmtx);
	    setInput(index, val);
	    sync();
	  }
	return;
	break;
      case LBPropertyInfo::STORE:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _runtime->_stores->writeValue(index,val);
	else
	  {
	    LBMutexLocker lock(_synchmtx);
	    _runtime->_stores->writeValue(index,val);
	  }
	return;
      case LBPropertyInfo::STATIC:
	_metadata->_statics->writeValue(index, val);
	return;
      }

    throw LBException("Set value to property of invalid attribute");

  }

  void LBCompositionStruct::setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, attr);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Failed to pass permission check for property mutation: "+errs);
    
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr()) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    switch( attr )
      {
      case LBPropertyInfo::IN:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  setInput(index, valptr);
	else
	  {
	    LBMutexLocker lock(_synchmtx);
	    setInput(index, valptr);
	    sync();
	  }
	return;
	break;
      case LBPropertyInfo::STORE:
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _runtime->_stores->writeValue(index,valptr);
	else
	  {
	    LBMutexLocker lock(_synchmtx);
	    _runtime->_stores->writeValue(index,valptr);
	  }
	return;
      case LBPropertyInfo::STATIC:
	_metadata->_statics->writeValue(index, valptr);
	return;
      }

    throw LBException("Set value to property of invalid attribute");

  }

  void LBCompositionStruct::setInput(int index, const LBObjPtr& valptr)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::IN);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), valptr.getConstRealPtr()) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    if ( ! prepareToRun(errs) )
      throw LBException("Error to prepare the composition struct to run: "+errs);
    int jid = _metadata->_inpmap.getInputJointid(index);
    
    // reflect the negative joint id useage: indicating unlinked input property
    if ( _runtime->_ins )
      if ( ! _runtime->_ins->writeValueByExternal(index,valptr) )
	throw LBException("Failed to set input property"+pinfo->name().toString());
    if ( jid >=0 )
      if ( ! _runtime->_livejoints[jid]->setNewValue(valptr) )
	throw LBException("Failed to set input property for composition struct");
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      if ( ! _asynchmain->start() ) // start asynch thread if not started
	throwError("Failed to start asynch struct main thread");
  }

  void LBCompositionStruct::setInput(int index, const LBObject& val)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(index, LBPropertyInfo::IN);
    if ( pinfo == 0 )
      throw( LBException("Invalid property index: ") );
    string errs;
    if ( !LubanTypeChecking::checkType(pinfo->typeSpec(), &val) )
      throw LBException("Object of invalid type set for property "+pinfo->name().toString() );

    if ( ! prepareToRun(errs) )
      throw LBException("Error to prepare the composition struct to run: "+errs);
    int jid = _metadata->_inpmap.getInputJointid(index);

    // reflect the negative joint id useage: indicating unlinked input property
    if ( _runtime->_ins )
      if ( ! _runtime->_ins->writeValueByExternal(index,val) )
	throw LBException("Failed to set input property"+pinfo->name().toString());
    if ( jid >=0 )
      if ( ! _runtime->_livejoints[jid]->setNewValue(val) )
	throw LBException("Failed to set input property for composition struct");
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      if ( ! _asynchmain->start() ) // start asynch thread if not started
	throwError("Failed to start asynch struct main thread");
  }


  void LBCompositionStruct::sync()
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
      {
	throwError("sync action does not apply to asynch struct");
	return;
      }
    string errs;
    if ( ! prepareToRun(errs) )
      {
	throwError("Error to prepare the composition struct to run: "+errs);
	return;
      }

    // start all asynch cells
    int numasynchs = _metadata->_components._asynchcells?_metadata->_components._asynchcells->size():0;
    for( int i=0; i<numasynchs; i++)
      if ( ! _runtime->_livecomps[(*_metadata->_components._asynchcells)[i]]->wakeupAsynch() )
	{
	  throwError("Can not start asynch component inside composition\n");
	  return;
	}

    int numzeroins = _metadata->_components._zeroincells?_metadata->_components._zeroincells->size():0;
    if ( numzeroins != 0 && ! _runtime->_zeroinsEvaled )
      {
	for( int i=0; i<numzeroins; i++)
	  {
	    try {
	      _runtime->_livecomps[(*_metadata->_components._zeroincells)[i]]->sync();
	    }
	    catch ( LBException& e )
	      {
		errs += "Exception when evluatiing components in composition: "+e.msg();
		throwError(errs);
		return;
	      }
	    catch (...)
	      {
		errs += "Unknown exception happened synch struct eval";
		throwError(errs);
		return;
	      }
	  }
	_runtime->_zeroinsEvaled = true;
      }

    bool runevalq = true;
    while ( true )
      {
	if ( runevalq )
	  {
	    int cellid;
	    _runtime->_evalque.resetCursor();
	    while ( _runtime->_evalque.nextToEval(cellid) )
	      {
		try {
		  _runtime->_livecomps[cellid]->sync(); // the sync action of the cell could add more cellid to the eval que
		}
		catch ( LBException& e )
		  {
		    errs += "Exception when evluatiing components in composition: "+e.msg();
		    throwError(errs);
		    return;
		  }
		catch (...)
		  {
		    errs += "Unknown exception happened in main sync thread of asynch struct";
		    throwError(errs);
		    return;
		  }
	      }
	    runevalq = false;
	  }

	if ( _runtime->_asynchproxy )
	  {
	    // check if any asynch cell generates new data
	    if ( _runtime->_asynchproxy->hasNewData() )
	      {
		for( int i=0; i<_runtime->_asynchproxy->asynchJoints(); i++)
		  if ( _runtime->_asynchproxy->hasNewData(i) )
		    {
		      _runtime->_livejoints[_runtime->_asynchproxy->liveJointIndex(i)]->asynchFlush(); // this will add cell to evalq
		      _runtime->_asynchproxy->informDataConsumed(i);
		      runevalq = true;
		    }
	      }
	    else // check if any potential data generating asynch cell active
		{
		  if ( _runtime->_asynchproxy->waitForNewDataIfAnyActive() == 0 )
		    break;
		  runevalq = false;
		}
	  }
	else
	  break;
      }

    //flush output downstream
    if ( _runtime->_outs && _runtime->_parentcomp ) 
      _runtime->_outs->flush();

  }

  bool LBCompositionStruct::asynchCompMainLoop(string& errs)
  {
    if ( _metadata->_itfc->mode() == LBStructInterface::SYNCH )
      {
	if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	errs += "async thread main loop can not run on synch struct\n";
	throwError(errs, false);
	return false;
      }
    if ( ! prepareToRun(errs) )
      {
	if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	errs += "Error to prepare the composition struct to run\n";
	throwError(errs, false);
	return false;
      }

    // start all asynch cells
    int numasynchs = _metadata->_components._asynchcells?_metadata->_components._asynchcells->size():0;
    for( int i=0; i<numasynchs; i++)
      if ( ! _runtime->_livecomps[(*_metadata->_components._asynchcells)[i]]->wakeupAsynch() )
	{
	  if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	  errs += "Failed to start asynch cells in the composition\n";
	  throwError(errs, false);
	  return false;
	}

    int numzeroins = _metadata->_components._zeroincells?_metadata->_components._zeroincells->size():0;
    if ( numzeroins != 0 && ! _runtime->_zeroinsEvaled )
      {
	for( int i=0; i<numzeroins; i++)
	  {
	    try {
	      _runtime->_livecomps[(*_metadata->_components._zeroincells)[i]]->sync();
	    }
	    catch ( LBException& e )
	      {
		if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
		errs += "Exception when evluatiing components in composition: "+e.msg();
		throwError(errs, false);
		return false;
	      }
	    catch (...)
	      {
		if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
		errs += "Unknown exception happened in main sync thread of asynch struct";
		throwError(errs, false);
		return false;
	      }
	  }
	_runtime->_zeroinsEvaled = true;
      }

    while ( ! _asynchmainstop )
      {
	// check if any asynch cell generates new data
	if ( _runtime->_asynchproxy->hasNewData() )
	  {
	    for( int i=0; i<_runtime->_asynchproxy->asynchJoints(); i++)
	      if ( _runtime->_asynchproxy->hasNewData(i) )
		{
		  _runtime->_livejoints[_runtime->_asynchproxy->liveJointIndex(i)]->asynchFlush(); // this will add cell to evalq
		  _runtime->_asynchproxy->informDataConsumed(i);
		}
	    
	    int cellid;
	    _runtime->_evalque.resetCursor();
	    while ( _runtime->_evalque.nextToEval(cellid) )
	      try {
		_runtime->_livecomps[cellid]->sync(); // the sync action of the cell could add more cellid to the eval que
	      }
	      catch( LBException &e)
		{
		  if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
		  errs += "Exception when evluatiing components in composition: "+e.msg();
		  throwError(errs, false);
		  return false;
		}
	      catch (...)
		{
		  if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
		  errs += "Unknown exception happened in main sync thread of asynch struct";
		  throwError(errs, false);
		  return false;
		}
	  }
	else // check if any potential data generating asynch cell active
	  if ( _runtime->_asynchproxy->numActive() == 0 )
	    {
	      if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(false);
	      if ( ! _runtime->_asynchproxy->waitForNewData() ) return true;
	      if ( _runtime->_parentcomp ) _runtime->_parentcomp->informComponentAct(true);
	    }
	  else
	    if ( ! _runtime->_asynchproxy->waitForNewData() )
	      return true;
      }

    return true;

  }

  void LBCompositionStruct::informComponentAct(bool wakeup)
  {
    if ( wakeup )
      _runtime->_asynchproxy->incActive();
    else
      _runtime->_asynchproxy->decActive();
  }

  bool LBCompositionStruct::readDataJoint(int jid, int pid, LBObjPtr& obj)
  {
    return  _runtime->_livejoints[jid]->readValue(pid, obj);
  }

  bool LBCompositionStruct::writeDataJoint(int jid, const LBObjPtr& valptr)
  {
    return _runtime->_livejoints[jid]->setNewValue(valptr);
  }

  bool LBCompositionStruct::writeDataJoint(int jid, const LBObject& val)
  {
    return _runtime->_livejoints[jid]->setNewValue(val);
  }

  LBObjPtr LBCompositionStruct::readStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr)
  {
    switch ( attr )
      {
      case LBPropertyInfo::STORE:
	return _runtime->_stores->readValue(prpindex);
      case LBPropertyInfo::STATIC:
	return _metadata->_statics->readValue(prpindex);
      }
    throw LBException("Fatal error, invalid attribut for property reading from internal component");
  }

  void LBCompositionStruct::writeStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr)
  {
    switch ( attr )
      {
      case LBPropertyInfo::STORE:
	_runtime->_stores->writeValue(prpindex, valptr);
	return;
      case LBPropertyInfo::STATIC:
	_metadata->_statics->writeValue(prpindex, valptr);
	return;
      }
    throw LBException("Fatal error, invalid attribut for property reading from internal component");
  }

  void LBCompositionStruct::writeStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr, const LBObject& val)
  {
    switch ( attr )
      {
      case LBPropertyInfo::STORE:
	_runtime->_stores->writeValue(prpindex, val);
	return;
      case LBPropertyInfo::STATIC:
	_metadata->_statics->writeValue(prpindex, val);
	return;
      }
    throw LBException("Fatal error, invalid attribut for property reading from internal component");
  }

  LBWriteLockedObjPtr LBCompositionStruct::getWriteLockedStoreNStaticPropertyByInternalComp(int prpindex, LBPropertyInfo::ExecAttr attr)
  {
    switch ( attr )
      {
      case LBPropertyInfo::STORE:
	return _runtime->_stores->getWriteLockedObj(prpindex);
      case LBPropertyInfo::STATIC:
	return _metadata->_statics->getWriteLockedObj(prpindex);
      }
    throw LBException("Fatal error, invalid attribut for property reading from internal component");
  }

  bool LBCompositionStruct::wakeupAsynch()
  {
    if ( ! _asynchmain )
      throw LBException("Attempt to start asynch thread on a synch struct" );
    if ( _asynchmain->isdead() )
      throw LBException("asynch struct internal error: "+_asynchmain->errmsg() );
    if ( ! _asynchmain->start() ) // start thread if it has not been started yet
      throw LBException("Failed to start asynch struct main thread");
    return true;
  }

  bool LBCompositionStruct::initializeProperties()
  {
    string errs;
    if ( ! _runtime )
      return prepareToRun(errs);
  }

  bool LBCompositionStruct::linkOutputProperty(const LBSymbol& outprpname, int jointid)
  {
    int prpidx=-1;
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(outprpname, prpidx);
    if ( !pinfo || pinfo->attr() != LBPropertyInfo::OUT )
      return false;
    _runtime->_outs->linkTo(prpidx, jointid);
  }

  bool LBCompositionStruct::prepareOutputForCompositionUse()
  {
    int outnum = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
    if ( outnum == 0 )
      return true;
    if ( ! _runtime )
      {
	string errs;
	if ( ! prepareToRun(errs ))
	  return false;
      }
    if ( ! _runtime->_outs )
      {
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _runtime->_outs = static_cast<LBStructOutPad*>(new LBStructOutPadAsynchComp(outnum));
	else
	  _runtime->_outs = new LBStructOutPadSynchComp(outnum);
      }
    return true;
  }

  bool LBCompositionStruct::setRunTimeParentComposition(LBCompositionStruct *pcomp)
  {
    if ( _runtime->_outs )
      _runtime->_outs->setParentComposition(pcomp);
    _runtime->_parentcomp = pcomp;
    return true;
  }

  bool LBCompositionStruct::prepareOutputForProcessUse()
  {
    int outnum = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
    if ( outnum == 0 )
      return true;
    if ( ! _runtime )
      {
	string errs;
	if ( ! prepareToRun(errs ))
	  return false;
      }
    if ( ! _runtime->_outs )
      {
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  _runtime->_outs = new LBStructOutPadAsynch(outnum);
	else
	  _runtime->_outs = new LBStructOutPadLocal(outnum);
      }
    return true;
  }


  bool LBCompositionStruct::prepareToRun(string& errs)
  {
    if ( _metadata->prepareMetaData(errs) )
      {
	if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH )
	  {
	    LBMutexLocker lock( _synchmtx);
	    prepareRunTime();
	  }
	else
	  prepareRunTime();
	return true;
      }
    return false;
  }

  void LBCompositionStruct::prepareRunTime()
  {
    // to be done
    // need to add input/store initialization code here
    // this is a single thread function

    if ( _runtime ) return;
    int storeprps = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STORE);
    int ins = _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
    int joints = _metadata->_joints.size();
    int comps = _metadata->_components.size();
    int toplevel = _metadata->_components.toplevel();
    bool asynch = _metadata->_itfc->mode() == LBStructInterface::ASYNCH;
    LBCompositionStructRunTime *newruntime = new LBCompositionStructRunTime(ins, storeprps, joints, comps, toplevel, asynch );
    int i = 0;
    for(; i < joints; i++)
      newruntime->_livejoints[i] = new LiveJoint(_metadata->_joints.getJoint(i),i, this);
    for( i=int(ComponentList::OUTPUTS)+1; i<comps; i++)
      {
	LBStruct *s = _metadata->_components.getLiveComp(i); // the mapping of output property to DataJoint id should be done at metadata prepareation , at the resolve external symbol linking time
	s->setRunTimeParentComposition(this);
	newruntime->_livecomps[i]=s;
      }

    // prepare for asynch joints
    if ( _metadata->_itfc->mode() == LBStructInterface::ASYNCH || _metadata->_components._asynchcells )
      {
	newruntime->_asynchproxy = new AsynchThreadsNDataProxy(_metadata->_joints.size());
	for(i=0; i<_metadata->_joints.size(); i++)
	  if ( _metadata->_components.isAsynch(_metadata->_joints.getJoint(i)._djtkey._cellid) )
	    newruntime->_asynchproxy->addAsynchJoint(i);
      }
    _runtime = newruntime;
    return;
  }


  bool LBCompositionStruct::addPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].inc();
  }
 
  bool LBCompositionStruct::removePending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].dec();    
  } 

  bool LBCompositionStruct::waitUntilNoPending(int staticprpindex)
  {
    if ( staticprpindex < 0 || staticprpindex >= _metadata->_itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC) )
      return false;
    return _metadata->_staticscounters[staticprpindex].waitDownToZero();
  } 

  LBObjPtr LBCompositionStruct::getStaticPropertyValue(int index) 
  {
    if ( !_metadata->_statics )
      throw LBException("Attempt to get non-existing static property");
    return _metadata->_statics->readValue(index);
  }

  void LBCompositionStruct::setStaticPropertyValue(int index, const LBObject& val)
  {
    if ( !_metadata->_statics )
      throw LBException("Attempt to set non-existing static property");
    _metadata->_statics->writeValue(index,val);
  }


  void LBCompositionStruct::setStaticPropertyValue(int index, const LBObjPtr& valptr)
  {
    if ( !_metadata->_statics )
      throw LBException("Attempt to set non-existing static property");
    _metadata->_statics->writeValue(index,valptr);
  }


  void LBCompositionStruct::waitUntilInputEmptyOrMainFinish()
  {
    if ( _runtime && _runtime->_asynchproxy )
      _runtime->_asynchproxy->waitUntilNoData();
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBCompositionStruct, luban_wait, "waitfinish", "void waitfinish()");
  LBObject *LBCompositionStruct::luban_wait(const LBVarArgs *args)
  {
    int numargs = args ? args->numArgs() : 0;
    if ( numargs )
      throw LBException("no arguement for wait() function");
    waitUntilInputEmptyOrMainFinish();
    return 0;
  }


  LBWriteLockedObjPtr LBCompositionStruct::getWriteLockedStaticProperty(int staticprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(staticprpindex, LBPropertyInfo::STATIC );
    if ( pinfo == 0 )
      throw( LBException("Invalid static property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _metadata->_statics->getWriteLockedObj(staticprpindex);
  }

  LBWriteLockedObjPtr LBCompositionStruct::getWriteLockedStoreProperty(int storeprpindex)
  {
    const LBPropertyInfo *pinfo = _metadata->_itfc->propertyInfo(storeprpindex, LBPropertyInfo::STORE );
    if ( pinfo == 0 )
      throw( LBException("Invalid store property index") );
    string errs;
    if ( ! LubanPermissionChecking::checkWritePermission(*pinfo, errs) )
      throw LBException("Write permission violation: "+errs);
    return _runtime->_stores->getWriteLockedObj(storeprpindex);
  }


  // helper class metadata
  LBCompositionStruct::LBCompositionStructMetaData::LBCompositionStructMetaData(LBStructInterface *itfc)
    :_itfc(itfc), _components(itfc->mode()==LBStructInterface::ASYNCH), _joints(), _extsyms(), _inpmap(itfc->numberOfPropertiesByAttr(LBPropertyInfo::IN)), _statics(0), _staticscounters(0), _preparemtx(), _status(UNPREPARED), _errmsg()
  {
    int numstatics = itfc->numberOfPropertiesByAttr(LBPropertyInfo::STATIC);
    if ( numstatics>0 )
      {
	_statics = new LocalPropertyStorage(numstatics, true);
	_staticscounters = new CounterWaiter[numstatics];
      }
  }

  LBCompositionStruct::LBCompositionStructMetaData::~LBCompositionStructMetaData()
  {
    delete _itfc;
    if ( _statics )
      {
	delete _statics;
	delete _staticscounters;
      }
  }

  int LBCompositionStruct::LBCompositionStructMetaData::newNamedTypedCell(const LBSymbol& cellname, const LBFullSymbol& structtypename, bool threading, string& errs)
  {
    int cid = _components.addNamedTyped(cellname, structtypename, threading, errs);
    if ( cid >= 0 )
      _extsyms.addSymbol(structtypename, cid);
    return cid;
  }

  int LBCompositionStruct::LBCompositionStructMetaData::newNamedAdhocCell(const LBSymbol& cellname, bool asynch, bool threading, string& errs)
  {
    return _components.addNamedAdhoc(cellname, asynch, threading, errs);
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::setNamedAdhocCell(const LBSymbol& cname, LBProcessStruct *proc)
  {
    int cid = _components.index(cname);
    if ( cid < 0 )
      return false;
    return _components.setAdhocProc(cid, proc);
  }

  int LBCompositionStruct::LBCompositionStructMetaData::newNonameAdhocCell(bool threading)
  {
    return _components.addNonameAdhoc(threading);
  }
 
  bool LBCompositionStruct::LBCompositionStructMetaData::setNonameAdhocCell(int cid, LBProcessStruct *proc)
  {
    return _components.setAdhocProc(cid, proc);
  }
 
  int LBCompositionStruct::LBCompositionStructMetaData::newNonameTypedCell(const LBFullSymbol& structtypename, bool threading, string& errs)
  {
    int cid = _components.addNonameTyped(structtypename, threading);
    if ( cid >= 0 )
      _extsyms.addSymbol(structtypename, cid);
    return cid;
  }

  const LBSymbol& LBCompositionStruct::LBCompositionStructMetaData::getCellName(int cid) const
  {
    return _components.name(cid);
  }

  int LBCompositionStruct::LBCompositionStructMetaData::getCellId(const LBSymbol& nm) const
  {
    return _components.index(nm);
  }

  const LBStructInterface *LBCompositionStruct::LBCompositionStructMetaData::getTypedCellInterface(int cellid) const
  {
    return _components.getTypedInterface(cellid);
  }

  int LBCompositionStruct::LBCompositionStructMetaData::getOutputDataJointIdForAdhocCell(int cellid)
  {
    return _joints.getJointIndex(cellid);
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::link(int upcellid, int downcellid, int& jointid, int& jointportid, string& errs)
  {
    if ( _components.isTyped(upcellid) || _components.isTyped(downcellid) )
      {
	errs += "Cells are not adhoc embeded process as expected\n";
	return false;
      }
    int jid = _joints.getJointIndex(upcellid);
    int pid = _joints.getPortIndex(jid, downcellid);
    _components.recordDownStream(upcellid, downcellid);
    _components.recordUpStream(downcellid, upcellid);
    jointid = jid;
    jointportid = pid;
    return true;
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::link(int upcellid, const LBSymbol& outprpname, int downcellid, int& jointid, int& jointportid, string& errs)
  {
    if ( ! _components.isTyped(upcellid) || _components.isTyped(downcellid) )
      {
	errs += "Up stream cell is not adhoc or downstream cell is not typed as expected\n";
	return false;
      }
    int jid = _joints.getJointIndex(upcellid, outprpname);
    int pid = _joints.getPortIndex(jid, downcellid);
    _components.recordDownStream(upcellid, outprpname, jid, downcellid);
    _components.recordUpStream(downcellid, upcellid);
    jointid = jid;
    jointportid = pid;
    return true;
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::link(int upcellid, const LBSymbol& upoutprpname, int downcellid,const LBSymbol& downinprpname,  int downprpidx, string& errs) // typed-typed
  {
    if (  ! _components.isTyped(downcellid) || ! _components.isTyped(upcellid) )
      {
	errs += "Cells are not typed struct as expected\n";
	return false;
      }
    int jid = _joints.getJointIndex(upcellid, upoutprpname);
    int pid = _joints.getPortIndex(jid, downcellid, downinprpname, downprpidx);
    _components.recordDownStream(upcellid, upoutprpname, jid, downcellid);
    _components.recordUpStream(downcellid, downinprpname, jid, pid, upcellid );
    return true;
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::link(int upcellid, int downcellid,const LBSymbol& downinprpname,  int downprpidx, string& errs) // adhoc-typed/output
  {
    if ( ! _components.isTyped(downcellid) || _components.isTyped(upcellid) )
      {
	errs += "down stream cell is not adhoc or up tream cell is not typed as expected\n";
	return false;
      }
    int jid = _joints.getJointIndex(upcellid);
    int pid = _joints.getPortIndex(jid, downcellid, downinprpname, downprpidx);
    _components.recordDownStream(upcellid, downcellid);
    _components.recordUpStream(downcellid, downinprpname, jid, pid, upcellid );
    return true;
  }


  bool LBCompositionStruct::LBCompositionStructMetaData::linkBatchInputs(const LBFullSymbol& structtypename, int tocellid, string& errs)
  {
    // check interface inheritance, instead of port names
    // do the same to procedure struct
    if ( ! _itfc->hasInterface(structtypename) )
      {
	errs += structtypename.toString() + " is not a parent interface of the composition to be used to refere to batch input properties of interface\n";
	return false;
      }
    const LBStructInterface *citfc = _components.getTypedInterface(tocellid);
    if ( ! citfc || ! citfc->hasInterface(structtypename) )
      {
	errs += structtypename.toString() + " is not a parent interface of the cell "+getCellName(tocellid).toString()+" to be used to refere to batch input properties from interface\n";
	return false;
      }

     _components.recordDownStream(structtypename, tocellid);
     _components.recordUpStream(tocellid, structtypename);

    return true;
  }

  bool LBCompositionStruct::LBCompositionStructMetaData::linkAllInputs(int tocellid, string& errs)
  {
    const LBFullSymbol& structtypename = *_itfc->name();
    const LBStructInterface *citfc = _components.getTypedInterface(tocellid);
    if ( ! citfc || ! citfc->hasInterface(structtypename) )
      {
	errs += structtypename.toString() + " is not a parent interface of the cell "+getCellName(tocellid).toString()+" to be used to refere to batch input properties from interface\n";
	return false;
      }

     _components.recordDownStream(structtypename, tocellid);
     _components.recordUpStream(tocellid, structtypename);

    return true;

  }

  bool LBCompositionStruct::LBCompositionStructMetaData::linkAllOutputs(int fromcellid, string& errs)
  {
    const LBStructInterface *fitfc = _components.getTypedInterface(fromcellid);
    if ( ! fitfc || ! fitfc->name() )
      {
	errs += getCellName(fromcellid).toString()+" is not typed cell to be used to refere to batch output properties to interface\n";
	return false;
      }

    const LBFullSymbol *fname = fitfc->name();
    if ( !_itfc->hasInterface(*fname) )
      {
	errs += fname->toString() + " is not a parent interface of the composition "+_itfc->name()->toString()+" to be used to refere to batch output properties to interface\n";
	return false;
      }

     _components.recordDownStream(fromcellid, *fname);
     _components.recordUpStream(*fname,fromcellid);

    return true;

  }

  bool LBCompositionStruct::LBCompositionStructMetaData::prepareMetaData(string& errs)
  {
    if ( _status == PREPARED ) return true;
    if ( _status == ERRED )
      {
	errs += _errmsg;
	return false;
      }
    LBMutexLocker plock(_preparemtx);
    if ( _status == PREPARED ) return true;
    if ( _status == ERRED )
      {
	errs += _errmsg;
	return false;
      }
    if ( !resolveBatchLinks(errs)  )
	{
	_errmsg = errs;
	_status = ERRED;
	return false;
      }

    if ( ! resolvePropertyReferences(errs) )
      {
	_errmsg = errs;
	_status = ERRED;
	return false;
      }
    bool resort;
    if ( ! resolveExternalSymbols(errs, resort) )
      {
	_errmsg = errs;
	_status = ERRED;
	return false;
      }
    if ( resort && ! _components.topsort(errs) )
      {
	_errmsg = errs;
	_status = ERRED;
	return false;
      }

    _status = PREPARED;
    return true;

  }

  // a function to hold the join id policy in ComponentInfo
  static inline int convertJointId(int jid)
  {
    if ( jid >= 0 ) return jid;
    return -jid-1;
  }
    

  bool LBCompositionStruct::LBCompositionStructMetaData::resolveExternalSymbols(string& errs, bool& resort)
  {
    int allsym = _extsyms.size();
    for( int i=0; i<allsym; i++)
      {
	LBStruct *st = LubanSymbolResolver::resolveStructSymbol(_extsyms.symbolName(i), errs);
	if ( ! st )
	  {
	    errs += "Can not resolve struct symbol "+_extsyms.symbolName(i).toString()+"\n";
	    return false;
	  }

	LBStruct *stcopy = static_cast<LBStruct*>(st->clone());
	stcopy->initializeProperties();
	stcopy->prepareOutputForCompositionUse();

	int cid = _extsyms.cellId(i);
	ComponentInfo *cinfo = _components.getComponentInfo(cid);
	if ( cinfo->_downmap )
	  {
	    ComponentInfo::DownPortMap::Iterator it(*cinfo->_downmap);
	    while ( it.next() )
	      stcopy->linkOutputProperty( *it.currentPair()->key, convertJointId(*it.currentPair()->value) );
	  }
	cinfo->_str = stcopy;
      }
    if ( _components.toplevel() == -1 )
      resort = true;

    return true;

  }

  bool LBCompositionStruct::LBCompositionStructMetaData::resolvePropertyReferences(string& errs)
  {
    // match inputs first
    ComponentInfo *cinfo = _components.getComponentInfo(int(ComponentList::INPUTS));
    if ( cinfo->_downmap)
      {
	ComponentInfo::DownPortMap::Iterator it(*cinfo->_downmap);
	while ( it.next() )
	  {
	    int inprpidx = -1;
	    const LBPropertyInfo *pinfo = _itfc->propertyInfo( *it.currentPair()->key, inprpidx);
	    if ( ! pinfo || pinfo->attr() != LBPropertyInfo::IN )
	      {
		errs += "Reference to property "+it.currentPair()->key->toString()+ "that is not part of the input interface\n";
		return false;
	      }
	    _inpmap.setMap( inprpidx, convertJointId(*it.currentPair()->value) );
	  }
      }


    // match outputs, just do checking
    cinfo = _components.getComponentInfo(int(ComponentList::OUTPUTS));
    if ( cinfo->_upmap )
      {
	ComponentInfo::UpPortMap::Iterator it(*cinfo->_upmap);
	while ( it.next() )
	  {
	    int outprpidx = -1;
	    const LBPropertyInfo *pinfo = _itfc->propertyInfo( *it.currentPair()->key, outprpidx);
	    if ( ! pinfo || pinfo->attr() != LBPropertyInfo::OUT )
	      {
		errs += "Reference to property "+it.currentPair()->key->toString()+ "that is not part of the output interface\n";
		return false;
	      }
	  }
      }

    return true;

  }
  
  bool LBCompositionStruct::LBCompositionStructMetaData::resolveBatchLinks(string& errs)
  {
    // we go one step down from INPUTS and one step up from OUTPUTS 
    // that's all possible batch links according to Luban's design
    ComponentInfo *cinfo = _components.getComponentInfo(int(ComponentList::INPUTS));
    if ( cinfo->_batchdowns )
      for( int i=0; i<cinfo->_batchdowns->size(); i++)
	{
	  const LBStructInterface *bitfc = LubanSymbolResolver::resolveStructInterface((*cinfo->_batchdowns)[i]._structtypename, errs);
	  if ( ! bitfc )
	    {
	      errs += "Refered struct interface "+(*cinfo->_batchdowns)[i]._structtypename.toString()+" does not exist\n";
	      return false;
	    }
	  if ( !_itfc->hasInterface((*cinfo->_batchdowns)[i]._structtypename ) )
	    {
	      errs += "Refered struct interface "+(*cinfo->_batchdowns)[i]._structtypename.toString()+" is not part of the current struct interface\n";
	      return false;
	    }

	  int downcellid = (*cinfo->_batchdowns)[i]._cellid;
	  const LBStructInterface *downcellitfc = _components.getTypedInterface(downcellid);
	  if ( ! downcellitfc )
	    {
	      errs += "Batch link to a non-typed component";
	      return false;
	    }
	  int numins = bitfc->numberOfPropertiesByAttr(LBPropertyInfo::IN);
	  int jointid, portid, downprpid;
	  for(int j=0; j<numins; j++)
	    {
	      const LBPropertyInfo *pinfo = bitfc->propertyInfo(j, LBPropertyInfo::IN);
	      const LBPropertyInfo *downprpinfo = downcellitfc->propertyInfo(pinfo->name(), downprpid);
	      if ( ! downprpinfo || downprpinfo->attr() != LBPropertyInfo::IN )
		{
		  errs += "Batch link to a non-existing input property "+pinfo->name().toString();
		  return false;
		}
	      jointid = _joints.getJointIndex(int(ComponentList::INPUTS), pinfo->name());
	      portid=_joints.getPortIndex(jointid, downcellid, pinfo->name(),downprpid);
	      if ( ! ( cinfo->_downmap && cinfo->_downmap->find(pinfo->name(), jointid) ) )
		cinfo->downStream(pinfo->name(), -(jointid+1), downcellid ); //negative to indicate it is from batch link
	      if ( ! _components.recordUpStream(downcellid, pinfo->name(), -(jointid+1), portid, int(ComponentList::INPUTS)) )
		{
		  errs += "Can not instantiate batch link due to confliction on property "+pinfo->name().toString();
		  return false;
		}
		   
	    }
	}

    cinfo = _components.getComponentInfo(int(ComponentList::OUTPUTS));
    if ( cinfo->_batchups )
      for( int i=0; i<cinfo->_batchups->size(); i++)
	{
	  const LBStructInterface *bitfc = LubanSymbolResolver::resolveStructInterface((*cinfo->_batchups)[i]._structtypename, errs);
	  if ( ! bitfc )
	    {
	      errs += "Refered struct interface "+(*cinfo->_batchups)[i]._structtypename.toString()+" does not exist\n";
	      return false;
	    }
	  if ( !_itfc->hasInterface((*cinfo->_batchups)[i]._structtypename ) )
	    {
	      errs += "Refered struct interface "+(*cinfo->_batchups)[i]._structtypename.toString()+" is not part of the current struct interface\n";
	      return false;
	    }
	  int upcellid = (*cinfo->_batchups)[i]._cellid;
	  int numouts = bitfc->numberOfPropertiesByAttr(LBPropertyInfo::OUT);
	  int portid, jointid, itfc_outpidx;
	  for(int j=0; j<numouts; j++)
	    {
	      const LBPropertyInfo *pinfo = bitfc->propertyInfo(j, LBPropertyInfo::OUT);
	      ComponentInfo::TypedUpLink uplink;
	      if ( cinfo->_upmap && cinfo->_upmap->find(pinfo->name(), uplink) )
		{
		  if ( uplink._jointid >= 0 )
		    continue;
		  else
		    {
		      errs += "Conflicting setting of output property "+pinfo->name().toString()+"\n";
		      return false;
		    }
		}
	      const LBPropertyInfo *itfc_outpinfo = _itfc->propertyInfo(pinfo->name(), itfc_outpidx);
	      if ( ! itfc_outpinfo || itfc_outpinfo->attr() != LBPropertyInfo::OUT )
		{
		  errs += "Batch link to a non-existing output property "+pinfo->name().toString();
		  return false;
		}

	      jointid = _joints.getJointIndex(upcellid, pinfo->name());
	      portid = _joints.getPortIndex(jointid, int(ComponentList::OUTPUTS), pinfo->name(), itfc_outpidx);
	      cinfo->upStream(pinfo->name(), -(jointid+1), portid, upcellid ); //negative to indicate it is from batch link
	      _components.recordDownStream(upcellid, pinfo->name(), -(jointid+1), int(ComponentList::OUTPUTS));
	    }
	}

    return true;

  }
  
  static int symbolHash(const LBSymbol& s) 
  { return s.hash(); }

  static bool symbolEqual(const LBSymbol& s1, const LBSymbol& s2) 
  { return s1 == s2; }
  
  LBCompositionStruct::ComponentList::ComponentList(bool asynch)
    : _namedcellmap(symbolHash, symbolEqual, 6 /* 64 buckets */), _complist(), _asynchcells(0), _zeroincells(0), _toporder(-1)
  {
    ComponentInfo *inputs = new ComponentInfo(int(INPUTS), false);
    inputs->_asynch = asynch;
    _complist.push_back(inputs);
    _complist.push_back(new ComponentInfo(int(OUTPUTS), false) );
  }
  
  LBCompositionStruct::ComponentList::~ComponentList()
  {
    for( int i=0; i<_complist.size(); i++)
      delete _complist[i];
    if ( _asynchcells ) delete _asynchcells;
    if ( _zeroincells ) delete _zeroincells;
  }

  int LBCompositionStruct::ComponentList::index(const LBSymbol& compname) const
  {
    int idx=-1;
    _namedcellmap.find(compname, idx);
    return idx;
  }

  int LBCompositionStruct::ComponentList::size() const
  {
    return _complist.size();
  }

  const LBSymbol& LBCompositionStruct::ComponentList::name(int compid) const
  {
    return _complist[compid]->name();
  }

  bool LBCompositionStruct::ComponentList::isAsynch(int compid) const
  {
    return _complist[compid]->_asynch;
  }

  bool LBCompositionStruct::ComponentList::isTyped(int compid) const
  {
    return _complist[compid]->_typed;
  }

  int LBCompositionStruct::ComponentList::topOrder(int compid) const
  {
    return _complist[compid]->_toporder;
  }
  
  const LBStructInterface* LBCompositionStruct::ComponentList::getTypedInterface(int cellid) const
  {
    return _complist[cellid]->typedInterface();
  }

  LBStruct* LBCompositionStruct::ComponentList::getLiveComp(int cellid) const
  {
    LBStruct *st = _complist[cellid]->_str;
    if ( st )
      return static_cast<LBStruct*>(st->clone());
    return 0;
  }
  
  int LBCompositionStruct::ComponentList::addNamedTyped(const LBSymbol& cname, const LBFullSymbol& structtypename, bool threading, string& errs)
  {
    int idx;
    if ( _namedcellmap.find(cname, idx) )
      {
	errs += cname.toString()+" cell already exists in composition\n";
	return -1;
      }
    idx = _complist.size();
    _complist.push_back(new ComponentInfo(cname, structtypename, threading));
    _namedcellmap[cname]=idx;
    return idx;
  }

  int LBCompositionStruct::ComponentList::addNamedAdhoc(const LBSymbol& cname, bool asynch, bool threading, string& errs)
  {
    int idx;
    if ( _namedcellmap.find(cname, idx) )
      {
	errs += cname.toString()+" cell already exists in composition\n";
	return -1;
      }
    idx = _complist.size();
    _complist.push_back(new ComponentInfo(cname, asynch, threading));
    _namedcellmap[cname]=idx;
    return idx;    
  }
  
  int LBCompositionStruct::ComponentList::addNonameAdhoc(bool threading)
  {
    int idx = _complist.size();
    _complist.push_back(new ComponentInfo(idx,threading));
    return idx;    
  }

  int LBCompositionStruct::ComponentList::addNonameTyped(const LBFullSymbol& structtypename, bool threading)
  {
    int idx = _complist.size();
    _complist.push_back(new ComponentInfo(structtypename, threading));
    return idx;    
  }
  
  bool LBCompositionStruct::ComponentList::setAdhocProc(int compid, LBProcessStruct *proc)
  {
    return _complist[compid]->setAdhocProc(proc);
  }
  
  bool LBCompositionStruct::ComponentList::recordDownStream(int cellid,  int downstreamcellid)
  {
    return _complist[cellid]->downStream(downstreamcellid);
  }
  
  bool LBCompositionStruct::ComponentList::recordDownStream(int cellid,  const LBSymbol& outprp, int jointid, int downstreamcellid) 
  {
    return _complist[cellid]->downStream(outprp, jointid, downstreamcellid);
  }
   
  bool LBCompositionStruct::ComponentList::recordUpStream(int cellid, int upstreamcellid)
  {
    return _complist[cellid]->upStream( upstreamcellid);
  }

  bool LBCompositionStruct::ComponentList::recordUpStream(int cellid, const LBSymbol& inputprpname, int jointid, int portid, int upstreamcellid)
  {
    return _complist[cellid]->upStream(inputprpname, jointid, portid,  upstreamcellid);
  }

  bool LBCompositionStruct::ComponentList::recordUpStream(int cellid, const LBFullSymbol& structtypename)
  {
    return _complist[cellid]->upStream(structtypename, int(INPUTS));
  }

  bool LBCompositionStruct::ComponentList::recordDownStream(const LBFullSymbol& structtypename, int downcellid)
  {
    return _complist[int(INPUTS)]->downStream(structtypename, downcellid);
  }

  bool LBCompositionStruct::ComponentList::recordUpStream(const LBFullSymbol& structtypename, int upcellid )
  {
    return _complist[int(OUTPUTS)]->upStream(structtypename, upcellid);
  }

  bool LBCompositionStruct::ComponentList::recordDownStream( int cellid, const LBFullSymbol& structtypename)
  {
    return _complist[cellid]->downStream(structtypename, int(OUTPUTS));
  }
							    
  bool LBCompositionStruct::ComponentList::topsort(string& errs)
  {
    // first gather initial starting nodes, zero in and asynch
    int sz = _complist.size();
    std::deque<int> zeroinq;
    std::vector<int> indegreevec(sz);
    zeroinq.push_back(int(INPUTS));
    indegreevec[int(INPUTS)] = 0;
    _complist[int(INPUTS)]->_toporder = 0;
    for( int i=int(OUTPUTS)+1; i < sz; i++)
      {
	if ( _complist[i]->isAsynch() )
	  {
	    zeroinq.push_back(i);
	    indegreevec[i] = 0;
	    if ( ! _asynchcells )
	      _asynchcells = new std::vector<int>();
	    _asynchcells->push_back(i);
	    _complist[i]->_toporder = 0;
	  }
	else
	  {
	    int ins = _complist[i]->indegree();
	    indegreevec[i]=ins;
	    if ( ins == 0 )
	      {
		zeroinq.push_back(i);
		if ( ! _zeroincells )
		  _zeroincells = new std::vector<int>();
		_zeroincells->push_back(i);
		_complist[i]->_toporder = 0;
	      }
	  }
      }

    // use zero in degree algorithm
    while ( zeroinq.size() )
      {
	int zeroin = zeroinq.front(); zeroinq.pop_front();
	int nextord = _complist[zeroin]->_toporder+1;
	for( int i=0; i<_complist[zeroin]->_downs.size(); i++)
	  {
	    int downcell = _complist[zeroin]->_downs[i];
	    if ( _complist[downcell]->isAsynch() ) // ignore asynch cell
	      continue;
	    if ( --indegreevec[downcell] == 0 )
	      {
		zeroinq.push_back(downcell);
		if ( _complist[downcell]->_toporder < nextord )
		  {
		    _complist[downcell]->_toporder = nextord;
		    if ( _toporder < nextord )
		      _toporder = nextord;
		  }
	      }
	  }
      }

    // check unsorted component
    string cellincycle;
    bool cyclic=false;
    for( int i=int(OUTPUTS)+1; i < sz; i++)
      {
	if ( _complist[i]->_toporder == -1 )
	  {
	    cyclic = true;
	    if ( ! _complist[i]->isAnonymous() )
	      cellincycle += " "+_complist[i]->name().toString();
	  }
      }

    if ( cyclic )
      {
	errs += "Invalid composition, cyclic path involving: "+cellincycle;
	return false;
      }

    return true;

  }

  LBCompositionStruct::ComponentInfo::ComponentInfo(const LBSymbol& cname, const LBFullSymbol& structtypename, bool threading)
    : _name(cname), _typed(true), _anonymous(false), _structtypename(structtypename), _asynch(false), _threading(threading), _toporder(-1), _ups(), _downs(), _downmap(0), _upmap(0), _batchups(0), _batchdowns(0), _str(0),_typeditfc(0), _erred(false), _errmsg()
  {
    string errs;
    _typeditfc = LubanSymbolResolver::resolveStructInterface(_structtypename, errs);
    if ( _typeditfc )
      {
	_downmap = new DownPortMap(symbolHash, symbolEqual, 3);
	_upmap = new UpPortMap(symbolHash, symbolEqual, 3);
	_asynch = _typeditfc->mode() == LBStructInterface::ASYNCH;
      }
    else
      {
	_erred = true;
	_errmsg = errs;
      }
  }

  LBCompositionStruct::ComponentInfo::ComponentInfo(const LBSymbol& cname, bool asynch, bool threading)
    : _name(cname), _typed(false), _anonymous(false), _structtypename(), _asynch(asynch), _threading(threading), _toporder(-1), _ups(), _downs(), _downmap(0), _upmap(0), _batchups(0), _batchdowns(0), _str(0),_typeditfc(0), _erred(false), _errmsg()
  {
  }

  LBCompositionStruct::ComponentInfo::ComponentInfo(int cellid, bool threading)
    : _name(), _typed(false), _anonymous(true), _structtypename(), _asynch(false), _threading(threading), _toporder(-1), _ups(), _downs(), _downmap(0), _upmap(0), _batchups(0), _batchdowns(0), _str(0),_typeditfc(0), _erred(false), _errmsg()
  {
    static const LBSymbol i("input"), o("output");
    if ( cellid == ComponentList::INPUTS )
      {
	_typed = true;
	_downmap = new DownPortMap(symbolHash, symbolEqual, 3);
	_name = i;
      }
    if ( cellid == ComponentList::OUTPUTS )
      {
	_name= o;
	_typed = true;
	_upmap = new UpPortMap(symbolHash, symbolEqual, 3);
      }
    std::ostringstream ost;
    ost<<"anonymous."<<cellid;
    _name=LBSymbol(ost.str());
  }

  LBCompositionStruct::ComponentInfo::ComponentInfo(const LBFullSymbol& structtypename, bool threading)
    : _name(), _typed(true), _anonymous(true), _structtypename(structtypename), _asynch(false), _threading(threading), _toporder(-1), _ups(), _downs(), _downmap(0), _upmap(0), _batchups(0), _batchdowns(0), _str(0),_typeditfc(0), _erred(false), _errmsg()
  {
    _name = LBSymbol(string("output:")+structtypename.toString());
    string errs;
    _typeditfc = LubanSymbolResolver::resolveStructInterface(_structtypename, errs);
    if ( _typeditfc )
      {
	_downmap = new LBHashMap<LBSymbol, int>(symbolHash, symbolEqual, 3);
	_upmap = new LBHashMap<LBSymbol, TypedUpLink>(symbolHash, symbolEqual, 3);
	_asynch = _typeditfc->mode() == LBStructInterface::ASYNCH;
      }
    else
      {
	_erred = true;
	_errmsg = errs;
      }
  }

  const LBSymbol& LBCompositionStruct::ComponentInfo::name() const
  {
    return _name;
  }

  const LBStructInterface* LBCompositionStruct::ComponentInfo::typedInterface() const
  {
      return _typeditfc;
  }

  bool LBCompositionStruct::ComponentInfo::setAdhocProc(LBProcessStruct *proc)
  {
    if ( _erred )
      return false;
    if ( _typed )
      {
	_erred = true;
	_errmsg = "Adhoc process struct being set to typed component cell";
	return false;
      }
    _str = static_cast<LBStruct*>(proc);
    _asynch = proc->interface().mode() == LBStructInterface::ASYNCH;
  }

  bool LBCompositionStruct::ComponentInfo::downStream(int downcellid)
  {
    if ( _erred )
      return false;
    if ( _typed )
      {
	_erred = true;
	_errmsg = "Typed component cell mistaken as adhoc process struct cell";
	return false;
      }
    if ( find( _downs.begin(), _downs.end(), downcellid) != _downs.end() ) // STL stupidness
      return true;
    _downs.push_back(downcellid);
    return true;
  }

  bool LBCompositionStruct::ComponentInfo::downStream(const LBSymbol& outprpname,int jointid, int downcellid)
  {
    if ( _erred )
      return false;
    if ( !_typed )
      {
	_erred = true;
	_errmsg = "Adhoc component cell mistaken as typed struct cell";
	return false;
      }
    if (  find( _downs.begin(), _downs.end(), downcellid) == _downs.end() ) // STL stupidness
      _downs.push_back(downcellid);
    if ( _typeditfc )
      {
	int dumy;
	const LBPropertyInfo *po = _typeditfc->propertyInfo(outprpname,dumy);
	if ( ! po || po->attr() != LBPropertyInfo::OUT )
	  {
	    _erred = true;
	    _errmsg = "Attempted to link non-existing output property "+outprpname.toString();
	    return false;
	  }
      }
    (*_downmap)[outprpname] = jointid;
    return true;
  }


  bool LBCompositionStruct::ComponentInfo::upStream(int upcellid)
  {
    if ( _erred )
      return false;
    if ( _typed )
      {
	_erred = true;
	_errmsg = "Typed component cell mistaken as adhoc process struct cell";
	return false;
      }
    if ( find( _ups.begin(), _ups.end(), upcellid) != _ups.end() ) // STL stupidness
      return true;
    _ups.push_back(upcellid);
    return true;
  }

  bool LBCompositionStruct::ComponentInfo::upStream(const LBSymbol& inprpname,int jointid, int portid, int upcellid)
  {
    if ( _erred )
      return false;
    if ( !_typed )
      {
	_erred = true;
	_errmsg = "Adhoc component cell mistaken as typed struct cell";
	return false;
      }
    if ( find( _ups.begin(), _ups.end(), upcellid) == _ups.end() ) // STL stupidness
      _ups.push_back(upcellid);
    int dumy;
    if ( _typeditfc )
      {
	const LBPropertyInfo *po = _typeditfc->propertyInfo(inprpname,dumy);
	if ( ! po || po->attr() != LBPropertyInfo::IN )
	  {
	    _erred = true;
	    _errmsg = "Attempted to link non-existing input property "+inprpname.toString();
	    return false;
	  }
      }

    // prevent confliction
    TypedUpLink uplk;
    if ( _upmap->find(inprpname, uplk ) )
      {
	if (   jointid == uplk._jointid || jointid < 0 && uplk._jointid >=0 )
	  return true; // batch link won't overwrite explicit link

	_erred = true;
	_errmsg = "Conflict link to input property "+inprpname.toString();
	return false;
      }
    
    (*_upmap)[inprpname] = TypedUpLink(jointid, portid);
    return true;
  }

  bool LBCompositionStruct::ComponentInfo::upStream(const LBFullSymbol& structtypename, int upcellid)
  {
    if ( _erred )
      return false;
    if ( !_typed )
      {
	_erred = true;
	_errmsg = "Adhoc component cell mistaken as typed struct cell";
	return false;
      }
    if ( find( _ups.begin(), _ups.end(), upcellid) == _ups.end() ) // STL stupidness
      _ups.push_back(upcellid);
    if ( ! _batchups )
      _batchups = new std::vector<BatchLink>();
    _batchups->push_back(BatchLink(structtypename, upcellid));
    return true;
  }

  bool LBCompositionStruct::ComponentInfo::downStream(const LBFullSymbol& structtypename, int downcellid)
  {
    if ( _erred )
      return false;
    if ( !_typed )
      {
	_erred = true;
	_errmsg = "Adhoc component cell mistaken as typed struct cell";
	return false;
      }
    if ( find( _downs.begin(), _downs.end(), downcellid) == _downs.end() ) // STL stupidness
      _downs.push_back(downcellid);
    if ( ! _batchdowns )
      _batchdowns = new std::vector<BatchLink>();
    _batchdowns->push_back(BatchLink(structtypename, downcellid));
    return true;
  }

  LBCompositionStruct::ComponentInfo::~ComponentInfo()
  {
    if ( _downmap ) delete _downmap;
    if ( _upmap ) delete _upmap;
    if ( _batchups ) delete _batchups;
    if ( _batchdowns ) delete _batchdowns;
    if ( _str ) delete _str;
  }

  static int djKeyHash(const LBCompositionStruct::DataJointKey& s) 
  { return s.hash(); }

  static bool djKeyEqual(const LBCompositionStruct::DataJointKey& s1, const LBCompositionStruct::DataJointKey& s2) 
  { return s1 == s2; }
  

  LBCompositionStruct::DataJointList::DataJointList()
    : _jointvec(), _jointmap(djKeyHash, djKeyEqual, 9 /* 512 buckets */ )
  {}

  LBCompositionStruct::DataJointList::~DataJointList()
  {
    for(int i=0; i<_jointvec.size(); i++)
      delete _jointvec[i];
  }

  int LBCompositionStruct::DataJointList::getJointIndex(int cellid)
  {
    int idx=-1;
    DataJointKey k(cellid);
    if ( _jointmap.find(k, idx) )
      return idx;
    idx = _jointvec.size();
    _jointvec.push_back(new DataJoint(k));
    _jointmap[k] = idx;
    return idx;
  }

  int LBCompositionStruct::DataJointList::getJointIndex(int cellid, const LBSymbol& outputname)
  {
    int idx=-1;
    DataJointKey k(cellid, outputname);
    if ( _jointmap.find(k, idx) )
      return idx;
    idx = _jointvec.size();
    _jointvec.push_back(new DataJoint(k));
    _jointmap[k] = idx;
    return idx;
  }

  int LBCompositionStruct::DataJointList::getPortIndex(int jointid, int downcellid)
  {
    return _jointvec[jointid]->getPort(downcellid);
  };

  int LBCompositionStruct::DataJointList::getPortIndex(int jointid, int downcellid, const LBSymbol& inprpname, int prpidx)
  {
    return _jointvec[jointid]->getPort(downcellid, inprpname, prpidx);
  };

  int LBCompositionStruct::DataJointList::size() const
  {
    return _jointvec.size();
  }

  const LBCompositionStruct::DataJoint&  LBCompositionStruct::DataJointList::getJoint(int jid) const
  {
    return *_jointvec[jid];
  }

  LBCompositionStruct::DataJoint::DataJoint(const DataJointKey& k)
    : _portvec(), _portmap(djKeyHash, djKeyEqual, 4 /* 16 buckets */ ), _djtkey(k)
  {}

  int LBCompositionStruct::DataJoint::getPort(int downcellid)
  {
    PortKey pk(downcellid);
    int idx = -1;
    if ( _portmap.find(pk, idx) )
      return idx;
    idx = _portvec.size();
    _portvec.push_back(pk);
    _portmap[pk]=idx;
    return idx;
  }

  int LBCompositionStruct::DataJoint::getPort(int downcellid, const LBSymbol& inprpname, int inprpidx)
  {
    PortKey pk(downcellid, inprpname, inprpidx);
    int idx = -1;
    if ( _portmap.find(pk, idx) )
      return idx;
    idx = _portvec.size();
    _portvec.push_back(pk);
    _portmap[pk]=idx;
    return idx;
  }

  LBCompositionStruct::LiveJoint::LiveJoint(const DataJoint& djt, int jointid, LBCompositionStruct *parentcomp)
    : _jointid(jointid), _asynch(false), _parentcomp(parentcomp), _asynchq(0), _asynchqmtx(0), _liveports(djt._portvec.size(), (LivePort*)0), _stop(false)
  {
    _asynch = parentcomp->metadata()->_components.isAsynch(djt._djtkey._cellid);

    bool alldownasynch = true;
    for( int i=0; i<djt._portvec.size(); i++)
      {
	int downcellid = djt._portvec[i]._cellid;
	int downprpid = djt._portvec[i]._prpindex;
	bool downasynch = parentcomp->metadata()->_components.isAsynch(downcellid);
	alldownasynch = alldownasynch && downasynch;
	_liveports[i] = new LivePort(downcellid, downprpid, downasynch);
      }

    if ( _asynch && ! alldownasynch )
      {
	_asynchq = new std::deque<LBObjPtr>();
	_asynchqmtx = new LBMutex();
      }

  }

  LBCompositionStruct::LiveJoint::LiveJoint(const LiveJoint& lj, LBCompositionStruct *parentcomp)
    : _jointid(lj._jointid), _asynch(lj._asynch), _parentcomp(parentcomp), _asynchq(0), _asynchqmtx(0), _liveports(lj._liveports.size(),(LivePort*)0), _stop(false)
  {
    for( int i=0; i<lj._liveports.size(); i++)
      _liveports[i] = new LivePort(*lj._liveports[i]);

    if ( lj._asynchq )
      {
	_asynchq = new std::deque<LBObjPtr>();
	_asynchqmtx = new LBMutex();
      }
  }

  LBCompositionStruct::LiveJoint::~LiveJoint()
  {
    if ( _asynchq ) delete _asynchq;
    if ( _asynchqmtx ) delete _asynchqmtx;
    for( int i=0; i<_liveports.size(); i++)
      delete _liveports[i];
  }

  bool LBCompositionStruct::LiveJoint::setNewValue(const LBObjPtr& valptr)
  {
    if ( _stop ) return false;
    for( int i=0; i<_liveports.size(); i++)
      {
	LivePort *port = _liveports[i];

	// handle the setting of output property first
	if ( port->_downcell == int(ComponentList::OUTPUTS) )
	  {
	    if ( _parentcomp->_runtime->_outs->writeValueByInternal(port->_downprpid, valptr) )
	      continue;
	    return false;
	  }

	if ( port->_asynch )
	  {
	    if ( ! port->isAdhoc() )
	      _parentcomp->_runtime->_livecomps[port->_downcell]->setInput(port->_downprpid, valptr);
	    else
	      {
		{
		  LBMutexLocker lock(*port->_adhocasynchqmtx);
		  port->_adhocasynchq->push_back(valptr);
		  port->_adhocnewdata->broadcast();
		}
		LBProcessStruct *prcstr = dynamic_cast<LBProcessStruct*>(_parentcomp->_runtime->_livecomps[port->_downcell]);
		if ( ! prcstr )
		  throw LBException("Fatal error, adhoc component is not a process struct");
		prcstr->signalNewData();
	      }
	  }
	else
	  {
	    if ( ! _asynch )
	      {
		if ( ! port->isAdhoc() )
		  _parentcomp->_runtime->_livecomps[port->_downcell]->setInput(port->_downprpid, valptr);
		else
		  port->_adhocsynchobj = valptr;
		int toporder = _parentcomp->metadata()->_components.topOrder(port->_downcell);
		_parentcomp->_runtime->_evalque.addCell(port->_downcell, toporder);
	      }
	  }
      }

    if ( _asynch && _asynchq )
      {
	LBMutexLocker lock(*_asynchqmtx);
	_asynchq->push_back(valptr);
	// wake up sync thread
	_parentcomp->_runtime->signalNewData(_jointid);
      }

    return true;
  }
							    
  bool LBCompositionStruct::LiveJoint::setNewValue(const LBObject& val)
  {
    if ( _stop ) return false;
    for( int i=0; i<_liveports.size(); i++)
      {
	LivePort *port = _liveports[i];

	// handle the setting of output property first
	if ( port->_downcell == int(ComponentList::OUTPUTS) )
	  {
	    if ( _parentcomp->_runtime->_outs->writeValueByInternal(port->_downprpid, val) )
	      continue;
	    return false;
	  }

	if ( port->_asynch )
	  {
	    if ( ! port->isAdhoc() )
	      _parentcomp->_runtime->_livecomps[port->_downcell]->setInput(port->_downprpid, val);
	    else
	      {
		LBObjPtr valptr(val.clone());
		{
		  LBMutexLocker lock(*port->_adhocasynchqmtx);
		  port->_adhocasynchq->push_back(valptr);
		  port->_adhocnewdata->broadcast();
		}
		LBProcessStruct *prcstr = dynamic_cast<LBProcessStruct*>(_parentcomp->_runtime->_livecomps[port->_downcell]);
		if ( ! prcstr )
		  throw LBException("Fatal error, adhoc component is not a process struct");
		prcstr->signalNewData();
	      }
	  }
	else
	  {
	    if ( ! _asynch )
	      {
		if ( ! port->isAdhoc() )
		  _parentcomp->_runtime->_livecomps[port->_downcell]->setInput(port->_downprpid, val);
		else
		  port->_adhocsynchobj.assign(val);
		int toporder = _parentcomp->metadata()->_components.topOrder(port->_downcell);
		_parentcomp->_runtime->_evalque.addCell(port->_downcell, toporder);
	      }
	  }
      }

    if ( _asynch && _asynchq )
      {
	LBObjPtr valptr(val.clone());
	{
	  LBMutexLocker lock(*_asynchqmtx);
	  _asynchq->push_back(valptr);
	}
	// wake up sync thread
	_parentcomp->_runtime->signalNewData(_jointid);
      }

    return true;

  }

  void LBCompositionStruct::LiveJoint::asynchFlush()
  {
    if ( ! _asynchq )
      throw LBException("Fatal error, calling asynchFlush function on a synch data joint");

    LBObjPtr val;
    {
      LBMutexLocker lock(*_asynchqmtx);
      val = _asynchq->front();
      _asynchq->pop_front();
    }

    // only take care of the synch down ports
    for( int i=0; i<_liveports.size(); i++)
      {
	LivePort *port = _liveports[i];
	if ( ! port->_asynch )
	  {
	    if ( ! port->isAdhoc() )
	      _parentcomp->_runtime->_livecomps[port->_downcell]->setInput(port->_downprpid, val);
	    else
	      port->_adhocsynchobj = val;
	    int toporder = _parentcomp->metadata()->_components.topOrder(port->_downcell);
	    _parentcomp->_runtime->_evalque.addCell(port->_downcell, toporder);
	  }
      }
  }

  bool LBCompositionStruct::LiveJoint::readValue(int portid, LBObjPtr& val)
  {
    LivePort *port = _liveports[portid];
    if ( ! port->_adhocasynchq )
      {
	val = port->_adhocsynchobj;
	return true;
      }

    if ( _stop ) return false;
    LBMutexLocker lock(*port->_adhocasynchqmtx);
    while ( true )
      {
	if ( port->_adhocasynchq->size()>0 )
	  {
	    val = port->_adhocasynchq->front();
	    port->_adhocasynchq->pop_front();
	    return true;
	  }
	else
	  {
	    port->_adhocnewdata->wait(*port->_adhocasynchqmtx);
	    if ( _stop ) return false;
	  }
      }

    return false; // should never be here
  }

  void LBCompositionStruct::LiveJoint::dismissThreads()
  {
    _stop = true;
    for( int i=0; i<_liveports.size(); i++)
      {
	LivePort *port = _liveports[i];
	if ( port->_adhocnewdata )
	  port->_adhocnewdata->broadcast();
      }
  }


  LBCompositionStruct::LiveJoint::LivePort::LivePort(int downcell, int prpidx, bool asynch)
    : _downcell(downcell), _downprpid(prpidx), _asynch(asynch),_adhocasynchq(0), _adhocasynchqmtx(0), _adhocnewdata(0),_adhocsynchobj()
  {
    if ( _asynch && _downprpid < 0 )
      {
	_adhocasynchq = new std::deque<LBObjPtr>();
	_adhocasynchqmtx = new LBMutex();
	_adhocnewdata = new LBCondVar();
      }
  }

  LBCompositionStruct::LiveJoint::LivePort::LivePort(const LBCompositionStruct::LiveJoint::LivePort& lp )
    : _downcell(lp._downcell), _downprpid(lp._downprpid), _asynch(lp._asynch),_adhocasynchq(0), _adhocasynchqmtx(0), _adhocnewdata(0),_adhocsynchobj(lp._adhocsynchobj )
  {
    if ( lp._adhocasynchq )
      {
	_adhocasynchq = new std::deque<LBObjPtr>();
	_adhocasynchqmtx = new LBMutex();
	_adhocnewdata = new LBCondVar();
      }
  }

  LBCompositionStruct::LiveJoint::LivePort::~LivePort()
  {
    if ( _adhocasynchq )
      {
	delete _adhocasynchq;
	delete _adhocasynchqmtx;
	delete _adhocnewdata;
      }
  }

  LBCompositionStruct::ExternalStructSymbolTable::ExternalStructSymbolTable()
    : _syms(), _cids()
  {
  }

  void LBCompositionStruct::ExternalStructSymbolTable::addSymbol(const LBFullSymbol& structtypename, int componentid)
  {
    _syms.push_back(structtypename);
    _cids.push_back(componentid);
  }

  int LBCompositionStruct::ExternalStructSymbolTable::size()
  {
    return _syms.size();
  }

  const LBFullSymbol&  LBCompositionStruct::ExternalStructSymbolTable::symbolName(int i)
  {
    return _syms[i];
  }

  int LBCompositionStruct::ExternalStructSymbolTable::cellId(int i)
  {
    return _cids[i];

  }

  LBCompositionStruct::InputPropertyToDataJointMap::InputPropertyToDataJointMap(int numins)
    : _inprp2jid(numins, -1), _unlinked(0)
  {
  }
 
  int LBCompositionStruct::InputPropertyToDataJointMap::getInputJointid(int inprpindex)
  {
    return _inprp2jid[inprpindex];
  }

  void LBCompositionStruct::InputPropertyToDataJointMap::setMap(int inprpindex, int jointid)
  {
    if ( jointid < 0 )
      _unlinked++;
    _inprp2jid[inprpindex]=jointid;
  }

  int LBCompositionStruct::InputPropertyToDataJointMap::numUnlinked()
  {
    return _unlinked;
  }

  int LBCompositionStruct::InputPropertyToDataJointMap::size()
  {
    return _inprp2jid.size();
  }


  LBCompositionStruct::LBCompositionStructRunTime::LBCompositionStructRunTime(int ins, int storeprps, int joints, int comps, int toplevel, bool asynch)
    : _livejoints(joints,(LiveJoint*)0), _livecomps(comps,(LBStruct*)0), _evalque(toplevel, comps), _zeroinsEvaled(false), _ins(0), _stores(0), _outs(0), _asynchproxy(0), _parentcomp(0)
  {
    if ( ins && !asynch )
	_ins = new LBStructInPadLocal(ins);

    if ( storeprps )
      _stores = new LocalPropertyStorage(storeprps, true);

  }

  LBCompositionStruct::LBCompositionStructRunTime::LBCompositionStructRunTime(const LBCompositionStruct::LBCompositionStructRunTime& rt, LBCompositionStruct * parentcomp)
    : _livejoints(rt._livejoints.size(),0), _livecomps(rt._livecomps.size(),0), _evalque(rt._evalque.toplevel(),rt._livecomps.size() ), _zeroinsEvaled(rt._zeroinsEvaled), _ins(0), _stores(0), _outs(0), _asynchproxy(0), _parentcomp(0)
  {
    int i=0;
    for(; i<_livejoints.size(); i++)
      _livejoints[i] = new LiveJoint(*rt._livejoints[i], parentcomp);
    for(i=int(LBCompositionStruct::ComponentList::OUTPUTS)+1; i<_livecomps.size(); i++)
      {
	_livecomps[i] = static_cast<LBStruct*>(rt._livecomps[i]->clone());
	_livecomps[i]->setRunTimeParentComposition(parentcomp);
      }
    if ( rt._ins )
      _ins = rt._ins->clone();
    if ( rt._outs )
      _outs = rt._outs->clone();
    if ( rt._stores )
      _stores = new LocalPropertyStorage(*rt._stores);
    if ( rt._asynchproxy )
      _asynchproxy = new AsynchThreadsNDataProxy(*rt._asynchproxy);
  }

  LBCompositionStruct::LBCompositionStructRunTime::~LBCompositionStructRunTime()
  {
    int i=0;
    for(; i<_livecomps.size(); i++)
      delete _livecomps[i];
    for(i=0; i<_livejoints.size(); i++)
      delete _livejoints[i];

    if (_ins ) delete _ins;
    if ( _outs ) delete _outs;
    if ( _stores ) delete _stores;
    if ( _asynchproxy ) delete _asynchproxy;

  }

  void LBCompositionStruct::LBCompositionStructRunTime::signalNewData(int jointid)
  {
    _asynchproxy->informNewData(jointid);
  }

  LBCompositionStruct::CompositionEvalQueue::CompositionEvalQueue(int toplevel, int numofcomps)
    : _cqvec(toplevel+1), _cursor(0), _compsinq(numofcomps,false)
  {}

  void LBCompositionStruct::CompositionEvalQueue::addCell(int cellid, int toplevel)
  {
    if ( ! _compsinq[cellid] )
      {
	_cqvec[toplevel].push_back(cellid);
	_compsinq[cellid] = true;
      }
  }

  bool LBCompositionStruct::CompositionEvalQueue::nextToEval(int& cellid)
  {
    while( _cursor<_cqvec.size() && _cqvec[_cursor].size() == 0 ) _cursor++;
    if ( _cursor >= _cqvec.size() )  return false;
    cellid = _cqvec[_cursor].front();
    _cqvec[_cursor].pop_front();
    _compsinq[cellid] = false;
    return true;
  }

  void LBCompositionStruct::CompositionEvalQueue::resetCursor()
  {
    _cursor=0;
  }

  int LBCompositionStruct::CompositionEvalQueue::toplevel() const
  {
    return _cqvec.size();
  }


  LBCompositionStruct::AsynchThreadsNDataProxy::AsynchThreadsNDataProxy(int sz)
    : _jointid2idx(sz,-1),_idx2jointid(), _datacounts(), _totalcount(0), _mainwaiting(false), _mtx(), _newdata(), _actives(0), _allgohome(false)
  {}

  LBCompositionStruct::AsynchThreadsNDataProxy::AsynchThreadsNDataProxy(const  LBCompositionStruct::AsynchThreadsNDataProxy& jinfo)
    : _jointid2idx(jinfo._jointid2idx),_idx2jointid(jinfo._idx2jointid), _datacounts(jinfo._idx2jointid.size(),0), _totalcount(0), _mainwaiting(false), _mtx(), _newdata(), _actives(0), _allgohome(false)
  {}

  bool LBCompositionStruct::AsynchThreadsNDataProxy::addAsynchJoint(int jointid)
  {
    _jointid2idx[jointid] = _datacounts.size();
    _datacounts.push_back(0);
    _idx2jointid.push_back(jointid);
    return true;
  }

  bool LBCompositionStruct::AsynchThreadsNDataProxy::hasNewData()
  {
    LBMutexLocker lock(_mtx);
    if ( _allgohome ) return false;
    return _totalcount > 0;
  }

  bool LBCompositionStruct::AsynchThreadsNDataProxy::hasNewData(int idx)
  {
    LBMutexLocker lock(_mtx);
    if ( _allgohome ) return false;
    return _datacounts[idx]>0;
  }

  void LBCompositionStruct::AsynchThreadsNDataProxy::informDataConsumed(int idx)
  {
    LBMutexLocker lock(_mtx);
    _datacounts[idx]--;
    _totalcount--;
    if ( _totalcount == 0 && _mainwaiting )
      _nodata.broadcast();
  }

  void LBCompositionStruct::AsynchThreadsNDataProxy::informNewData(int jointid)
  {
    int idx = _jointid2idx[jointid];
    {
      LBMutexLocker lock(_mtx);
      _datacounts[idx]++;
      _totalcount++;
      _newdata.signal();
    }
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::waitForNewData()
  {
    if ( _allgohome ) return 0;
    LBMutexLocker lock(_mtx);
    if ( _allgohome ) return 0;
    while ( _totalcount == 0 )
      {
	_mainwaiting = true;
	_nodata.broadcast();
	_newdata.wait(_mtx);
	_mainwaiting = false;
	if ( _allgohome ) return 0;
      }
    if ( _allgohome ) return 0;
    return _totalcount;
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::waitForNewDataIfAnyActive()
  {
    if ( _allgohome ) return 0;
    LBMutexLocker lock(_mtx);
    if ( _allgohome ) return 0;
    while ( _totalcount == 0 && _actives != 0 )
      {
	_mainwaiting = true;
	_nodata.broadcast();
	_newdata.wait(_mtx);
	_mainwaiting = false;
	if ( _allgohome ) return 0;
      }
    if ( _allgohome ) return 0;
    return _totalcount;
  }

  void LBCompositionStruct::AsynchThreadsNDataProxy::waitUntilNoData()
  {
    if ( _allgohome ) return;
    LBMutexLocker lock(_mtx);
    if ( _allgohome ) return;
    if ( _totalcount == 0 && _actives == 0 && _mainwaiting )
      return;
    _nodata.wait(_mtx);
    return;
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::liveJointIndex(int idx ) const
  {
    return _idx2jointid[idx];
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::asynchJoints() const
  {
    return _idx2jointid.size();
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::numActive()
  {
    LBMutexLocker lock(_mtx);
    return _actives;
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::incActive()
  {
    LBMutexLocker lock(_mtx);
    return ++_actives;
  }

  int LBCompositionStruct::AsynchThreadsNDataProxy::decActive()
  {
    LBMutexLocker lock(_mtx);
    if (  --_actives == 0 )
	_newdata.broadcast(); // wake up the new data dreamers, the main thread 

    return _actives;
  }

  void LBCompositionStruct::AsynchThreadsNDataProxy::dismissAll()
  {
    LBMutexLocker lock(_mtx);
    _allgohome = true;
    _newdata.broadcast(); // wake up all the new data dreamers, 
    _nodata.broadcast();
  }


  bool LBCompositionStruct::AsynchMainThread::start()
  {
    LBMutexLocker mlock(_initmtx);

    if ( ! _runningthread )
      {
	_runningthread = new LBThread(&_dsp);
	if ( _runningthread && _runningthread->isValid() )
	  {
	    _dsp._st = Dispatch::RUNNING;
	    if ( _dsp._compst->_runtime && _dsp._compst->_runtime->_parentcomp ) 
	      _dsp._compst->_runtime->_parentcomp->informComponentAct(true);
	    return true;
	  }
	_dsp._st = Dispatch::UNRUNABLE;
	_dsp._errmsg = "Can not start thread";
	return false;
      }
    return _runningthread->isValid() && _dsp._st == Dispatch::RUNNING;
  }

  bool LBCompositionStruct::AsynchMainThread::isdead() const
  {
    return _dsp._st == Dispatch::ENDED || _dsp._st == Dispatch::UNRUNABLE ;
  }

  const string& LBCompositionStruct::AsynchMainThread::errmsg() const
  {
    return _dsp._errmsg;
  }
  
  bool LBCompositionStruct::AsynchMainThread::join()
  {
    if ( _runningthread && _runningthread->isValid() )
      return _runningthread->join();
    return false;
  }

  LBCompositionStruct::AsynchMainThread::~AsynchMainThread()
  {
    if ( _runningthread ) delete _runningthread;
  }

  void LBCompositionStruct::AsynchMainThread::Dispatch::run()
  { 
    bool runok=true;
    string errs;
    try { 
      runok = _compst->asynchCompMainLoop(errs);
    }
    catch ( LBException& lbe)
      {
	_errmsg = lbe.msg();
	_st = UNRUNABLE;
      }
    catch (...)
      {
	_st = UNRUNABLE;
	_errmsg="Unknown exception in struct evaluation";
      }
    if ( ! runok )
      _errmsg="Asynch struct internal thread has ended abnormally: "+errs;
    else
      _errmsg="Asynch struct internal main thread ended, struct no longer active";
    _st = ENDED;
    _compst->throwError(_errmsg, false);
  }

  //The below to export member functions for Luban's reflection mechanism
  // actually functions defined at lbstruct level
  LBEXPORT_MEMBER_FUNC(Luban::LBCompositionStruct, luban_set, "pset", "void set(map kvpairss),set(string key, object value),set(vector keys, vector values)" ); 
  LBEXPORT_MEMBER_FUNC(Luban::LBCompositionStruct, luban_get, "pget", "object get(string key),vector get(vector keys)" ); 
  // to handle asynch struct internal thread
  LBEXPORT_MEMBER_FUNC(Luban::LBCompositionStruct, luban_start, "start", "null start() start internal thread of an asynch struct");


}
