#include <string>
#include <iostream>

#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/lbvector.hpp"

#include "lbtypes/lbset.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbbool.hpp"

namespace Luban
{
  using std::string;

  LBDEFINE(Luban::LBSet, 1, 0)

    //static, this one must be implemented, and at least cover default constructor
  LBSet* LBSet::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() == 0 )
      return new LBSet();
    
    switch ( args->numArgs() ) {
    case 1:
      {
	const LBObject* arg0 = args->getArg(0);
	// export copy constructor
	if ( arg0 )
	  {
	    const LBSet *vp = dynamic_cast<const LBSet*>(arg0);
	    if ( vp )
	      return vp->clone();
	  }
	return 0;
	break;
      }
    default:
      ;
    }
    
    return 0;
  }


  LBSet::LBSet() 
    : _imp( )
  {
  }

  string LBSet::toString() const
  {
    if ( size() )
      {
	ISet::Iterator it(_imp->_realset);
	string s ="{";
	bool gogo = it.next();
	while ( gogo )
	  {
	    s += (*it.currentKey()).toString();
	    if ( gogo = it.next() ) s += ',';
	  }
	s += '}';
	return s;
      }
    static const string emptyset("{}");
    return emptyset;
  }

  ostream& LBSet::toStream(ostream& ost) const
  {
    int len = size();
    ost << len << '\n';
    if ( len )
      {
	ISet::Iterator it(_imp->_realset);
	while( it.next() )
	  if ( *it.currentKey() )
	    {
	      ost.put('1'); ost.put(' ');
	      LBObject::instanceToStream(ost, *(*it.currentKey()) );
	    }
	  else
	    {
	      ost.put('0'); ost.put(' ');
	    }
      }
    return ost;
  }

  istream& LBSet::fromStream(istream& ist,  int major, int minor)
  {
    int length=-1;
    ist>>length;
    char slashn=ist.get();
    if ( ! ist || length < 0 || slashn != '\n'  ) 
      throw LBException("Corrupted stream, error recovering number of elements for a set from stream");
    if ( length ==0 )
      {
	clear();
	return ist;
      }

    LBSetImp *newimp = new LBSetImp();
    string err;
    int j=0;
    for(;  j < length && ist; j++ )
      {
	char onezero, spchar;
	ist.get(onezero); ist.get(spchar);
	if ( spchar != ' ' || onezero != '1' && onezero != '0' )
	  throw LBException("Corrupted stream, invalid delimiter for set stream");
	LBObject *key = 0;
	if ( onezero == '1' )
	  {
	    key = LBObject::instanceFromStream(ist,err);
	    if ( key == 0 )
	      {
		delete newimp;
		throw LBException( string("Error retrieving key object from stream for set, error: ")+string(err));
	      }
	  }
	LBObjPtr pkey(key);
	newimp->_realset.insert(pkey) ;
      }
    if ( !ist || j < length )  // premature exit of above loop
      {
	delete newimp;
	throw LBException( string("Failed to retrieve all objects from stream for set, error: ")+string(err));
      }

    _imp = LBSetImpPtr(newimp);

    return ist;
  }

  // the below two defines that two sets are equal only when they have the same guts
  LBDEFAULT_EQUALS_FUNC(Luban::LBSet)

  bool LBSet::operator==(const LBSet& x) const  
  {
    if ( _imp == x._imp ) return true ; 
    if ( size() == 0 && x.size() == 0 ) return true;
    if ( size() != x.size() ) return false;

    ISet::Iterator it(_imp->_realset);
    while( it.next() )
      {
	if ( ! x._imp->_realset.find(*it.currentKey()) )
	  return false;
      }

    return true;


  }

  bool LBSet::cast(LBObject *casttotarget) const
  {
    // can cast to vector and set
    LBVector *v = dynamic_cast<LBVector*>(casttotarget);
    if ( v )
      {
	v->clear();
	// for performance purpose, use the friend privilege to pass shared ref counted through pointer
	ISet::Iterator it(_imp->_realset);
	while( it.next() )
	    v->push( *it.currentKey() );

	return true;
      }

    // here is for casting to set, only for keys

    return false;
  }

  LBSet& LBSet::add(const LBObject& m)
  {
    const LBSet* mp = dynamic_cast<const LBSet*>(&m);
    if ( mp )
      {
	if ( mp->size() == 0 )
	  return *this;
	if ( size() == 0 )
	  {
	    _imp = mp->_imp;
	    return *this;
	  }
	      
	unicopy(_imp); // copy on write
	ISet::Iterator it(mp->_imp->_realset);
	while( it.next() )
	    _imp->_realset.insert( *it.currentKey());
	return *this;
      }

    throw LBException("Can not add set with type "+m.getType().toString());
    
  }

  LBSet& LBSet::sub(const LBObject& m)
  {
    const LBSet* mp = dynamic_cast<const LBSet*>(&m);
    if ( mp )
      {
	if ( mp->size() == 0 || size() == 0 )
	  return *this;
	unicopy(_imp); // copy on write
	ISet::Iterator it(mp->_imp->_realset);
	while( it.next() )
	    _imp->_realset.erase( *it.currentKey());
	return *this;
      }

    throw LBException("Can not minus set with type "+m.getType().toString());
    
  }

  // simple comparison of size
  bool LBSet::before(const LBObject& m) const
  {
    const LBSet* mp = dynamic_cast<const LBSet*>(&m);
    if ( mp )
      return size() < mp->size();
    throw LBException("Can not compare set with type "+m.getType().toString());
  }

  int LBSet::size() const
  {  return _imp ? _imp->_realset.size():0; }
  
  class LBSet::LBSetIterator : public LBConstIterator
  {
  public:
    LBSetIterator(const LBSet& m)
      :_pad(0), _end(true), _it(0 ), _row(&_pad, 1)
    {
      if ( m.size() )
	_it = new ISet::Iterator(m._imp->_realset);
    }
    ~LBSetIterator()
    { if ( _it ) delete _it; }
    bool next() 
    { 
      if ( _it && _it->next() )
	{
	  _end = false;
	  _pad = _it->currentKey()->operator->();
	  return true;
	}
      _end = true;
      return false;
    }
    const LBVarArgs* getCurrentRow() 
    { 
      if ( ! _end ) 
	return &_row; 
      return 0; 
    }
  private:
    const LBObject *_pad;
    bool _end;
    ISet::Iterator *_it;
    LBSimpleVarArgs _row;
  };

  LBConstIterator* LBSet::getIterator() const
  {   return new LBSetIterator(*this); }

  bool LBSet::contains(const LBObject* key) const
  {
    if ( size() == 0 ) return false;
    LBObjPtr ktemp( const_cast<LBObject*>(key) );
    bool result = _imp->_realset.find(ktemp);
    ktemp.release();
    return result;
  }

  bool LBSet::insert(const LBObject* k)
  {
    uniGuts(_imp); // copy on write
    LBObjPtr kp(k?k->clone():0);
    return _imp->_realset.insert(kp);
  }

  bool LBSet::insert(LBObject* k)
  {
    uniGuts(_imp); // copy on write
    LBObjPtr kp( k );    
    if ( _imp->_realset.insert(kp) )
      return true;
    kp.release();
    return false;
  }

  //private
  bool LBSet::insert(const LBObjPtr& lp)
  {
    uniGuts(_imp); // copy on write

    return _imp->_realset.insert(lp);

  }

  void LBSet::clear()
  {
    if ( size() == 0 )
      return;
    _imp = LBSetImpPtr();
  }

  bool LBSet::erase(const LBObject* key)
  {
    unicopy(_imp);
    LBObjPtr kp( const_cast<LBObject*>(key) );
    bool d=_imp->_realset.erase(kp);
    kp.release();
    return d;
  }


  LBEXPORT_MEMBER_FUNC(Luban::LBSet, luban_insert, "insert", "bool set::insert(object key)" ); 
  LBObject* LBSet::luban_insert(const LBVarArgs *args)
  {
    if ( args && ( args->numArgs()==1) )
      {
	const LBObject *key = args->getArg(0);
	bool res = insert(key);
	return new LBBool(res);
      }
    throw LBException(string("Invalid number of args passed to set::insert(object key)"));
  }

  LBEXPORT_MEMBER_FUNC(Luban::LBSet, luban_remove, "remove", "bool set::remove(object key)" ); 
  LBObject* LBSet::luban_remove(const LBVarArgs *args)
  {
    if ( args && ( args->numArgs()==1) )
      {
	const LBObject *key = args->getArg(0);
	bool res = erase(key);
	return new LBBool(res);
      }
    throw LBException(string("Invalid number of args passed to set::remove()"));
  }

    

}
