#ifndef __LBOBJECT_HPP__
#define __LBOBJECT_HPP__

#include <string>
#include <iostream>
#include "lbtypes/lbtypeinfo.hpp"

namespace Luban
{
  using std::string;
  using std::ostream;
  using std::istream;

  class LBSymbol;
  class LBVarArgs;
  class LBConstIterator;
  class LBMap;

  class LBObject
  {
  public:
    // no constructors and one destructor just to make the whole hiearchy destructors  virtual
    virtual ~LBObject() {}

    // interfaces which all LB data types must implement, REQUIRED

    // below five functions have default declaration and definition in LBDeclareMarcos.hpp
    // and LBDefineMacros.hpp
    virtual LBObject* clone() const = 0;
    virtual bool assign(const LBObject& lbo) = 0;
    virtual const LBTypeInfo& getType() const = 0;
    // memberFuncCall covers all functions which are not accessible by virtual funcs
    virtual LBObject* exportedMemberFunc(const LBSymbol& funcname, const LBVarArgs* lbargs) = 0;
    virtual LBMap* allMemberFuncs() const = 0;

    // the below are required to be implemented by developer
    virtual string toString() const = 0;   // for human readable printing
    virtual ostream& toStream(ostream& o) const = 0 ;  // for persistence and network transportation
    // major.minor is the version number of the object IN THE STREAM
    // which are passed in in case it is needed for backward compatible streaming
    virtual istream& fromStream(istream& i, int major=-1, int minor=-1) = 0;     // for persistence and network transportation
    virtual bool equals(const LBObject& another) const = 0;  


    // optional,
    // interfcaes for common Luban type operations
    // optional; if not implemented it will throw non-applicable operations exception when called 
    virtual int hash() const ;
    // return true if succeed to assign the casted value to the target, else return false and leave target unchanged
    virtual bool cast(LBObject* casttotarget ) const; // cast self and assign to the target, return false if fails
    virtual LBObject& add(const LBObject& lbo); // return reference to self
    virtual LBObject& sub(const LBObject& lbo); // return reference to self
    virtual LBObject& mul(const LBObject& lbo); // return reference to self
    virtual LBObject& div(const LBObject& lbo); // return reference to self
    virtual LBObject& mod(const LBObject& lbo); // return reference to self
    virtual LBObject& neg(); // return reference to self
    // post or prefix are handled at Luban code generator level, below two are prefix
    virtual LBObject& plusplus(); // return reference to self
    virtual LBObject& minusminus(); // return reference to self
    // below is for consistent < > == comparison for the same or compatible type
    virtual bool      before(const LBObject& lbo) const; // throw exception if non-comparable

    // optional, below are container related functions
    virtual void subscriptSet(const LBObject* index, const LBObject* value); // this is for a[b] = c;
    virtual LBObject* subscriptGet(const LBObject* index); // for a[b].func(c); return a pointer to internal data
    virtual const LBObject* subscriptConstGet(const LBObject* index) const; // for c = a[b]; return a const pointer to internal data
    virtual int size() const;
    virtual LBConstIterator* getIterator() const; // for "foreach( x in y )", return 0 by default. Implement it for container type
    virtual bool contains(const LBObject* x) const;
    virtual void clear(); // erase all elements




    // Below are  functions IMPLEMENTED at LBObject level

    // this is for general comparison of two lbobjects. It behaves like strcmp
    // returns 1, 0 or -1
    // 0: obj1 obj2 same order ( means either equal or undefined order ),   
    // 1: obj1 (after) obj2; -1: obj1(before)obj2
    // This is only intended to be used for sorting a container which 
    // contains mixed types of objects , not > < == comparison
    static int objcmp(const LBObject* obj1, const LBObject* obj2);


    // official serialize/de-serialize interface for all LBObjects
    // It adds meta data like typeinfo on top of the datas streamed 
    // by object's to/fromStream virtual functions
    static LBObject* instanceFromStream(istream& i, string &returnerrmsg);
    static ostream& instanceToStream(ostream& o, const LBObject& lbo);

    // this is a non-virtual function to intercept member function call from Luban
    // the purpose is to export the basic virtual functions for luban to call
    LBObject* memberFuncCall(const LBSymbol& funcname, const LBVarArgs* lbargs);

  protected:
    // LBTypeInfo system utility, this function is responsible to construct typeinfo
    // so the typeinfo's constructor can be private for safe
    static LBTypeInfo *creatLBTypeInfo( const type_info& cppt, const char* cppname,  LBTypeInfo::LBConstructor lbc, int major, int minor);

  };

  ostream& operator << (ostream&, const LBObject& lbo); // this is for human readable print out

}
#endif
