#ifndef __REFCOUNTEDPTR_HPP__
#define __REFCOUNTEDPTR_HPP__

#include "lbtypes/AtomicInt.hpp"  // this is for multi-thread safety

namespace Luban
{

  template< class T > class RefCountedPtrImp
  {
  public:
    AtomicInt _refs;
    T *_realptr;
  public:
    explicit RefCountedPtrImp(T* ptr) : _refs(1), _realptr(ptr) {}
    ~RefCountedPtrImp() { if ( _realptr ) delete _realptr; }
  };

  template<typename T> class RefCountedPtr;
    
  template <typename T> void unicopy(RefCountedPtr<T> &imp);
  template <typename T> void uniclone(RefCountedPtr<T> &imp);
  template <typename T> void uniorzero(RefCountedPtr<T> &imp);

  template< class T > class RefCountedPtr
  {
  public:
    explicit RefCountedPtr(T* ptr=0 );

    RefCountedPtr(const RefCountedPtr&);

    ~RefCountedPtr();

    RefCountedPtr& operator = ( const RefCountedPtr& );

    bool operator==(const RefCountedPtr& x) const;

    T* operator->();
    const T* operator->() const;

    T& operator*();
    const T& operator*() const;

    operator bool() const;

    bool isShared() const;
    T* getRealPtr();
    const T* getConstRealPtr() const;
    T* setRealPtr(T* realptr);
    T* release();
    
    friend void unicopy<>(RefCountedPtr<T> &imp);
    friend void uniclone<>(RefCountedPtr<T> &imp);
    friend void uniorzero<>(RefCountedPtr<T> &imp);

  protected:

    RefCountedPtrImp<T> *_imp;

  };



  template< class T >  RefCountedPtr<T>::RefCountedPtr(T* ptr)
    : _imp(0)
  {    if ( ptr ) _imp = new RefCountedPtrImp<T>(ptr); }

  template< class T >  RefCountedPtr<T>::RefCountedPtr(const RefCountedPtr& p)
    : _imp(0)
  {
    if ( p._imp && p._imp->_realptr )
      {
	p._imp->_refs.inc();
	_imp = p._imp;
      }
  }
    
  template< class T >  RefCountedPtr<T>::~RefCountedPtr()
  { if ( _imp && _imp->_refs.dec() == 0 ) delete _imp; }

  template<class T> RefCountedPtr<T>& RefCountedPtr<T>::operator = ( const RefCountedPtr& x )
  {
    if ( _imp == x._imp )
      return *this;
    if ( x._imp && x._imp->_realptr )
      {
	x._imp->_refs.inc();
	if ( _imp && _imp->_refs.dec() == 0 ) delete _imp;
	_imp = x._imp;
	return *this;
      }
    if ( ! _imp )
      return *this;
    if ( _imp->_refs.dec() < 1 ) 
      delete _imp;
    _imp = 0;
    return *this;
  }

  template<class T> bool RefCountedPtr<T>::operator==(const RefCountedPtr& x) const
  {
    // if both are zero pointer or pointing to the same thing, they are equal
    // not equal otherwise
    return( _imp == x._imp ) || ( ! _imp || ! _imp->_realptr ) &&  ( ! x._imp || ! x._imp->_realptr );
  }

  template<class T>  T* RefCountedPtr<T>::operator->() 
  { return _imp?_imp->_realptr:0; }

  template<class T>  const T* RefCountedPtr<T>::operator->() const
  { return _imp?_imp->_realptr:0; }

  template<class T>  T& RefCountedPtr<T>::operator*() 
  { return *_imp->_realptr; }

  template<class T>  const T& RefCountedPtr<T>::operator*() const
  { return *_imp->_realptr; }

  template<class T>  RefCountedPtr<T>::operator bool() const
  { return _imp && _imp->_realptr; }

  template<class T> bool RefCountedPtr<T>::isShared() const
  { return _imp && _imp->_refs.compareTo(1) > 0 ; }
  
  template<class T>  const T* RefCountedPtr<T>::getConstRealPtr()  const
  { return _imp?_imp->_realptr:0; }

  template<class T>  T* RefCountedPtr<T>::getRealPtr() 
  { return _imp?_imp->_realptr:0; }

  template<class T>  T* RefCountedPtr<T>::setRealPtr(T* newptr) 
  { 
    if ( _imp )
      {
	T *temp = _imp->_realptr;
	_imp->_realptr = newptr;
	return temp;
      }
    if ( newptr ) 
      _imp = new RefCountedPtrImp<T>(newptr);
    return 0;
  }
  
  template<class T>  T* RefCountedPtr<T>::release() 
  {
    if ( _imp )
      {
	T* temp = _imp->_realptr;
	_imp->_realptr = 0;
	return temp;
      }

    return 0;

  }


  // this function assume T has a virtual function called clone() to do copy
  template< class T > void uniclone( RefCountedPtr<T> &ptr )
  {
    if ( ! ptr._imp || ! ptr._imp->_realptr) return;
    if ( ptr._imp->_refs.compareTo(1) > 0 )
      {
	T* newptr = ptr._imp->_realptr->clone();
	if ( ptr._imp->_refs.dec() < 1 )
	  {
	    delete newptr;
	    ptr._imp->_refs.set(1);
	    return;
	  }
	ptr._imp = new RefCountedPtrImp<T>(newptr);
      }
  }

  // this function assume T has a copy constructor
  template< class T > void unicopy( RefCountedPtr<T> &ptr )
  {
    if ( ! ptr._imp || ! ptr._imp->_realptr) return;
    if ( ptr._imp->_refs.compareTo(1) > 0 )
      {
	T* newptr = new T(*ptr._imp->_realptr);
	if ( ptr._imp->_refs.dec() < 1 )
	  {
	    delete newptr;
	    ptr._imp->_refs.set(1);
	    return;
	  }
	ptr._imp = new RefCountedPtrImp<T>(newptr);
      }
  }

  template< class T > void uniorzero( RefCountedPtr<T> &ptr )
  {
    if ( ! ptr._imp || ! ptr._imp->_realptr) return;
    if ( ptr._imp->_refs.compareTo(1) > 0 )
      {
	if ( ptr._imp->_refs.dec() < 1 )
	  ptr._imp->_refs.set(1);
	else
	  ptr._imp = 0;
      }
  }


}

#endif
