#ifndef __LBPROCESSSTRUCT_HPP__
#define __LBPROCESSSTRUCT_HPP__

#include <vector>
#include <string>

#include "luban/lbstruct.hpp"
#include "luban/lbstructinterface.hpp"
#include "luban/lbvmcode.hpp"
#include "luban/lbstructiopad.hpp"
#include "luban/lbvectorstruct.hpp"
#include "luban/lbrwlockedobjptr.hpp"

#include "lbtypes/LBDeclareMacros.hpp"
#include "lbtypes/RefCountedPtr.hpp"
#include "lbtypes/HashUtil.hpp"
#include "lbtypes/lbiterator.hpp"
#include "lbtypes/LBObjPtr.hpp"
#include "lbtypes/lbsymbol.hpp"
#include "lbtypes/lbfullsymbol.hpp"
#include "lbtypes/lbexception.hpp"


#include "lbthread/lbthread.hpp"

namespace Luban
{

  class LBTypeSpec;
  class LBCompositionStruct;
  class LocalPropertyStorage;
  class CounterWaiter;
  using std::string;

  class LBProcessStruct : public LBStruct
  {
    LBDECLARE(Luban::LBProcessStruct);
  public:
    ~LBProcessStruct();
    LBProcessStruct();
    LBProcessStruct(LBStructInterface* itfc);
    LBProcessStruct(const LBProcessStruct& p);

    // used by composition struct code generation
    LBProcessStruct(const LBSymbol& selfname, bool asynch, LBCompositionStruct *comps);
    LBProcessStruct(int cellid, LBCompositionStruct *comps);

    LBProcessStruct& operator=(const LBProcessStruct& p); // only throw exception 

    // LBObject required interfaces
    string toString() const;
    ostream& toStream( ostream& o ) const ;
    istream& fromStream( istream& i, int major=-1, int minor=-1 );
    bool equals(const LBObject& another) const;

    // LBStruct required interfaces
    const LBStructInterface& interface() const;
    LBProcessStruct* structCall(const LBNamedArgs* args=0) const;
    void setBatchProperties(const LBNamedArgs* args=0);

    LBObjPtr getPropertyValue(const LBSymbol& s);
    LBObject* getPropertyPtr(const LBSymbol& s);
    void touchProperty(const LBSymbol& s);
    void setProperty(const LBSymbol& s, const LBObject& val); //exception if set failed
    void setProperty(const LBSymbol& s, const LBObjPtr& valptr); //exception if set failed

    LBObjPtr getPropertyValue(int index, LBPropertyInfo::ExecAttr attr);
    LBObject* getPropertyPtr(int index, LBPropertyInfo::ExecAttr attr);
    void touchProperty(int index, LBPropertyInfo::ExecAttr attr);
    void setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObject& val); //exception if set failed
    void setProperty(int index, LBPropertyInfo::ExecAttr attr, const LBObjPtr& valptr); //exception if set failed

    // property access help functions they have the locking mechanism and policy inside
  private:
    LBObjPtr readPropertyValue(int index, LBPropertyInfo::ExecAttr attr);
    LBObject* readPropertyUnsharedGuts(int index, LBPropertyInfo::ExecAttr attr);

  public:

    bool addPending(int staticprpindex); // this only applies to static property
    bool removePending(int staticprpindex); // this only applies to static property
    bool waitUntilNoPending(int staticprpindex); // this only applies to static property
    LBObjPtr getStaticPropertyValue(int index) ;
    void setStaticPropertyValue(int index, const LBObject& val) ;
    void setStaticPropertyValue(int index, const LBObjPtr& valptr);

    void waitUntilInputEmptyOrMainFinish(); // for asynch struct, caller hangs until the input q get to zero
    LBObject* luban_wait(const LBVarArgs *args); // to export above function

    LBWriteLockedObjPtr getWriteLockedStaticProperty(int staticprpindex); // this only applies to static property
    LBWriteLockedObjPtr getWriteLockedStoreProperty(int storeprpindex); // this only applies to store property of ASYNCH struct


    // helper interface for composition
    void setInput(int index, const LBObjPtr& valptr);
    void setInput(int index, const LBObject& val) ;
    void sync();
    bool linkOutputProperty(const LBSymbol& outprpname, int jointid); 
    bool prepareOutputForCompositionUse();
    bool prepareOutputForProcessUse();
    bool setRunTimeParentComposition(LBCompositionStruct *parentcomp);
    bool wakeupAsynch(); // start asynch struct internal thread if it has not been started
    
    // Process struct its own interfaces
    int numOfExternalSymbols() const;
    const LBFullSymbol* externalSymbol(int index) const;
    const LBObjPtr& evalForLastObj();
    void runAllCode(); // run all code for this struct
    bool runAllCodeAsynch(string& errs); // run all code for this struct asynch way

    void signalNewData(); //for asynch adhoc proc struct, to be called when new data arrives from upstream
//     void dismissNewDataWaiters(); //for asynch adhoc proc struct, to be called when parent composition are to be destroyed
//     bool isAdhocAsynch() const; // for asynch adhoc proc struct, to be called by parent composition to tell if it is adhoc and asynch

    friend class LubanCodeGenerator;
    friend class Dispatch;

  private:
    
    class LBProcessStructMetaData;
    LBProcessStructMetaData* metadata() { return _metadata.getRealPtr(); }

    // link time helper functions
    bool resolveExternalSymbolsIfNeeded(string& errs); // this is one toplevel function
    bool resolveExternalSymbols(string& errs); // this is one toplevel function

    bool resolveGlobalVar(int varindex, LBStruct* &st, int &prpindex, string &errs);
    bool reIndexAllGlobalVars(string& errs); // to be called after refered external symbol change
    bool reIndexAllPropertyOps(string& errs); // to be called at time the struct is first loaded, or its parent struct interface changes
    bool resolveAllNamedArgs(string& errs);  // to be called at time the struct is first loaded

    // before running
    void prepareRunTime();
    bool initializeProperties(); 
    void initLocalVarsTypeAndThreading(); // init local var value holder and associated locking mechanism

    // temp data types
    typedef LBObjPtr TempObj;
    typedef std::vector<TempObj> TempObjVec;
    class TempIterObjVec;

    // run time help functions
    bool runOneCode(int& ic, TempObjVec *tempobjs, TempIterObjVec *tempiterobjs, LBVMCode::Operand& lastop, string& errs); // run one line of code
    // below functions cover variables(global/local), local properties, temp variables, subscription and property ref
    bool readObjFromOperand(const LBVMCode::Operand& op, TempObjVec *tempobjs, LBObjPtr& objptr, string& errs); // get SNAP SHOT of the object
    bool writeObjToOperand(const LBVMCode::Operand& op,TempObjVec *tempobjs, const LBObject& obj, string& errs);
    bool writeObjToOperand(const LBVMCode::Operand& op,TempObjVec *tempobjs, const LBObjPtr& obj, string& errs);
    // LBWriteLockedObjPtr is an auto_ptr of a struct containing a write_lock_grabber and a LBObject *
    bool getWriteLockedObjFromOperand(const LBVMCode::Operand& op,TempObjVec *tempobjs, LBWriteLockedObjPtr& obj, string& errs);
    bool addPendingUpdate(const LBVMCode::Operand& vartoupdate, string& errs);
    bool removePendingUpdate(const LBVMCode::Operand& vartoupdate, string& errs);
    void throwError(const string& errmsg, bool throwit=true); // set all outputs to error value, cancel all runing threads
    bool writeValueToProperty(int pindex, LBPropertyInfo::ExecAttr attr, const LBObjPtr& objptr, string& errs);
    bool writeValueToProperty(int pindex, LBPropertyInfo::ExecAttr attr, const LBObject& obj, string& errs);

    // below used in destructor
    void cancelInternalThreads(); // try to cancel internal threads for asynch struct, to avoid using pthread_cancel which hangs

  private:

    class Dispatch : public LBRunable
    {
    public:
      enum Status { TORUN, RUNNING, FINISHED, RUNERR, TERMINATED } ;

      Dispatch(LBProcessStruct &ps, int startic, int endic, const LBVMCode::Operand& target)
	: _startic(startic), _endic(endic), _pstruct(ps), _target(target), _threadPoolIndex(-1), _status(TORUN), _errmsg(), _terminate(false)
      {
      }

      ~Dispatch() {}

      void run(); // runable required interface

      Status status() const { return _status; }
      void setPoolIndex(int poolindex) { _threadPoolIndex = poolindex; }
      void setTerminateFlag(bool t) { _terminate = t; }
      
    private:
      Dispatch();

      int _startic, _endic;
      LBProcessStruct &_pstruct;
      LBVMCode::Operand _target;
      int _threadPoolIndex;
      Status _status;
      string _errmsg;
      bool _terminate;
    };

    class LiveThread
    {
    public:
      LiveThread(LBProcessStruct &ps, int startic, int endic, const LBVMCode::Operand& targetvar )
	: _dispatch(ps, startic, endic, targetvar), _runningthread(0)
      {
      }
      ~LiveThread();

      bool start();
      bool join();
      bool isValid() const;
      void setPoolIndex(int poolindex);
      void setTerminateFlag(bool f);
    private:
      LiveThread();
      Dispatch _dispatch;
      LBThread *_runningthread;

    };

    class LiveThreadPool
    {
    public:
      LiveThreadPool();
      ~LiveThreadPool();

      int add(LiveThread *onethread);
      void setTerminationFlagAll(bool f);
      bool waitAll();
      bool joinAll();

      bool incRunning();
      bool decRunning();

      void reset();

    private:
      std::vector<LiveThread*> _allthreads;
      int _numrunning;

      LBMutex _addMutex;
      LBMutex _runningMutex;
      LBCondVar _zeroRunningCond;
    };

    class ExternalTypeSymbolTable
    {
    public:
      enum Expectation { ASTRUCT, ACONSTRUCTABLE };

      ExternalTypeSymbolTable();
      ~ExternalTypeSymbolTable();

      // below are called at compile time 
      // these functions only do local record keeping without actually loading the refered global symbol
      // External symbol resolving happens once for each defined struct type before evaluation
      // record the golbal symbol with expectation of a struct, return -1 if failed
      int getStructIndex(const LBFullSymbol& structname);
      bool getStructOp(const LBFullSymbol& structname, LBVMCode::Operand &resultop);
      // record the golbal symbol with expectation of a struct or a basic type
      bool getConstructableOp(const LBFullSymbol& symbolname, LBVMCode::Operand &resultop);
      // add symbol without expectation and operand

      int size() const;
      const LBFullSymbol* symbolName(int index) const;

      // dynamic linking call, called at link time
      bool resolveAllSymbols(string& errs);

      // called at run time 
      LBStruct* getInitializedStructCopy(int index) const; // caller owns the retured pointer
      LBStruct* getOriginalStructPointer(int index) const; // return the struct pointer owned by the name space
      const LBTypeSpec* getTypeSpec(int index) const;

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }
      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }

    private:
      ExternalTypeSymbolTable(const ExternalTypeSymbolTable& t); // not supposed to be used
      ExternalTypeSymbolTable& operator = (const ExternalTypeSymbolTable& t); // not supposed to be used

    private: // data
      class TypeSymbolInfo
      {
      private:
	friend class ExternalTypeSymbolTable;
	TypeSymbolInfo(const LBFullSymbol& s, Expectation e, int idx )
	  : _symbolname(s), _index(idx), _exp(e), _structobj(0),_structobjlocal(0), _typespecobj(0) {}
	~TypeSymbolInfo() { if ( _structobjlocal) delete _structobjlocal; }
	TypeSymbolInfo();

	LBFullSymbol _symbolname;
	Expectation _exp;
	int _index;

	// below are populated by dynamic linking
	LBStruct *_structobj; // namespace instance
	LBStruct *_structobjlocal; // local copy
	const LBTypeSpec *_typespecobj; // pointing to the instance in global namespace
      };

      LBHashMap<LBFullSymbol, TypeSymbolInfo*> _symbolmap;
      std::vector<TypeSymbolInfo*> _symbolvec;

    };

    class TypeExpInfo;
    class LocalVarOpInfoTable
    {
    public:
      LocalVarOpInfoTable();
      ~LocalVarOpInfoTable();

      bool addVar(const LBSymbol& vname, TypeExpInfo* tinfo, LBVMCode::Operand &resultop, bool threading );
      bool getVar(const LBSymbol& vname, LBVMCode::Operand &resultop, bool threading );
      int size() const;
      const LBSymbol *varName(int index) const;
      const LBTypeSpec *typeSpec(int index) const;
      bool threading(int idx) const;

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      LocalVarOpInfoTable(const LocalVarOpInfoTable& t);
      LocalVarOpInfoTable& operator = (const LocalVarOpInfoTable& t);

    private:
      class LocalVarInfo
      {
      public:
	LocalVarInfo(const LBSymbol& vname, TypeExpInfo* tinfo, bool threading, int index )
	  : _name(vname), _typeexpinfo(tinfo), _threading(threading), _index(index) {}

	LocalVarInfo(const LocalVarInfo& x)
	  : _name(x._name), _typeexpinfo(0), _threading(x._threading), _index(x._index)
	{
	  if ( x._typeexpinfo )
	    _typeexpinfo = new TypeExpInfo(*x._typeexpinfo);
	}

	~LocalVarInfo() { if ( _typeexpinfo ) delete _typeexpinfo; }
	LBSymbol _name;
	TypeExpInfo *_typeexpinfo;
	bool _threading;
	int _index;

      private:
	LocalVarInfo();
	LocalVarInfo& operator=(const LocalVarInfo& x);
      };

      std::vector<LocalVarInfo*> _varvec;
      LBHashMap<LBSymbol, LocalVarInfo*> _varmap;

    };


    class GlobalVarOpInfoTable
    {
    public:
      bool getGlobalVar(int extsymbolindex, const LBSymbol& prpname, LBVMCode::Operand &resultop);

      int getGlobalStructIndex(int index);
      const LBSymbol* getPropertyName(int index);
      int getPropertyIndex(int index); // return -1 if index not available
      bool setPropertyIndex(int index, int prpindex);

      int size() { return _globalvaropvec.size(); }

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      class GlobalVarOpInfo
      {
      public:
	GlobalVarOpInfo()
	  : _structindex(-1), _prpname(), _prpindex(-1)
	{
	}
	GlobalVarOpInfo(int sidx, const LBSymbol& pname)
	  : _structindex(sidx), _prpname(pname), _prpindex(-1)
	{
	}
	int _structindex;
	LBSymbol _prpname;
	int _prpindex; // to be set at run time
      };

      std::vector<GlobalVarOpInfo> _globalvaropvec;

    };

    class LocalPropertyOpInfoTable
    {
    public:
      LocalPropertyOpInfoTable();

      bool getPropertyOp(const LBSymbol& prpname, LBPropertyInfo::ExecAttr attr, int prpindex, LBVMCode::Operand &resultop, bool threading, string& errs);
      
      LBPropertyInfo::ExecAttr getAttr(int opindex) const;
      int getIndex(int opindex) const;
      const LBSymbol* getName(int opindex) const;
      bool getThreading(const LBSymbol& pname) const;

      int size() const;

      bool setIndex(int opindex, int propertyIndex);

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      LocalPropertyOpInfoTable(const LocalPropertyOpInfoTable& t);
      LocalPropertyOpInfoTable& operator = (const LocalPropertyOpInfoTable& t);

    private:
      class PropertyOpInfo
      {
      public:
	PropertyOpInfo(const LBSymbol& pnm, LBPropertyInfo::ExecAttr attr, bool threading)
	  : _propertyname(pnm), _propertyattr(attr), _propertyindex(-1), _threading(threading)
	{
	}
	LBSymbol _propertyname;
	LBPropertyInfo::ExecAttr _propertyattr;
	int _propertyindex;
	bool _threading;
      };

      LBHashMap<LBSymbol, int> _propertymap;
      std::vector<PropertyOpInfo> _propertyvec;
    };

    class VarPropertyOpInfoTable
    {
    public:

      LBVMCode::Operand getPropertyOp(const LBVMCode::Operand& varop, const LBSymbol& prpname);

      const LBSymbol *propertyName(int opindex) const ; 
      const LBVMCode::Operand *structOp(int opindex) const;

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      struct VarPropertyInfo
      {
	VarPropertyInfo(const LBVMCode::Operand& varop, const LBSymbol& prpname)
	  : _op(varop), _prpname(prpname)
	{}
	LBVMCode::Operand _op;
	LBSymbol _prpname;
      };

      std::vector<VarPropertyInfo> _prpinfovec;

    };      

    class MemberFuncOpInfoTable
    {
    public:
      // insert
      LBVMCode::Operand getMemberFuncOp(const LBVMCode::Operand& varop, const LBSymbol& funcname);

      // query
      const LBVMCode::Operand *objOperand(int index) const;
      const LBSymbol *funcName(int index) const;

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      struct MemberFuncInfo
      {
	MemberFuncInfo(const LBVMCode::Operand& varop, const LBSymbol& funcname)
	  : _op(varop), _funcname(funcname)
	{}
	LBVMCode::Operand _op;
	LBSymbol _funcname;
      };

      std::vector<MemberFuncInfo> _funcinfovec;

    };      

    class SubscriptOpInfoTable
    {
    public:
      LBVMCode::Operand getSubscriptOp(const LBVMCode::Operand& varop, const LBVMCode::Operand& subexpop);
      
      const LBVMCode::Operand *objOp(int index);
      const LBVMCode::Operand *subOp(int index);

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }

    private:
      struct SubOpInfo
      {
	SubOpInfo(const LBVMCode::Operand& varop,  const LBVMCode::Operand& subexpop)
	  : _varop(varop), _subexpop(subexpop)
	{}
	LBVMCode::Operand _varop;
	LBVMCode::Operand _subexpop;
      };

      std::vector<SubOpInfo> _subopinfovec;

    };

    class TypeExpInfo
    {
    public:
      TypeExpInfo()
	: _typespec(), _typetree(0), _homens()
      {}

      TypeExpInfo(LBTypeSpec *tspec, Tree typetree, const LBFullSymbol& homens)
	: _typespec(static_cast<LBObject*>(tspec)), _typetree(typetree), _homens(homens)
      {}

      TypeExpInfo(LBTypeSpec *tspec)
	: _typespec(static_cast<LBObject*>(tspec)), _typetree(0), _homens()
      {}

      TypeExpInfo(const TypeExpInfo& t)
	: _typespec(t._typespec), _typetree(0), _homens(t._homens)
      {
	if ( t._typetree )
	  _typetree = t._typetree->clone();
      }

      ~TypeExpInfo()
      {
	if ( _typetree ) delete _typetree;
      }

      const LBTypeSpec* typeSpec();
      const LBObjPtr& getTypeSpecAsObj() const
      { return _typespec; }

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }

    private:

      LBObjPtr _typespec;
      Tree _typetree;
      LBFullSymbol _homens;
    };

    class LocalVarObj
    {
    public:
      LocalVarObj() : _obj(), _locks(0), _typespec(0) {}
      ~LocalVarObj();
      void reset(); // clear state
      void initThreading(); // add mutex and rw lock for multi-threading use
      void setType(const LBTypeSpec* tp); // add type spec
      
      bool addPending();    // increase pending thread update
      bool removePending(); // decrease pending thread update
      bool waitUntilNoPending();

      LBObjPtr readValue() const;
      bool writeValue(const LBObjPtr& objptr);
      bool writeValue(const LBObject& obj);

      LBWriteLockedObjPtr getWriteLockedObj();
    private:
      class LockStruct;
      LBObjPtr _obj;
      LockStruct *_locks;
      const LBTypeSpec *_typespec;

    };

    class TempIterObj
    {
    public:
      TempIterObj(LBConstIterator *it, const LBObjPtr& obj)
	: _it(it), _obj(obj)
      {}

      TempIterObj()
	:_it(0), _obj()
      {}

      ~TempIterObj() { if ( _it ) delete _it; }

      LBConstIterator* getIterator() { return _it; }
      void reset()
      {
	delete _it;
	_it = 0;
	_obj = LBObjPtr();
      }
      void reset(LBConstIterator *it, const LBObjPtr& obj)
      {
	if ( _it ) delete _it;
	_it = it;
	_obj = obj;
      }
    private:
      TempIterObj(const TempIterObj& i);
      TempIterObj& operator = (const TempIterObj& i);

    private:
      LBConstIterator *_it;
      LBObjPtr _obj;
    };

    class TempIterObjVec
    {
    public:
      TempIterObjVec( int sz )
	: _iterobjvec(sz)
      {}
      ~TempIterObjVec()
      {
	for( int i = 0; i < _iterobjvec.size(); i++)
	  if ( _iterobjvec[i] )
	    delete _iterobjvec[i];
      }

      TempIterObj& operator[](int idx)
      {
	if ( _iterobjvec[idx] )
	  return *_iterobjvec[idx];
	TempIterObj *iter = new TempIterObj();
	_iterobjvec[idx] = iter;
	return *iter;
      }

    private:
      TempIterObjVec( );
      std::vector<TempIterObj*> _iterobjvec;
    };

    class LBStructEvalArgs : public LBVectorizingInfo
    {
    public:
      LBStructEvalArgs();
      ~LBStructEvalArgs();

      bool addNamedArg(const LBSymbol& name, const LBVMCode::Operand& op, bool vecit, bool parallel);
      void addStructINs(const LBVMCode::Operand& structop, bool vecthem, bool parallel);
      void addThisINs(bool vecthem, bool parallel);

      int numArgs() const // only returns correct number after the struct and this INs are resolved
      { 
	if ( ! _namedargs ) return 0;
	return _namedargs->size();
      }

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


      // LBVectorizing interface
      int numVecNames() const;
      const LBSymbol* vecName(int index) const ;
      bool parallel() const;

      

      friend class LBProcessStruct;

    private:
      LBStructEvalArgs(const LBStructEvalArgs& a);
      LBStructEvalArgs& operator = (const LBStructEvalArgs& a);

      void addNamedArgFromStructType(const LBSymbol& nm, int structidx, int inprpidx, bool vec);
      void addNamedArgFromThisINs(const LBSymbol& nm, int inprpidx, bool vec);

      std::vector<int> *getVecedArgs() 
      { 
	if ( ! _vecedargs )
	  _vecedargs = new std::vector<int>();
	return _vecedargs;
      };

      class OneNamedArg
      {
      public:
	enum Source { DIRECT, STYPE,  STHIS };
	OneNamedArg(const LBSymbol& name, const LBVMCode::Operand& op, bool vec)
	  : _name(name), _src(DIRECT), _directop(op), _stypeindex(-1), _thisindex(-1),_vec(vec)
	{}

	OneNamedArg(const LBSymbol& nm, int structidx, int inprpidx, bool vec)
	  : _name(nm), _src(STYPE), _directop(), _stypeindex(structidx), _thisindex(inprpidx), _vec(vec)
	{}

	OneNamedArg(const LBSymbol& nm, int inprpidx, bool vec)
	  : _name(nm), _src(STHIS), _directop(), _stypeindex(-1), _thisindex(inprpidx), _vec(vec)
	{}

	LBSymbol _name;
	Source _src;
	LBVMCode::Operand _directop;
	int _stypeindex;
	int _thisindex;
	bool _vec;
      };

      class BatchNamedArg
      {
      public:
	BatchNamedArg(const LBVMCode::Operand& sop, bool vecit)
	  : _structtypeop(sop), _vec(vecit)
	{}

	LBVMCode::Operand _structtypeop;
	bool _vec;
      };

      typedef std::vector<OneNamedArg> NamedArgVec;
      NamedArgVec *_namedargs;
      typedef LBHashMap<LBSymbol, OneNamedArg*> NamedArgMap;
      NamedArgMap *_namedargmap;
      std::vector<int> *_vecedargs;

      typedef std::vector<BatchNamedArg> StructTypeOpVec;
      StructTypeOpVec *_structtypes;

      bool _getthisins, _vecthisins;

      bool _par;
    };

    class LBAnonymousArgs
    {
    public:
      LBAnonymousArgs()
	: _args()
      {}
      void pushArg(const LBVMCode::Operand& op ) { _args.push_back(op); }
      const LBVMCode::Operand* getArgOp(int index) const
      {
	if ( index < 0 || index >= size() )
	  return 0;
	return &_args[index];
      }
      int size() const { return _args.size(); }

      ostream& toStream(ostream& ost) const
      {
	// to be done
	throw LBException("Not implemented yet");
      }

      istream& fromStream(istream& ist)
      {
	// to be done
	throw LBException("Not implemented yet");
      }


    private:
      std::vector<LBVMCode::Operand> _args;
    };


    class AsynchMainThread
    {
    public:
      AsynchMainThread(LBProcessStruct *s): _dsp(s), _runningthread(0), _initmtx() {}
      ~AsynchMainThread();

      bool start();
      bool join();
      bool isdead() const; // check if the thread is dead
      const string& errmsg() const;

      class Dispatch : public LBRunable
      {
      public:
	friend class AsynchMainThread;
	enum Status { TOBERUN, RUNNING, UNRUNABLE, ENDED};
	Dispatch( LBProcessStruct *pst ): _pst(pst), _st(TOBERUN), _errmsg() {}

	void run();

      private:
	LBProcessStruct *_pst;
	Status _st;
	string _errmsg;
      };

    private:
      Dispatch _dsp;
      LBThread *_runningthread;
      LBMutex _initmtx;

    };

    class ParentCompDataJointOps
    {
    public:
      int newJTReadOp(int jid, int pid)
      {
	int idx = _jids.size();
	_jids.push_back(jid);
	_pids.push_back(pid);
	return idx;
      }
      bool getJointInfo(int opindex, int& jid, int&pid)
      {
	if ( opindex < 0 || opindex >= _jids.size() )
	  return false;
	jid = _jids[opindex];
	pid = _pids[opindex];
	return true;
      }
    private:
      std::vector<int> _jids;
      std::vector<int> _pids;
    };

    class LBProcessStructMetaData
    {
    public:
      LBProcessStructMetaData(LBStructInterface* itfc);
      LBProcessStructMetaData();
      ~LBProcessStructMetaData();

      // constructors for composition internal proc struct
      LBProcessStructMetaData(const LBSymbol& cellname, bool asynch, LBCompositionStruct *compmom);
      LBProcessStructMetaData(int cellid, LBCompositionStruct *compmom);

      // functions to populate the procedural code, used by LubanCodeGenerator
      LBVMCodeSequence& code();
      // function to parse and verify the type expression, extract external type symbols 
      // and allocate a operand for this variable, return false if failed
      bool addLocalVariable(const LBSymbol& name, Tree typeexp, const LBFullSymbol& homens, LBVMCode::Operand& resultvarop, string& errs ); // I hate exception
      // below function can return operand of  local variable
      // create the variable if not already exist, return false if failed
      bool getLocalVariableOrCell(const LBSymbol& name, LBVMCode::Operand& resultvarop, bool forwriting, string& errs);
      bool getLocalVariable(const  LBSymbol& name, LBVMCode::Operand& resultvarop, string& errs ); // local variable only
      // this function find/create an operand that pointing to the external global static property
      // this operand will be dynamically linked ( again) at each run time, 
      bool getGlobalVariable(const LBFullSymbol fullstructname, const LBSymbol& propertyname, LBVMCode::Operand& resultvarop);
      // below function will resovle the struct name, and create an operand for it
      // the struct name will be saved and it will be re-resolved at each run time, return false if failed
      bool getGlobalStructType(const LBFullSymbol& fullstructname,  LBVMCode::Operand &resultop, string& errs);
      // below function expect the name can be resolved to a struct or a basic type
      bool getGlobalConstructableType(const LBFullSymbol& fulltypename,  LBVMCode::Operand &resultop);

      // below function get operand for property, return false if the property does not exist
      bool getLocalPropertyOperand(const LBSymbol& pname, LBPropertyInfo::ExecAttr exat, LBVMCode::Operand &pop, string& errs);
      LBVMCode::Operand getVarPropertyOperand(const LBVMCode::Operand& varop, const LBSymbol& propertyname);
      LBVMCode::Operand getSubscriptOperand(const LBVMCode::Operand& varop,const LBVMCode::Operand& subexpop);
      LBVMCode::Operand getMemberFuncOperand(const LBVMCode::Operand& varop,  const LBSymbol& funcname);

      LBVMCode::Operand newTemp(); // this function generate operand for temp object
      LBVMCode::Operand newTempIter(); // this function generate operand for temp iterators 
      LBVMCode::Operand newLiteral(LBObject* obj); // generate literal objects 
      // below function will instantiate the tree, while save the parse tree, so it can be verified and intantiated
      // at each run time. ( Type expression may refer external type symbol )
      bool newLiteralTypeSpec(Tree typetree, const LBFullSymbol& homens,  LBVMCode::Operand &resultop, string& errs); // type spec objects
      LBVMCode::Operand newNamedArgsOperand(LBProcessStruct::LBStructEvalArgs* args ); // this function generates op for named arg
      LBVMCode::Operand newAnonymousArgsOperand(LBProcessStruct::LBAnonymousArgs* args ); // this function generates op for anonymous arg
      void setStaticLastOp(const LBVMCode::Operand& staticlitop); // this is for a struct with only constant expressiopn

      bool findRootOperand(const LBVMCode::Operand& subop, LBVMCode::Operand& rootop);
      void startDispatch() { _multithread = true; _compilingDispatch = true; }
      void endDispatch() { _compilingDispatch = false; }


      void openLoop();        // push one more open loop into stack
      bool closeLoop(int pointtoontinue, int pointtobreak); // remove an open loop and patch all register cont and breaks
      bool addContinue(int contstindex);  // add the index of a cont statement to currne open loop  
      bool addBreak(int breakstindex);    // add the index of a break statement to currne open loop 

      ostream& toStream(ostream& ost) const; // to stream out all guts
      istream& fromStream(istream& ist); // to stream out all guts


      // composition internal process struct related functions
      bool getVarPropertyOperand(const LBSymbol& varname, const LBSymbol& prpname, LBVMCode::Operand& resultop, string& errs);
      LBVMCode::Operand getAdhocOutputOperand(); // the only output operand for adhoc cell
      // the operand from below functions pointing to parent composition data joint
      bool getAdhocInputOperand(int upstreamcello, LBVMCode::Operand& resultop, string& errs); // link from an adhoc upstream
      bool getAdhocInputOperand(int upstreamcello, const LBSymbol& prpname, LBVMCode::Operand& resultop, string& errs); // link from a typed upstream
      bool getAdhocInputOperand(const LBSymbol& inputprpname, LBVMCode::Operand& resultop, string& errs); // link from input property 


      bool isAdhoc(); // tell if this struct is an adhoc cell in a composition

    private:
      LBProcessStructMetaData(const LBProcessStructMetaData& m);
      LBProcessStructMetaData& operator=(const LBProcessStructMetaData& m);

    private:
      friend class LBProcessStruct;
      friend class Dispatch;

      LBStructInterface *_itfc;

      LBVMCodeSequence _code;
      LocalVarOpInfoTable _localvops; // local variable info, compile time determined info
      GlobalVarOpInfoTable _globalvops; // refered global variables, need link time reset
      LocalPropertyOpInfoTable _localpropertyops; // local properties, need link time reset it may depend external interface inhereitance

      // property reference for varibale or temp obj instance
      VarPropertyOpInfoTable _varpropertyops;
      // subscript for variable or temp obj instance
      SubscriptOpInfoTable _subscriptops;
      // op for member func call instances
      MemberFuncOpInfoTable _memberfuncops;

      // literals
      typedef std::vector<LBObjPtr> ObjVec;
      ObjVec _literalobjs; // configured at compile/link time, no change at run time
      typedef std::vector<TypeExpInfo*> TypeLiteralVec;
      TypeLiteralVec _literaltypeexps; // need link time re-processing because of it may depend on external types
      
      // Temps and local variable values
      int _numtemps;   //  number of temporaribly variables
      int _numtempiters;   // number of temporary iterator variable

      typedef std::vector<LBStructEvalArgs*> NamedArgObjVec;
      NamedArgObjVec _namedargs;  // contents determined at compile time,need link time reset it may depend externa interface inhereitance

      typedef std::vector<LBAnonymousArgs*> AnonArgObjVec;
      AnonArgObjVec _anonargs;  // contents determined at compile time, should be persisted

      bool _multithread;

      ExternalTypeSymbolTable _extsymbols; // central of link time processing

      bool _extsresolved; // indicator of external symbol resolved or not

      bool _nsRecorded;  // is this struct a defined struct type in namespace
      LocalPropertyStorage *_statics; // pointing to the same stroage as in the _properties member
      CounterWaiter *_staticscounters;

      LBObjPtr _staticlastobj;

      // Following are the temp working area for code generator to work on loops, breaks and continues
      typedef std::vector<int> VInt;
      typedef std::vector<VInt> VVInt;
      VVInt *_loopbreaks;
      VVInt *_loopcontinues;
      bool _compilingDispatch;

      // following are the container composition struct information for internal process struct 
      int _cellid;
      LBSymbol _cellname;
      LBCompositionStruct *_compmom;
      ParentCompDataJointOps _jtreadops;
      int _jtwriteindex;

      LBMutex _initmtx;
    };

    class LBProcessStructRunTime
    {
    private:
      friend class LBProcessStruct;
      friend class Dispatch;
      friend class AsynchMainThread;

      LBProcessStructRunTime(int numvars, bool multithread);
      ~LBProcessStructRunTime();

      void reset(); // reset for next evaluation

      typedef std::vector<LocalVarObj> LocalVarObjVec;
      LocalVarObjVec *_localvars;

      LiveThreadPool *_livethreads;  // threads dispatched from the one run of the code

      enum STATUS { EVALERROR, EVALRUNNING, EVALOK, UNEVALED };
      STATUS _status;
      string _errmsg;
      
      LBObjPtr _lastobj; // the last result obj of the code excuting

      LBCompositionStruct *_parentcomp;
      LBObjPtr _adhocoutput; // for adhoc synch struct output cacheing
      bool _adhocoutputUpdated;

      CounterWaiter *_adhocDataCounter; // for adhoc asynch struct to count input

    };

    class LBProcessStructProperties
    {
    public:
      LBProcessStructProperties()
	: _ins(0), _outs(0), _outlocks(0), _stores(0)
      {}
      LBProcessStructProperties(const LBProcessStructProperties& p);
      ~LBProcessStructProperties();

//       LBObjPtr getPropertyValue(int index, LBPropertyInfo::ExecAttr attr);
//       LBObject* getPropertyUniquePtr(int index, LBPropertyInfo::ExecAttr attr); 

      LBStructInPad *_ins; // in properties set/get interface
      LBStructOutPad *_outs; // out properties set/get interface
      typedef std::vector<LBMutex*> LockVec;
      LockVec *_outlocks; // out property internal multi-thread write locks
      LocalPropertyStorage *_stores; // instance specific local store type property storage
    };      

    typedef RefCountedPtr<LBProcessStructMetaData> MetaDataPtr;
    MetaDataPtr _metadata;
    LBMutex *_synchmtx; // struct-wide lock for synch struct
    LBProcessStructRunTime *_runtime;
    LBProcessStructProperties *_properties;
    AsynchMainThread *_asynchmainthread; // the main thread for asynch struct code run
    bool _mainstop; // for destructor use to stop the asynch main thread, avoid using thread_cancel which is messy
  };

}

#endif
