#include <fstream>
#include <sstream>
#include <string>

#include "lbtypes/lbobject.hpp"
#include "lbtypes/lbchar.hpp"
#include "lbtypes/lbstring.hpp"
#include "lbtypes/lbint.hpp"
#include "lbtypes/lbtypefactory.hpp"
#include "lbtypes/lbtypeinfo.hpp"
#include "lbtypes/LBDefineMacros.hpp"
#include "lbtypes/lbexception.hpp"
#include "lbtypes/lbvarargs.hpp"
#include "lbtypes/RefCountedPtr.hpp"

#include "lbstd/file.hpp"


namespace Std
{
  using std::string;
  using std::ifstream;
  using std::ofstream;
  using std::ostringstream;

  using Luban::LBObject;
  using Luban::LBChar;
  using Luban::LBInt;
  using Luban::LBString;
  using Luban::LBVarArgs;
  using Luban::LBSymbol;
  using Luban::LBException;

  LBDEFINE(Std::File, 1, 0 )

  File* File::staticConstructor(const LBVarArgs* args)
  {
    if ( args == 0 || args->numArgs() != 2 )
      throw LBException("to open a file std::file(string filename, char r/w)");

    const LBString *fname = dynamic_cast<const LBString*>(args->getArg(0));
    const LBChar *rw = dynamic_cast<const LBChar*>(args->getArg(1));
    if ( ! fname || ! rw )
      throw LBException("to open a file std::file(string filename, char r/w)");
    
    AccessType acs = READ;
    char a = char(*rw);
    if ( a  == 'w' )
      acs = WRITE;
    else
      if ( a == 'a' )
	acs = APPEND;
      else
	if ( a != 'r' )
	  throw LBException("to open a file std::file(string filename, char r/w/a)");

    File *result = new File(fname->str(), acs);
    if ( result->isValid() )
      return result;
    
    throw LBException("Failed to open file");

  }

  LBDEFAULT_EQUALS_FUNC(Std::File)

  File::File(const string& fname, AccessType acs)
    :_filename(fname), _rfile( 0 ), _wfile(0)
  {
    switch ( acs )
      {
      case READ:
	_rfile.setRealPtr(new ifstream(fname.c_str()));
	break;
      case WRITE:
	_wfile.setRealPtr(new ofstream(fname.c_str()));
	break;
      case APPEND:
	_wfile.setRealPtr(new ofstream(fname.c_str(), std::ios_base::app));
	break;
      default:
	;
      }
  }

  string File::toString() const
  {
    static const string s("std::file ");
    return s+_filename;
  }

  ostream& File::toStream(ostream& ost) const
  {
    throw LBException("std::file object can not be streamed out");
    return ost;
  }

  istream& File::fromStream(istream& ist, int major, int minor)
  {
    throw LBException("std::file object can not be streamed in");
    return ist;
  }

  bool File::put(const LBObject& obj)
  {
    if ( _wfile )
      {
	(*_wfile)<< obj.toString();
	return bool(*_wfile);
      }
    return false;
  }

  bool File::putLine(const LBObject& obj)
  {
    if ( _wfile )
      {
	(*_wfile)<<obj.toString()<<"\n";
	return bool(*_wfile);
      }
    return false;
  }

  bool File::readChar(char& c)
  {
    if ( _rfile )
      {
	c=_rfile->get();
	return bool(*_rfile);
      }
    return false;
  }
   
  bool File::readLine(string& aline)
  {
    if ( _rfile )
      return getline(*_rfile, aline);
    return false;
  }

  bool File::readAll(string& all)
  {
    if ( _rfile && *_rfile )
      {
	char c;
	while( *_rfile && _rfile->get(c) )
	  all += c;
	return true;
      }
    return false;
  }

  bool File::readable() const
  {
    return _rfile && *_rfile;
  }

  bool File::writeable() const
  {
    return _wfile && *_wfile;
  }

  bool File::putObj(const LBObject& obj)
  {
    if ( writeable() )
      {
	ostringstream ost;
	LBObject::instanceToStream(ost, obj);
	string objstr = ost.str();
	*_wfile << objstr;
	return true;
      }
    throw LBException("File is not writeable");
    return false;
  }

  LBObject* File::getObj()
  {
    if ( readable() )
      {
	string errs;
	LBObject *obj = LBObject::instanceFromStream(*_rfile,errs );
	if ( obj )
	  return obj;
	throw LBException("Failed to restore object from file: "+errs);
	return 0;
      }
    throw LBException("File is not readable");
    return 0;
  }
    
  LBEXPORT_MEMBER_FUNC(Std::File, luban_put, "write", "void write(obj1, obj2, obj3....)" ); 
  LBObject* File::luban_put(const LBVarArgs *args)
  {
    if ( ! _wfile )
      throw LBException("File is opened for read");
    if ( ! args || args->numArgs() == 0 )
      return 0;
    for( int i=0; i<args->numArgs(); i++)
      {
	const LBObject *obj = args->getArg(i);
	if ( obj ) 
	  put( *obj );
	else
	   *_wfile << "null";
      }
    return 0;
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_putobj, "writeobj", "void writeobj(obj1, obj2, obj3....)" ); 
  LBObject* File::luban_putobj(const LBVarArgs *args)
  {
    if ( ! args || args->numArgs() == 0 )
      return 0;
    for( int i=0; i<args->numArgs(); i++)
      {
	const LBObject *obj = args->getArg(i);
	if ( obj ) 
	  putObj( *obj );
	else
	  throw LBException("Can not persist null value guts");
      }
    return 0;
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_getobj, "readobj", "object readobj()" ); 
  LBObject* File::luban_getobj(const LBVarArgs *args)
  {
    if ( args && args->numArgs() != 0 )
      throw LBException("std::file::readobj() does not take arguements");
    return getObj( );
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_putline, "writeline", "void writeline(obj1, obj2, obj3....)" ); 
  LBObject* File::luban_putline(const LBVarArgs *args)
  {
    if ( ! _wfile )
      throw LBException("File is opened for read");
    if ( ! args || args->numArgs() == 0 )
      {
	*_wfile<<'\n';
	return 0;
      }
    for( int i=0; i<args->numArgs(); i++)
      {
	const LBObject *obj = args->getArg(i);
	if ( obj ) 
	  putLine(*obj);
	else
	  *_wfile << "null\n";
      }
    
    return 0;
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_get, "read", "string read(int n=1)" ); 
  LBObject* File::luban_get(const LBVarArgs *args)
  {
    if ( ! readable() )
      throw LBException("File is not readable");
    int toread = 1;
    if ( args && args->numArgs() ) 
      {
	if ( args->numArgs() != 1 )
	  throw LBException("file::read function take one integer argument");

	const LBInt *n = dynamic_cast<const LBInt*>(args->getArg(0));
	if ( ! n )
	  throw LBException("file::read function take one integer argument");	      
	toread = int(*n);
      }

    char c;
    string s;
    while ( --toread >= 0 && readChar(c) )
      s += c;
    return new LBString(s);
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_getline, "readline", "string readline()" ); 
  LBObject* File::luban_getline(const LBVarArgs *args)
  {
    if ( ! readable() )
      throw LBException("File is not readable");
    if ( args && args->numArgs() )
      {
	throw LBException("file::readline() does not take arguments");
	return 0;
      }
    string ln;
    if ( readLine(ln) )
      return new LBString(ln);
    throw LBException("Failed to readline from file");
    return 0;
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_getall, "readall", "string readall()" ); 
  LBObject* File::luban_getall(const LBVarArgs *args)
  {
    if ( ! _rfile )
      throw LBException("File is opened for write only");

    if ( args && args->numArgs() )
      {
	throw LBException("file::readall() does not take arguments");
	return 0;
      }
    string ln;
    if ( readAll(ln) )
      return new LBString(ln);
    return 0;
  }

  LBEXPORT_MEMBER_FUNC(Std::File, luban_close, "close", "void close()" ); 
  LBObject* File::luban_close(const LBVarArgs *args)
  {
    if ( args && args->numArgs() )
      {
	throw LBException("file::close() does not take arguments");
	return 0;
      }
    if ( _rfile )
      _rfile->close();
    else
      if ( _wfile )
	_wfile->close();
    return 0;
  }



}
